;*******************************************************************************
; File: rave.pro
;
; Purpose: This file contains the rave procedure which creates the 3D Object Graphics tree
;          for RAVE (RAPID Visualization Environment).  Rave is called upon initialization
;          and for every change during run-time.   This file also contains all helper
;          routines which need intimate knowledge of the 3D graphics tree.
;
; Routines:  rave
;            raveChangeView
;            raveChangeBackgroundColor
;            raveMenuBarHandler
;
; Author:  LASP/Steve Monk
;
; History: 10/01/03 SM - Release 1.0.
;
;*******************************************************************************


;*******************************************************************************
; Routine: rave
; Purpose: Create or change the 3D Object Graphics tree.
;          Called upon initialization, and for every change during run-time.
;*******************************************************************************
pro rave, state, initialize=initialize

common rave_common, pos, window, view, model, trackball, old_state, $
                    xs, ys, zs, xrange, yrange, zrange

if not keyword_set( initialize) then initialize = 0
if n_elements(pos) eq 0 then initialize = 1

if initialize then begin

  ;Create the object graphics tree.  Each node is listed by its uname.
  ;window---palette
  ;   |
  ; scene
  ;   |
  ; view---model---Plot3DBox---xAxis
  ;   |        |           |---yAxis
  ;   |        |           |---zAxis
  ;   |        |           |---x,y,z axis titles, fonts
  ;   |        |           |---bounding box lines and rule lines
  ;   |        |    
  ;   |        |---EarthTransform---Earth
  ;   |        |    
  ;   |        |---Curvature---(lots added by curvature.pro)
  ;   |        |    
  ;   |        |---c1---OrbitTrack
  ;   |        |    |
  ;   |        |    |---OrbitProjection---projections onto XY,XZ,YZ planes
  ;   |        |    |                 |---S/C markers in XY,YZ,YZ planes
  ;   |        |    |
  ;   |        |    |---Transform---GlobeTransform---c1_Globe (uvalue = fltarr(16,9) = 
  ;   |        |                |                              alog10 of averaged electron data)
  ;   |        |                |---B (uvalue = fltarr(3) = averaged B field data)
  ;   |        |                |   |---arrowShaft   (uname = "c1_B" to facilitate picking)
  ;   |        |                |   |---arrowHead    (uname = "c1_B" to facilitate picking)
  ;   |        |                |
  ;   |        |                |---PitchTransform---PitchContour1
  ;   |        |                                 |---PitchContour2
  ;   |        |                                 |---etc
  ;   |        |    
  ;   |        |---c2 (same as above)
  ;   |        |---c3 (same as above)
  ;   |        |---c4 (same as above)
  ;   |       
  ;   |---annotations---eventTimeStr "04:00:01"
  ;                 |---"c1"
  ;                 |---"c2"
  ;                 |---"c3"
  ;                 |---"c4"
  ;                 |---ColorBar
  ;                 |---ColorBarAxis
  
  getOrbitData, state.minTimeStr, state.maxTimeStr, T, pos
  ;Note: pos = fltarr(N,3,4)  ;pos[ N_pts, x|y|z, Cluster1|Cluster2|Cluster3|Cluster4]
  
  reverse_x = 1  ;1=x-axis increases R to L, i.e. sun is to the L
  reverse_y = 1  ;1=y-axis increases top to bottom
  xrange = [min( pos[*,0,*]), max( pos[*,0,*])]
  if reverse_x then xrange = reverse( xrange)
  yrange = [min( pos[*,1,*]), max( pos[*,1,*])]
  if reverse_y then yrange = reverse( yrange)
  zrange = [min( pos[*,2,*]), max( pos[*,2,*])]
  if keyword_set(km) then units = ' (km)' else units = ' (Re)'
  xtitles = ['X!IGSE!N'+units, 'X!IGSE!N'+units, 'Y!IGSE!N'+units, 'X!IGSE!N'+units]
  ytitles = ['Y!IGSE!N'+units, 'Z!IGSE!N'+units, 'Z!IGSE!N'+units, 'R=(Y!E2!N+Z!E2!N)!E1/2!N!IGSE!N'+units]
  
  ;Bug Note: zclip=[1.0,-1.0] causes the near and far (in z) corners of the cube to disappear when
  ;zooming in.  view->SetProperty, zclip=[3.0,-3.0] fixes it, but causes the axis titles to be much
  ;larger, regardless of where/when the call is made.  Since this is worse than the cube corners
  ;disappearing, I left it as is.
  
  view->SetProperty, viewplane_rect=[-1.02,-1.02,2.02,2.02], color=state.background_color, zclip=[1.5,-1.5]
  palette = obj_new( 'IDLgrPalette')
  palette->loadct, 39
  window->setProperty, palette=palette
  
  ;Reset the model's transform, because if reload when zoomed out, bounding box axes have fewer tick marks
  ;and too big of labels and titles.
  identityMatrix = [[1.0, 0.0, 0.0, 0.0], $
                    [0.0, 1.0, 0.0, 0.0], $
                    [0.0, 0.0, 1.0, 0.0], $
                    [0.0, 0.0, 0.0, 1.0]]
  model->SetProperty, transform=identityMatrix  
  ax=30 & ay=0 & az=30 & scale_factor=1.2
  model->rotate, [1,0,0], -90
  model->rotate, [0,1,0], az
  model->rotate, [1,0,0], ax
  model->scale, scale_factor, scale_factor, scale_factor
  
  first_time = (model->count() eq 0)
  if first_time then begin
    annotations_model = obj_new( 'IDLgrModel', name='annotations')
    view->add, annotations_model
    ;Note: I removed the ONGLASS keyword, because it prevents the text from scaling when you resize the window.
    ;However, the text is worse quality.  Setting the z-value to 1 in location seems to help.
    annotations_model->add, obj_new( 'IDLgrText', 'hh:mm:ss', location=[0.0,0.95,1], align=0.5, $
                                     name='eventTimeStr', color=state.foreground_color, $
                                     font=obj_new( 'IDLgrFont', size=16.0, thick=3))
    ;Add color bar and axis to R edge of 3-D plot window.
    drawColorBar, state
    createPitchAnglePlot, state, palette
  endif
  
  if not first_time then begin
    ;Remove existing objects that need re-creating because they use xs, xs, zs output by rave_axes call,
    ;or because their displayed positions would be wrong when bounding box changes.
    object_names = ['Plot3DBox', 'EarthTransform', 'Curvature', 'c1', 'c2', 'c3', 'c4']
    for i=0,n_elements( object_names)-1 do begin
      obj = model->GetByName( object_names[i])
      model->remove, obj
      obj_destroy, obj
    endfor
  endif
  
  ;Add xAxis, yAxis, zAxis and titles to the plot3dBoxObj node.
  plot3dBoxObj = obj_new( 'IDLgrModel', name='Plot3DBox')
  model->add, plot3dBoxObj
  xtitle = 'X '+units          ;This doesn't work in object graphics: +'!C!C!M4 Sun'
  ytitle = 'Y!IGSE!N'+units
  ztitle = 'Z!IGSE!N'+units
  rave_axes, xrange, yrange, zrange, window, plot3dBoxObj, $
             xtitle=xtitle, ytitle=ytitle, ztitle=ztitle, $
             xs=xs, ys=ys, zs=zs, state.foreground_color
    
  SC_rgbColors = [state.foreground_color[0],244,137,73]
  white = [255,255,255]
  
  globeRadius = 0.1  ;radius in normalized coordinates
  lons = 16 & lats = 9
  mesh_obj, 4, v_16x9,  p_16x9,  fltarr(lons+1,lats+1) + globeRadius    ;create a 16x9 sphere
  
  ;Make a 16x16 sphere for the earth with radius Re.
  lats = 16
  mesh_obj, 4, v, p, fltarr(lons+1,lats+1) + abs( xs[1])   ;create a 16x16 sphere, 1 Re in radius
  earth = obj_new( 'IDLgrPolygon', v, polygons=p, style=2, color=white, name='earth')
  poly_shades = intarr(16,16) + 32  ;with color table 39, index 32 = blue/black for night-side.
  poly_shades[4:11,*] = 191         ;with color table 39, index 191 = yellow = [255,255,0] for day-side.
  convertFaceColorsToVertexColors, v, p, poly_shades, vertexColors
  earth->setProperty, vert_colors=vertexColors
  earthTransform = obj_new( 'IDLgrModel', name='EarthTransform')
  model->add, earthTransform
  earthTransform->add, earth
  earthTransform->Translate, xs[0], ys[0], zs[0]
  
  
  for SC=0,3 do begin
    SCName = 'c' + strtrim(SC+1,2)
    
    if first_time then begin
      annotations_model->add, obj_new( 'IDLgrText', SCName, name=SCName, color=SC_rgbColors[SC], $
                                       font=obj_new( 'IDLgrFont', size=14.0, thick=4), $
                                       location=[-0.99, 0.95-SC*0.06, 1])
    endif
    
    SCObj = obj_new( 'IDLgrModel', name=SCName)
    model->add, SCObj
    
    ;Note: coordinates used to create primitives are by default in normalized coordinates,
    ;relative to the origin of the nearest parent IDLgrModel coordinate system.  
    ;Or you can pass in data coordinates by specifying a conversion to normalized
    ;coordinates in the x/y/zcoord_conv keywords.
    orbitTrack = obj_new( 'IDLgrPolyline', pos[*,0,SC], pos[*,1,SC], pos[*,2,SC], $
                          color=SC_rgbColors[SC], thick=3, name='OrbitTrack', $
                          xcoord_conv=xs, ycoord_conv=ys, zcoord_conv=zs)
    SCObj->add, orbitTrack
    
    ;When a S/C moves, change the polyline in the orbitProjection object.
    orbitProjectionObj = obj_new( 'IDLgrModel', name='OrbitProjection')
    SCObj->add, orbitProjectionObj
    n_vertices = (size( pos,/dim))[0]
    ones = fltarr( n_vertices)+1
    ;Project orbit onto x-y plane at Z bottom plane.
    orbitProjectionObj->add, obj_new( 'IDLgrPolyLine', color=SC_rgbColors[SC], $
                                      pos[*,0,SC], pos[*,1,SC], ones*zrange[0], $
                                      xcoord_conv=xs, ycoord_conv=ys, zcoord_conv=zs, $
                                      name='XYPlaneProjection')
    ;Project orbit onto y-z plane at X backplane.
    orbitProjectionObj->add, obj_new( 'IDLgrPolyLine', color=SC_rgbColors[SC], $
                                      ones*xrange[1], pos[*,1,SC], pos[*,2,SC], $
                                      xcoord_conv=xs, ycoord_conv=ys, zcoord_conv=zs, $
                                      name='YZPlaneProjection')

    ;Project orbit onto x-z plane at Y backplane
    orbitProjectionObj->add, obj_new( 'IDLgrPolyLine', color=SC_rgbColors[SC], $
                                      pos[*,0,SC], ones*yrange[1], pos[*,2,SC], $
                                      xcoord_conv=xs, ycoord_conv=ys, zcoord_conv=zs, $
                                      name='XZPlaneProjection')
    
    psym = 6
    size = [0.005,0.005]
    symbolObj = obj_new( 'IDLgrSymbol', psym, color=SC_rgbColors[SC], thick=2, size=size, name='MarkerSymbol')
    orbitProjectionObj->add, obj_new( 'IDLgrPolyLine', name='XYPlaneMarker', symbol=symbolObj, $
                                      xcoord_conv=xs, ycoord_conv=ys, zcoord_conv=zs)
    orbitProjectionObj->add, obj_new( 'IDLgrPolyLine', name='XZPlaneMarker', symbol=symbolObj, $
                                      xcoord_conv=xs, ycoord_conv=ys, zcoord_conv=zs)
    orbitProjectionObj->add, obj_new( 'IDLgrPolyLine', name='YZPlaneMarker', symbol=symbolObj, $
                                      xcoord_conv=xs, ycoord_conv=ys, zcoord_conv=zs)
    
    ;To display new RAPID data, change the vertexColors in a Globe node.
    globeObj  = obj_new( 'IDLgrPolygon', v_16x9,  polygons=p_16x9,  style=2, color=state.foreground_color, $
                         name=SCname + '_Globe')          ;Display IES_CLEANBM data.
    
    ;To change a B arrow, change the polygon data in a B node.
    Bobj = obj_new( 'IDLgrModel', name='B')
    arrowShaftObj = obj_new( 'IDLgrPolyline', color=state.foreground_color, name=SCname+'/B')
    arrowHeadObj  = obj_new( 'IDLgrPolygon',  color=state.foreground_color, name=SCname+'/B')
    Bobj->Add, arrowShaftObj
    Bobj->Add, arrowHeadObj
        
    ;Rotate the GlobeTransform about the z axis so that sector 13 is approximately centered on the
    ;+x (sun) direction.  Positive rotations go counter-clockwise when looking down from the N pole.
    ;Without rotation, the start of sector 0 is at -x, the start of sector 1 is 22.5 degrees clockwise
    ;of -x (when looking down from the N pole), etc.  A 11.25 degree rotation centers sector 0 on -x, 
    ;180 more centers it on +x, and -3*22.5 centers sector 13 instead of sector 0 on +x.  An additional
    ;angle of 3*22.5 - 71.417 accounts for the sun being slightly off-center in sector 13.
    angle = 11.25 + 180 - 71.417
    globeTransformObj = obj_new( 'IDLgrModel', name='GlobeTransform')
    globeTransformObj->Rotate, [0,0,1], angle
    globeTransformObj->add, globeObj
    
    ;Save this fixed rotation matrix as the uvalue of the GlobeTransform object, because we'll want to
    ;restore it each time step before adding a small spin axis rotation.  The spin axis probably changes
    ;very little for our typical time ranges, but do it every time step anyway, just in case there's
    ;ever a spin axis manoever inside our time range.
    globeTransformObj->GetProperty, transform=t
    globeTransformObj->SetProperty, uvalue=t
    
    ;To move or resize a globe and the B vector, change the matrix in the Transform node.
    transformObj = obj_new( 'IDLgrModel', name='Transform')
    transformObj->add, globeTransformObj
    transformObj->add, Bobj
    SCObj->add, transformObj
    
    ;Create circles representing constant latitude contours at latitudes -60, -30, 0, 30, 60.
    pitchTransformObj = obj_new( 'IDLgrModel', name='PitchTransform')
    n_equator_pts = 100
    latitudes = [-60, -30, 0, 30, 60]
    for lat=0,n_elements( latitudes)-1 do begin
      n_pts = fix( n_equator_pts * cos( latitudes[ lat] / !radeg))
      coords = fltarr(3, n_pts)
      coords[0,*] = indgen( n_pts) * 360./n_pts
      coords[1,*] = latitudes[ lat]
      coords[2,*] = globeRadius
      coords = cv_coord( from_sphere=coords, /deg, /to_rect)
      contourLineObj = obj_new( 'IDLgrPolyline', coords, thick=1, $
                                vert_colors=[state.background_color[0], state.foreground_color[0]], $
                                name='PitchAngleContours')
      pitchTransformObj->add, contourLineObj
    endfor
    transformObj->add, pitchTransformObj
    
  endfor  ;loop through all 4 S/C
  
  old_state = state  ;Save state so can later optimize based on what has changed.
  
endif  ;initialize eq 1


;Set eventTimeStr text display.
eventTimeStr = view->GetByName( 'annotations/eventTimeStr')
eventTimeStr->setProperty, string=strmid( T_TO_STR( state.event_time), 9)

;Set event_Ts and interpolate for the positions at the event time.
event_Ts = dblarr(4) + state.event_time
interpolateXYZROnEventTimes, event_Ts, event_XYZRs, spin_axes=spin_axes

SC_colors = [state.foreground_color[0],244,137,73]   ;white/black,red,green,blue

for SC=0,3 do begin
  SCname = 'c' + strtrim( SC+1, 2)
  ;Get objects from the tree.
  SCObj        = model->GetByName( SCname)
  SC_transform = model->GetByName( SCname + '/Transform')
  SC_globe_transform = model->GetByName( SCname + '/Transform/GlobeTransform')
  SC_globe     = model->GetByName( SCname + '/Transform/GlobeTransform/' + SCname + '_Globe')
  
  if state.show.ElectronFluxGlobes then begin
    getRapidData, SC, state.event_time, state.averaging_interval, T, rapidData, have_rapid_data
    if have_rapid_data then begin
      n_sectors   = (size( rapidData, /dim))[0]
      if n_sectors ne 16 then throw,'n_sectors = ' + strtrim( n_sectors, 2) + ' not supported!'
      d = rapidData[*,*]   ;[16,9]
      W = where( finite(d) eq 0b or d le 0.0, count, complement=WW)
      if count gt 0 then begin
        percent_missing_data = count / (16*9.) * 100
        percent_str = (percent_missing_data lt 1.0) ? '<1' : strtrim( fix( percent_missing_data), 2)
        ;print, percent_str + $
        ;  '% of ' + SCname + "'s sector/detector values were zero."
        if state.show.FillDataGaps then begin
          fillDataGaps, d
        endif else begin
          d[w] = 10.0^state.logRangeForColorScaling[0]
        endelse
      endif
      
      if state.show.SphericalHarmonicFit then begin
        SHA_TEST = 0   ;This is under development!
        if SHA_TEST then begin
          if 0 then begin
            sha_test, state.event_time, d
          endif else begin
            if SC eq 0 then begin
              getSphericalHarmonic, 2, 0, part='Re', d
              getSphericalHarmonicExpansionCoefficients, d, 0.0d, s, Lmax=2, method='IDL', /ver
            endif
            if SC eq 1 then begin
              getSphericalHarmonic, 2, 1, part='Re', d
              getSphericalHarmonicExpansionCoefficients, d, 0.0d, s, Lmax=2, method='IDL',/ver
            endif
            if SC eq 2 then begin
              getSphericalHarmonic, 2, -1, part='Re', d
              getSphericalHarmonicExpansionCoefficients, d, 0.0d, s, Lmax=2, method='IDL',/ver
            endif
            if SC eq 3 then begin
              getSphericalHarmonic, 2, 2, part='Re', d
              getSphericalHarmonicExpansionCoefficients, d, 0.0d, s, Lmax=2, method='IDL',/ver
            endif
          endelse
          d = 10.0^d
        endif
        
        SHA_METHOD = 'ccSHT'  ;'Ulysses', 'ccSHT'
        if SHA_METHOD eq 'ccSHT' then begin
          if 0 then verbose = 1 else verbose = 0
          getSphericalHarmonicExpansionCoefficients, d, state.event_time, s, Lmax=2, method='IDL', verbose=verbose
          addSphericalHarmonicExpansionCoefficients, SC, s
        endif else begin  ;SHA_METHOD = 'Ulysses'
          getSphericalHarmonicExpansionCoefficients, SC, state, s
          addSphericalHarmonicExpansionCoefficients, SC, s
        endelse
      endif
      d = alog10(d)
    endif else begin  ;have_rapid_data is false
      d = fltarr(16,9) + state.logRangeForColorScaling[0]
    endelse
    SC_globe->SetProperty, hide=0, uvalue=d
        
    test_orientation = 0
    if test_orientation then begin
      ;To prove the orientation and color mapping is correct, illuminate a few selected
      ;pixels with index 191 = [255,255,0] = yellow (using color table 39).
      d = bytarr(16,9) + SC_colors[SC]
      d[13, 5:8] = 191  ;Illuminate the top half of the sector in the x direction (sun direction).
      d[9,5]     = 191  ;Illuminate a pixel above the equator in the sector in the +y direction.
      d[0,8]     = 191  ;Illuminate a pixel near the N pole in sector 0.
      colorGlobe, SC_globe, d, [0,255]
    endif else begin
      colorGlobe, SC_globe, d, state.logRangeForColorScaling
    endelse
    
    if have_rapid_data then begin
      SC_globe->setProperty, style=2                  ;color-filled polygon faces
    endif else begin
      ;wire-frame with hidden line removal
      SC_globe->setProperty, style=1, /HIDDEN_LINES, vert_colors=0, color=state.foreground_color
    endelse

  endif else begin  ;state.show.ElectronFluxGlobes is false
    SC_globe->SetProperty, /HIDE
  endelse  
  
  ;Translate the S/C globe and B vector to the S/C position at the event time and scale them.
  identityMatrix = [[1.0, 0.0, 0.0, 0.0], $
                    [0.0, 1.0, 0.0, 0.0], $
                    [0.0, 0.0, 1.0, 0.0], $
                    [0.0, 0.0, 0.0, 1.0]]
  SC_transform->SetProperty, transform=identityMatrix  
  SC_transform->Scale, state.globe_scaling_factor, state.globe_scaling_factor, state.globe_scaling_factor
  SC_transform->Translate, xs[0]+xs[1]*event_XYZRs[0,SC], $
                           ys[0]+ys[1]*event_XYZRs[1,SC], $
                           zs[0]+zs[1]*event_XYZRs[2,SC]  
  
  ;Add a rotation so that the S pole of the globe is aligned with the spin axis, which is usually
  ;slightly off from its nominal value of [0,0,-1] (the S ecliptic pole).
  ;spin_axis = [0, -90, 1]
  ;spin_axis = [187.6, -84.1, 1]
  spin_axis = reform( spin_axes[*,SC])
  if spin_axis[0] gt 180.0 then spin_axis[0] = -(360.0 - spin_axis[0])
  spin_axis = cv_coord( from_sphere=spin_axis, /DEGREES, /TO_RECT)
  SC_globe_transform->GetProperty, uvalue=t     ;Start with transform which aligns sector 13 with the sun.
  SC_globe_transform->SetProperty, transform=t
  rotateOneVectorIntoAnother, [0,0,-1], spin_axis, SC_globe_transform
    
  if state.show.MagneticField or state.show.PitchAngleContours then begin
    getMagData, SC, state.event_time, B, state.averaging_interval, success, max_Btot=max_Btot
  endif
  
  SC_B = SC_transform->GetByName( 'B')
  if state.show.MagneticField then begin
    if success then begin
      SC_B->SetProperty, hide=0, uvalue=[B.Bx, B.By, B.Bz]
      globeRadius = 0.1  ;globe radius in normalized coordinates
      drawMag, [B.Bx, B.By, B.Bz], SC_B, max_Btot, xs, ys, globeRadius, state
    endif else begin
      SC_B->SetProperty, hide=1
    endelse
  endif else begin
    SC_B->SetProperty, hide=1
  endelse    
  
  SC_pitch_transform = SC_transform->GetByName( 'PitchTransform')
  if state.show.ElectronFluxGlobes and state.show.PitchAngleContours then begin
    if success then begin
      SC_pitch_transform->SetProperty, hide=0
      rotateOneVectorIntoAnother, [0,0,1], [B.Bx, B.By, B.Bz], SC_pitch_transform, /RESET
    endif else begin
      SC_pitch_transform->SetProperty, hide=1
    endelse
  endif else begin
    SC_pitch_transform->SetProperty, hide=1
  endelse
  
  SC_orbitProjection = SCObj->GetByName( 'OrbitProjection')
  if state.show.OrbitProjections then begin
    SC_orbitProjection->SetProperty, hide=0
    projectOrbitsOntoCubeSides, event_XYZRs, SC, SC_orbitProjection, xrange, yrange, zrange
  endif else begin
    SC_orbitProjection->SetProperty, hide=1
  endelse
  
  SC_orbitTrack = SCObj->GetByName( 'OrbitTrack')
  SC_orbitTrack->SetProperty, hide=(state.show.OrbitTracks ? 0 : 1)
    
endfor

;Can't use HIDE, because PitchAngleScene occludes the 3D scene, so add/remove PitchAngleScene.
pitch_angle_view_is_visible = obj_valid( state.Scene->GetByName( 'PitchAngleView'))
if state.show.PitchAnglePlot then begin
  if not pitch_angle_view_is_visible then begin
    state.Scene->Add, state.pitch_angle_view
    getPitchAngleData, restoreStartTime=STR_TO_T( state.minTimeStr), $
                       restoreStopTime=STR_TO_T( state.maxTimeStr), $
                       band=state.band
  endif
  drawPitchAnglePlot, state
endif else begin
  if pitch_angle_view_is_visible then begin
    state.Scene->Remove, state.pitch_angle_view
  endif
endelse

hide = state.show.ElectronFluxGlobes ? 0 : 1
colorBar = view->GetByName( 'annotations/ColorBar')
colorBar->SetProperty, hide=hide
colorBarAxis = view->GetByName( 'annotations/ColorBarAxis')
colorBarAxis->SetProperty, hide=hide

raveCurvature, state

window->draw, state.scene
old_state = state  ;Save state so can later optimize based on what has changed.
return
END


;*******************************************************************************
; Routine:  raveChangeView
;
; Purpose:  Change the view direction to one of several pre-defined view directions.
;*******************************************************************************
pro raveChangeView, view_specification_str, model

;Get object references for changing axis text orientation.
plot3DBox = model->GetByName( 'Plot3DBox')
xTitle = plot3DBox->GetByName( 'XTitle')
yTitle = plot3DBox->GetByName( 'YTitle')
zTitle = plot3DBox->GetByName( 'ZTitle')

identityMatrix = [[1.0, 0.0, 0.0, 0.0], $
                  [0.0, 1.0, 0.0, 0.0], $
                  [0.0, 0.0, 1.0, 0.0], $
                  [0.0, 0.0, 0.0, 1.0]]
scale_factor = 1.2
rotation_inc = 10
translation_inc = 0.1
zoom_inc = 0.1
case view_specification_str of
  'Standard': begin
    model->SetProperty, transform=identityMatrix  
    ax=30 & ay=0 & az=30
    model->rotate, [1,0,0], -90
    model->rotate, [0,1,0], az
    model->rotate, [1,0,0], ax
    xTitle->SetProperty, UpDir=[0,1,0], Baseline=[1,0,0]
    yTitle->SetProperty, UpDir=[-1,0,0], Baseline=[0,1,0]
    zTitle->SetProperty, UpDir=[-1,0,0], Baseline=[0,0,1]
    model->scale, scale_factor, scale_factor, scale_factor
  end
  'Southward (Top View)': begin
    model->SetProperty, transform=identityMatrix  
    xTitle->SetProperty, UpDir=[0,1,0], Baseline=[1,0,0]
    yTitle->SetProperty, UpDir=[-1,0,0], Baseline=[0,1,0]
    zTitle->SetProperty, UpDir=[-1,0,0], Baseline=[0,0,1]
    model->scale, scale_factor, scale_factor, scale_factor
  end
  'Tailward': begin
    model->SetProperty, transform=identityMatrix  
    model->rotate, [1,0,0], -90
    model->rotate, [0,1,0], 90
    xTitle->SetProperty, UpDir=[0,1,0], Baseline=[1,0,0]
    yTitle->SetProperty, UpDir=[0,0,1], Baseline=[0,-1,0]
    zTitle->SetProperty, UpDir=[0,1,0], Baseline=[0,0,1]
    model->scale, scale_factor, scale_factor, scale_factor
  end
  'X-Z Plane': begin
    model->SetProperty, transform=identityMatrix  
    model->rotate, [1,0,0], -90
    xTitle->SetProperty, UpDir=[0,0,1], Baseline=[1,0,0]
    yTitle->SetProperty, UpDir=[-1,0,0], Baseline=[0,1,0]
    zTitle->SetProperty, UpDir=[-1,0,0], Baseline=[0,0,1]
    model->scale, scale_factor, scale_factor, scale_factor
  end
  'Reverse': begin
    print,'Reverse'
  end
  
  ;The following are in response to short-cut arrow keys.
  'RotateLeft': begin
    model->rotate, [0,1,0], -rotation_inc
  end
  'RotateRight': begin
    model->rotate, [0,1,0], rotation_inc
  end
  'RotateUp': begin
    model->rotate, [1,0,0], -rotation_inc
  end
  'RotateDown': begin
    model->rotate, [1,0,0], rotation_inc
  end
  'ZoomUp': begin
    model->scale, 1 + zoom_inc, 1 + zoom_inc, 1 + zoom_inc
  end
  'ZoomDown': begin
    model->scale, 1 - zoom_inc, 1 - zoom_inc, 1 - zoom_inc
  end
  else: begin
    print,'Unknown menu selection: ' + view_specification_str
    return
  end        
endcase

;Note: window->draw, state.scene  is now done by the caller.
END


;*******************************************************************************
; Routine: raveMenuBarHandler
;
; Purpose: Handle menu selections to the RAVE 3-D display "File" menu.
;
;*******************************************************************************
pro raveMenuBarHandler, type_str, rave_tlb, print_filename=print_filename, redraw

common rave_common, pos, window, view, model, trackball, old_state, $
                    xs, ys, zs, xrange, yrange, zrange

redraw = 0
WIDGET_CONTROL, rave_tlb, get_uvalue=state
window->getProperty, palette=palette

switch type_str of
  
  'File.Print.Postscript': begin
    ;This is taken from http://www.dfanning.com/ographics_tips/object_eps.html but doesn't work.
    ;Ghostview gives error reading the file, only displays a short line instead of the image.
    printerObj = obj_new( 'IDLgrPrinter', color_model=1, n_colors=256, palette=palette, $
                          print_quality=2)
    ok = dialog_printersetup( printerObj)
    if not ok then return
    printerObj->draw, state.scene
    printerObj->NewDocument
    obj_destroy, printerObj
    break
  end
  
  'File.Print.Encapsulated Postscript': begin
    ;This is taken from http://www.dfanning.com/ographics_tips/object_eps.html.
    ;Currently only the color bar is colored; everything else is B/W.
    geom = WIDGET_INFO( WIDGET_INFO( rave_tlb, FIND_BY_UNAME='DrawWidget'), /GEOM)
    dimensions = [geom.xsize, geom.ysize]
    cm_per_pixel = 2.54/72  ;72 dots per inch
    clipboard = obj_new( 'IDLgrClipboard', dimensions=dimensions, resolution=[1,1]*cm_per_pixel)
    filename = 'rave.eps'
    clipboard->Draw, view, filename=filename, /PostScript  ;, /Vector
    obj_destroy, clipboard
    cd, current=cur
    print,'Wrote ' + cur + papco_get_pathsep() + filename
    break
  end
  
  'File.Print.Jpeg':
  'File.Print.Tiff':
  'File.Print.PNG': begin
    geom = WIDGET_INFO( WIDGET_INFO( rave_tlb, FIND_BY_UNAME='DrawWidget'), /GEOM)
    width = geom.xsize & height = geom.ysize
    buffer = obj_new( 'IDLgrBuffer', dim=[width, height], graphics_tree=view, $
                       color_model=1, n_colors=256, palette=palette, resolution=[1,1]*2.54/72/1.4)
    ;Note: if specify color_model=0, palette=palette, get blank jpeg file.
    buffer->Draw, state.scene
    imageObj = buffer->Read()
    imageObj->SetProperty, interpolate=1   ;Not sure this improves quality...
    imageObj->GetProperty, data=image
    palette->GetProperty, red_values=r, green_values=g, blue_values=b
    case type_str of
      'File.Print.Jpeg': begin
        set_plot,'Z'
        device, z_buffering=0, set_resolution=[width, height]
        loadct, 39
        tv, image
        if not keyword_set( print_filename) then print_filename = 'rave.jpeg'
        result = execute( "saveimage, '" + print_filename + "', /jpeg, quality=100")
        device, /close
        set_plot,'X'
      end
      'File.Print.Tiff': begin
        if not keyword_set( print_filename) then print_filename = 'rave.tif'
        result = execute( "write_tiff, print_filename, rotate( image,7), red=r, green=g, blue=b")
        ;spawn,'xv rave.tif'
      end
      'File.Print.PNG': begin
        if not keyword_set( print_filename) then print_filename = 'rave.png'
        result = execute( "write_png, print_filename, image, r, g, b")
        ;spawn,'xv rave.png'
      end
    endcase
    cd, current=cwd
    if result eq 0 then begin
      print,'Failed to write ' + cwd + papco_get_pathsep() + print_filename
    endif else begin
      print, 'Wrote ' + cwd + papco_get_pathsep() + print_filename
    endelse
    buffer->SetProperty, graphics_tree=obj_new()
    obj_destroy, buffer
    break
  end
  
  'File.Save.Spherical Harmonic Coefficients': begin
    saveSphericalHarmonicExpansionCoefficients
    break
  end
  
  'File.Movie.Start Saving Frames': begin
    state.movie_filename_base = 'movie'
    ;Start frame number AFTER last existing movie file's frame number.
    f = findfile( 'movie*', count=count)
    if count eq 0 then begin
      state.movie_frame_number = 1
    endif else begin
      frame_number_start_index = strpos( f[0], '_', /REVERSE_SEARCH) + 1
      f = strmid( f, frame_number_start_index)
      f = (reverse( f[ sort(f)]))[0]
      last_existing_frame_number = fix( strmid( f, 0, strpos( f, '.')))
      state.movie_frame_number = last_existing_frame_number + 1
    endelse
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
    redraw = 1
    break
  end
  
  'File.Movie.Stop Saving Frames': begin
    state.movie_filename_base = ''
    state.movie_frame_number = 1
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
    break
  end
  
  'File.Quit': begin
    WIDGET_CONTROL, rave_tlb, /DESTROY
    break
  end
  
  else: begin
    print,'raveMenuBarHandler: Unknown menu selection: ' + type_str
    break
  end
endswitch

END


;*******************************************************************************
pro raveChangeBackgroundColor, state
;*******************************************************************************
common rave_common, pos, window, view, model, trackball, old_state, $
                    xs, ys, zs, xrange, yrange, zrange
view->SetProperty, color=state.background_color
objArr = [view->GetByName( 'annotations/eventTimeStr')]
objArr = [objArr, view->GetByName( 'annotations/c1')]
objArr = [objArr, model->GetByName( 'Plot3DBox/XTitle')]
objArr = [objArr, model->GetByName( 'Plot3DBox/xAxis')]
objArr = [objArr, model->GetByName( 'Plot3DBox/YTitle')]
objArr = [objArr, model->GetByName( 'Plot3DBox/yAxis')]
objArr = [objArr, model->GetByName( 'Plot3DBox/ZTitle')]
objArr = [objArr, model->GetByName( 'Plot3DBox/zAxis')]
objArr = [objArr, model->GetByName( 'Plot3DBox/BoundingBox')]
objArr = [objArr, model->GetByName( 'Plot3DBox/RuleLines')]
objArr = [objArr, model->GetByName( 'c1/OrbitTrack')]
objArr = [objArr, model->GetByName( 'c1/OrbitProjection/XYPlaneProjection')]
objArr = [objArr, model->GetByName( 'c1/OrbitProjection/YZPlaneProjection')]
objArr = [objArr, model->GetByName( 'c1/OrbitProjection/XZPlaneProjection')]
objArr = [objArr,  view->GetByName( 'annotations/ColorBarAxis')]
;objArr = [objArr,  view->GetByName( 'annotations/ColorBarAxisTitle')]  ;Title not currently being added

;The IDLgrSymbol object is special: it doesn't have a super-class, and can't be put into
;the graphics tree, so we can't get it by name.
obj = model->GetByName( 'c1/OrbitProjection/YZPlaneMarker')
obj->GetProperty, symbol=symbolObj
symbolObj->SetProperty, color=state.foreground_color[0]

for i=0,3 do begin
  pitchTransform = model->GetByName( 'c' + strtrim(i+1,2) + '/Transform/PitchTransform')
  if obj_valid( pitchTransform) then begin
    contourArr = pitchTransform->Get( /ALL, count=count)
    for j=0,count-1 do begin
      objArr = [objArr, contourArr[j]]
    endfor
  endif
endfor

curvatureObj = model->GetByName( 'Curvature')
if obj_valid( curvatureObj) then begin
  transformArr = curvatureObj->Get( /ALL, count=count)
  for i=0,count-1 do begin
    obj = transformArr[i]->GetByName( 'B')
    objArr = [objArr, obj->Get( position=0)]
    objArr = [objArr, obj->Get( position=1)]
    ;objArr = [objArr, transformArr[i]->GetByName( 'B/arrowHead')]
    objArr = [objArr, transformArr[i]->GetByName( 'CurvatureArcRotation/CurvatureArc')]
  endfor
endif

for i=0,n_elements(objArr)-1 do begin
  if obj_valid( objArr[i]) then begin
    objArr[i]->GetProperty, color=color
    color = (n_elements( color) eq 3) ? state.foreground_color : state.foreground_color[0]
    objArr[i]->SetProperty, color=color
  endif else begin
    print,'raveChangeBackgroundColor: objArr[' + strtrim(i,2) + '] is invalid!'
  endelse
endfor
END
