;*******************************************************************************
; Method:  rave_axes
; Purpose: Create x, y and z axes in object graphics. 
; Author:  Steve Monk with code snippets from David Fanning.
; usage:   rave_axes, xrange, yrange, zrange, window, model, $
;                     xtitle='xtitle', ytitle='ytitle', ztitle='ztitle'
; Inputs:
;   xrange, yrange, zrange - 2 element float vectors contain the desired
;                            range for the axes.  If the second element
;                            is less than the first element, then the
;                            axis will be reversed.
;   window                 - a IDLgrWindow object
;   model                  - a IDLgrModel object; the method adds 3 axes and
;                            3 titles to the model.
;   xtitle, ytitle, ztitle - Keywords containing strings for the axis titles.
;
; Outputs:
 ;   xrange, yrange, zrange - scaled so distance is the same in all directions (isotropic)
;    xs, ys, zs - The *coord_conv values to use in scaling anything you add to
;                 the model.
;    Many objects are added to the model input object.
;    The axes' coord_conv normalizes their data range to the range [-0.5, 0.5].
;*******************************************************************************


;*******************************************************************************
function normalize, range, position=position

; This is a utility routine to calculate the scaling vector
; required to position a vector of specified range at a
; specific position given in normalized coordinates. The
; scaling vector is given as a two-element array like this:
;
; scalingVector = [translationFactor, scalingFactor]
;
; The scaling vector should be used with the [XYZ]COORD_CONV
; keywords of a graphics object or model. For example, if you
; wanted to scale an X axis into the data range of -0.5 to 0.5,
; you might type something like this:
;
;   xAxis->GetProperty, Range=xRange
;   xScale = Normalize(xRange, Position=[-0.5, 0.5])
;   xAxis, XCoord_Conv=xScale
;*******************************************************************************
on_error, 1
IF N_Params() EQ 0 THEN Message, 'Please pass range vector as argument.'

IF (N_Elements(position) EQ 0) THEN position = [0.0, 1.0] ELSE $
    position=Float(position)
range = Float(range)

scale = [((position[0]*range[1])-(position[1]*range[0])) / (range[1]-range[0]), $
          (position[1]-position[0])/(range[1]-range[0])]

RETURN, scale
END


;*******************************************************************************
function tickNameFormatter, axisDirection, tickmarkIndex, tickmarkValue
;*******************************************************************************
return, strtrim( string( format='(F6.1)', tickmarkValue), 2)
END


;*******************************************************************************
pro rave_axes, xrange, yrange, zrange, thisWindow, thisModel, $
               xtitle=xtitle, ytitle=ytitle, ztitle=ztitle, xs=xs, ys=ys, zs=zs, $
               foreground_color
;*******************************************************************************

; Set xrange, yrange, zrange so that all the axes have the same data range
; as the axis with the biggest data range, so that distance is the same in
; all directions.

ranges = [[xrange],[yrange],[zrange]]  ;fltarr(2,3)
maxRange = max( abs( ranges[1,*]-ranges[0,*]))
reversed = 2*(ranges[1,*] gt ranges[0,*]) - 1   ;intarr(1,3), -1 = reversed, 1 = not reversed
color = foreground_color

for i=0,2 do begin
  ranges[*,i] = total( ranges[*,i])/2.0 + [-1,1] * maxRange/2.0 * reversed[i]
endfor
xrange = ranges[*,0]
yrange = ranges[*,1]
zrange = ranges[*,2]

axisFont = Obj_New('IDLgrFont', 'Helvetica', Size=16)

; Create title objects for the axes and plot. Color them the foreground color. 
; Title objects will be located carefully below for reversible axes.
; Be sure you DON'T add the title objects directly to the axis objects.

xTitleObj = Obj_New('IDLgrText', xtitle, Color=color, Font=axisFont, $
                    /Enable_Formatting, Vertical_Alignment=1, name='XTitle')
yTitleObj = Obj_New('IDLgrText', ytitle, Color=color, Font=axisFont, $
                    /Enable_Formatting, name='YTitle')
zTitleObj = Obj_New('IDLgrText', ztitle, Color=color, Font=axisFont, $
                    /Enable_Formatting, name='ZTitle')

; I create a tick length variable, because I find it
; convenient to use a unit of tick length in positioning
; the axis titles.

ticklen = 0.03

; Create axes objects and color them the foreground color.
; Note how I set the axis annotation to 10 point Helvetica. 
; Another reason for positioning the axis title independently of the axis
; is that the title can have other colors and sizes than the axis annotation. 
; This is NOT possible if the title is part of the axis.

xAxis = Obj_New("IDLgrAxis", 0, name='xAxis', Color=color, Ticklen=ticklen, $
                Range=xrange, /exact, tickformat='tickNameFormatter')
xAxis->GetProperty, Ticktext=xAxisText
xAxisText->SetProperty, Font=axisFont

yAxis = Obj_New("IDLgrAxis", 1, name='yAxis', Color=color, Ticklen=ticklen, $
                Range=yrange, /exact, tickformat='tickNameFormatter')
yAxis->GetProperty, Ticktext=yAxisText
yAxisText->SetProperty, Font=axisFont

zAxis = Obj_New("IDLgrAxis", 2, name='zAxis', Color=color, Ticklen=ticklen, $
                Range=zrange, /exact, tickformat='tickNameFormatter')
zAxis->GetProperty, Ticktext=zAxisText
zAxisText->SetProperty, Font=axisFont

; The axes may not use exact axis scaling, so the ranges may
; have changed from what they were originally set to. Get
; and update the range variables.

if 0 then begin  ;Not necessary since we specified /exact. SM
xAxis->GetProperty, CRange=xrange
yAxis->GetProperty, CRange=yrange
zAxis->GetProperty, CRange=zrange
endif

; Scale each axis into the range -0.5 to 0.5.  The normalized range is -0.5 to 0.5
; so when the display is rotated we don't have to worry about translations. 
; In other words, the rotations occur about the point (0,0,0).

xs = Normalize( xrange, Position=[-0.5,0.5])
ys = Normalize( yrange, Position=[-0.5,0.5])
zs = Normalize( zrange, Position=[-0.5,0.5])

; Scale the axes and place them in the coordinate space.
; Note that not all values in the Location keyword are used.
; (I've put really large values into the positions that are not
; being used to demonstate this.) For example, with the X axis only
; the Y and Z locations are used.

xAxis->SetProperty, Location=[9999.0, -0.5, -0.5], XCoord_Conv=xs
yAxis->SetProperty, Location=[-0.5, 9999.0, -0.5], YCoord_Conv=ys
zAxis->SetProperty, Location=[-0.5,  0.5, 9999.0], ZCoord_Conv=zs

; Add the axes objects to the model.

thisModel->Add, xAxis
thisModel->Add, yAxis
thisModel->Add, zAxis

; Add the title objects to the model.

thisModel->Add, xTitleObj
thisModel->Add, yTitleObj
thisModel->Add, zTitleObj

; Once we have a window, find the size of the character box surrounding the
; axis annotation, and calculate a location for the axis titles. Note that the Y
; dimension of the X axis text box is always about 75% of what it *should* be. This
; is the reason the X axis title always appears too close to the axis compared
; to the Y and Z axis in the normal default placement. That is why you see that
; number multiplied by 1.5 for the XTitleObj below. (The values -0.5, 0.5, and 0
; are the endpoints and middle, respectively, of my axis in my viewport rectangle.)

; To orient the text properly, you must supply the proper baseline and up direction
; vectors to the Y and Z titles. The X title does not need this, since the X "situation"
; is the default case. For example, read the Y title orientation like this: draw the
; text parallel to the Y axis (Baseline=[0,1,0]), with the up direction in the -X direction
; (UpDir=[-1,0,0]).
; The algorithm for all three axes are similar, although the details of setting the text
; baseline and up direction vectors vary for each axis.

d = thisWindow->GetTextDimensions( xAxisText)
xTitleObj->SetProperty, Location=[0, -0.5 - d[1]*1.5-ticklen, -0.5], $
                        Alignment=0.5

d = thisWindow->GetTextDimensions( yAxisText)
yTitleObj->SetProperty, Location=[-0.5 - d[0]-ticklen, 0, -0.5], $
                        Baseline=[0,1,0], UpDir=[-1,0,0], Alignment=0.5

d = thisWindow->GetTextDimensions( zAxisText)
zTitleObj->SetProperty, Location=[-0.5 - d[0]-ticklen, 0.5, 0], $
                        Baseline=[0,0,1], UpDir=[-1,0,0], Alignment=0.5

xAxis->SetProperty, TextBaseline=[reversed[0], 0, 0]
yAxis->SetProperty, TextUpDir=[0, 1*reversed[1], 0]
zAxis->SetProperty, TextUpDir=[0, 0, 1*reversed[2]]

;Draw the rest of the box.

;vertices of vertical lines:
v = [[-1,-1,-1],[-1,-1,1],[ 1,-1,-1],[ 1,-1,1],[ 1, 1,-1],[ 1, 1,1]]

;vertices of horizontal lines at top of box parallel to x-axis:
v = [[v],[[-1,-1, 1],[ 1,-1,1],[-1, 1, 1],[ 1, 1,1]]]

;vertices of horizontal lines at top of box parallel to y-axis:
v = [[v],[[-1,-1, 1],[-1, 1,1],[ 1,-1, 1],[ 1, 1,1]]]

;vertices of horizontal lines at bottom of box:
v = [[v], [[1, -1,-1],[ 1, 1,-1],[-1, 1,-1],[ 1, 1,-1]]]
v = v*0.5
n_vertices = (size(v,/dim))[1]
polylines = intarr( n_vertices/2*3)
for i=0,n_elements( polylines)-1,3 do begin
  polylines[i] = 2
  polylines[i+1] = (i/3)*2
  polylines[i+2] = (i/3)*2+1
endfor
thisModel->add, obj_new( 'IDLgrPolyline', v, polylines=polylines, color=color, name='BoundingBox')

modValue = 1  ;2
;Create dotted lines at every or every other major tick mark, depending on modValue.
v = [0,0,0]
xAxis->GetProperty, tickvalues=tickvalues
for i=0,n_elements( tickvalues)-1 do begin
  if fix( tickvalues[i]*10) mod modValue eq 0 then begin
    v = [[v], [[tickvalues[i], yrange[0], zrange[0]], [tickvalues[i], yrange[1], zrange[0]]]]
    v = [[v], [[tickvalues[i], yrange[1], zrange[0]], [tickvalues[i], yrange[1], zrange[1]]]]
  endif
endfor
yAxis->GetProperty, tickvalues=tickvalues
for i=0,n_elements( tickvalues)-1 do begin
  if fix( tickvalues[i]*10) mod modValue eq 0 then begin
    v = [[v], [[xrange[0], tickvalues[i], zrange[0]], [xrange[1], tickvalues[i], zrange[0]]]]
    v = [[v], [[xrange[1], tickvalues[i], zrange[0]], [xrange[1], tickvalues[i], zrange[1]]]]
  endif
endfor
zAxis->GetProperty, tickvalues=tickvalues
for i=0,n_elements( tickvalues)-1 do begin
  if fix( tickvalues[i]*10) mod modValue eq 0 then begin
    v = [[v], [[xrange[0], yrange[1], tickvalues[i]], [xrange[1], yrange[1], tickvalues[i]]]]
    v = [[v], [[xrange[1], yrange[0], tickvalues[i]], [xrange[1], yrange[1], tickvalues[i]]]]
  endif
endfor
v = v[*,1:*]
n_vertices = (size(v,/dim))[1]
polylines = intarr( n_vertices/2*3)
for i=0,n_elements( polylines)-1,3 do begin
  polylines[i] = 2
  polylines[i+1] = (i/3)*2
  polylines[i+2] = (i/3)*2+1
endfor
thisModel->add, obj_new( 'IDLgrPolyline', v, polylines=polylines, color=color, lines=[1,'1010'x], $
                         xcoord_conv=xs, ycoord_conv=ys, zcoord_conv=zs, name='RuleLines')
END


;*******************************************************************************
; This is a test program.
;*******************************************************************************
;device, get_screen_size=sze
;orb3d_window_vertical_size = sze[1]*0.6
;resolution = [1,1]*orb3d_window_vertical_size
;window = obj_new( 'IDLgrWindow', dimension=resolution,     $
;                  location=[sze[0]-resolution[0]-10, 20],  $
;                  title='Rapid Visualization Environment  (RAVE)')
;view = obj_new( 'IDLgrView', viewplane_rect=[-1,-1,2,2], color=[0,0,0], zclip=[1.0,-1.0])
;model = obj_new( 'IDLgrModel')
;ax=30 & ay=0 & az=30 & scale_factor=1.2
;;model->rotate, [1,0,0], -90
;;model->rotate, [0,1,0], az
;;model->rotate, [1,0,0], ax
;model->scale, scale_factor, scale_factor, scale_factor
;
;units = ' (Re)'
;xtitle = 'X!IGSE!N'+units+'!C!C!M4 Sun'
;ytitle = 'Y!IGSE!N'+units
;ztitle = 'Z!IGSE!N'+units
;xrange = [-10,20]
;yrange = [20,-50]
;zrange = [-20,30]
;view->add, model
;rave_axes, xrange, yrange, zrange, window, model, $
;           xtitle=xtitle, ytitle=ytitle, ztitle=ztitle, xs=xs, ys=ys, zs=zs, color=[255,255,255]
;
;white = [255,255,255]
;lons = 16 & lats = 9 & radius = 0.04               ;radius in normalized coordinates
;mesh_obj, 4, v, p, fltarr(lons+1,lats+1) + radius  ;create a sphere
;globeObj = obj_new( 'IDLgrPolygon', v, polygons=p, style=2, color=white, $
;                    xcoord_conv=[xs[0],1], ycoord_conv=[ys[0],1], zcoord_conv=[zs[0],1])
;
;palette = obj_new( 'IDLgrPalette')
;palette->loadct, 39
;globeObj->SetProperty, palette=palette
;
;test_orientation = 1
;if test_orientation then begin
;  ;As a test of the orientation, illuminate the top half of the sun sector.
;  poly_shades = bytarr(16,9) + 200  ;orange?
;  poly_shades[13,5:8] = 255         ;white
;endif
;for i=0,8 do poly_shades[*,i] = shift( poly_shades[*,i], -5)
;
;n_vertices = (size( v, /dimensions))[1]
;vertexColors = bytarr( n_vertices)
;for i=0,n_elements(p)-1,5 do begin
;  polygonIndex = i/5
;  firstVertexInThisPolygon = p[i+1]
;  vertexColors[ firstVertexInThisPolygon] = poly_shades[ polygonIndex mod 16, polygonIndex/16]
;endfor
;
;globeObj->SetProperty, vert_colors=vertexColors
;
;scModel = obj_new( 'IDLgrModel')
;scModel->add, globeObj
;translation = [20,0,0]
;scModel->translate, xs[1]*translation[0], 0, 0
;model->add, scModel
;
;xobjview, model, /block, scale=1.0, background=[0,0,0]
;stop
;
;window->draw, view
;stop
;END
