;*******************************************************************************
; File: rave_main
;
; Purpose: RAVE (RAPID Visualization Environment) displays color-coded globes representing
;          RAPID electron flux, in a 3-dimensional view.  Each globe moves on a colored line
;          representing the spacecraft's orbit track.  An arrow at each globe represents the
;          magnetic field at that spacecraft.  When the user advances the time slider, the 
;          globes move along their orbit tracks, and display electron data averaged over
;          a user-specified averaging interval at the new time.
;          The 3D display can be rotated and zoomed by left clicking and dragging the mouse.
;          The color coding can be changed by left clicking and dragging on the color bar.
;          Select the "Help" menu item under the "Help" menu (in the upper right-hand corner)
;          for detailed help.
;
;
; Routines:  rave_main               - The main program creates the widget interface and event 
;                                      handlers for RAVE, calls rave.pro to create the 3D graphics
;                                      panel, and initializes reading of data.
;            rave_update             - Wrapper around rave.pro; called by event handler and PAPCO.
;            raveWidgetEventHandler  - Main event handler for widget events.
;            rave3DMouseEventHandler - Dispatches events in the graphics window to 
;                                      colorBarEventHandler, TrackballMgr or raveHelp.
;            setTimeSliderValue      - Time slider helper routine.
;            raveHelp                - Displays help text.
;
; Inputs:  minTimeStr   - The start time string of the orbit tracks as: 'yyyy/ddd-hh:mm:ss' 
;          maxTimeStr   - The stop time string
;          eventTimeStr - The initial time at which the S/C are displayed.
;          run_env      - 'STAND-ALONE', 'TCAD', 'PAPCO'
;          group_leader - When the group_leader widget quits, RAVE will quit too.
;
; Outputs: rave_tlb     - May be useful to the caller - the top level base of the 
;                         widget GUI.
;
; Usage:   RAVE can be run stand-alone, from TCAD or from PAPCO.
;          RAVE requires IDL version 5.6 or greater.
;          To run stand-alone, do:
;          % papco
;          IDL> @rave.bat
;          IDL> rave_main, '2002/223-14:15:00', '2002/223-15:00:00', '2002/223-14:22:30'
;
;
; Author:  LASP/Steve Monk
;
; History: 10/01/03 SM - Release 1.0.
;
;*******************************************************************************
pro rave_main, minTimeStr, maxTimeStr, eventTimeStr, rave_tlb, run_env=run_env, group_leader=group_leader

common rave_common, pos, window, view, model, trackball, old_state, $
                    xs, ys, zs, xrange, yrange, zrange
common rave_main_common, rave_tlb_save

forward_function getKey, string_to_dt, dt_to_string, t_to_str, str_to_t, new_dt
;!EXCEPT = 2  ;Report math errors as they occur, with line numbers.  Good for debugging.
if !version.release lt 5.6 then begin
  print,'RAVE requires IDL version 5.6 or greater!'
  return
endif
if not keyword_set( run_env) then run_env = 'STAND-ALONE'

case run_env of
  
  'STAND-ALONE': begin
    if n_params() lt 3 then begin
      print,'Usage:   rave_main, minTimeStr, maxTimeStr, eventTimeStr'
      print,'Example: rave_main, ''2001/239-04:00:00'', ''2001/239-04:40:00'', ''2001/239-04:01:00''
      return
    endif
    device,decomposed=0
    loadct,39
    papco, /set_only
  end
  
  'TCAD': begin
    ;Create TCAD window, unless it's already up.
    if !D.window eq -1 then begin
      device, get_screen_size=SZE
      tcad_window_vertical_size = SZE[1]*0.3
      resolution = [1,1]*tcad_window_vertical_size
      xsize = SZE[0]*0.95
      ysize = SZE[1] - tcad_window_vertical_size - 25*2 - 70  ;2 window title bars plus fudge
      window, 0, xsize=xsize, xpos=SZE[0]-xsize,$
                 ysize=ysize, ypos=SZE[1]-ysize, title='TCAD'
      call_procedure, 'TCAD_PLOT_LISTS__LOAD', filename='rave_tcad_pl.pro'
      call_procedure, 'TCAD_PLOT__DRAW_PAGE'
    endif
    call_procedure, 'TCAD_DISPLAY__GET_START_STOP_TIMES', start_time_dt, stop_time_dt
    minTimeStr = dt_to_string( start_time_dt)
    maxTimeStr = dt_to_string( stop_time_dt)
    eventTimeStr = minTimeStr
    papco, /set_only
    
    ;User can change times in TCAD, then click RAVE button again; we destroy the rave display, and
    ;create a new one.  (I tried sending a reload command to RAVE with the new times, but the
    ;vertical line display in TCAD fouls up and stays fouled up, despite a fix to TCAD)
    
    if n_elements( rave_tlb_save) gt 0 then begin
      if WIDGET_INFO( rave_tlb_save, /VALID_ID) then begin
        if xregistered( 'rave_main') then begin
          WIDGET_CONTROL, rave_tlb_save, /DESTROY  
        endif
        rave_tlb_save = 0
      endif
    endif
  end
  
  'PAPCO': begin
    if n_params() lt 4 then begin
      print,'rave_main: Wrong number of arguments for PAPCO mode!'
      return
    endif
  end
endcase

;Create the RAVE GUI widget hierarchy.

;Store this state structure in the UVALUE of the top level base (rave_tlb)
state = {minTimeStr                     : minTimeStr,   $
         maxTimeStr                     : maxTimeStr,   $
         event_time                     : STR_TO_T( eventTimeStr),    $
         band                           : 1,            $
         data_type                      : 'IES_CLEANBM',$
         logRangeForColorScaling        : [-1.0, 2.0],  $
         step_size_in_seconds           : 30,           $
         averaging_interval             : 30,           $
         slider_value                   : 0,            $
         generate_time_slider_step_events : 0,          $
         curvature_lines                : 1,            $
         show                           : {ElectronFluxGlobes     : 0, $ ;Set defaults at cw_bgroup calls.
                                           FillDataGaps           : 0, $ ;Ditto
                                           MagneticField          : 0, $ ;Ditto
                                           MagneticFieldCurvature : 0, $ ;Ditto
                                           OrbitTracks            : 0, $ ;Ditto
                                           OrbitProjections       : 0, $ ;Ditto
                                           PitchAngleContours     : 0, $ ;Ditto
                                           PitchAnglePlot         : 0, $ ;Ditto
                                           SphericalHarmonicFit   : 0}, $
         globe_scaling_factor           : 1.4,          $
         movie_filename_base            : '',           $
         movie_frame_number             : 0,            $ 
         run_env                        : run_env,      $
         foreground_color               : byte( [255,255,255]),$
         background_color               : byte( [0,0,0]),      $
         current_draw_widget_size       : 0,            $
         help_is_active                 : 0,            $
         help_button                    : 0L,           $
         rave_tlb                       : 0L,           $
         scene                          : obj_new(),    $
         window                         : obj_new(),    $
         pitch_angle_view               : obj_new(),    $
         pitch_angle_plot_model         : obj_new()}

;Position the RAVE panel approximately, so it won't move too much and cause distraction when we
;later position it exactly (after its size has been determined).
device, get_screen_size=sze
xsize = sze[1]*0.6
ysize = sze[1]*0.6
state.current_draw_widget_size = ysize
rave_tlb = WIDGET_BASE( /ROW, TITLE='Cluster  RAPID  3-D  Visualization', /TLB_KILL_REQUEST_EVENTS, $
                        MBAR=menuBarBase, /TLB_SIZE_EVENTS, uname='RaveTopLevelBase', uvalue=state, $
                        XOFFSET=sze[0] - xsize - 40, YOFFSET=sze[1] - ysize - 40, $
                        event_pro='raveWidgetEventHandler')
state.rave_tlb = rave_tlb
rave_tlb_save = rave_tlb
if keyword_set( GROUP_LEADER) then WIDGET_CONTROL, rave_tlb, GROUP_LEADER=GROUP_LEADER

leftBase = WIDGET_BASE( rave_tlb, /COLUMN)
mainControlsPanel = WIDGET_BASE( leftBase, /ROW, uname='MainControlsPanel')

if !version.release ge 5.6 then begin
  tabWidget = WIDGET_TAB( mainControlsPanel)
  mainControlsLeft  = WIDGET_BASE( tabWidget, title='Initialize',  /COLUMN)
  preferencesWidget = WIDGET_BASE( tabWidget, title='Preferences', /COLUMN)
  runWidget         = WIDGET_BASE( tabWidget, title='Run-Time',    /ROW)
  WIDGET_CONTROL, tabWidget, SET_TAB_CURRENT=2
  mainControlsCenter = WIDGET_BASE( runWidget, /COLUMN)
  mainControlsRight  = WIDGET_BASE( runWidget, /COLUMN)
endif else begin
  mainControlsLeft   = WIDGET_BASE( mainControlsPanel, /COLUMN)
  mainControlsCenter = WIDGET_BASE( mainControlsPanel, /COLUMN)
  mainControlsRight  = WIDGET_BASE( mainControlsPanel, /COLUMN)
endelse

font = '6x13'

;Populate preferences panel. (IDL 5.6 only)
if n_elements( preferencesWidget) gt 0 then begin
  unames = ['Black','White']
  tracking_base = WIDGET_BASE( preferencesWidget, uname='BackgroundTracking', /TRACKING_EVENTS)
  junk = CW_BGROUP2(  tracking_base, label_left='Background:', unames, uvalue='BackgroundColor', $
                      uname='BackgroundColor', button_uvalue=unames, /return_name, column=2, $
                      /EXCLUSIVE, font=font, ids=ids, set_value=0)
  for i=0,n_elements( ids)-1 do WIDGET_CONTROL, ids[i], SET_UNAME=unames[i]
  unames = ['GSE','GSM']
  tracking_base = WIDGET_BASE( preferencesWidget, uname='CoordinatesTracking', /TRACKING_EVENTS)
  junk = CW_BGROUP2(  tracking_base, label_left='Coordinates:', unames, uvalue='Coordinates', $
                      uname='Coordinates', button_uvalue=unames, /return_name, column=2, $
                      /EXCLUSIVE, font=font, ids=ids, set_value=0)
  for i=0,n_elements( ids)-1 do WIDGET_CONTROL, ids[i], SET_UNAME=unames[i]
  WIDGET_CONTROL, ids[1], SENSITIVE=0
  junk = CW_FIELD2( preferencesWidget, title='Curvature Lines:', uvalue='CurvatureLines', $
                    uname='CurvatureLines', xsize=3, value=state.curvature_lines, $
                    font=font, fieldfont=font, /RETURN_EVENTS, /TRACKING_EVENTS)
  junk = CW_TEXT_WITH_ARROWS( preferencesWidget, string( state.globe_scaling_factor, format='(F3.1)'), $
                              font=font, uvalue='GlobeSize', uname='GlobeSize', $
                              label='Globe Size:', /TRACKING_EVENTS)
endif

;Populate mainControlsLeft, now the "Initialize" panel.
junk = WIDGET_LABEL( mainControlsLeft, value='Initialization:', /ALIGN_CENTER, font=font)
mainControlsLeft = WIDGET_BASE( mainControlsLeft, /COLUMN, /FRAME)
timeSelectionBase = WIDGET_BASE( mainControlsLeft, /COLUMN, /FRAME)
junk = CW_FIELD2( timeSelectionBase, title='from:', value=minTimeStr, uname='StartTime', uvalue='StartTime', $
                  font=font, fieldfont=font, /TRACKING_EVENTS)
junk = CW_FIELD2( timeSelectionBase, title='  to:', value=maxTimeStr, uname='StopTime', uvalue='StopTime', $
                  font=font, fieldfont=font, /TRACKING_EVENTS)
dataChoiceBase = WIDGET_BASE( mainControlsLeft, /COLUMN)
junk = WIDGET_LABEL( dataChoiceBase, value='Data Choice:', /ALIGN_CENTER, font=font)
data_choices = ['IES_CLEANBM', 'IES_EPAD_16']  ;Must match same variable in event handler below!
junk = CW_BGROUP2( dataChoiceBase, data_choices, row=n_elements( data_choices), /EXCLUSIVE, font=font, /FRAME, $
                   uname='DataChoice', ids=ids, uvalue='DataChoice')
for i=0,n_elements( ids)-1 do WIDGET_CONTROL, ids[i], /TRACKING_EVENTS, SET_UNAME=data_choices[i]
WIDGET_CONTROL, junk, SET_VALUE=0
;for i=1,n_elements( data_choices)-1 do WIDGET_CONTROL, ids[i], SENSITIVE=0
bandBase = WIDGET_BASE( mainControlsLeft, /ROW)
junk = WIDGET_SLIDER( bandBase, min=0, max=7, value=1, title='energy band', uname='Band', $
                      uvalue='Band', font=font, /DRAG, /TRACKING_EVENTS)
junk = WIDGET_LABEL( bandBase, uname='BandEnergy', value='53-70 keV  ', font=font, /TRACKING_EVENTS)
buttonBase = WIDGET_BASE( mainControlsLeft, /ROW, /ALIGN_CENTER)
junk = WIDGET_BUTTON( buttonBase, value='Reload', uname='Reload', uvalue='Reload',font=font, /TRACKING_EVENTS)


;Populate mainControlsCenter and mainControlsRight, both on the "Run-Time" panel.
junk = WIDGET_LABEL( mainControlsRight, VALUE='Time Step Control:', font=font, /ALIGN_CENTER)
timeBase = WIDGET_BASE( mainControlsRight, /COLUMN, /FRAME)
junk = WIDGET_LABEL( timeBase, VALUE='2003/001-00:00:00', uname='EventTimeLabel', font=font)
junk = WIDGET_SLIDER( timeBase, /SUPPRESS, uname='TimeSlider', uvalue='TimeSlider', /ALIGN_CENTER, /DRAG, $
                      /TRACKING_EVENTS)
buttonBase = WIDGET_BASE( timeBase, /ROW, /ALIGN_CENTER)
animation_button = WIDGET_BUTTON( buttonBase, value='Animate', uvalue='Animate', uname='Animate', $
                                  font=font, /TRACKING_EVENTS)
framedBase = WIDGET_BASE( timeBase, /COLUMN, /FRAME)
junk = WIDGET_LABEL( framedBase, VALUE='(seconds) ', font=font, /ALIGN_RIGHT)
junk = CW_TEXT_WITH_ARROWS( framedBase, '30 ', uvalue='TimeStepInterval', label='Time Step', $
                            uname='TimeStepInterval', /ALIGN_RIGHT, font=font, /TRACKING_EVENTS)
junk = CW_TEXT_WITH_ARROWS( framedBase, '30 ', uvalue='AveragingInterval', label='Averaging Period', $
                            uname='AveragingInterval', /ALIGN_RIGHT, font=font, /TRACKING_EVENTS)
lockButtonBase = WIDGET_BASE( framedBase, /ROW, /ALIGN_CENTER, /TRACKING_EVENTS, uname='LockTogetherBase')
junk = CW_BGROUP2(  lockButtonBase, ['Lock Together'], uvalue='LockTogether', uname='LockTogether', $
                    button_uvalue=['LockTogether'], row=1, /NONEXCLUSIVE, font=font, set_value=[1], $
                    ids=ids)
WIDGET_CONTROL, ids[0], /TRACKING_EVENTS, SET_UNAME='LockTogether'


;showBase = WIDGET_BASE( mainControlsCenter, /COLUMN)
junk = WIDGET_LABEL( mainControlsCenter, value='Show:', /ALIGN_CENTER, font=font)
showBase = WIDGET_BASE( mainControlsCenter, /COLUMN, /FRAME)     ;Moved in place of old color bar.

labels  = ['Electron Flux Globes', 'Fill Globe Data Gaps', 'Pitch Angle Contours', 'Spherical Harmonic Fit']
button_uvalues = ['ElectronFluxGlobes', 'FillDataGaps', 'PitchAngleContours', 'SphericalHarmonicFit']
values = [1,1,1,0]
for i=0,n_elements( values)-1 do begin   ;Set default values of buttons in state.show.
  junk = execute( 'state.show.' + button_uvalues[i] + ' = ' + strtrim( values[i],2))
endfor
junk = CW_BGROUP2( showBase, labels, uvalue='Show1', /NONEXCLUSIVE, font=font, /FRAME, $
                   row=n_elements( labels), set_value=values, ids=ids, button_uvalue=button_uvalues)
;Note: CW_BGROUP doesn't know about TRACKING_EVENTS, and will crash if it gets one.  Had to copy
;the code from $IDL_DIR/lib and modify it to pass through TRACKING_EVENTS.
for i=0,n_elements( ids)-1 do WIDGET_CONTROL, ids[i], /TRACKING_EVENTS, SET_UNAME=button_uvalues[i]

;Disable "Spherical Harmonic Fit" checkbox unless external library is installed.
help, /routine, name='getccSHTLibName', output=output
if strpos( output[ n_elements( output)-1], 'GETCCSHTLIBNAME') ne 0 then begin
  WIDGET_CONTROL, ids[3], SENSITIVE=0
endif else begin
  f = findfile( call_function( 'getccSHTLibName'), count=count)
  if count eq 0 then begin
    WIDGET_CONTROL, ids[3], SENSITIVE=0
  endif
endelse

labels = ['Magnetic Field', 'Magnetic Field Curvature', 'Orbit Tracks', 'Orbit Projections', $
          'Pitch Angle Plot']
button_uvalues = ['MagneticField', 'MagneticFieldCurvature', 'OrbitTracks', 'OrbitProjections', $
                  'PitchAnglePlot']
values = [1,0,1,0,0]
for i=0,n_elements( values)-1 do begin   ;Set default values of buttons in state.show.
  junk = execute( 'state.show.' + button_uvalues[i] + ' = ' + strtrim( values[i],2))
endfor
junk = CW_BGROUP2( showBase, labels, uvalue='Show2', /NONEXCLUSIVE, font=font, $
                   row=n_elements( labels), set_value=values, ids=ids, button_uvalue=button_uvalues)
for i=0,n_elements( ids)-1 do WIDGET_CONTROL, ids[i], /TRACKING_EVENTS, SET_UNAME=button_uvalues[i]

WIDGET_CONTROL, rave_tlb, set_uvalue=state

helpPanel = WIDGET_BASE( leftBase, MAP=0, /COLUMN, uname='HelpPanel', /FRAME)
junk = WIDGET_LABEL( helpPanel, value='Help')
helpText = WIDGET_TEXT( helpPanel, font=font, /WRAP, SCROLL=1, uname='HelpText')
buttonBase = WIDGET_BASE( helpPanel, /ROW, /ALIGN_CENTER)
junk = WIDGET_BUTTON( buttonBase, value='Quit Help', uvalue='QuitHelp',font=font)



;Populate rightBase with 3-D graphics panel
rightBase = WIDGET_BASE( rave_tlb, /COLUMN)
graphicsDisplayHeaderBase = WIDGET_BASE( rightBase, /ROW)
junk = CW_PDMENU( menuBarBase, /MBAR, $
                  ['1\File', $
                   '1\Print', $
                   '0\Jpeg', '0\Tiff', '2\PNG', $  ; '0\Postscript', '2\Encapsulated Postscript', $
                   '1\Save', $
                   '2\Spherical Harmonic Coefficients', $
                   '1\Movie', $
                   '0\Start Saving Frames', '2\Stop Saving Frames', $
                   '2\Quit', $
                   '1\Help', $
                   '2\Help'], $
                   UNAME="MenuBar", UVALUE="MenuBar", /RETURN_FULL_NAME, /HELP, font=font, ids=ids)
                   
;By inspection, the last element in ids is the 'Help.Help' button.  If change help menu, change this too!
state.help_button = ids[ n_elements( ids)-1]
                   
junk = CW_PDMENU( graphicsDisplayHeaderBase, $
                  ['1\Views', $
                   '0\Standard', '0\Southward (Top View)', '0\Tailward', '2\X-Z Plane'], $
;                   '0\Standard', '0\Southward (Top View)', '0\Tailward', '0\X-Z Plane', '2\Reverse'], $
                  UVALUE="ViewMenu", /RETURN_FULL_NAME, font=font)
drawBase = WIDGET_BASE( rightBase)
wDraw = WIDGET_DRAW( drawBase, GRAPHICS_LEVEL=2, /BUTTON_EVENTS, RETAIN=0, /EXPOSE_EVENTS, $
                     XSIZE=xsize, YSIZE=ysize, uname='DrawWidget', uvalue='DrawWidget', $
                     EVENT_FUNC='rave3DMouseEventHandler')
if !version.release ge 5.6 then WIDGET_CONTROL, wDraw, /DRAW_KEYBOARD_EVENTS  ;arrow key shortcuts, etc.
WIDGET_CONTROL, rave_tlb, /REALIZE, MAP=0, UPDATE=0

WIDGET_CONTROL, wDraw, GET_VALUE=window
scene = obj_new( 'IDLgrScene')
view  = obj_new( 'IDLgrView')
model = obj_new( 'IDLgrModel')
view->add, model
scene->add, view
state.scene = scene
state.window = window
WIDGET_CONTROL, rave_tlb, set_uvalue=state
TrackballMgr, graphicsDisplayHeaderBase, window, scene, model, xsize, ysize, toolTipfont=font

WIDGET_CONTROL, rave_tlb, MAP=1, update=1

;Position the RAVE panel near the bottom right corner of the screen.
geom = WIDGET_INFO( rave_tlb, /GEOMETRY)
xsize = geom.scr_xsize
ysize = geom.scr_ysize
device, get_screen_size=sze
WIDGET_CONTROL, rave_tlb, MAP=0
WIDGET_CONTROL, rave_tlb, XOFFSET=sze[0] - xsize - 30, YOFFSET=sze[1] - ysize - 40
WIDGET_CONTROL, rave_tlb, MAP=1

;These calls are necessary because these *__realize routines aren't called late enough via
;IDL's automated call-back scheme, and must be called again here.
cw_text_with_arrows__realize, WIDGET_INFO( rave_tlb, FIND_BY_UNAME='TimeStepInterval'),  /draw
cw_text_with_arrows__realize, WIDGET_INFO( rave_tlb, FIND_BY_UNAME='AveragingInterval'), /draw
cw_text_with_arrows__realize, WIDGET_INFO( rave_tlb, FIND_BY_UNAME='GlobeSize'),         /draw

;Load Cluster data and draw 3-D display.
if minTimeStr ne '' and maxTimeStr ne '' then begin
  rave_update, state, /REINIT
endif

if run_env eq 'PAPCO' then begin
  XMANAGER, 'rave_main', rave_tlb, /NO_BLOCK, /JUST_REG, EVENT_HANDLER='raveWidgetEventHandler'
endif else begin
  XMANAGER, 'rave_main', rave_tlb, /NO_BLOCK, EVENT_HANDLER='raveWidgetEventHandler'
endelse

END


;*******************************************************************************
; Routine:  rave3DMouseEventHandler
; Purpose:  Main 3-D WIDGET_DRAW mouse event handler.
;           Distinguish between mouse select events (color bar stretch/contract)
;           and rotate/translate/zoom events.
;*******************************************************************************

function rave3DMouseEventHandler, ev

common rave3DMouseEventHandler_common, directManipulationIsInProgress, buttonDownTime, colorBar, colorBarAxis
common rave_common, pos, window, view, model, trackball, old_state, $
                    xs, ys, zs, xrange, yrange, zrange

if n_elements( directManipulationIsInProgress) eq 0 then directManipulationIsInProgress = 0

if tag_names( ev, /STRUCTURE_NAME) eq 'WIDGET_DRAW' then begin
  if ev.type eq 4 then begin  ;Expose event.
    window->Draw, view    
    return,0
  endif
  if ev.type eq 0 and ev.press eq 1 then begin      ;Left button press.
    buttonDownTime = systime(1)
    location = [ev.x, ev.y]
    objects = window->Select( view, location)
    if obj_valid( objects[0]) then begin
      objects[0]->GetProperty, name=name
      if name eq 'ColorBar' then begin
        colorBar = objects[0]
        colorBarEventHandler, ev, colorBar
        directManipulationIsInProgress = 1
        return,0
      endif
    endif
  endif 
  
  ;This determines if it was a quick press and release click.   If so, see if there's an object
  ;in the 3-D panel under the click point, and get help on it.
  if ev.type eq 1 and ev.release eq 1 then begin  ;Left button release.
    if systime(1) - buttonDownTime le 1 then begin
      location = [ev.x, ev.y]
      objects = window->Select( view, location)
      if obj_valid( objects[0]) then begin
        objects[0]->GetProperty, name=name3D
        WIDGET_CONTROL, ev.top, get_uvalue=state
        raveHelp, state, name3D=name3D, obj3D=objects[0]
      endif
    endif
  endif
  
  if directManipulationIsInProgress then begin
    if ev.type eq 1 and ev.release eq 1 then begin  ;Left button release.
      directManipulationIsInProgress = 0
    endif
    colorBarEventHandler, ev, colorBar
    window->draw, view
    return,0
  endif
endif

result = TrackballMgr__handleDrawEvent( ev)
return, result
END


;*******************************************************************************
pro raveWidgetEventHandler, event             ;Main controls event handler
;*******************************************************************************

common rave_common, pos, window, view, model, trackball, old_state, $
                    xs, ys, zs, xrange, yrange, zrange

if (tag_names( event, /STRUCTURE_NAME) eq 'WIDGET_KILL_REQUEST') then begin
  WIDGET_CONTROL, event.top, /DESTROY
  return
endif

rave_tlb = WIDGET_INFO( event.top, FIND_BY_UNAME='RaveTopLevelBase')
WIDGET_CONTROL, rave_tlb, get_uvalue=state

if (tag_names( event, /STRUCTURE_NAME) eq 'WIDGET_BASE') then begin
  available_draw_widget_area = event.y - 100
  draw_widget = WIDGET_INFO( event.top, FIND_BY_UNAME='DrawWidget')
  WIDGET_CONTROL, draw_widget, DRAW_XSIZE=available_draw_widget_area, $
                               DRAW_YSIZE=available_draw_widget_area
  state.current_draw_widget_size = available_draw_widget_area
  WIDGET_CONTROL, rave_tlb, set_uvalue=state
  return
endif

if (tag_names( event, /STRUCTURE_NAME) eq 'WIDGET_TRACKING') then begin
  raveHelp, state, event=event
  return
endif


WIDGET_CONTROL, event.id, GET_UVALUE=uval
if (size( uval))[1] ne 7 then return       ;All known widget uvalues should be strings.
parent = WIDGET_INFO( event.id, /PARENT)
redraw = 0
reinit = 0

case uval of
  
  'TimeSlider': begin
    minTime = STR_TO_T( state.minTimeStr)
    maxTime = STR_TO_T( state.maxTimeStr)
    event_time_label_widget = WIDGET_INFO( parent, FIND_BY_UNAME='EventTimeLabel')
    
    if event.drag eq 1 then begin   ;User is dragging slider.
      offset_from_minTime =  event.value/100.0 * (maxTime - minTime)
      offset_in_step_size_units = round( offset_from_minTime / (state.step_size_in_seconds/86400.))
      state.event_time = minTime + (offset_in_step_size_units * state.step_size_in_seconds)/86400.
      ;state.event_time = minTime + event.value/100.0 * (maxTime - minTime)
      state.slider_value = event.value
      WIDGET_CONTROL, event_time_label_widget, set_value=T_TO_STR( state.event_time)
      WIDGET_CONTROL, rave_tlb, set_uvalue=state
      break
    endif
    
    update_event_time = 0
    
    if state.generate_time_slider_step_events then begin
      ;This mechanism generates a series of events, just as if the user were clicking right of
      ;the slider.  Can be used to step continuously through a time range, while automatically
      ;generating movie frames, spherical harmonic coefficients, etc.
      new_event_time = state.event_time + state.step_size_in_seconds / 86400.
      if new_event_time lt maxTime then begin
        update_event_time = 1
        event = {WIDGET_SLIDER, id: event.id, top: rave_tlb, handler: rave_tlb, $
                 value: 0, drag: 2}
        ;Note: We don't specify slider value, because slider position is set below in the
        ;update_event_time block.  Set drag=2 to identify that this is a generated event.
        ;Thus if the last generated event is queued, but state.generate_time_slider_step_events
        ;has been turned off, we will ignore this last event.
        ;print,'Sending time slider step event, current state.event_time: ' + t_to_str( state.event_time)
        WIDGET_CONTROL, event.id, SEND_EVENT=event
      endif else begin
        animate_button = WIDGET_INFO( WIDGET_INFO( parent, /parent), FIND_BY_UNAME='Animate')
        WIDGET_CONTROL, animate_button, set_value='Animate'
        state.generate_time_slider_step_events = 0
        WIDGET_CONTROL, rave_tlb, set_uvalue=state
      endelse
    endif else if abs( event.value - state.slider_value) eq 10 and event.drag ne 2 then begin  
      ;User clicked next to slider.  The default slider ranges from 0 to 100, with 10 the
      ;default interval when user clicks next to the slider, as opposed to dragging it.
      increment_sign = (event.value gt state.slider_value)*2 - 1
      new_event_time = state.event_time + increment_sign * state.step_size_in_seconds / 86400.
      if new_event_time gt maxTime or new_event_time lt minTime then break
      update_event_time = 1      
    endif else if event.drag ne 2 then begin  ;End of a drag, ignore last generated event (drag=2)
      offset_from_minTime =  event.value/100.0 * (maxTime - minTime)
      offset_in_step_size_units = round( offset_from_minTime / (state.step_size_in_seconds/86400.))
      new_event_time = minTime + (offset_in_step_size_units * state.step_size_in_seconds)/86400.
      ;new_event_time = minTime + event.value/100.0 * (maxTime - minTime)
      update_event_time = 1
    endif
    
    if update_event_time then begin
      ;print,'Updating slider and event time widget to: ' + t_to_str( new_event_time)
      state.event_time = new_event_time
      WIDGET_CONTROL, event_time_label_widget, set_value=T_TO_STR( new_event_time)
      new_slider_value = fix(((new_event_time - minTime) / (maxTime - minTime)) * 100)
      WIDGET_CONTROL, event.id, set_value=new_slider_value
      state.slider_value = new_slider_value
      WIDGET_CONTROL, rave_tlb, set_uvalue=state
      redraw = 1
    endif
  
  end
  
  'TimeStepInterval': begin
    if !version.os_name eq 'Mac OS X' then begin
      WIDGET_CONTROL, event.id, map=0
      WIDGET_CONTROL, event.id, map=1
      id = WIDGET_INFO( rave_tlb, FIND_BY_UNAME='AveragingInterval')
      WIDGET_CONTROL, id, map=0
      WIDGET_CONTROL, id, map=1
    endif
    WIDGET_CONTROL, event.id, get_value=string_value
    step_size_in_seconds = fix( string_value[0])
    fac = 1
    if event.up_arrow then fac   = 2
    if event.down_arrow then fac = 0.5
    step_size_in_seconds = step_size_in_seconds * fac
    if step_size_in_seconds ge 1000 then begin
      step_size_in_seconds = 999
      junk = WIDGET_MESSAGE( 'The largest allowed step size is 999 seconds!', /INFO)
    endif
    if step_size_in_seconds lt 1 then begin
      step_size_in_seconds = 1
      junk = WIDGET_MESSAGE( 'The smallest allowed step size is 1 second!', /INFO)
    endif
    if step_size_in_seconds eq 32 then step_size_in_seconds = 30
    if step_size_in_seconds eq 15 then step_size_in_seconds = 16
    string_value = strtrim( string( format='(I3)', step_size_in_seconds), 2)
    WIDGET_CONTROL, event.id, set_value=string_value
    WIDGET_CONTROL, WIDGET_INFO( parent, FIND_BY_UNAME='LockTogether'), get_value=lock_together
    state.step_size_in_seconds = step_size_in_seconds
    if lock_together eq 1 then begin
      state.averaging_interval = step_size_in_seconds
      WIDGET_CONTROL, WIDGET_INFO( parent, FIND_BY_UNAME='AveragingInterval'), set_value=string_value
    endif
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
    redraw = 1
  end
  
  'AveragingInterval': begin
    if !version.os_name eq 'Mac OS X' then begin
      WIDGET_CONTROL, event.id, map=0
      WIDGET_CONTROL, event.id, map=1
      id = WIDGET_INFO( rave_tlb, FIND_BY_UNAME='TimeStepInterval')
      WIDGET_CONTROL, id, map=0
      WIDGET_CONTROL, id, map=1
    endif
    WIDGET_CONTROL, event.id, get_value=string_value
    averaging_interval = float( string_value[0])
    fac = 1
    if event.up_arrow then fac   = 2
    if event.down_arrow then fac = 0.5
    averaging_interval = averaging_interval * fac
    if averaging_interval ge 1000 then begin
      averaging_interval = 999
      junk = WIDGET_MESSAGE( 'The largest allowed averaging_interval is 999 seconds!', /INFO)
    endif
    if averaging_interval lt 1 then begin
      averaging_interval = 1
      junk = WIDGET_MESSAGE( 'The smallest allowed averaging_interval is 1 second!', /INFO)
    endif
    if averaging_interval eq 32 then averaging_interval = 30
    if averaging_interval eq 15 then averaging_interval = 16
    string_value = strtrim( string( format='(I3)', averaging_interval), 2)    
    strip_trailing_zeros, string_value
    WIDGET_CONTROL, event.id, set_value=string_value
    WIDGET_CONTROL, WIDGET_INFO( parent, FIND_BY_UNAME='LockTogether'), get_value=lock_together
    state.averaging_interval = averaging_interval
    if lock_together eq 1 then begin
      state.step_size_in_seconds = averaging_interval
      WIDGET_CONTROL, WIDGET_INFO( parent, FIND_BY_UNAME='TimeStepInterval'), set_value=string_value
    endif
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
    redraw = 1
  end
  
  'Animate': begin
    WIDGET_CONTROL, event.id, get_value=value
    if value eq 'Animate' then begin
      value = ' Stop '
      WIDGET_CONTROL, event.id, set_value=value
      ;See rave_main ("state.generate_time_slider_step_events") for an explanation of this nifty feature.
      slider_id = WIDGET_INFO( rave_tlb, FIND_BY_UNAME='TimeSlider')
      WIDGET_CONTROL, slider_id, get_value=slider_value
      if slider_value lt 99 then begin
        state.generate_time_slider_step_events = 1
        WIDGET_CONTROL, rave_tlb, set_uvalue=state
        event = {WIDGET_SLIDER, id: slider_id, top: rave_tlb, handler: rave_tlb, $
                 value: state.slider_value + 10, drag: 0} 
        WIDGET_CONTROL, event.id, SEND_EVENT=event
      endif
    endif else begin
      value = 'Animate'
      WIDGET_CONTROL, event.id, set_value=value
      state.generate_time_slider_step_events = 0
      WIDGET_CONTROL, rave_tlb, set_uvalue=state
    endelse
  end
  
  'GlobeSize': begin
    WIDGET_CONTROL, event.id, get_value=string_value
    scaling = float( string_value[0])
    if event.up_arrow then scaling   = scaling + 0.2
    if event.down_arrow then begin
      if scaling le 0.2 then begin
        scaling = scaling / 2
      endif else begin
        scaling = scaling - 0.2
      endelse
    endif
    string_value = strtrim( string( format='(F4.2)', scaling), 2)    
    strip_trailing_zeros, string_value
    WIDGET_CONTROL, event.id, set_value=string_value
    state.globe_scaling_factor = scaling
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
    redraw = 1
  end
  
  'Show1': begin
    W = where( tag_names( state.show) eq strupcase( event.value))
    state.show.(W[0]) = event.select
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
    redraw = 1
  end
  
  'Show2': begin
    W = where( tag_names( state.show) eq strupcase( event.value))
    state.show.(W[0]) = event.select
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
    redraw = 1
  end
  
  'CurvatureLines': begin
    state.curvature_lines = event.value
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
  end

  'DataChoice': begin
    data_choices = ['IES_CLEANBM', 'IES_EPAD_16']  ;Must match same variable in rave_main above!
    state.data_type = data_choices[ event.value]
    band_widget              = WIDGET_INFO( rave_tlb, FIND_BY_UNAME='Band')
    band_energy_label_widget = WIDGET_INFO( rave_tlb, FIND_BY_UNAME='BandEnergy')
    if state.data_type eq 'IES_CLEANBM' then begin
      WIDGET_CONTROL, band_widget, set_slider_max=7, set_value=1
      WIDGET_CONTROL, band_energy_label_widget, set_value='53-70 keV'
      state.band = 1
    endif else begin
      WIDGET_CONTROL, band_widget, set_slider_max=1, set_value=0
      WIDGET_CONTROL, band_energy_label_widget, set_value='92-202 keV'
      state.band = 0
    endelse
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
  end

  'Band': begin
    band_energy_label_widget = WIDGET_INFO( rave_tlb, FIND_BY_UNAME='BandEnergy')
    ;for i=0,5 do print,[input_header.ch_positions[0,i,0,0],input_header.ch_positions[1,i,0,0]]*2.2 + 6.5
    if state.data_type eq 'IES_CLEANBM' then begin
      labels = ['42-53', '53-70', '70-97', '97-127', '127-171', '171-237', '237-325', '325-453'] + ' keV'
    endif else begin
      labels = ['92-202', '202-411'] + ' keV'
    endelse
    WIDGET_CONTROL, band_energy_label_widget, set_value=labels[ event.value]
  end
  
  'Reload': begin
    startTimeWidget  = WIDGET_INFO( rave_tlb, FIND_BY_UNAME='StartTime')
    stopTimeWidget   = WIDGET_INFO( rave_tlb, FIND_BY_UNAME='StopTime')
    dataChoiceWidget = WIDGET_INFO( rave_tlb, FIND_BY_UNAME='DataChoice')
    bandWidget       = WIDGET_INFO( rave_tlb, FIND_BY_UNAME='Band')
    WIDGET_CONTROL, startTimeWidget, get_value=start_time_str
    WIDGET_CONTROL, stopTimeWidget,  get_value=stop_time_str
    validateStartStopTimes, start_time_str[0], stop_time_str[0], minTime, maxTime, ok
    if not ok then return
    state.minTimeStr = T_TO_STR( minTime)
    state.maxTimeStr = T_TO_STR( maxTime)
    WIDGET_CONTROL, bandWidget, get_value=band
    state.band = band
    if not (state.event_time ge minTime and state.event_time le maxTime) then begin
      state.event_time = minTime + state.step_size_in_seconds/86400.
    endif
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
    setTimeSliderValue, rave_tlb, state.event_time
    ;Automatically resize the globe_scaling_factor to be appropriate for the time range and
    ;size of the bounding box.  Log interpolate between known reasonable timeRange/globeSize pairs.
    time_ranges_in_hours = [1,2,4,8,16]
    scaling_factors = alog10([1.6,1.2,0.8,0.4,0.3])
    new_scaling_factor = 10^interpol( scaling_factors, time_ranges_in_hours, (maxTime-minTime)*24)
    new_scaling_factor = round( new_scaling_factor*10)/10.0 > 0.1   ;Round to nearest 0.1
    new_scaling_factor = new_scaling_factor - 0.4                   ;Fudge for Dan's preferences
    state.globe_scaling_factor = new_scaling_factor
    string_value = strtrim( string( format='(F4.2)', new_scaling_factor), 2)    
    strip_trailing_zeros, string_value
    WIDGET_CONTROL, WIDGET_INFO( state.rave_tlb, FIND_BY_UNAME='GlobeSize'), set_value=string_value
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
    redraw = 1
    reinit = 1
  end
  
  'ColorBar': begin
    state.logRangeForColorScaling = event.range
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
    redraw = 1
  end
  
  'MenuBar': begin
    
    if event.value eq 'Help.Help' then begin
      ;Note: I put this code here instead of in raveMenuBarHandler because
      ;if it needs changing, so will the 'QuitHelp' block below.
      state.help_is_active = 1
      helpPanel = WIDGET_INFO( rave_tlb, FIND_BY_UNAME='HelpPanel')
      scr_xsize = (WIDGET_INFO( helpPanel, /GEOM)).scr_xsize - 10
      helpText = WIDGET_INFO( helpPanel, FIND_BY_UNAME='HelpText')
      WIDGET_CONTROL, helpText, scr_xsize=scr_xsize, scr_ysize=150, MAP=1
      WIDGET_CONTROL, rave_tlb, set_uvalue=state
      raveHelp, state, activate=1
      WIDGET_CONTROL, state.help_button, set_uvalue='Help.Quit', set_value='Quit'
      
    endif else if event.value eq 'Help.Quit' then begin
      state.help_is_active = 0
      WIDGET_CONTROL, rave_tlb, set_uvalue=state
      WIDGET_CONTROL, WIDGET_INFO( rave_tlb, FIND_BY_UNAME='HelpPanel'), MAP=0
      WIDGET_CONTROL, state.help_button, set_uvalue='Help.Help', set_value='Help'
      
    endif else begin   ;Must be a "File" menu choice.
      raveMenuBarHandler, event.value, rave_tlb, redraw
      if WIDGET_INFO( rave_tlb, /VALID_ID) then begin
        ;raveMenuBarHandler may have changed the state (e.g. start/stop movie), so get it again.
        ;rave_tlb will be invalid if event.value was 'File.Quit'.
        WIDGET_CONTROL, rave_tlb, get_uvalue=state
      endif
    endelse
  end
  
  'QuitHelp': begin
    state.help_is_active = 0
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
    WIDGET_CONTROL, WIDGET_INFO( rave_tlb, FIND_BY_UNAME='HelpPanel'), MAP=0
    WIDGET_CONTROL, state.help_button, set_uvalue='Help.Help', set_value='Help'
  end
  
  'ViewMenu': begin
    ;event.value is something like "Views.Standard".  Pass "Standard" to raveChangeView.
    raveChangeView, strmid( event.value, strpos( event.value, '.')+1), model
    redraw = 1
  end
  
  'BackgroundColor': begin
    state.background_color = (event.value eq 'Black') ? [1,1,1] : [255,255,255]
    state.foreground_color = (event.value eq 'Black') ? [255,255,255] : [1,1,1]
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
    raveChangeBackgroundColor, state
    redraw = 1
  end
  
  'DrawWidget': begin
    if tag_names( event, /STRUCTURE_NAME) eq 'TRACKBALLDRAGFINISHED' then begin
      ;The trackballmgr event handler sends this event when the user lets up on
      ;the mouse button after dragging it around (rotate, translate or scale).
      ;Trackballmgr has already redrawn the display with window->Draw, view, 
      ;but we do it again via rave.pro so as to output a movie frame, if enabled.
      redraw = 1
    endif else begin
      ;WIDGET_DRAW event forwarded to us from trackballmgr: a key press/release event
      ;with a field added containing the rotate/translate/zoom mode.  Note that after
      ;the user clicks on one of the rotate/translate/zoom buttons, the arrow keys change
      ;which of these buttons has focus, instead of generating a widget_draw event.
      ;Click on the draw window to give it back focus for the arrow keys.
      if event.press eq 1 and event.type eq 6 then begin  ;non-ASCII key was pressed
        case event.mode of
          0: view_str = 'Rotate'
          1: view_str = 'Translate'
          2: view_str = 'Zoom'
          else: view_str = ''
        endcase
        case event.key of
          5: view_str = view_str + 'Left'    ;left-arrow key
          6: view_str = view_str + 'Right'   ;right-arrow key
          7: view_str = view_str + 'Up'      ;up-arrow key
          8: view_str = view_str + 'Down'    ;down-arrow key
          else: return
        endcase
        ;Note that after a short-cut arrow key operation, we don't usually set redraw=1.
        ;The user can click once in the draw window without moving anything to cause
        ;redraw=1 and a movie frame, if desired.
        raveChangeView, view_str, model
        window->draw, state.scene
      endif
    endelse
  end
  else: ;
  
endcase

if redraw then rave_update, state, reinit=reinit
END


;*******************************************************************************
pro rave_update, state, reinit=reinit, set_new_event_time=set_new_event_time
;*******************************************************************************

WIDGET_CONTROL, state.rave_tlb, set_uvalue=state

if keyword_set( reinit) or  keyword_set( set_new_event_time) then begin
  new_event_time = state.event_time
  minTime = STR_TO_T( state.minTimeStr)
  maxTime = STR_TO_T( state.maxTimeStr)
  event_time_label_widget = WIDGET_INFO( state.rave_tlb, FIND_BY_UNAME='EventTimeLabel')
  time_slider_widget = WIDGET_INFO( state.rave_tlb, FIND_BY_UNAME='TimeSlider')
  
  WIDGET_CONTROL, event_time_label_widget, set_value=T_TO_STR( new_event_time)
  new_slider_value = fix(((new_event_time - minTime) / (maxTime - minTime)) * 100)
  WIDGET_CONTROL, time_slider_widget, set_value=new_slider_value
  state.slider_value = new_slider_value
  WIDGET_CONTROL, state.rave_tlb, set_uvalue=state
endif

if keyword_set( reinit) eq 1 then begin
  WIDGET_CONTROL, /HOURGLASS
  start_time = STR_TO_T( state.minTimeStr)
  stop_time  = STR_TO_T( state.maxTimeStr)
  getMagData, restoreStartTime=start_time, restoreStopTime=stop_time
  print,'Calling getRapidData for: ' + state.minTimeStr + ' - ' + state.maxTimeStr
  getRapidData, restoreStartTime=start_time, restoreStopTime=stop_time, $
                data_type=state.data_type, band=state.band
endif

rave, state, initialize=reinit
  
;Send an event to make the cursor go from hourglass back to normal.
if reinit eq 1 then begin
  WIDGET_CONTROL, state.rave_tlb, SEND_EVENT={id: state.rave_tlb, top: state.rave_tlb, $
                                              handler: state.rave_tlb}
endif

;Draw two vertical lines on the time-series display, which show the averaging interval
;and the time.

case state.run_env of
  
  'STAND-ALONE': begin
    ;  No time-series display, so nothing to do.
  end
  
  'TCAD': begin
    wset,0
    ;Note: if ever do direct graphics in RAVE, will have to save/restore system variables between
    ;RAVE and TCAD.  Not necessary now because RAVE uses object graphics exclusively.
    ;Whenever relinquishing control to user, the TCAD window's system variables should be restored
    ;by calling a to-be-written TCAD routine to get them, and TCAD should save its system variables
    ;to same common block or state structure after drawing so they're always current.  Don't make the
    ;mistake of storing TCAD's system variables in RAVE or visa-versa - this led to a bug before I
    ;got rid of the system variables save/restore scheme entirely.
    call_procedure, 'TCAD_DISPLAY__GET_START_STOP_TIMES', start_time_dt, stop_time_dt
    averaging_window = [state.event_time - state.averaging_interval/2.0/86400., $
                        state.event_time + state.averaging_interval/2.0/86400.]
    averaging_window_dt = [NEW_DT( averaging_window[0]), NEW_DT( averaging_window[1])]
    call_procedure, 'SPECIAL_CURSOR_NONMODAL', start_time_dt, stop_time_dt, $
                     averaging_window_dt, time_label_format='h$:m$:s$'
  end
  
  'PAPCO': begin
    ; Don't know if possible to draw averaging window lines in PAPCO...
  end
endcase
  
if state.movie_filename_base ne '' then begin
  cd,current=cwd
  if state.run_env eq 'TCAD' then begin  ;Combine RAVE and TCAD images into one image, then write PNG file.
      
    ;Get RAVE image into a pixel array, then add side margins.
    ;Notes: 1) resolution 1.4 factor makes color bar axis clearer than with 1.0 factor.
    ;       2) side_margin 0.1 factor makes each side bar width 10% of current RAVE image width.
    ;       3) height = width/2.5 sets aspect ratio for TCAD image.
    state.window->getProperty, palette=palette
    buffer = obj_new( 'IDLgrBuffer', dim=[1,1]*state.current_draw_widget_size, $
                      color_model=1, n_colors=256, palette=palette, resolution=[1,1]*2.54/72/1.4)
    buffer->Draw, state.scene
    imageObj = buffer->Read()
    imageObj->GetProperty, data=rave_image
    obj_destroy, [imageObj, buffer]
    side_margin = bytarr( 0.1 * state.current_draw_widget_size, state.current_draw_widget_size)
    rave_image = [side_margin, rave_image, side_margin]
      
    ;Get TCAD image into a pixel array.
    call_procedure, 'SPECIAL_CURSOR_NONMODAL', /erase
    P=!P & X=!X & Y=!Y & Z=!Z
    width = (size( rave_image, /DIM))[0]
    height = round( width/2.5)
    ;height = round( width/3.0)
    height = height + (10 - (height mod 10))  ;Make height even multiple of 10 pixels
    set_plot,'Z'
    device, z_buffering=0, set_resolution=[ width, height]
    call_procedure, 'TCAD_PLOT__DRAW_PAGE'
    call_procedure, 'SPECIAL_CURSOR_NONMODAL', start_time_dt, stop_time_dt, $
                    averaging_window_dt, time_label_format='h$:m$:s$'
    tcad_image = tvrd()      
    call_procedure, 'SPECIAL_CURSOR_NONMODAL', /erase
    device, /close
    set_plot, 'X'
    !P=P & !X=X & !Y=Y & !Z=Z
    call_procedure, 'SPECIAL_CURSOR_NONMODAL', start_time_dt, stop_time_dt, $
                    averaging_window_dt, time_label_format='h$:m$:s$'
    
    ;Combine the RAVE image and the TCAD image into one image.
    combo_image = [[rave_image],[tcad_image]]
    filename = state.movie_filename_base + '_combo_' + string( state.movie_frame_number, format='(I3.3)') + '.png'
    palette->GetProperty, red_values=r, green_values=g, blue_values=b
    write_png, filename, combo_image, r, g, b
    ;spawn, 'xv ' + cwd + papco_get_pathsep() + filename + ' &'
      
  endif else begin
      
    filename = state.movie_filename_base + '_rave_' + strtrim( state.movie_frame_number,2) + '.png'
    raveMenuBarHandler, 'File.Print.PNG', state.rave_tlb, print_filename=filename
      
  endelse
    
  print, 'Wrote ' + cwd + papco_get_pathsep() + filename
  state.movie_frame_number = state.movie_frame_number + 1
  WIDGET_CONTROL, state.rave_tlb, set_uvalue=state
endif

END


;*******************************************************************************
pro setTimeSliderValue, rave_tlb, event_time
;*******************************************************************************
  WIDGET_CONTROL, rave_tlb, get_uvalue=state
  minTime = STR_TO_T( state.minTimeStr)
  maxTime = STR_TO_T( state.maxTimeStr)
  slider_value = fix( (event_time - minTime) / (maxTime - minTime) * 100)
  timeSliderWidget = WIDGET_INFO( rave_tlb, FIND_BY_UNAME='TimeSlider')
  WIDGET_CONTROL, timeSliderWidget, set_value=slider_value
  state.event_time = event_time
  state.slider_value = slider_value
  ;print,'setTimeSliderValue: set value to ' + strtrim( slider_value, 2) + ', ' + T_TO_STR( event_time)
  WIDGET_CONTROL, WIDGET_INFO( rave_tlb, FIND_BY_UNAME='EventTimeLabel'), $
                  set_value=T_TO_STR( event_time)
  WIDGET_CONTROL, rave_tlb, set_uvalue=state
END


;*******************************************************************************
pro raveHelp, state, event=event, name3D=name3D, obj3D=obj3D, activate=activate
;*******************************************************************************

text = 'Move the mouse over a control widget above, or click on an object in the 3D graphics window.'
fgcolor = state.foreground_color[0] eq 0 ? 'black' : 'white'

if n_elements( activate) eq 1 then begin
  if activate then begin
    helpText = WIDGET_INFO( state.rave_tlb, FIND_BY_UNAME='HelpText')
    WIDGET_CONTROL, helpText, set_value=text
  endif
  return
endif

if keyword_set( event) then begin
  helpText = WIDGET_INFO( event.top, FIND_BY_UNAME='HelpText')
  if event.enter eq 0 then return
  uname = WIDGET_INFO( event.id, /UNAME)
  text = ''
  switch uname of
  
    'Animate': begin
      text = $
        ['Click "Animate" to continuously step through the entire time range, as if you continued clicking ' + $
         'to the right of the "Time Step Control" slider.   The "Animate" button changes to "Stop" ' + $
         'so you can stop it at any time.', $
         'This is useful for generating a movie, after "File/Start Saving Frames" has been selected.']
      break
    end
    
    'AveragingInterval': begin
      text = $
        ['"Averaging Period" changes the width of a window centered around the current time, over which ' + $
         'the magnetic field and electron flux data are averaged.',$
         'Click on the up-arrow to double the width, or on the down-arrow to half the width.  ' + $
         'Or edit the text field and hit return.', $
         'The resolution of the underlying data is 4 seconds for both the magnetic field data and the electron data.', $
         "If the averaging interval for the electron data isn't long enough, many data gaps (black) will " + $
         'be present when the  "Fill Data Gaps" check box is cleared.']
      break
    end
    
    'BackgroundTracking': begin
      text = $
        'Check "Black" or "White" for the background color.  A white background is useful with File/Print.'
      break
    end
    
    'Band':
    'BandEnergy': begin
      text = $
        'This is the energy range of the data shown on the electron flux globes.'
      break
    end
    
    'CoordinatesTracking': begin
      text = 'GSE coordinates are used, in units of Re.'
      break
    end
    
    'CurvatureLines': begin
      n = state.curvature_lines
      text = ['Enter the number of Magnetic Field Curvature vectors and curves ' + $
              'to show simultaneously, and hit return.', $
              'Currently only the vectors and curve for the last ' + $
              (n le 1 ? 'time step is' : strtrim( n,2) + ' time steps are') + ' shown.']
      break
    end
    
    'ElectronFluxGlobes': begin
      text = $
        'Check the "Electron Flux Globes" checkbox to display color-coded RAPID electron flux on spheres ' + $
        'representing the 4 Cluster spacecraft.  The color bar shows the colors associated with the flux ' + $
        'levels in units of [counts]s-1cm-2sr-1keV-1. ' + $
        'The data displayed on the globes is averaged over the number of seconds indicated in the ' + $
        '"Averaging Period" box.'
      break
    end
    
    'FillDataGaps': begin
      text = 'Check the "Fill Data Gaps" checkbox to automatically fill in missing or zero data on the ' + $
             'electron globes by averaging adjacent non-zero data.  To decrease the amount of filled ' + $
             'data, increase the averaging period.'  
      break
    end
    
    'GlobeSize': begin
      text = 'Click on the up-arrow to increase the electron globe size, or on the down-arrow ' + $
        'to decrease it.   Or edit the text field and hit return.'
      break
    end
    
    'IES_CLEANBM': begin
      text = 'IES_CLEANBM data is rarely available; it has 16 azimuthal sectors, 9 elevations and ' + $
        '8 energy bands.'
      break
    end
    
    'IES_EPAD_16': begin
      text = 'IES_EPAD_16 data is always available; it has 16 azimuthal sectors, 3 elevations ' + $
        'and 2 energy bands.   The on-board software chooses which 3 of the 9 elevations to sample ' + $
        'for each spin.  These often vary enough so that when averaging over several spins, near-full ' + $
        'elevation coverage may be obtained.'
      break
    end
    
    'LockTogether': begin
      text = 'Check the "Lock Together" checkbox to make "Time Step" and "Averaging Period" ' + $
             'have the same value.'
      break
    end
    
    'PitchAngleContours': begin
      text = $
      'Check the "Pitch Angle Contours" checkbox to draw constant pitch angle contours on the electron globes.  ' + $
        'Contours at 30, 60 and 90, 120 and 150 degrees are oriented around the magnetic field vector.'
      break
    end
    
    'SphericalHarmonicFit': begin
      text = $
        ['Check the "Spherical Harmonic Fit" checkbox to save spherical harmonic expansion coefficients at each ' + $
         'time step.  These can later be displayed as a time-series plot in TCAD or PAPCO.', $
         '1) The series of coefficients grows whenever you go to a time LATER than the current time.  ' + $
         'If you go to a time BEFORE the current time, the series is started over again.', $
         '2) To save the series of coefficients to a save file, select "File/Save/Spherical Harmonic Coefficients"' + $
         'from the File menu.  Four files are created: Cluster_sha_cx_yyyy_ddd.save, where x is 1,2,3,4, ' + $
         'yyyy is the year and ddd is the day of year.']
      break
    end
    
    'MagneticField': begin
      text = $  
        'Check the "Magnetic Field" checkbox to display the magnetic field vector at each spacecraft.  ' + $
        'The vector has been averaged over the time period set in "Averaging Period".  ' + $
        "The vector's length is scaled according to the longest vector in the time range."
      break
    end
    
    'MagneticFieldCurvature': begin
      text = $  
        ['Check the "Magnetic Field Curvature" checkbox to show the vectors associated with the magnetic ' + $
         'field curvature.', $
         'The magnetic field vector at the Cluster mesocenter is colored ' + fgcolor + ', ' + $
         'the curvature vector is light red, and the osculating plane normal is light blue.', $
         'The magnetic field line at the mesocenter is represented by a semi-circle whose radius equals the ' + $
         'curvature radius.', $
         'Clear the checkbox to erase them.', $
         'The number of simultaneously shown curvature vectors and semi-circles can be changed in the ' + $
         '"Preferences" panel''s "Curvature Lines" box.']
      break
    end
    
    'OrbitTracks': begin
      text = $
        'Check the "Orbit Tracks" checkbox to show the orbit tracks of the 4 Cluster spacecraft.  ' + $
        'The orbit track is color coded by spacecraft: c1 = ' + fgcolor + ', c2 = red, c3 = green, c4 = blue.  ' + $
        'The orbit track is displayed over the time range given in the "Initialize" panel.'
      break
    end
    
    'OrbitProjections': begin
      text = $
        'Check the "Orbit Projections" checkbox to show projections of the orbit tracks onto the walls of the ' + $
        "bounding box.  Each spacecraft's location is indicated by a small dot on its orbit projection."
      break
    end
    
    'PitchAnglePlot': begin
      text = 'Check the "Pitch Angle Plot" checkbox to pitch angle plots.'
      break
    end
    
    'StartTime':
    'StopTime': begin
      text = $
        ['Set "from:" and "to:" to the start/stop times to load data over.  Enter the times in this format: ' + $
         '"yyyy/ddd-hh:mm:ss".  Hit return after entering each time.', $
         'Because the burst-mode data used by the program is very large, we recommend restricting the time range ' + $
         'to a few hours at most, depending on how much memory the host machine has available.', $
         'The start and stop times must be on the same day.']
      break
    end
    
    'TimeSlider': begin
      text = $
        ['Click to the right or left of the "Time Step Control" slider to advance/backup the time by the ' + $
         'number of seconds in the "Time Step" box.   Or drag the slider to the desired time.  Exact positioning ' + $
         'can be achieved by making "Time Step" shorter.', $
         'The available time range can be changed in the "Initialize" panel.', $
         "Bug note: if you drag the slider, and the mouse isn't positioned over the slider when you let up " + $
         "on the mouse button, the 3D display won't update.  Single click on the slider again to make it update."]
      break
    end
    
    'TimeStepInterval': begin
      text = $
        'Change "Time Step" to the time step in seconds the program takes when you click ' + $
        'right or left of the "Time Step Control" slider.   Click on the up-arrow to double the time, ' + $
        'or on the down-arrow to half the time.   Or edit the text field and hit return.'
      break
    end
    
    else: ;
  endswitch
    
  if text[0] eq '' then begin
    text = 'No help yet for ' + uname
  endif
  if text[0] ne '' then begin
    WIDGET_CONTROL, helpText, set_value=text
  endif
endif



if keyword_set( name3D) then begin
  helpText = WIDGET_INFO( state.rave_tlb, FIND_BY_UNAME='HelpText')
  text = ''
  switch name3D of
  
  'BoundingBox':
  'xAxis':
  'yAxis':
  'zAxis':
  'XTitle':
  'YTitle':
  'ZTitle':
  'RuleLines': begin
    text = 'The coordinate system is GSE, in units of Re.'
    break
  end
  
  'c1/B':
  'c2/B':
  'c3/B':
  'c4/B': begin
    obj3D->GetProperty, parent=parent
    parent->GetProperty, uvalue=B
    text = ['Magnetic field at Cluster ' + strmid( name3D, 1, 1) + ' = ' + $
            vectorAsString( B, format='(F7.2)') + ' nT', $
            strtrim( state.averaging_interval, 2) + ' second average']            
    break
  end
  
  
  'ColorBar': begin
    text = $
      ['The color bar shows the flux levels associated with the colors displayed on the electron flux globes.  ' + $
       'The units are [counts] s-1 cm-2 sr-1 keV-1. ', $
       'To change the colors for high count rates, grab the upper part of the color bar with the mouse ' + $
       'and move it up or down.', $
       'To change the colors for low count rates, move the lower part up or down.']
    break
  end
  
  'CurvatureVector': begin
    obj3D->GetProperty, parent=Cobj
    Cobj->GetProperty, parent=xformObj
    xformObj->GetProperty, uvalue=cv
    text = ['Magnetic field curvature vector, radius = ' + $
            strtrim( string( cv.radius, format='(F15.2)'),2) + ' Re', $
            'azimuthal angle = ' + angleAsString( cv.phi_c), $
            'polar angle = ' + angleAsString( cv.theta_c)]
    break
  end
  
  'eventTimeStr': begin
    text = 'This time is the center of the averaging interval'
    break
  end
  
  'c1_Globe':
  'c2_Globe':
  'c3_Globe':
  'c4_Globe': begin
    text = $
      ['Cluster ' + strmid(name3D,1,1) + ' electron flux globe.', $
       'The colors on a globe represent the electron flux seen by a RAPID instrument.  ' + $
       'The color bar shows the color coding in units of [counts]s-1cm-2sr-1keV-1.' + $
       'The data is averaged over the time period given in "Averaging Period", centered on the time ' + $
       'shown at the top of the 3D graphics window.', $
       'There are 9 detectors pointing at different polar angles which sweep across 16 azimuthal angles ' + $
       'during a spacecraft spin period.  The globes are oriented using the spin axes in the Cluster ' + $
       'Auxiliary Parameters file, with sector 13 pointing towards the sun.']
    break
  end
  
  'MesocenterB': begin
    obj3D->GetProperty, parent=Bobj
    Bobj->GetProperty, parent=xformObj
    xformObj->GetProperty, uvalue=cv
    text = ['Magnetic field at mesocenter = ' + vectorAsString( cv.B, format='(F7.2)') + ' nT', $
            strtrim( state.averaging_interval, 2) + ' second average, centered on ' + $
            strmid( t_to_str( cv.T), 9)]
    break
  end
  
  'OrbitTrack': begin
    text = $
      'The orbit tracks show the position of the Cluster spacecraft over the start/stop times given ' + $
      'in the "Initialize" panel'
    break
  end
  
  'OsculatingPlaneNormal': begin
    obj3D->GetProperty, parent=Cobj
    Cobj->GetProperty, parent=xformObj
    xformObj->GetProperty, uvalue=cv
    text = ['Osculating Plane Normal', $
            'azimuthal angle = ' + angleAsString( cv.phi_n), $
            'polar angle = ' + angleAsString( cv.theta_n)]
    break
  end
  
  else: ;
  endswitch
  if text[0] eq '' then begin
    text = 'No help yet for ' + name3D
  endif
  if text[0] ne '' then begin
    WIDGET_CONTROL, helpText, set_value=text
  endif
endif

END


;*******************************************************************************
;Main program
;*******************************************************************************

;rave_main, '2001/239-03:55:00', '2001/239-04:40:00'
;END
