;*******************************************************************************
; File:  rave_subroutines.pro
;
; Purpose:  Provide data access routines (wrappers around PAPCO routines with buffering),
;           and routines to deal with RAVE's 3D display.
; Routines:  
;            convertFaceColorsToVertexColors
;            colorBarEventHandler
;            colorGlobe
;            createPitchAnglePlot
;            drawColorBar
;            drawMag
;            drawArrow
;            drawPitchAnglePlot
;            fillDataGaps
;            fillDataGapsNearestNeighbor
;            getMagData
;            getOrbitData
;            getPitchAngleData
;            getRapidData
;            interpolateXYZROnEventTimes
;            projectOrbitsOntoCubeSides
;            read_rapid_B_papco
;            read_rapid_papco
;            rotateOneVectorIntoAnother
;            validateStartStopTimes
;            throw
;
;Obsolete/Unused Routines:
;            drawPlane
;            drawPlaneOrbitIntersections
;            fillDataGapsRenka
;            oldDrawPlane
;
; Author:  LASP/Steve Monk
;
; History: 10/01/03 SM - Release 1.0.
;
;*******************************************************************************

forward_function t_to_str, str_to_t, convert_t90_to_Date, convert_date_to_t90


;*******************************************************************************
pro getOrbitData, start_time_str, stop_time_str, T_out, D_out, success, tcad=tcad
;*******************************************************************************
  common Cluster_aux_common, T_save, D_save, T, D, pos
  success = 0
  start_time = str_to_t( start_time_str)
  stop_time  = str_to_t( stop_time_str)
  
  ;Get the data from CDAWeb's netCDF file(s) using PAPCO's CDF module.
  ;The environmental variable PAPCO_CDF_DATA must be defined, e.g. $PAPCO_DLNK/cdf.
  
  read_data = 0
  if n_elements(T_save) lt 2 then begin
    read_data = 1
  endif else if start_time lt T_save[0] or stop_time gt T_save[ n_elements( T_save)-1] then begin
    read_data = 1
  endif
  
  if read_data then begin
    print, 'In getOrbitData'    
    result = papco_check_data_env('PAPCO_CDF_DATA', PATH=path)
    if result eq 0 then begin
      print,'getOrbitData: Cannot get cluster aux data: PUM CDAcdf module not loaded or PAPCO_CDF_DATA not defined'
      return
    endif 

    COMMON mjdt, mjdt_start, mjdt_end
    start_time = str_to_t( start_time_str)
    stop_time  = str_to_t( stop_time_str)
    original_mjdt_start = mjdt_start
    original_mjdt_end   = mjdt_end
    mjdt_start = t_to_mjdt( start_time)
    mjdt_end   = t_to_mjdt( stop_time)

    COMMON papco_cdf, papco_cdf_control
    control_struct = papco_cdf_control
    control_struct.cdf_type    = 'aux'
    control_struct.cdf_file    = path+'Cluster_aux/2002/cl_sp_aux_20020811_v01.cdf'
    control_struct.cdfvar_name = 'sc_r_xyz_gse__CL_SP_AUX,'   + $  ;Position of reference S/C
                                 'sc_dr1_xyz_gse__CL_SP_AUX,' + $  ;Position of c1 relative to reference
                                 'sc_dr2_xyz_gse__CL_SP_AUX,' + $
                                 'sc_dr3_xyz_gse__CL_SP_AUX,' + $
                                 'sc_dr4_xyz_gse__CL_SP_AUX,' + $
                                 'sc_at1_lat__CL_SP_AUX,'     + $  ;GSE lat of c1 spin axis
                                 'sc_at1_long__CL_SP_AUX,'    + $  ;GSE lon of c1 spin axis
                                 'sc_at2_lat__CL_SP_AUX,'     + $
                                 'sc_at2_long__CL_SP_AUX,'    + $
                                 'sc_at3_lat__CL_SP_AUX,'     + $
                                 'sc_at3_long__CL_SP_AUX,'    + $
                                 'sc_at4_lat__CL_SP_AUX,'     + $
                                 'sc_at4_long__CL_SP_AUX,'    + $
                                 'gse_gsm__CL_SP_AUX'               ;Rotation angle from GSE to GSM
    info = papco_getplotinfostruct()
    info.usr_ptr1 = ptr_new( control_struct)
    
    ;See papco_lib/CDAWlib/read_myCDF.pro for an explanation of this ugly hack which speeds up
    ;reading of the aux file by 10x.
    COMMON speedup_read_myCDF_common, speedup_read_myCDF
    speedup_read_myCDF = 1    
    r_papco_cdf, info  ;, /verb
    speedup_read_myCDF = 0
    
    mjdt_start = original_mjdt_start
    mjdt_end   = original_mjdt_end

    COMMON get_error, get_err_no, get_err_msg
    if get_err_no ne 0 then throw,'r_papco_cdf failed to read orbit data between ' + $
                                  start_time_str + ' and ' + stop_time_str
    
    COMMON papco_cdf_data, input_data
    T = cdf_to_t( input_data.epoch__CL_SP_AUX.dat)                    ;dblarr(n)
    D = replicate( {pos_gse       : fltarr(3),   $
                    dpos          : fltarr(3,4), $
                    axis_lat_gse  : fltarr(4),   $
                    axis_lon_gse  : fltarr(4),   $
                    gse_gsm_angle : 0.0}, n_elements(T))
    ;Copy data from CDF structure to D array of structures.
    D.pos_gse         = input_data.sc_r_xyz_gse__CL_SP_AUX.dat        ;fltarr(3,N)
    D.dpos[*,0]       = input_data.sc_dr1_xyz_gse__CL_SP_AUX.dat
    D.dpos[*,1]       = input_data.sc_dr2_xyz_gse__CL_SP_AUX.dat
    D.dpos[*,2]       = input_data.sc_dr3_xyz_gse__CL_SP_AUX.dat
    D.dpos[*,3]       = input_data.sc_dr4_xyz_gse__CL_SP_AUX.dat
    D.axis_lat_gse[0] = input_data.sc_at1_lat__CL_SP_AUX.dat
    D.axis_lat_gse[1] = input_data.sc_at2_lat__CL_SP_AUX.dat
    D.axis_lat_gse[2] = input_data.sc_at3_lat__CL_SP_AUX.dat
    D.axis_lat_gse[3] = input_data.sc_at4_lat__CL_SP_AUX.dat
    D.axis_lon_gse[0] = input_data.sc_at1_long__CL_SP_AUX.dat
    D.axis_lon_gse[1] = input_data.sc_at2_long__CL_SP_AUX.dat
    D.axis_lon_gse[2] = input_data.sc_at3_long__CL_SP_AUX.dat
    D.axis_lon_gse[3] = input_data.sc_at4_long__CL_SP_AUX.dat
    D.gse_gsm_angle   = input_data.gse_gsm__CL_SP_AUX.dat
    
    input_data = 0
    T_save = T & D_save = D
  endif
  
  T = T_save & D = D_save
  W = where( T ge start_time and T le stop_time, CO)
  
  if CO lt 1 then throw,'No orbit data found between ' + start_time_str + ' and ' + stop_time_str
  
  ;Extend one more point on both sides of start/stop times so can interpolate.
  if W[CO-1] ne n_elements(T)-1 then W = [W,W[CO-1]+1]
  if W[0] ne 0 then W = [W[0]-1,W]
  T = T[W] & D = D[W]
  
  if keyword_set( TCAD) then begin
    T_out = T
    D_out = D
    success = 1
    return
  endif
  
  N = n_elements(T)
  pos = fltarr(N,3,4)  ;pos[ N_pts, x|y|z, Cluster1|Cluster2|Cluster3|Cluster4]
  for I=0,2 do begin
    for SC=0,3 do begin
      pos[*,I,SC] = D[*].pos_gse[I] + D[*].dpos[I,SC]
    endfor
  endfor
  if not keyword_set( km) then pos = pos/6378.
  T_out = T
  D_out = pos
END


;*******************************************************************************
; No longer using this routine because CLUSTER_EPH_SETS IDL save files don't have
; spin axis orientation which RAVE needs.
pro getOrbitData_old, start_time_str, stop_time_str, T_out, pos_out, success, tcad=tcad
;*******************************************************************************
  common Cluster_aux_common, T_save, D_save, T, D, pos
  success = 0
  start_time = str_to_t( start_time_str)
  stop_time  = str_to_t( stop_time_str)
  
  ;Read a PAPCO cluster ephemeris save set file, using PAPCO routines modified
  ;to work outside of PAPCO.  The environmental variable CLUSTER_EPH_SETS must be set.
  read_data = 0
  if n_elements(T_save) eq 0 then begin
    read_data = 1
  endif else if start_time lt T_save[0] or stop_time gt T_save[ n_elements( T_save)-1] then begin
    read_data = 1
  endif
  
  if read_data then begin
    print, 'In getOrbitData_old'
    common mjdt, mjdt_start, mjdt_end
    common cluster_eph_data, input_header, input_data
    original_mjdt_start = mjdt_start
    original_mjdt_end   = mjdt_end
    mjdt_start = t_to_mjdt( str_to_t( start_time_str))
    mjdt_end   = t_to_mjdt( str_to_t( stop_time_str))
    r_cluster_eph
    mjdt_start = original_mjdt_start
    mjdt_end   = original_mjdt_end
    n = n_elements( input_data)
    if n le 1 then throw,'No orbit data found in cdf file between ' + start_time_str + ' and ' + stop_time_str
    T = cdf_to_t( input_data.epoch)
    D = replicate( {pos_gse : fltarr(3), dpos : fltarr(3,4)}, n)
    D.pos_gse   = input_data.sc_r_xyz_gse
    D.dpos[*,0] = input_data.sc_dr1_xyz_gse
    D.dpos[*,1] = input_data.sc_dr2_xyz_gse
    D.dpos[*,2] = input_data.sc_dr3_xyz_gse
    D.dpos[*,3] = input_data.sc_dr4_xyz_gse
    input_header = 0 & input_data = 0
    T_save = T & D_save = D
  endif
  
  T = T_save & D = D_save
  W = where( T ge start_time and T le stop_time, CO)
  
  if CO lt 1 then throw,'No orbit data found between ' + start_time_str + ' and ' + stop_time_str
  
  ;Extend one more point on both sides of start/stop times so can interpolate.
  if W[CO-1] ne n_elements(T)-1 then W = [W,W[CO-1]+1]
  if W[0] ne 0 then W = [W[0]-1,W]
  T = T[W] & D = D[W]
  
  if keyword_set( TCAD) then begin
    T_out = T
    pos_out = D
    success = 1
    return
  endif
  
  N = n_elements(T)
  pos = fltarr(N,3,4)  ;pos[ N_pts, x|y|z, Cluster1|Cluster2|Cluster3|Cluster4]
  for I=0,2 do begin
    for SC=0,3 do begin
      pos[*,I,SC] = D[*].pos_gse[I] + D[*].dpos[I,SC]
    endfor
  endfor
  if not keyword_set( km) then pos = pos/6378.
  T_out = T
  pos_out = pos
END


;*******************************************************************************
pro interpolateXYZROnEventTimes, event_Ts, event_vals, spin_axes=spin_axes
;*******************************************************************************
  common Cluster_aux_common, T_save, D_save, T, D, pos
  if n_elements( event_Ts) ne 4 then $
    throw, 'event_Ts must be an array of 4 TCAD time doubles, one for each spacecraft!'
  event_vals = fltarr(3,4)  ;event_vals[ x|y|z, SC]
  spin_axes = [[0,-90.,1],[0,-90.,1],[0,-90.,1],[0,-90.,1]]   ;[lon,lat,radius],SC
  W = where( T ge event_Ts[0], co)
  for SC=0,3 do begin
    if event_Ts[SC] ne 0.0d then begin
      if co gt 0 then begin
        spin_axes[0,SC] = D[W[0]].axis_lon_gse[SC]
        spin_axes[1,SC] = D[W[0]].axis_lat_gse[SC]
      endif
      for I=0,2 do begin
        ;Usage of interpol:  new_Ys = interpol( old_Ys, old_Xs, new_Xs)
        val = interpol( pos[*,I,SC], T, [event_Ts[SC]])
        event_vals[I,SC] = val[0]
      endfor
    endif
  endfor
END


;*******************************************************************************
pro getRapidData, SC, inputTime, averaging_interval, T, outputData, success, data_type=data_type, $
                  restoreStartTime=restoreStartTime, restoreStopTime=restoreStopTime, $
                  band=band, no_average=no_average
;*******************************************************************************
  common RAPID_DATA_COMMON, Ts, Ds, D_data_type, old_inputTime, old_restore_start_stop_times, $
                            old_band, old_data_type
  success = 0
  if (n_elements(Ts) eq 0) or keyword_set( restoreStartTime) then begin
    if n_params() gt 0 then throw,'No non-keyword arguments may be passed when initializating!'
    if not (keyword_set( restoreStartTime) and keyword_set( restoreStopTime)) then $
      throw, 'restoreStart/StopTime keywords must be set on first call!'
    if not keyword_set( band) then band = 0
    if n_elements(band) gt 1 then $
      throw, 'band must be a scalar!'
    if n_elements(Ts) eq 0 then begin
      old_restore_start_stop_times = [0d,0d]
      old_inputTime = -1
      old_band = -1
      old_data_type = ''
    endif
    
    ;This is for convenience during development - often I restart the program, but the data remains
    ;in common blocks, so don't take the time to re-read it.
    times_are_unchanged = (old_restore_start_stop_times[0] eq restoreStartTime) and $
                          (old_restore_start_stop_times[1] eq restoreStopTime)
    if times_are_unchanged and (old_band eq band) and (old_data_type eq data_type) then begin
      print,'getRapidData: Not re-loading data as requested, because it appears to be loaded already!'
      return
    endif
    
    old_band = band
    old_restore_start_stop_times = [restoreStartTime, restoreStopTime]
    old_data_type = data_type
    
    ;First look for savesets generated by this routine:  yymmdd_hh-hh.cal_dat where
    ;the first hh is the start hour, the second hh is the stop hour (rounded up).
    split_time, restoreStartTime, year, doy, start_hour, mm, ss, month, day
    yymmdd = strmid( strtrim( year,2), 2) + string( format='(I2.2)', month) + string( format='(I2.2)', day)
    start_hh = string( format='(I2.2)', start_hour)
    split_time, restoreStopTime, year, doy, stop_hour, mm, ss, month, day
    stop_hour = (stop_hour + 1) < 24
    stop_hh = string( format='(I2.2)', stop_hour)
    T0=0 & T1=0 & T2=0 & T3=0 & D0=0 & D1=0 & D2=0 & D3=0
        
    D_data_type = data_type    
    for SC=0,3 do begin
      ;data_type is 'IES_CLEANBM' or 'IES_EPAD_16'
      dir = getenv('PAPCO_DLNK') + '/cluster/rapid/ifdata/RAPID_' + strtrim(SC+1,2) + '/' + data_type + '/'
      found = 0
      if data_type eq 'IES_CLEANBM' then begin
        files = findfile( dir + yymmdd + '*.cal_dat', count=count)
        for i=0,count-1 do begin
          result = stregex( files[i], '[0-9]{2}-[0-9]{2}', /extract)
          if n_elements( result) gt 0 then begin
            filename = files[i]
            file_start_hour = fix( strmid( result, 0, 2))
            file_stop_hour  = fix( strmid( result, 3, 2))
            if file_start_hour le start_hour and file_stop_hour ge stop_hour then begin
              print,'Restoring ' + filename
              restore, filename
              found = 1
              break
            endif
          endif
        endfor
      endif
      if not found then begin    ;Get PAPCO data, round times to nearest hour.
        hour_rounded_start_time_str = t_to_str( restoreStartTime)  ;yyyy/ddd-hh:mm:ss
        strput, '00:00', hour_rounded_start_time_str, 12           ;yyyy/ddd-hh:00:00
        hour_rounded_stop_time_str  = t_to_str( restoreStopTime + 1/24.0d)
        start_year_doy = strmid( hour_rounded_start_time_str, 0, 8)
        if start_year_doy eq strmid( hour_rounded_stop_time_str, 0, 8) then begin
          strput, '00:00', hour_rounded_stop_time_str, 12
        endif else begin                                           ;don't cross day boundary
          hour_rounded_stop_time_str = start_year_doy + '-23:59:00'
        endelse
        read_rapid_papco, hour_rounded_start_time_str, hour_rounded_stop_time_str, $
                          data_type, SC, T, D, found, /calibrate
        if found and data_type eq 'IES_CLEANBM' then begin
          filename = yymmdd + '_' + start_hh + '-' + stop_hh + '.cal_dat'
          if file_test( dir, /WRITE) then begin
            save, T, D, filename=dir+filename
          endif
        endif
      endif
      if found then begin
        success = 1
        case SC of
        0: begin &  T0=T & D0=D.data[*,*,band]  & end
        1: begin &  T1=T & D1=D.data[*,*,band]  & end
        2: begin &  T2=T & D2=D.data[*,*,band]  & end
        3: begin &  T3=T & D3=D.data[*,*,band]  & end
        endcase
      endif
    endfor
    
    Ts = create_struct('t0',T0,'t1',T1,'t2',T2,'t3',T3)
    Ds = create_struct('d0',D0,'d1',D1,'d2',D2,'d3',D3)
    
    ;Find max/min values so can scale color bar range.
    ;WARNING!  I abandoned work on this because setting the color bar range to the max/min found
    ;was bad;  max is way to large (~7), min is slightly too big (0.5, should be 0 or -0.5)
    ;min = 1e10 & max = 0
    ;for SC=0,3 do begin
    ;  SC_min = min( (Ds.(SC))[where( Ds.(SC) gt 0)], max=SC_max)
    ;  min = min < SC_min
    ;  max = max > SC_max
    ;endfor
    ;max = alog10( max)
    ;min = alog10( min)
    ;print,''
    ;print,'min/max=',min,max
    ;print,''
      
    if success then begin
      for SC=0,3 do begin
        W = where( Ts.(SC) ge restoreStartTime and Ts.(SC) lt restoreStopTime, co)
        if co gt 0 then begin
          Ts.(SC) = (Ts.(SC))[W]
          Ds.(SC) = (Ds.(SC))[*,*,W]
        endif
      endfor
    endif
    return
  endif      ;End of restore data block.
  
  
  success = 1
  bin_width = averaging_interval/86400./2
  W = where( Ts.(SC) ge inputTime-bin_width and Ts.(SC) lt inputTime+bin_width, count)
  if count eq 0 then begin
    t1_str = strmid( t_to_str( inputTime-bin_width), 9)
    t2_str = strmid( t_to_str( inputTime+bin_width), 9)
    print, "Couldn't find any " + D_data_type + ' data for Cluster ' + strtrim(SC+1,2) + $
           ' between ' + t1_str + ' and ' + t2_str
    success = 0
    return
  endif
  T = (Ts.(SC))[W]
  D = (Ds.(SC))[*,*,W]
  
  if keyword_set( no_average) then begin
    outputData = D
    if SC eq 2 and D_data_type eq 'IES_CLEANBM' then begin  ;Cluster 3 has two bad detector/sectors
      outputData[12,2,*] = (outputData[11,2,*] + outputData[14,2,*] + outputData[12,1,*] + outputData[12,3,*])/4
      outputData[13,2,*] = (outputData[11,2,*] + outputData[14,2,*] + outputData[13,1,*] + outputData[13,3,*])/4
    endif
    return
  endif
  
  ;WARNING!  POSSIBLE BUG!
  ;Note: this standard averaging method may not be appropriate for this data - need to ask Dan.
  ;Alternative is to not count 0 data, i.e. toss it out, instead of assuming it's a valid part
  ;of the statistics.
  T = total(T)/n_elements(T)
  outputData = D[*,*,0]
  for i=1,count-1 do begin
    outputData = outputData + D[*,*,i]
  endfor
  outputData = outputData / count
  if SC eq 2 and D_data_type eq 'IES_CLEANBM' then begin  ;Cluster 3 has two bad detector/sectors
    outputData[12,2] = (outputData[11,2] + outputData[14,2] + outputData[12,1] + outputData[12,3])/4
    outputData[13,2] = (outputData[11,2] + outputData[14,2] + outputData[13,1] + outputData[13,3])/4
  endif
  return
END



;*******************************************************************************
pro read_rapid_papco, start_time_str, stop_time_str, data_type, SC, T, D, $
                      success, calibrate=calibrate
;*******************************************************************************

success = 0
COMMON cluster_rapid  ;Only cluster_rapid_control is used here.
control_struct = cluster_rapid_control
control_struct.sc_id = SC     ;0,1,2,3
control_struct.data_src = 0   ;ral if data
control_struct.make = 1
;select correct product indec for chosen data type
idx = where(IF_names EQ data_type, c)
IF c NE 0 THEN control_struct.product = idx(0) ELSE $
message, 'Cannot find data type '+data_type
IF keyword_set( calibrate) THEN control_struct.cal = 3

plotinfo = papco_getplotinfostruct()
plotinfo.usr_ptr1 = ptr_new( control_struct)

;fill_cluster_rapid_plotinfo, plotinfo, control_struct

COMMON cluster_rapid_data, rapid_header, rapid_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg
COMMON cluster_rapid_slice

;When run from TCAD, not PAPCO, mjdt_start is undefined the first time.
;If it's undefined or a scalar, set it to 0.  If we're in PAPCO, it will be
;a structure, not a scalar, with sze[0] eq 1.

sze = size( mjdt_start)
if sze[0] eq 0 then begin
  mjdt_start = 0
  mjdt_end   = 0
endif
original_mjdt_start = mjdt_start
original_mjdt_end   = mjdt_end
mjdt_start = t_to_mjdt( str_to_t( start_time_str))
mjdt_end   = t_to_mjdt( str_to_t( stop_time_str))

panel = [0, 1, 1] & orbitNo = 0 & IsRead = 0
output = 2
draw_cluster_rapid, panel, PlotInfo, orbitNo, get_data_call, slice_type, IsRead, $
                    OUTPUT = output

mjdt_start = original_mjdt_start
mjdt_end   = original_mjdt_end

if get_err_no ne 0 then return

D = data        ;plot routine returns 'data' in cluster_rapid_slice_1 common block.

T = (D.endtime + D.time)/2.0
strs = call_function( 'tai2utc', T, /ecs)  ;yyyy/mm/dd hh:mm:ss.sss
N = n_elements( strs)
T = dblarr(N)
for i=0L,N-1 do begin
  tcad_time_string__parse_date_time,strs[i],dt
  T[i] = dt.days
endfor
success = 1
return
END


;*********************************************************************************
; Routine: fillDataGaps
; Purpose: Main driver routine called by rave.pro to do something about data gaps.
; Inputs:  d should be the data array (not alog10( data)), fltarr(16,9)
; Notes: 
; 1) The Renka method tends to overdo trending (too much weight given to the gradient),
;    and little control is available (unless you did your own triangulation, too hard).
;    Filled values are often much higher or much lower than their surrounding values.
; 2) The nearest neighbor averaging method doesn't do quite enough trending, but is
;    visually better than the Renka method.
; 3) Algorithm:  First fill all the 0 values which are surrounded by 4 non-zero values;
;                then fill all those surrounded by 3 non-zero values, etc. until we fill
;                values which have only one non-zero neighbor.  Each step is repeated
;                (in case the previous iteration caused new candidates to appear) until 
;                no more fills can be made.  Then the required number of neighbors
;                is decreased by 1.
;*********************************************************************************
pro fillDataGaps, d
  sze = size( d, /dim)
  if sze[0] ne 16 or sze[1] ne 9 then throw, 'Input argument d is the wrong size!'
  invalid_data_count = 16*9+1
  for n_required_neighbors=4,1,-1 do begin
    while 1 do begin
      fillDataGapsNearestNeighbor, d, n_required_neighbors
      invalid_data = where( d le 0)
      previous_invalid_data_count = invalid_data_count
      invalid_data_count = invalid_data[0] eq -1 ? 0 : n_elements( invalid_data)
      ;print, strtrim( invalid_data_count, 2) + ' invalid data points, n_required_neighbors = ' + $
      ;       strtrim( n_required_neighbors,2)
      if invalid_data_count eq 0 or invalid_data_count eq previous_invalid_data_count then break
    endwhile
  endfor
END


;*********************************************************************************
; Routine: fillDataGapsRenka
; Purpose: Fill 0 values in d without changing non-zero values using Renka's method.
;          See $IDL_DIR/lib/sph_scat.pro for paper reference.
;*********************************************************************************
pro fillDataGapsRenka, d, valid_data, invalid_data, verbose=verbose
;Set the lat/lon positions of the sectors/detectors in lat_arr and lon_arr.
lon_arr = fltarr( 16, 9)
lat_arr = fltarr( 16, 9)
for i=0,8 do begin
  lon_arr[*,i] = findgen(16)/16 * 360
endfor
for i=0,15 do begin
  lat_arr[i,*] = findgen(9)/9 * 180 - 90 + 180./9/2 + 2 ;without +2, hangs IDL in sph_scat
endfor
lon_arr = reform( lon_arr, 16*9)
lat_arr = reform( lat_arr, 16*9)

;Extract the valid data points and their lat/lons from d.  Note that the index arrays
;resulting from the 'where' call can be used on 2-D AND 1-D reformed arrays the same.
valid_data = where( finite( d) and d gt 0, count, complement=invalid_data)
if count eq 16*9 then return
valid_subset_of_d = d[ valid_data]
lon_arr = lon_arr[ valid_data]
lat_arr = lat_arr[ valid_data]

n_detectors = 9
output_lats = findgen( n_detectors)/n_detectors * 180 - 90 + 180./n_detectors/2
output_lons = findgen(16)/16 * 360
output_bounds = [min( output_lons), min( output_lats), max( output_lons), max( output_lats)]

sze = size( valid_data, /DIM)
data = alog10( reform( valid_subset_of_d, total( sze)))
interpolated_d_1dim = sph_scat( lon_arr, lat_arr, data, $
                                bounds=output_bounds, nlon=16, nlat=n_detectors)
d_interpolated = reform( interpolated_d_1dim, 16, n_detectors)

if keyword_set( verbose) then begin
  for i=8,6,-1 do begin
    print,'d_interpolated[*,' + strtrim(i,2) + '] = '
    print,d_interpolated[*,i]
  endfor
endif

;Replace the zero data points by interpolated data, without changing valid data.
d[ invalid_data] = 10.0^d_interpolated[ invalid_data]

END


;*********************************************************************************
; Routine: fillDataGapsNearestNeighbor
; Purpose: Set the value of every point in the raw input which is le 0 to the
;          average of its n neighboring points, if it has that many neighbors;
;          if not, leave it as is.
; Inputs:  raw_d should be data (not alog10( data))
;          n_required_neighbors can be 4,3,2,1
;*********************************************************************************
pro fillDataGapsNearestNeighbor, raw_d, n_required_neighbors

W = where( raw_d le 0, invalid_data_count, complement=WW)
if invalid_data_count eq 0 then return
if WW[0] eq -1 then return

sze = size( raw_d, /dim)
d = fltarr( sze[0], sze[1])
d[WW] = alog10( raw_d[WW])
new_raw_d = raw_d

;Loop through each lat ring.
for lat=0,8 do begin
  ;Loop through each 0 value in this lat ring, changing those values to the average
  ;of their neighbors as we go if they have n_required_neighbors.
  W = where( raw_d[*,lat] le 0, invalid_points_count)
  if invalid_points_count eq 0 then continue
  for i=0,invalid_points_count-1 do begin
    lon = W[i]
    prev_lon = (lon-1) lt 0 ? 15 : lon-1
    next_lon     = (lon+1) gt 15 ? 0 : lon+1
    prev_lat = (lat-1)
    next_lat     = (lat+1)
    pts = [0]
    if raw_d[ prev_lon, lat] gt 0 then pts = [pts,d[ prev_lon, lat]]
    if raw_d[ next_lon, lat] gt 0 then pts = [pts,d[ next_lon, lat]]
    if next_lat le 8 then begin
      if raw_d[ lon, next_lat] gt 0 then pts = [pts,d[ lon, next_lat]]
    endif
    if prev_lat ge 0 then begin
      if raw_d[ lon, prev_lat] gt 0 then pts = [pts,d[ lon, prev_lat]]
    endif
    if n_elements(pts) eq 1 then continue
    pts = pts[1:*]
    if n_elements(pts) lt n_required_neighbors then continue
    ave = total( pts)/n_elements(pts)
    new_raw_d[ lon, lat] = 10^ave
  endfor
endfor
raw_d = new_raw_d
END


;*******************************************************************************
; Routine: getMagData
; Purpose: 1) When called only with restoreStartTime and restoreStopTime keywords,
;             calls read_rapid_B_papco to read in data.
;          2) When called without the restore keywords, calls read_rapid_B_papco,
;             which gets the data from its common block, and averages the B data
;             around inputTime +/- averaging_interval/2.
; Notes:
; 1) When you want to change the start/stop times of the in-memory data inside the
;    read_rapid_B_papco common block, you must manually call this routine, specifying
;    the restore keywords.
; 2) When called without the restore keywords, the routine clips the averaging range
;    to be within restoreStartTime..restoreStopTime.
;*******************************************************************************

pro getMagData, SC_in, inputTime, outputData, averaging_interval, success, $
                restoreStartTime=restoreStartTime, restoreStopTime=restoreStopTime, $
                max_Btot=max_Btot

COMMON MAG_DATA_COMMON, max_Btot_save, restoreStartTimeSave, restoreStopTimeSave
success = 0
if n_elements( max_Btot_save) eq 0 or keyword_set( restoreStartTime) then begin
  start_time_str = t_to_str( restoreStartTime)
  stop_time_str =  t_to_str( restoreStopTime)
  max_Btot_save = 0
  for SC=0,3 do begin
    read_rapid_B_papco, start_time_str, stop_time_str, SC, T, D, found
    ;D is an array of structures with structure tags Bx, By, Bz, e.g. D[0].Bx
    if found then max_Btot_save = max_Btot_save > max( sqrt( D.Bx^2 + D.By^2 + D.Bz^2))
  endfor
  restoreStartTimeSave = restoreStartTime
  restoreStopTimeSave  = restoreStopTime
endif
max_Btot = max_Btot_save

if n_params() ne 5 then return

bin_width = averaging_interval/86400./2
start_time_str = t_to_str( (inputTime-bin_width) > restoreStartTimeSave)
stop_time_str  = t_to_str( (inputTime+bin_width) < restoreStopTimeSave)
read_rapid_B_papco, start_time_str, stop_time_str, SC_in, T, D, found
if not found then return

W = where( T ge inputTime-bin_width and T lt inputTime+bin_width, count)
if count eq 0 then return
D = D[W]
outputData = D[0]
Bmag = sqrt( D.Bx^2 + D.By^2 + D.Bz^2)
BmagAve = total( Bmag) / count                ;Average magnitudes.
for i=0,2 do begin                         
  D.(i) = D.(i) / Bmag
  outputData.(i) = total( D.(i)) / count      ;Average the i'th component of the unit vectors.
  outputData.(i) = outputData.(i) * BmagAve   ;Multiply by averaged magnitude.
endfor
success = 1
return
END


;*******************************************************************************
; Routine: read_rapid_B_papco
; Purpose: Read and cache magnetic field data for a given S/C.
; Inputs:  start_time_str  - The time range data is requested for, each as:
;          stop_time_str     'yyyy/ddd-hh:mm:ss'
;          SC              - The spacecraft data is requested for, integer or string
;                            i.e. 0,1,2,3 or '0','1','2','3'.
; Outputs: T, D  - Time and data arrays between start_time_str and stop_time_str
;          found - 1 if data was found between start_time_str and stop_time_str,
;                  0 otherwise.
; Notes:
; 1) The routine maintains a cache of data for each S/C.  The T and D arrays
;    for a given S/C (e.g. T0, D0) contain all the available data between
;    c0_cache_times[0] and c0_cache_times[1], as returned by read_papco_cdf.
; 2) The cache_times variables are used to prevent re-reading the same
;    cdf file, when the time asked for wasn't available.
;*******************************************************************************

pro read_rapid_B_papco, start_time_str, stop_time_str, SC, T, D, found      

COMMON read_rapid_B_papco, T0, T1, T2, T3, D0, D1, D2, D3, initialized, $
                           c0_cache_times, c1_cache_times, c2_cache_times, c3_cache_times

;if SC eq 0 then print,'read_rapid_B_papco: ',start_time_str,'  ',stop_time_str

if n_elements( initialized) eq 0 then begin  ;First call
  initialized = 1
  T0=0 & T1=0 & T2=0 & T3=0  &  D0=0 & D1=0 & D2=0 & D3=0
  c0_cache_times = [0d,0d]
  c1_cache_times = [0d,0d]
  c2_cache_times = [0d,0d]
  c3_cache_times = [0d,0d]
endif

case SC of
  '0': begin & T = T0 & D = D0 & cache_times = c0_cache_times & end
  '1': begin & T = T1 & D = D1 & cache_times = c1_cache_times & end
  '2': begin & T = T2 & D = D2 & cache_times = c2_cache_times & end
  '3': begin & T = T3 & D = D3 & cache_times = c3_cache_times & end
endcase
  
start_time = str_to_t( start_time_str)
stop_time  = str_to_t( stop_time_str)

if start_time ge cache_times[0] and stop_time le cache_times[1] then begin
  W = where( T ge start_time and T le stop_time, co)
  if co gt 0 then begin
    T = T[W]
    D = D[W]
    found = 1
    return
  endif

  if T[0] lt start_time and T[ n_elements(T)-1] gt stop_time then begin
    ;start_time/stop_time range is within T range, but is so short that no points
    ;fall within it.  So interpolate for a single point.
    D_out = D[0]
    for i=0,2 do begin
      D_out.(i) = interpol( D.(i), T, (stop_time - start_time)/2)
    endfor
    T = (stop_time - start_time)/2
    D = D_out
    found = 1
    return
  endif

  if start_time ge cache_times[0] and stop_time le cache_times[1] then begin
    found = 0
    return
  endif
endif

case SC of
  '0': c0_cache_times = [start_time, stop_time]
  '1': c1_cache_times = [start_time, stop_time]
  '2': c2_cache_times = [start_time, stop_time]
  '3': c3_cache_times = [start_time, stop_time]
endcase
  
T = 0 & D = 0 & found = 0
SC_str = strtrim( SC+1, 2)
result = papco_check_data_env('PAPCO_CDF_DATA', PATH=path)
if result eq 0 then begin
  print,'read_rapid_B_papco: Cannot get cluster_fgm data: PUM CDAcdf module not loaded or PAPCO_CDF_DATA not defined'
  return
endif 

COMMON mjdt, mjdt_start, mjdt_end
original_mjdt_start = mjdt_start
original_mjdt_end   = mjdt_end
mjdt_start = t_to_mjdt( start_time)
mjdt_end   = t_to_mjdt( stop_time)

COMMON papco_cdf  ;papco_cdf_control
control_struct = papco_cdf_control
control_struct.cdf_type    = 'cluster_fgm'
control_struct.cdf_file    = path+'cluster_fgm/2000/c'+SC_str+'_pp_fgm_20001101_v01.cdf'
control_struct.cdfvar_name ='B_xyz_gse__C' + SC_str + '_PP_FGM'
info = papco_getplotinfostruct()
info.usr_ptr1 = ptr_new( control_struct)

r_papco_cdf, info  ;, /verb

mjdt_start = original_mjdt_start
mjdt_end   = original_mjdt_end

COMMON get_error, get_err_no, get_err_msg
if get_err_no ne 0 then return
      
;Get time array
COMMON papco_cdf_data, cluster_mfi_data
pos = (where( strpos(tag_names(cluster_mfi_data), 'EPOCH') ne -1))[0]
cdf_times = cluster_mfi_data.(pos).dat
T = cdf_to_t( cdf_times)                    ;dblarr(n)
;Get data array
pos = (where( strpos(tag_names(cluster_mfi_data),strupcase(control_struct.cdfvar_name)) ne -1))[0]
tmp = transpose(cluster_mfi_data.(pos).dat)   ;fltarr(n,3)
N = (size(tmp,/dim))[0]
D = replicate( {Bx:0.0, By:0.0, Bz:0.0}, N)
D.Bx = tmp[*,0]
D.By = tmp[*,1]
D.Bz = tmp[*,2]
nodata = cluster_mfi_data.(pos).FILLVAL
case SC of
  '0': begin & T0 = T & D0 = D & end
  '1': begin & T1 = T & D1 = D & end
  '2': begin & T2 = T & D2 = D & end
  '3': begin & T3 = T & D3 = D & end
endcase
W = where( T ge start_time and T le stop_time, co)
if co eq 0 then return
T = T[W]
D = D[W]
found = 1

return
END


;*******************************************************************************
pro drawMag, B_in, Bobj, max_Btot, xs, ys, globeRadius, state
;*******************************************************************************
  B = B_in
  if xs[1] lt 0 then B[0] = -B[0]
  if ys[1] lt 0 then B[1] = -B[1]
  B_mag = sqrt( total( B^2))
  B_unit = B/B_mag
  if state.show.ElectronFluxGlobes then begin
    arrow_start = B_unit * globeRadius  ;Start arrow on surface of electron flux globe.
  endif else begin
    arrow_start = [0.0, 0.0, 0.0]       ;Start arrow at globe center on orbit track.
  endelse
  fac = 0.15/max_Btot                   ;Scale so maximum B length is 0.15 in normalized coordinates.
  arrow_end = arrow_start + B*fac
  drawArrow, arrow_start, arrow_end, Bobj, color=state.foreground_color
END

  
;*******************************************************************************
; Routine: drawArrow
; Purpose: Set the geometry of an arrow (vector) in a 3-D graphics object.
; Inputs:  arrow_start - The start of the arrow, in normalized coordinates, e.g. [0.0,0.0,0.0].
;          arrow_end   - The tip of the arrow head, in normalized coordinates.
;          parentObj   - A graphics object which contains these two graphics objects as children:
;                        1) A IDLgrPolyline which will represent the arrow shaft.
;                        2) A IDLgrPolygon which will represent the arrowhead.
;                        On entry these objects must be instantiated, but don't need any attributes set.
; Outputs: The two children graphics object of parentObj have their geometry (data) set
;          and are ready to be drawn.
;*******************************************************************************

pro drawArrow, arrow_start, arrow_end, parentObj, arrow_head_radius=arrow_head_radius, $
               arrow_head_length=arrow_head_length, thick=thick, color=color
  if not keyword_set( arrow_head_radius) then arrow_head_radius = 0.012
  if not keyword_set( arrow_head_length) then arrow_head_length = arrow_head_radius*2
  if not keyword_set( thick) then thick=3
  if not keyword_set( color) then color=[255,255,255]
  
  ;Get objects from object tree.
  shaftObj = parentObj->Get( position=0)  
  headObj  = parentObj->Get( position=1)  
  
  ;Set the arrow shaft's geometry.
  shaftObj->SetProperty, data=[[arrow_start],[arrow_end]], thick=thick, color=color
  
  ;Draw arrow head as a cone (surface of revolution):
  x_axis = [1,0,0] & y_axis = [0,1,0] & z_axis = [0,0,1]
  dir = arrow_end - arrow_start
  dir = dir / sqrt( total( dir^2))
  
  dir_dot_x = abs(total(dir*x_axis))
  dir_dot_y = abs(total(dir*y_axis))
  dir_dot_z = abs(total(dir*z_axis))
  max = max([dir_dot_x, dir_dot_y, dir_dot_z])
  case max of
    dir_dot_x:  perp = crossp(dir, x_axis)
    dir_dot_y:  perp = crossp(dir, y_axis)
    dir_dot_z:  perp = crossp(dir, z_axis)
  endcase
  perp = perp/sqrt( total( perp^2))
  pt_on_base_radius = arrow_end - arrow_head_length*dir + perp*arrow_head_radius
  n_faces = 16
  mesh_obj, 6, v, p, [[arrow_end], [pt_on_base_radius]], p1=n_faces, p2=arrow_end, p3=dir
  n_vertices = (size(v))[2]
  headObj->SetProperty, data=v, polygons=p, color=color, thick=thick
END


;*******************************************************************************
;Purpose:  Apply a rotation to the input transform object which rotates 
;          vector_to_rotate into stationary_vector.
;          If the RESET keyword is set, start from the identity matrix, otherwise
;          add the rotation to the existing transform.
;Example:  To align the pitch angle contours with the B vector, vector_to_rotate
;          is the N pole, stationary_vector is B, and the pitch angle contour
;          polyline objects are children of the input transform object.
;*******************************************************************************
pro rotateOneVectorIntoAnother, vector_to_rotate, stationary_vector, $
                                transform, RESET=RESET

unit_vector_to_rotate = vector_to_rotate / sqrt( total( vector_to_rotate^2))
unit_stationary_vector = stationary_vector / sqrt( total( stationary_vector^2))
N_pole = [0,0,1]
perp = crossp( unit_stationary_vector, unit_vector_to_rotate)
perp = perp/sqrt( total( perp^2))
angle = acos( total( unit_vector_to_rotate * unit_stationary_vector)) * !radeg   ;0..180
if keyword_set( RESET) then begin
  identityMatrix = [[1.0, 0.0, 0.0, 0.0], $
                    [0.0, 1.0, 0.0, 0.0], $
                    [0.0, 0.0, 1.0, 0.0], $
                    [0.0, 0.0, 0.0, 1.0]]
  transform->SetProperty, transform=identityMatrix  
endif
transform->rotate, perp, angle
END

  
;******************************************************************************
; Routine: colorGlobe
; Purpose: Convert d values (already alog10) to color table indices according to
;          logRangeForColorScaling.  Rotate so sun sector is synched with globe
;          orientation, and apply the colors to the globe.
;******************************************************************************
pro colorGlobe, SC_globe, d, logRangeForColorScaling
    
colors = bytscl( d, min=logRangeForColorScaling[0], max=logRangeForColorScaling[1])
poly_shades = reform( colors)
;Reverse the order of sectors, leave order of detectors as is.  I.e. d[15,*] = d[0,*]
poly_shades = transpose( rotate( poly_shades,3))

SC_globe->GetProperty, data=v, polygons=p
convertFaceColorsToVertexColors, v, p, poly_shades, vertexColors
SC_globe->setProperty, vert_colors=vertexColors
END
  
  
;******************************************************************************
; Routine: convertFaceColorsToVertexColors
; Purpose:  Rearrange the poly_shades values into the vertexColors array.  
;           With flat shading, only the first vertex in a polygon needs a color.
;           Each polygon has 4 vertices.
;******************************************************************************
pro convertFaceColorsToVertexColors, v, p, poly_shades, vertexColors    
  n_vertices = (size( v, /dim))[1]
  n_sectors  = (size( poly_shades, /dim))[0]  ;fltarr(16,9) or fltarr(16,16)
  vertexColors = bytarr( n_vertices)
  for i=0,n_elements(p)-1,5 do begin
    polygonIndex = i/5
    firstVertexInThisPolygon = p[i+1]
    vertexColors[ firstVertexInThisPolygon] = poly_shades[ polygonIndex mod n_sectors, polygonIndex/n_sectors]
  endfor
END 


;******************************************************************************
pro projectOrbitsOntoCubeSides, event_vals, SC, orbitProjectionObj, xrange, yrange, zrange
;******************************************************************************
  marker = orbitProjectionObj->GetByName( 'XYPlaneMarker')
  pos = [event_vals[0,SC], event_vals[1,SC], event_vals[2,SC]]
  marker->SetProperty, data=[[pos[0],pos[1],zrange[0]], [pos[0],pos[1],zrange[0]]]
  marker = orbitProjectionObj->GetByName( 'XZPlaneMarker')
  marker->SetProperty, data=[[pos[0], yrange[1], pos[2]], [pos[0], yrange[1], pos[2]]]
  marker = orbitProjectionObj->GetByName( 'YZPlaneMarker')
  marker->SetProperty, data=[[xrange[1], pos[1], pos[2]], [xrange[1], pos[1], pos[2]]]
END


;******************************************************************************
pro validateStartStopTimes, start_time_str, stop_time_str, start_time, stop_time, ok
;******************************************************************************
ok = 1
z = execute( 'tcad_time_string__parse_date_time, start_time_str[0], start_time_dt, ok')
if (z eq 0) or (ok eq 0) then begin
  junk = WIDGET_MESSAGE( ['Start time string is invalid!', $
                          'Must be of the form: yyyy/ddd-hh:mm:ss', $
                          'or yyyy/mm/dd-hh:mm:ss'], /ERROR)
  ok = 0
  return
endif
z = execute( 'tcad_time_string__parse_date_time, stop_time_str[0], stop_time_dt, ok')
if (z eq 0) or (ok eq 0) then begin
  junk = WIDGET_MESSAGE( ['Stop time string is invalid!', $
                          'Must be of the form: yyyy/ddd-hh:mm:ss', $
                          'or yyyy/mm/dd-hh:mm:ss'], /ERROR)
  ok = 0
  return
endif
start_time = start_time_dt.days
stop_time  = stop_time_dt.days
return
END


;******************************************************************************
;This function finds the intersection point of a line and a plane.  The line is
;defined by the two points v1 and v2.  The plane is defined by a point on the
;plane, p, and a normal to the plane, n.  If the line is parallel to the plane,
;there is no intersection point, and the function returns 1 and sets valid to 0.
;Otherwise the function returns the intersection point as fltarr(3), and sets
;valid to 1.
;******************************************************************************
function intersectPlaneWithLine, p, n, v1, v2, valid=valid
  DEGENERATE_TOLERANCE = 0.000002
  valid = 1
  denominator = total((v1 - v2) * n)
  if denominator eq 0.0 then begin  ;Line is parallel to plane, so no intersection pt.
    valid = 0
    return,1
  endif
  numerator = total((p - v2) * n)
  t = numerator / denominator
  omt = 1.0 - t
  if  1.0 lt t*DEGENERATE_TOLERANCE or -1.0 gt t*DEGENERATE_TOLERANCE then begin
    valid = 0
    return,1
  endif
  intersection = t*v1 + omt*v2
  return,intersection
END


;*******************************************************************************
pro drawPlane, origin, plane_normal, valid=valid
;*******************************************************************************
; norm_pts defines intersection of the plane with the unit bounding cube.
; Using the default view, these are in order: lower-right, lower-left, upper-left,
; upper-right, and lower-right repeated.
  drawPlaneOrbitIntersections, origin, plane_normal
  norm_pts = fltarr(3,5)
  norm_pts[*,0] = intersectPlaneWithLine( origin, plane_normal, [0.0,0,0], [1.0,0,0], valid=valid)
  if not valid then return
  norm_pts[*,1] = intersectPlaneWithLine( origin, plane_normal, [0.0,1,0], [1.0,1,0], valid=valid)
  if not valid then return
  norm_pts[*,2] = intersectPlaneWithLine( origin, plane_normal, [0.0,1,1], [1.0,1,1], valid=valid)
  if not valid then return
  norm_pts[*,3] = intersectPlaneWithLine( origin, plane_normal, [0.0,0,1], [1.0,0,1], valid=valid)
  if not valid then return
  norm_pts[*,4] = norm_pts[*,0]
  
  data_pts = convert_coord( norm_pts, /norm, /to_data, /t3d)
  if !X.Crange[1] gt !X.Crange[0] then begin
    data_pts[0,*] = !X.Crange[1] < data_pts[0,*] > !X.Crange[0]
  endif else begin
    data_pts[0,*] = !X.Crange[0] < data_pts[0,*] > !X.Crange[1]
  endelse
  if !Y.Crange[1] gt !Y.Crange[0] then begin
    data_pts[1,*] = !Y.Crange[1] < data_pts[1,*] > !Y.Crange[0]
  endif else begin
    data_pts[1,*] = !Y.Crange[0] < data_pts[1,*] > !Y.Crange[1]
  endelse
  plots,data_pts,thick=2,/t3d,/data
END


;*******************************************************************************
pro drawPlaneOrbitIntersections, origin, plane_normal
;*******************************************************************************
  SC_colors = [255,244,137,73]   ;white,yellow,red,blue
  common Cluster_aux_common, T_save, D_save, T, D, pos
;pos = fltarr(N,3,4)  ;pos[ N_pts, x|y|z, Cluster1|Cluster2|Cluster3|Cluster4]
  n_pts = n_elements(T)
  d = fltarr( n_pts)
  p0 = convert_coord( origin, /norm, /to_data, /t3d)  ;Get origin in same coordinates as orbit.
  for SC=0,3 do begin
  
    ;Calculate the "signed distance" between each orbit point p and the plane.
    ;Each of these is the projection of the vector from the origin to p, onto n,
    ;the plane normal.
    for I=0,n_pts-1 do begin
      d[I] = total( plane_normal * (pos[I,*,SC]-p0))
    endfor
    
    ;Find the first sign change, where two points, p1 and p2, are on opposite sides
    ;of the plane.
    mask = d + abs(d)  ;0 where d is negative, 2*d elsewhere
    W = where( mask eq 0, CO, complement=W2)
    if CO eq 0 or CO eq n_pts then continue   ;no sign change
    if W[0] ne 0 then begin
      p1 = pos[ W[0]-1, *, SC]
      p2 = pos[ W[0],   *, SC]
    endif else begin
      p1 = pos[ W2[0]-1, *, SC]
      p2 = pos[ W2[0],   *, SC]
    endelse
    
    ;Find the point where the line defined by p1 and p2 intersects the plane.
    p = intersectPlaneWithLine( p0, plane_normal, p1, p2, valid=valid)
    if not valid then continue
    
    ;Draw 3 lines from the intersection point to the 3 bounding cube planes.
    ;Make sure the end point of each line at its bounding plane doesn't go outside of
    ;the Crange values.
    pts = fltarr(3,4)
    pts[*,0] = p
    pts[*,1] = [!X.Crange[1], p[1], p[2]]  ;yz plane
    pts[*,2] = [p[0], !Y.Crange[1], p[2]]  ;xz plane
    pts[*,3] = [p[0], p[1], !Z.Crange[0]]  ;xy plane
    for I=1,3 do begin
      plots, pts[*,[0,I]], /t3d, /data, color=SC_colors[SC]
    endfor
  endfor
END


;*******************************************************************************
pro oldDrawPlane, origin, plane_normal
;*******************************************************************************
  norm_pts = [[origin[0],0,0],$
              [origin[0],1,0],$
              [origin[0],1,1],$
              [origin[0],0,1],$
              [origin[0],0,0]]
  data_pts = convert_coord( norm_pts, /norm, /to_data, /t3d)
  if !X.Crange[1] gt !X.Crange[0] then begin
    data_pts[0,*] = !X.Crange[1] < data_pts[0,*] > !X.Crange[0]
  endif else begin
    data_pts[0,*] = !X.Crange[0] < data_pts[0,*] > !X.Crange[1]
  endelse
  if !Y.Crange[1] gt !Y.Crange[0] then begin
    data_pts[1,*] = !Y.Crange[1] < data_pts[1,*] > !Y.Crange[0]
  endif else begin
    data_pts[1,*] = !Y.Crange[0] < data_pts[1,*] > !Y.Crange[1]
  endelse
  plots,data_pts,thick=2,/t3d,/data
END


;*******************************************************************************
function getKey, arrow_key=arrow_key
;*******************************************************************************
  arrow_key = ''
  k = get_kbrd(1)
  if (byte(k))[0] eq 27 then begin
    k = get_kbrd(1)
    k = get_kbrd(1)
    case k of
      'A': begin
        arrow_key = 'up_arrow'
        return,'up_arrow'
      end
      'B': begin
        arrow_key = 'down_arrow'
        return,'down_arrow'
      end
      'C': begin
        arrow_key = 'right_arrow'
        return,'right_arrow'
      end
      'D': begin
        arrow_key = 'left_arrow'
        return,'left_arrow'
      end
      else: ;
    endcase
  endif else begin
    return,k
  endelse
  return,''
END


;*******************************************************************************
pro drawColorBar, state
;*******************************************************************************
  
common rave_common, pos, window, view, model, trackball, old_state, $
                    xs, ys, zs, xrange, yrange, zrange

  annotations_model = view->GetByName( 'annotations')
  first_time = not obj_valid( annotations_model->GetByName( 'ColorBarAxis'))
  if first_time then begin
    axis_yrange = [1e-2, 1e4]    ;This may eventually be a preference, but ycoord_conv[1] must change too...?
    axis_yrange_log = alog10( axis_yrange)
    exps = fix( indgen( axis_yrange_log[1] - axis_yrange_log[0] + 1) + axis_yrange_log[0])
    labels = '10!U' + strtrim( exps,2)
    ticktext = obj_new( 'IDLgrText', labels, color=state.foreground_color, /ENABLE_FORMATTING, $
                        /ONGLASS, font=obj_new('IDLgrFont'))
    ;title = obj_new( 'IDLgrText', '[cts]s!U-1!Ncm!U-2!Nsr!U-1!NkeV!U-1', /ENABLE_FORMATTING, $
    ;                 name='ColorBarAxisTitle', color=state.foreground_color, $
    ;                 font=obj_new('IDLgrFont'))
    ;Note: this title takes too much space, so I didn't add it.
    xpos = 0.95
    colorBarAxis = obj_new( 'IDLgrAxis', direction=1, color=state.foreground_color, /log, name='ColorBarAxis', $
                            xcoord_conv=[xpos, 1], ycoord_conv=[0, 0.2], $  ;title=title, $
                            ticklen=0.2/8, range=axis_yrange, major=n_elements( labels), $
                            ticktext=ticktext)
    ;Note: I couldn't figure how to make the axis tick labels scale when user changes window size;
    ;they always stay the same size.  Tried zcoord_conv = [1,1], location=[0,0,1], etc.
    colorBarAxis->GetProperty, crange=axis_yrange, ycoord_conv=ycoord_conv
    ;Note: IDL positions the y-axis so that 0 (10^0 for log axis) is positioned at ycoord_conv[0].
    image_lower_left_corner = [xpos, axis_yrange[0] * ycoord_conv[1]]
    xsize = 10
    image_height = (axis_yrange[1] - axis_yrange[0]) * ycoord_conv[1]       ;In normalized coordinates
    image = obj_new( 'IDLgrImage', name='ColorBar', dimensions=[xsize, image_height], $
                     location=image_lower_left_corner)
    window->getProperty, palette=palette
    image->SetProperty, palette=palette
    annotations_model->add, image
    annotations_model->add, colorBarAxis
  endif
  
  column = indgen(256,/byte)
  xsize = 10
  arr1 = bytarr( xsize, 256)
  for i=0,xsize-1 do arr1[i,*] = column
  
  ;The range logRangeForColorScaling[0]..logRangeForColorScaling[1] is represented by the
  ;256 pixels in arr1, and we have a conversion factor from a log range to pixels.  We use this
  ;conversion factor to make two more image arrays:  arr0 below the 256 color bar, and arr2 above
  ;the 256 color bar, with the y size of these arrays are determined by how close the color bar axis
  ;limits are to the logRangeForColorScaling limits.
  
  fac_log_to_pixels = 256 / (state.logRangeForColorScaling[1] - state.logRangeForColorScaling[0])
  colorBarAxis = annotations_model->GetByName( 'ColorBarAxis')
  colorBarAxis->GetProperty, crange=axis_yrange
  arr0_ysize = fix( (state.logRangeForColorScaling[0] - axis_yrange[0]) * fac_log_to_pixels) > 1
  arr2_ysize = fix( (axis_yrange[1] - state.logRangeForColorScaling[1]) * fac_log_to_pixels) > 1
  arr0 = bytarr( xsize, arr0_ysize) + arr1[0,0]
  arr2 = bytarr( xsize, arr2_ysize) + arr1[0,255]
  arr = [[arr0],[arr1],[arr2]]
  colorBar = annotations_model->GetByName( 'ColorBar')
  colorBar->SetProperty, data=arr
  ;window->draw, state.scene
END


;*******************************************************************************
pro colorBarEventHandler, ev, colorBar
;*******************************************************************************

common colorBarEventHandler_common, oldMouseLogValue, axisXValue, sideBeingDragged  ;'top' or 'bottom'
common rave_common, pos, window, view, model, trackball, old_state, $
                    xs, ys, zs, xrange, yrange, zrange

colorBar->GetProperty, parent=parent
colorBarAxis = parent->GetByName( 'ColorBarAxis')
colorBarAxis->GetProperty, ycoord_conv=ycoord_conv
rave_tlb = WIDGET_INFO( ev.top, FIND_BY_UNAME='RaveTopLevelBase')
WIDGET_CONTROL, rave_tlb, get_uvalue=state

if ev.type eq 0 and ev.press eq 1 then begin      ;Left button press.
  
  ;Bug Note:  This method doesn't work under Mac OS X/PowerBook G4; PickData doesn't return 1,
  ;even when its x input is the same as an x input which Help shows is definitely over the colorBarAxis.
  if 0 then begin
  for x=ev.x,ev.x-20,-1 do begin
    ;PickData returns 1 only if user clicks EXACTLY on the vertical axis, which is nearly
    ;impossible to do, so we step left of the click until we find the vertical axis.
    result = window->PickData( view, colorBarAxis, [x, ev.y], data_coords)
    if result eq 1 then break
  endfor
  if result ne 1 then begin
    print,"colorBarEventHandler: Couldn't find color bar axis!"
    sideBeingDragged = ''
    return
  endif
  endif
  
  ;Convert the ev.y value to an exponent of 10 on the color bar axis.
  ;ev.y has a range from 0, the bottom of draw area, to the top, given by state.current_draw_widget_size.
  ;The bounds of the normalized 3-D drawing area is -1 to 1.
  ;The colorBarAxis's ycoord_conv[1] = 0.2 is the conversion factor from each increment of an exponent of 10, 
  ;to an offset in the normalized 3-D drawing area coordinates.  10^0 is at normalized position 0, 10^1 is at
  ;y_coord_conv[1], 10^-1.5 is at -1.5*ycoord_conv[1], etc.  To get the exponent of 10 the click occurred on,
  ;scale ev.y to the range [-1,1], then divide by ycoord_conv[1].
  
  
  data_coords=[0, (ev.y/float(state.current_draw_widget_size)*2 - 1) / ycoord_conv[1]]
  
  mid_point = (state.logRangeForColorScaling[1] + state.logRangeForColorScaling[0])/2
  sideBeingDragged = (data_coords[1] ge mid_point) ? 'top' : 'bottom'
  oldMouseLogValue = data_coords[1]
  WIDGET_CONTROL, ev.id, /DRAW_MOTION                ;Enable mouse motion events.
  return
endif  

if ev.type eq 2 then begin                           ;Drag event
  if sideBeingDragged eq '' then return
  
  data_coords=[0, (ev.y/float(state.current_draw_widget_size)*2 - 1) / ycoord_conv[1]]

  delta_log_value = data_coords[1] - oldMouseLogValue
  colorBarAxis->GetProperty, crange=axis_yrange
  new_log_range = state.logRangeForColorScaling
  if sideBeingDragged eq 'top' then begin
    new_log_range[1] = state.logRangeForColorScaling[1] + delta_log_value
    if new_log_range[1] ge axis_yrange[1] then begin
      new_log_range[1] = axis_yrange[1]
    endif else if new_log_range[1] le (state.logRangeForColorScaling[0] + 1) then begin
      new_log_range[1] = state.logRangeForColorScaling[0] + 1
    endif
  endif else begin
    new_log_range[0] = state.logRangeForColorScaling[0] + delta_log_value
    if new_log_range[0] le axis_yrange[0] then begin
      new_log_range[0] = axis_yrange[0]
    endif else if new_log_range[0] ge (state.logRangeForColorScaling[1] - 1) then begin
      new_log_range[0] = state.logRangeForColorScaling[1] - 1
    endif
  endelse
  if total( new_log_range eq state.logRangeForColorScaling) ne 2 then begin
    state.logRangeForColorScaling = new_log_range
    WIDGET_CONTROL, rave_tlb, set_uvalue=state
    drawColorBar, state
    for SC=0,3 do begin
      SCname = 'c' + strtrim(SC+1,2)
      SC_globe = model->GetByName( SCname + '/Transform/GlobeTransform/' + SCname + '_Globe')
      SC_globe->GetProperty, uvalue=d    ;alog10( d)
      if n_elements(d) eq 0 then begin
        print,'d is undefined!'
        continue
      endif
      colorGlobe, SC_globe, d, state.logRangeForColorScaling
    endfor
  endif
  oldMouseLogValue = data_coords[1]
  return
endif

if ev.type eq 1 and ev.release eq 1 then begin      ;Left button release.
  WIDGET_CONTROL, ev.id, DRAW_MOTION=0              ;Disable motion events.
endif
END


;*******************************************************************************
pro drawPlane, method    
;*******************************************************************************
  if method eq 1 then begin
    ;Draw a plane.  Have to draw the plane in the Z-buffer, so that it looks different
    ;behind than in front.
    device,set_graphics=6  ;set to xor mode
    pat = bytarr(3,3) & pat[1,1] = 255
    polyfill,[.5,.5,.5,.5],[0,0,1,1],[0,1,1,0],/norm,/t3d,pattern=pat
    
    ;Couldn't get these keywords to polyfill to work as advertised:
    ;,transparent=255, image_coord=[[10,10],[10,10],[10,10],[10,10]]
    
    device,set_graphics=3
  endif
  
  if method eq 2 then begin
    ;Draw a plane.  Have to draw the plane in the Z-buffer, so that it looks different
    ;behind than in front.
    device,set_graphics=6 ;set mode to XOR
    polyfill,[.5,.5,.5,.5],[0,0,1,1],[0,1,1,0],/norm,/t3d,color=255
    device,set_graphics=3
  endif
  
  if method eq 3 then begin
    ;Draw a plane (method 2)
    ;z_inc = (!Z.Crange[1] - !Z.Crange[0])/10
    ;for z=!Z.Crange[0],!Z.Crange[1],z_inc
    ;  plots,!X.Crange[0
    ;endfor
    ;device,set_graphics=6  ;set to xor mode
    polyfill,[.5,.5,.5,.5],[0,0,1,1],[0,1,1,0],/norm,/t3d,/line_fill,lines=1
    polyfill,[.5,.5,.5,.5],[0,0,1,1],[0,1,1,0],/norm,/t3d,/line_fill,orientation=90,lines=1
    ;device,set_graphics=3
  endif
  
  if method eq 4 then begin
    ;Draw a plane.  Have to draw the plane in the Z-buffer, so that it looks different
    ;behind than in front.
    device,set_graphics=6  ;set to xor mode
    ;pat = bytarr(20,20) & for I=0,19 do for J=(I mod 2),19,2 do pat[I,J] = 255
    pat = bytarr(5,5) & pat[2,2] = 255
    polyfill,[.5,.5,.5,.5],[0,0,1,1],[0,1,1,0],/norm,/t3d,pattern=pat $
      ,transparent=255 ;, image_coord=[[10,10],[10,10],[10,10],[10,10]]
    
    device,set_graphics=3
  endif
END


;*******************************************************************************
pro throw, error_string    ;Simulates the java 'throw' method.
;*******************************************************************************
print,error_string
help, calls=calls
for i=1,n_elements(calls)-1 do $
  print,calls[i]
print,"To return to the routine which called THROW, type '.out'"
stop
END


;*******************************************************************************
; Purpose: Make it easy to recycle rave when developing.
;*******************************************************************************
pro a

common rave_common, pos, window, view, model, trackball, old_state, $
                    xs, ys, zs, xrange, yrange, zrange

WIDGET_CONTROL, old_state.rave_tlb, /DESTROY
resolve_routine, ['rave','rave_main'], /COMPILE_FULL_FILE
rave_main, run_env='TCAD'
END



;*******************************************************************************
function vectorAsString, v, format=format
;*******************************************************************************
  s = '(' + strtrim( string( v[0], format=format), 2) + ' ' + $
            strtrim( string( v[1], format=format), 2) + ' ' + $
            strtrim( string( v[2], format=format), 2) + ')'
  return,s
END


;*******************************************************************************
function angleAsString, angle_in_degrees
;*******************************************************************************
  s = strtrim( string( angle_in_degrees, format='(F8.2)'), 2)
  return,s
END


;*******************************************************************************
pro rave_subroutines      ;This helps in compiling.
;*******************************************************************************
END
