;+
; NAME:
;       RAVE_XPLOT
;
; PURPOSE:
;       Create all the objects needed for an x-y plot in object graphics.
;
; AUTHORS:
;       FANNING SOFTWARE CONSULTING
;       David Fanning, Ph.D.
;       1645 Sheely Drive
;       Fort Collins, CO 80526 USA
;       Phone: 970-221-0438
;       E-mail: davidf@dfanning.com
;       Coyote's Guide to IDL Programming: http://www.dfanning.com
;
;       Modified 09/12/03 SM:
;       - This is no longer a stand-alone widget program, but a re-usable routine which
;         returns the IDLgrModel root of the plot's object hierarchy.
;       - Removed many bells & whistles.
;
; CATEGORY:
;       Widgets, IDL 5 Object Graphics.
;
; CALLING SEQUENCE:
;       XPlot, x, y, model
;
; REQUIRED INPUTS:
;       x: A vector of input values used as the independent data.
;       y: A vector of input values used as the dependent data.
;       model: This is only required if the OPLOT keyword is set.
;              An IDLgrModel containing all the objects: IDLgrPlot, IDLgrAxis, etc.
;
; OUTPUTS:
;       model: This is only set if the OPLOT keyword is NOT set.
;              An IDLgrModel containing all the objects: IDLgrPlot, IDLgrAxis, etc.
;
; OPTIONAL KEYWORD PARAMETERS:
;
;       _EXTRA: This keyword collects otherwise undefined keywords that are
;       passed to new Plot command. To some extent these are similar to the
;       old IDL Plot command. For example: Linestyle=2, Thick=3,
;       XRange=[-100,100], etc.
;
;       POSITION: A four-element array of the form [x0, y0, x1, y1] for locating
;       the axes of the plot in the display window. The coordinates are in
;       "normalized" units, which means the extent of the window is from 0.0 to 1.0.
;       The default value of POSITION is [0.15, 0.15, 0.925, 0.925].
;
;       PSYM: The index of a plotting symbol to use on the plot. Integers 0-7
;       are valid values.
;
;       SYMSIZE: Sets the size of the symbols. By default, symbols are sized
;       so that they are 0.015 percent of the axis range.
;
;       VECTOR: Set this keyword if you want the printed output to be in
;       vector (as opposed to bitmap) form. This is faster, but not as accurate.
;
;       TITLE: A string used as the title of the plot.
;
;       XTITLE: A string used as the X title of the plot.
;
;       YTITLE: A string used as the Y title of the plot.
;
;       xAxisKeywords: A structure with keywords accepted by IDLgrAxis::Init, e.g.
;                      xAxisKeywords={exact: 1}
;                      Caveats:
;                      - Don't include TITLE in the structure; use the XTITLE keyword.
;                      - Don't include RANGE in the structure; use the XRANGE keyword.
;
;       yAxisKeywords: A structure with keywords accepted by IDLgrAxis::Init, e.g.
;                      yAxisKeywords={exact: 1}
;                      Caveats:
;                      - Don't include TITLE in the structure; use the YTITLE keyword.
;                      - Don't include RANGE in the structure; use the YRANGE keyword.
;
;       OPLOT:  The zero-based child index under the model of the IDLgrPlot to modify.
;               When OPLOT is specified, the model argument is used as an input and
;               not changed on return.  If the indexed object isn't an IDLgrPlot, i.e.
;               the IDLgrPlot hasn't been created yet, then a new IDLgrPlot is created
;               and initialized with all the same values as the first IDLgrPlot.
;               Thus you don't have to re-specify XRANGE, YRANGE, THICK, etc.
;               Then the keywords in _extra are passed to IDLgrPlot::SetProperty.
;               Notes:
;               1) Whenever you specify a new value of the oplot keyword, make sure the value
;                  is one greater than the last oplot value, or 1 if it's the first call
;                  specifying oplot.
;               2) psym and symsize are not handled yet.
;
; EXAMPLE:
;       To use this program, pass a 1D vector or vectors, like this:
;
;        IDL> XPlot, RandomU(seed, 11) * 9, YRange=[0, 10]
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  1997-2002 Fanning Software Consulting.
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################



FUNCTION Normalize, range, Position=position

    ; This is a utility routine to calculate the scaling vector
    ; required to position a vector of specified range at a
    ; specific position given in normalized coordinates. The
    ; scaling vector is given as a two-element array like this:
    ;
    ;   scalingVector = [translationFactor, scalingFactor]
    ;
    ; The scaling vector should be used with the [XYZ]COORD_CONV
    ; keywords of a graphics object or model. For example, if you
    ; wanted to scale an X axis into the data range of -0.5 to 0.5,
    ; you might type something like this:
    ;
    ;   xAxis->GetProperty, Range=xRange
    ;   xScale = Normalize(xRange, Position=[-0.5, 0.5])
    ;   xAxis, XCoord_Conv=xScale

IF (N_Elements(position) EQ 0) THEN position = [0.0, 1.0] ELSE $
    position=Float(position)
range = Float(range)

scale = [((position[0]*range[1])-(position[1]*range[0])) / $
    (range[1]-range[0]), (position[1]-position[0])/(range[1]-range[0])]

RETURN, scale
END
;-------------------------------------------------------------------------



PRO Rave_XPlot, xx, yy, plotModel, _extra=_extra, PSym=psym, Title=title, SymSize=symSize, $
                XTitle=xtitle, YTitle=ytitle, Exact=exact, Vector=vector, Position=position, $
                axisColor=axisColor, xAxisKeywords=xAxisKeywords, yAxisKeywords=yAxisKeywords, $
                oplot=oplot

x = xx
y = yy

; Check keyword parameters.

IF N_Elements( psym) EQ 0 THEN psym = 0
IF N_Elements( position) EQ 0 THEN position = [0.15, 0.20, 0.97, 0.97]
IF N_Elements( title) EQ 0 THEN title = ''
IF N_Elements( symsize) EQ 0 THEN symsize = 1.0
IF N_Elements( xtitle) EQ 0 THEN xtitle = ''
IF N_Elements( ytitle) EQ 0 THEN ytitle = ''
IF N_Elements( axisColor) EQ 0 THEN axisColor=[255,255,0]  ;yellow
vector = Keyword_Set( vector)

if N_Elements( oplot) eq 1 then begin  ;oplot is defined;  it may be 0
  plotObj = plotModel->Get( position=oplot, ISA='IDLgrPlot')
  if not obj_valid( plotObj) then begin
    firstPlotObj = plotModel->Get( position=0, ISA='IDLgrPlot')
    if not obj_valid( firstPlotObj) then throw, "Can't specify OPLOT on first call!"
    firstPlotObj->GetProperty, all=all
    plotObj = obj_new( 'IDLgrPlot', _extra=all)
    plotModel->add, plotObj, position=oplot
  endif
  plotObj->SetProperty, datax=x, datay=y, _extra=_extra
  return
endif

; Create title objects for the axes.

xTitle = Obj_New('IDLgrText', xtitle, color=axisColor)
yTitle = Obj_New('IDLgrText', ytitle, color=axisColor)

; Make a symbol object. Color symbols cyan.

thisSymbol = Obj_New('IDLgrSymbol', psym, Color=[0, 255, 255])

; Make a font object.

helvetica10pt = Obj_New('IDLgrFont', 'Helvetica', Size=10)

; Create a plot object. The plot will be in the coordinate
; space 0->1. The view will be in the range -0.35->1.25 so
; that the plot axis annotation will be visible.

thisPlot = Obj_New("IDLgrPLOT", x, y, _Extra=_extra, Symbol=thisSymbol)

; Get the data ranges from the Plot Object.

thisPlot->GetProperty, XRange=xrange, YRange=yrange

; Create plot box style axes.
; The large values in the LOCATION keyword indicates which
; values are NOT used. The axes text is set to Helvetica
; 10 point font.

xAxis1 = Obj_New("IDLgrAxis", 0, Color=axisColor, Ticklen=0.025, $
                 Range=xrange, Title=xtitle, Location=[1000, position[1] ,0], $
                 _extra=xAxisKeywords)
xAxis1->GetProperty, Ticktext=xAxisText
xAxisText->SetProperty, Font=helvetica10pt

xAxis2 = Obj_New("IDLgrAxis", 0, Color=axisColor, Ticklen=0.025, $
                 /NoText, Range=xrange, TickDir=1, Location=[1000, position[3], 0], $
                 _extra=xAxisKeywords)

yAxis1 = Obj_New("IDLgrAxis", 1, Color=axisColor, Ticklen=0.025, $
                 Title=ytitle, Range=yrange, Location=[position[0], 1000, 0], $
                 _extra=yAxisKeywords)
yAxis1->GetProperty, Ticktext=yAxisText
yAxisText->SetProperty, Font=helvetica10pt

yAxis2 = Obj_New("IDLgrAxis", 1, Color=axisColor, Ticklen=0.025, $
                 /NoText, Range=yrange, TickDir=1, Location=[position[2], 1000, 0], $
                 _extra=yAxisKeywords)

; Because we may not be using exact axis ranging, the axes
; may extend further than the xrange and yrange. Get the
; actual axis range so that the plot, etc. can be scaled
; appropriately.

xAxis1->GetProperty, CRange=xrange
yAxis1->GetProperty, CRange=yrange

; Set up the scaling so that the axes for the plot and the
; plot data extends from 0->1 in the X and Y directions.

xs = Normalize( xrange, Position=[position[0], position[2]])
ys = Normalize( yrange, Position=[position[1], position[3]])

; Scale the plot data and axes into 0->1.

thisPlot->SetProperty, XCoord_Conv=xs, YCoord_Conv=ys
xAxis1->SetProperty, XCoord_Conv=xs
xAxis2->SetProperty, XCoord_Conv=xs
yAxis1->SetProperty, YCoord_Conv=ys
yAxis2->SetProperty, YCoord_Conv=ys

; Size the symbols appropriately for the plot.

xSymSize = (xrange[1] - xrange[0]) * 0.015 * symSize
ySymSize = (yrange[1] - yrange[0]) * 0.015 * symSize
IF Obj_Valid(thisSymbol) THEN thisSymbol->SetProperty, Size=[xSymSize, ySymSize]

  ; Create a plot title. Center it at a location above the plot.

helvetica14pt = Obj_New('IDLgrFont', 'Helvetica', Size=14)
plotTitle = Obj_New('IDLgrText', title, Color=axisColor, $
                    Location=[0.5, 1.05, 0.0], Alignment=0.5, Font=helvetica14pt)

; Create a plot model and add axes, plot, and plot title to it.

plotModel = Obj_New('IDLgrModel')

;thisPlot should be the first child of plotModel;  subsequent oplots add a
;IDLgrPlot object after the last IDLgrPlot object.  Thus the IDLgrPlot objects
;associated with the first plot and subsequent oplots can be accessed by zero-based
;child index.

plotModel->Add, thisPlot
plotModel->Add, xAxis1
plotModel->Add, xAxis2
plotModel->Add, yAxis1
plotModel->Add, yAxis2
plotModel->Add, plotTitle

if 0 then begin
; Create a view and add the plot model to it. Notice that the view
; is larger than the 0->1 plot area to accomodate axis annotation.
; The view will have a gray background.

plotView = Obj_New('IDLgrView', Viewplane_Rect=[0.0, 0.0, 1.0, 1.0], $
                   Location=[0,0], Color=[80,80,80])
plotView->Add, plotModel
endif

END
;------------------------------------------------------------------------


