; Routines for handling time
; Xinlin Version -  rld code modified by sm 3/7/00
; The value in DT.days is referred to as "modified UNIX system time":
; fractional-days since 1970/001-00:00:00.
; E.g. 1970/001-12:00:00 = 0.5
; 07/25/02 SM - Modifications to allow negative times, for handling the Dst
;               dataset starting in 1957.

; DEFINE_TIME: Defines a structure that holds a calendar date and clock time.

pro DEFINE_TIME

  ON_ERROR, 2

  ; Define the structure for holding a calendar date and clock time in
  ; modified UNIX system time format, fractional days since Jan 1, 1970.

  dt = {DT, DAYS: double(0.0)}  ;Changed to double 9/25/00 SM. TBD if get any problems
                                ;from this.  See discussion in CLOCK_TO_DT below.

  return

end


; NEW_DT: Create a new DT structure and optionally set its date and time.
; The output is determined as follows:
;
;   - If all parameters are omitted, then a single "null" DT structure is
;     returned with the days and time components set to zero.
;   - If only the SIZE parameter is present, then an array of null DT
;     structures is returned with the number of array elements determined
;     by the value of the SIZE parameter.
;   - If the DAYS and TIME parameters are present and are both scalar values
;     and the SIZE parameter is not present, then the output will be a
;     single DT structure with its components set to the input values.
;   - If the DAYS and TIME parameters are present and are scalar values
;     and the SIZE parameter is present, then the output will be an
;     array of DT structures with the components of each array element
;     set to the input parameter values.
;   - If one or both of the DAYS and TIME parameters are an array, then
;     the output will be an array of DT structures.  If one of these
;     parameters is an array then the other must be either an array
;     of the same length or a scalar value. The SIZE parameter is
;     ignored if either input parameter is an array.

function NEW_DT, days, SIZE=size

  ON_ERROR, 2

  np = N_PARAMS ()
  if np ne 0 and np ne 1 then MESSAGE, "Wrong number of parameters"

  ns = N_ELEMENTS (days)
  if KEYWORD_SET (SIZE) then nl = size else nl = 1
  if ns le 1 then ns = nl

  ; Create DT structure(s) with a null value

  dt = REPLICATE ({DT, DAYS: double(0.0)}, ns)

  if np eq 1 then begin

    ; Fill in the output DT structures with the supplied date and time values

    dt.days = days

  endif

  return, dt

end


; DT_ADD_SECS: Add a delta time, in seconds, to a date/time, returning 
; a date/time value. There are four possible permutations of input
; parameters, as shown below.  If both inputs are scalar, then the
; result is scalar; otherwise the result is an array with the same
; length as the input array.  For the last case below, both input
; arrays must have the same length.
;
;   DT_IN    SECS     RESULT
;
;   SCALAR   SCALAR   SCALAR -- dt_out = dt_in + secs
;   SCALAR   ARRAY    ARRAY  -  dt_out(i) = dt_in + secs(i)
;   ARRAY    SCALAR   ARRAY  -- dt_out(i) = dt_in(i) + secs
;   ARRAY    ARRAY    ARRAY  -- dt_out(i) = dt_in(i) + secs(i)

function DT_ADD_SECS, dt_in, secs

  ON_ERROR, 2

  len1 = N_ELEMENTS (dt_in)
  len2  = N_ELEMENTS (secs)

  minlen = len1 < len2
  maxlen = len1 > len2

  if ((minlen ne 1) and (minlen ne maxlen)) then $
    MESSAGE, "When both arguments are arrays, they must be the same length"

  dt_out = NEW_DT (dt_in.days + secs/double(86400.))  ;SM 9/25/00

  return, dt_out

end


; DT_DIF: Calculate the difference, in seconds, between two date/times. 
; DT_IN1 may be a scalar DT structure or an array of DT structures.  If
; it is an array, then DT_IN2 is subtracted from each array value and
; an array value is returned

function DT_DIF, dt_in1, dt_in2

  ON_ERROR, 2

  secs = (dt_in1.days - dt_in2.days)*86400.D0

return, secs

end


; CLOCK_TO_DT: Convert a clock time (yr, day-of-year, hour, minutes, seconds)
; to modified UNIX system time format.  Input parameters can be either scalar
; or an array.  If any parameter is an array, then the other parameters must be
; either a scalar value or an array of the same length.

function CLOCK_TO_DT, year, doy, hh, mm, ss

  ON_ERROR, 2

  ; Check the input parameters to see if any are arrays

  ny = N_ELEMENTS (year)
  nd = N_ELEMENTS (doy)
  nh = N_ELEMENTS (hh)
  nm = N_ELEMENTS (mm)
  ns = N_ELEMENTS (ss)

  nt = ny > nd > nh > nm > ns

  dt_out = NEW_DT (SIZE=nt)

  yr = long(year) - 1970
  ;...,1964,1968,1972,1976,1980,...are leap years
  dt_out.days = yr*365L + (yr gt 0)*(yr+1)/4 + (yr lt 0)*(yr-2)/4 + doy - 1  
  dt_out.days = dt_out.days + (hh*3600L + mm*60L + ss)/double(86400.)
  
  ;Note: we're using doubles now, because a float is only accurate to within about 43 seconds
  ;for a year of 1996.  Lose accuracy in seconds place because
  ;1996 +  1/86400. = xxxx.000000 instead of xxxx.000012 and in fact,
  ;1996 + 42/86400. = xxxx.000000 as well.
  ;print,format='(A,I2,A,F12.6)','CLOCK_TO_DT: ss = ',ss,', dt_out.days = ',dt_out.days  
  return, dt_out

end


; DT_TO_STRING: Provides the UTC corresponding to the date-time 
; as a 19-character ASCII string.  The input may be either a
; scalar or an array, and the output has the same number of
; elements.  Good for dates from Jan 1, 1901 to Dec 30, 2100.
; The null date (0) is represented as 1970/001-00:00:00.
; The "magic number" 1461 is the number of days in any 4 year
; period: 1461 = 365*3 + 366.

function DT_TO_STRING, dt_in

  ON_ERROR, 2
  
  offset = (1970-1901)*365 + 17     ;17 leap years since 1901: 1904,...,1968
  x = long(dt_in.days + offset)/1461 ;integer # of 4 yr periods since 1901 (3*365 + 366)
  y = dt_in.days + offset - x*1461  ;remainder in days (0..1460.999...)
  z = (fix(y)/365) < 3              ;integer # of full yr periods in y (0,1,2 or 3)

  year = 1901 + x*4 + z
  doy  = y - z*365 + 1
  sec_of_day = long( round((dt_in.days - floor( dt_in.days)) * 86400.))
  hh = sec_of_day/3600
  mm = (sec_of_day mod 3600) / 60
  ss =  sec_of_day mod 60
  
  ns = N_ELEMENTS (dt_in)
  if ns eq 1 then begin

    ; Input is scalar, so return a scalar value

    timestring_out = STRING (format="(I4, '/', I3.3, '-', I2.2, ':',"$
       +"I2.2, ':', I2.2)", year, doy, hh, mm, ss)

  endif else begin

    ; Input is an array, so return an array

    timestring_out = STRARR (ns)

    for i = 0, ns-1 do begin
      timestring_out(i) = STRING( format="(I4, '/', I3.3, '-', I2.2, ':',I2.2, ':', I2.2)", $    
                                  year(i), doy(i), hh(i), mm(i), ss(i))

    endfor
   
  endelse

  return, timestring_out

end


; STRING_TO_DT: Convert a date/time in ASCII character string format
; to date/time value. A date/time structure with the converted time is
; returned.  If the time string in not valid, then the return date
; and time are set to 0.  Currently the input can be a scalar
; value only

function STRING_TO_DT, time_string_in

  ON_ERROR, 2

  dt_out = NEW_DT ()
  hh = 0
  mm = 0
  ss = 0.

  ; Trim off any leading or trailing blanks and calculate length of string

  time_string = STRTRIM (time_string_in, 2)
  nl = STRLEN (time_string)

  ; Look for the slash that separates the year and day-of-year

  n = STRPOS (time_string, '/')

  if n gt 0 then begin

    ; We found a slash: interpet what is before it as the year

    year = FIX (STRMID (time_string, 0, n))

    ; Look for the dash that separates doy from hours

    time_string = STRMID (time_string, n+1, nl)
    n = STRPOS (time_string, '-')

    if n ge 0 then begin

      ; We found the dash: interpet what is before it as the day-of-year

      doy = FIX (STRMID (time_string, 0, n))

      ; Look for a colon separating hours from minutes

      time_string = STRMID (time_string, n+1, nl)
      n = STRPOS (time_string, ':')

      if n ge 0 then begin

        ; We found a colon: interpret what is before is as hours

        hh = FIX (STRMID (time_string, 0, n))

        ; Look for the colon separating minutes from seconds

        time_string = STRMID (time_string, n+1, nl)
        n = STRPOS (time_string, ':')

        if n gt 0 then begin

          ; We found the colon: interpret what is before it as minutes and
          ; what is after as seconds

          mm = FIX (STRMID (time_string, 0, n))
          ss = FLOAT (STRMID (time_string, n+1, nl))

        endif else begin

           ; We didn't find any seconds: so assume they are 0.0
 
           mm = FIX (STRMID (time_string, 0, nl))

        endelse

        dt_out = CLOCK_TO_DT (year, doy, hh, mm, ss)

      endif else begin

        ; We didn't find minutes or seconds: so assume they are 00:00.0

        hh = FIX (STRMID (time_string, 0, nl))

        dt_out = CLOCK_TO_DT (year, doy, hh, mm, ss)

      endelse

    endif else begin

      ; We didn't find hours, minutes or seconds, so assume 00:00:00.0

      doy = FIX (STRMID (time_string, 0, nl))

      dt_out = CLOCK_TO_DT (year, doy, hh, mm, ss)

    endelse

  endif

  return, dt_out

end




; split_time: Provides the UTC corresponding to the TCAD fractional days since 1970 as
; yr, doy, hour, min, sec.  The input may be either a scalar or an array, and the
; outputs have the same number of elements.  Good for dates from Jan 1, 1901 to
; Dec 30, 2100.  The null date (0) is represented as 1970/001-00:00:00.
; The "magic number" 1461 is the number of days in any 4 year
; period: 1461 = 365*3 + 366.

pro split_time, time_in, year, doy, hh, mm, ss, month, day

  offset = (1970-1901)*365 + 17     ;17 leap years since 1901: 1904,...,1968
  x = long(time_in + offset)/1461   ;integer # of 4 yr periods since 1901 (3*365 + 366)
  y = time_in + offset - x*1461     ;remainder in days (0..1460.999...)
  z = (fix(y)/365) < 3              ;integer # of full yr periods in y (0,1,2 or 3)

  year = 1901 + x*4 + z
  doy  = fix(y) - z*365 + 1
  sec_of_day = long( round((time_in - floor( time_in)) * 86400.))
  hh = sec_of_day/3600
  mm = (sec_of_day mod 3600) / 60
  ss =  sec_of_day mod 60
  
  if n_params() gt 6 then begin
    ;WARNING!  Code only works on scalar time_in from here on!
    days_in_month = [31,28,31,30,31,30,31,31,30,31,30,31]
    if year mod 4 eq 0 then days_in_month[1] = 29
    last_doy_in_month = intarr(12)
    for I=0,11 do last_doy_in_month[I] = total( days_in_month[0:I])  ;last_doy_in_month = [31,59,...]
    W = where( last_doy_in_month ge doy)
    month = W[0] + 1
    day = doy
    if day gt 31 then day = doy - last_doy_in_month[ month-2]
  endif
  return
END

function t90_to_t, t90
  ;T90 is a long, seconds since 1990/001-00:00:00.
  fractional_days_since_1990 = t90/86400.0d
  days_between_1990_and_1970 = ((1990 - 1970)*365 + 5)  ;1972, 1976, 1980, 1984, 1988
  return, fractional_days_since_1990 + days_between_1990_and_1970
END

function t_to_t90, t
  ;T90 is a long, seconds since 1990/001-00:00:00.
  days_between_1990_and_1970 = ((1990 - 1970)*365 + 5)  ;1972, 1976, 1980, 1984, 1988
  fractional_days_since_1990 = t - days_between_1990_and_1970
  return, long( fractional_days_since_1990*86400.0d)
END

forward_function convert_date_to_t90, str_to_t

function t_to_mjdt, t
  convert_t90_to_Date, t_to_t90(t), year, doy, hour, minutes, seconds
  mjdt = convert_date_to_t90( year=year, doy=doy, hour=hour, minutes=minutes, second=seconds, /mjdt)
  return, mjdt
END

forward_function utc2tai, tai2utc
function mjdt_to_t, mjdt         ;There may be a better way to do this...
  tai = utc2tai( {mjd:mjdt.mjd, time:mjdt.t*1000})
  str = tai2utc( tai,  /ecs)
  space_index =  strpos( str, ' ')
  str = strmid( str, 0, space_index) + '-' + strmid( str, space_index+1)
  tcad_time_string__parse_date_time, str, dt, flag
  return, dt.days
END

;cdf time is a double, msec since epoch, which is 1 B.C. (?)
function cdf_to_t, cdf
  return, cdf/86400000.0d - 719528.0d
END

pro tcad_time
  resolve_routine, 'DEFINE_TIME'
  resolve_routine, 'SPLIT_TIME'
  resolve_routine, 'NEW_DT', /is_function
  resolve_routine, 'DT_ADD_SECS', /is_function
  resolve_routine, 'DT_DIF', /is_function
  resolve_routine, 'CLOCK_TO_DT', /is_function
  resolve_routine, 'DT_TO_STRING', /is_function
  resolve_routine, 'STRING_TO_DT', /is_function
  resolve_routine, 'T90_TO_T', /is_function
  resolve_routine, 'T_TO_T90', /is_function
  resolve_routine, 'T_TO_MJDT', /is_function
  resolve_routine, 'MJDT_TO_T', /is_function
  resolve_routine, 'CDF_TO_T', /is_function
  return
END


pro test_tcad_time
leap_yrs = [1964,1968,1972,1976,1988,1992,1996,2000,2004]
for leap_yr_ind=0,n_elements( leap_yrs)-1 do begin
  leap_yr = leap_yrs[ leap_yr_ind]
  strs = [strtrim(leap_yr+0,2) + '/364-23:00:00', $
          strtrim(leap_yr+0,2) + '/365-23:00:00', $
          strtrim(leap_yr+0,2) + '/366-23:00:00', $
          strtrim(leap_yr+1,2) + '/001-00:00:00', $
          strtrim(leap_yr+1,2) + '/365-23:00:00', $
          strtrim(leap_yr+2,2) + '/001-00:00:00', $
          strtrim(leap_yr+2,2) + '/365-00:00:00', $
          strtrim(leap_yr+2,2) + '/365-23:00:00', $
          strtrim(leap_yr+3,2) + '/001-00:00:00', $
          strtrim(leap_yr+3,2) + '/365-00:00:00', $
          strtrim(leap_yr+4,2) + '/001-00:00:00', $
          strtrim(leap_yr+4,2) + '/365-00:00:00', $
          strtrim(leap_yr+4,2) + '/366-00:00:00', $
          strtrim(leap_yr+5,2) + '/001-00:00:00']
  for i=0,n_elements(strs)-1 do begin
    t = str_to_t( strs[i])
    print,strs[i] + ' = ' + strtrim(t,2)
    s = t_to_str( t)
    if strs[i] ne s then begin
      print,'str_to_t( ' + strs[i] + ') failed!'
      stop
    endif
  endfor
endfor

END
