;****************************************************************************
; TCAD_TIME_STRING - Implementation of Time Strings for TCAD
; Author: R. Davis  Last Update: 31 Jan 1999
;
; Component TCAD_TIME_STRING
;
; TCAD Time Strings come in three flavors.
;
;   Absolute Date/Time - Date and Time fully specified
;
;     ADATE [-] TIME
;
;     where ADATE is either [YYYY]/DOY or [YYYY]/MM/DD
;     and TIME has the general format HH[:MM[:SS[.SSS]]]
;
;   Relative Date/Time - Date specified relative to current day
;
;     RDATE TIME
;
;     where RDATE has the general form /-n
;     TIME has the same form as for the absolute format
;
;   Delta Time:
;
;     +TIME or - TIME
;
;     where TIME is as described above.  The time is interpreted
;     as a delta from some epoch.
;
; Methods:
;
;   PARSE_DATE_TIME  - Parse an absolute or relative date/time
;   PARSE_DELTA_TIME - Parse a delta time
;   PARSE_TIME       - General time parser (used by the other routines)
;
; Notes:
;
; 1) PARSE_DATE_TIME returns a DT date/time structure
; 2) PARSE_DELTA and PARSE_TIME return a double precision floating point value
;    which represents the delta time in seconds
; 3) All methods return the flag parameter as follows:
;    0 -- Not a valid time string
;    1 -- Valid time string and output value is set
; 4) In general spaces between parts of a time string are ignored, except
;    that if there is no dash between the date and time parts of a
;    string then there must be at least one space
;
;"$Revision: 1.1 $"
;"$Date: 2003/11/21 21:02:31 $"
;"$Author: friedel $"
;"$RCSfile: tcad_time_string.pro,v $"
;
;
;****************************************************************************

pro TCAD_TIME_STRING__PARSE_DATE_TIME, time_string_in, dt_out, flag

  ON_ERROR, 2

  flag = 0

  month = [0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334]

  time_string = time_string_in

  ; If the date and time are separated by a dash, turn it into a space

  n = RSTRPOS (time_string, '-')
  if n gt 0 and STRMID (time_string, n-1, 1) ne '/' then $
    STRPUT, time_string, ' ', n
 
  ; Squeeze down any white space between date and time to a single space

  time_string = STRCOMPRESS (time_string)

  ; Now cleave the string into a date part and a time part at the space

  time_string = STR_SEP (time_string, ' ')
 
  ; Process the date part

  date_string = LONG (STR_SEP (time_string(0), '/'))
  n = N_ELEMENTS (date_string)

  case n of 

   2: begin

    year = date_string(0)
    doy  = date_string(1)

    ; See if the year was specified

    if (year eq 0) then begin

      current_date = BIN_DATE (SYSTIME())
      year = current_date(0)

      ; See if the day number is relative

      if (doy le 0) then begin

        doy = month(current_date(1)-1) + current_date(2) + doy
        if year mod 4 eq 0 and current_date(1) gt 2 then doy = doy + 1

        if doy le 0 then begin

          ; DOY is in previous year

          year = year - 1
          if year mod 4 eq 0 then maxdays = 366 else maxdays = 365
          doy = max_days - doy

        endif

      endif

    endif

   end

   3: begin

    year  = date_string(0)
    mon   = date_string(1)
    day   = date_string(2)

    ; See if the year was specified

    if (year eq 0) then begin

      current_date = BIN_DATE (SYSTIME())
      year = current_date(0)

    endif

    ; Convert the month and day to day-of-year

      doy = month(mon-1) + day
      if year mod 4 eq 0 and mon gt 2 then doy = doy + 1

   end

   else: return

  end

  ; Convert date to DT format

  dt_out = CLOCK_TO_DT (year, doy, 0, 0, 0.)

  ; Parse the time, if present

  if N_ELEMENTS (time_string) gt 1 then begin

    TCAD_TIME_STRING__PARSE_TIME, time_string(1), time, flag

    if flag then dt_out = DT_ADD_SECS (dt_out, time)

  endif else flag = 1

  return

end


pro TCAD_TIME_STRING__PARSE_DELTA_TIME, time_string_in, time_out, flag

  ON_ERROR, 2

  flag = 0

  ; Get rid of all white space

  time_string = STRCOMPRESS (time_string_in, /REMOVE_ALL)
  nl = STRLEN (time_string)

  ; Look for the plus or minus sign that would signify that this is a
  ; delta time

  sign = STRMID (time_string, 0, 1)

  if sign ne '+' and sign ne '-' then return

  ; We did find a sign.  Parse the time

  time_string = STRMID (time_string, 1, nl)

  TCAD_TIME_STRING__PARSE_TIME, time_string, time_out, flag

  if sign eq '-' then time_out = -time_out

  flag = 1

  return

end


pro TCAD_TIME_STRING__PARSE_TIME, time_string_in, time_out, flag

  ON_ERROR, 2

  flag = 1

  ; Separate the time string into it's parts (HH, MM, SS)

  part = DOUBLE (STR_SEP (time_string_in, ':'))
  n = N_ELEMENTS (part)

  case n of 
   1:    time_out = part(0)*3600.D0
   2:    time_out = part(0)*3600.D0 + part(1)*60.D0
   3:    time_out = part(0)*3600.00 + part(1)*60.D0 + part(2)
   else: flag = 0
  end

  return

end

function string_to_t, str
  tcad_time_string__parse_date_time, str, dt_out
  T = dt_out.days
  return, T
END

function t_to_string, T
  str = dt_to_string( new_dt(T))
  return, str
END

function str_to_t, str
  tcad_time_string__parse_date_time, str, dt_out
  T = dt_out.days
  return, T
END

function t_to_str, T
  str = dt_to_string( new_dt(T))
  return, str
END

pro tcad_time_string
resolve_routine,'TCAD_TIME_STRING__PARSE_DATE_TIME'
resolve_routine,'TCAD_TIME_STRING__PARSE_DELTA_TIME'
resolve_routine,'TCAD_TIME_STRING__PARSE_TIME'
resolve_routine,'T_TO_STRING', /is_function
resolve_routine,'STRING_TO_T', /is_function
resolve_routine,'T_TO_STR',    /is_function
resolve_routine,'STR_TO_T',    /is_function
return
END
