;*******************************************************************************
; Purpose: TrackballMgr creates a Trackball object, buttons to choose rotation,
;          translation or zoom, and provides event handlers for the buttons
;          and for mouse movement.  The mouse event handler consults the
;          Trackball object, and depending on the value of common block variable
;          'mode', sets the transformation matrix to rotate, translate or scale
;          the model.
;
; Inputs: WIDGET_BASE, IDLgrWindow and IDLgrView or IDLgrScene
;         The buttons are added to the WIDGET_BASE;  the window and view or scene
;         are stored and used in the draw event handler to redraw the scene:
;         window->Draw, view or scene
;
; Event Handlers and Generated Event
; 1) The caller's code must supply 'TrackballMgr__handleDrawEvent' as the event handler
;    function in the EVENT_FUNC keyword for the caller's WIDGET_DRAW.  This event
;    handler generates an event of this type when the user lets up on the mouse
;    This event tells whatever event handler catches it that the user has let up on the
;    mouse button after dragging it around (rotate, translate or scale):
;    event = {id: draw_widget_id, top: ev.top, handler: ev.top, value: 'TrackballDragFinished'}
;
; 2) TrackballMgr automatically supplies 'TrackballMgr__handleButtonEvent' as the
;    EVENT_PRO routine for the buttons' event handler.
;
; Design Notes:
; 1) At the time I wrote this, I didn't know how to make it an object the caller could
;    have multiple instances of.  You can't supply an object method to the EVENT_PRO
;    keyword, and the non-object event handlers don't have access to instance since they're
;    not part of the object.  So it's not obvious how to have an object provide its own
;    event handlers.   The Coyote site's FSC_FileSelect.pro/FSC_FileSelect_Event_Handler
;    shows the *right* way to do it.  Sometime I should rewrite this code.
;*******************************************************************************

pro TrackballMgr__handleButtonEvent, ev

common TrackballMgr_common, window, view, model, trackball, buttonDown, $
                            rotateButton, translateButton, zoomButton, $
                            toolTipLabel, mode
;Display a tool tip help string whenever mouse goes over a button.
if TAG_NAMES( ev, /STRUCTURE_NAME) eq 'WIDGET_TRACKING' then begin
  ids = [rotateButton, translateButton, zoomButton]
  toolTipStrings = ['rotate with mouse', 'translate with mouse', 'zoom with mouse']
  for i=0,2 do begin
    if ids[i] eq ev.id then begin
      if ev.enter then begin
        WIDGET_CONTROL, toolTipLabel, SET_VALUE=toolTipStrings[i]
      endif else begin
        WIDGET_CONTROL, toolTipLabel, SET_VALUE=""
        return
      endelse
    endif
  endfor
  return
endif

WIDGET_CONTROL, ev.id, GET_UVALUE=uval
subdirectory = ['resource', 'bitmaps']
case uval of
  
  'rotateActiveButton': begin
    mode = -1  
    WIDGET_CONTROL, rotateButton, SET_UVALUE='rotateInactiveButton', /BITMAP, $
                    SET_VALUE=filepath( 'rotate.bmp', subdirectory=subdirectory)
  end
  
  'rotateInactiveButton': begin
    mode = 0
    WIDGET_CONTROL, rotateButton, SET_UVALUE='rotateActiveButton', /BITMAP, $
                    SET_VALUE=filepath( 'rotate_active.bmp', subdirectory=subdirectory)
    WIDGET_CONTROL, translateButton, SET_UVALUE='translateInactiveButton', /BITMAP, $
                    SET_VALUE=filepath( 'pan.bmp', subdirectory=subdirectory)
    WIDGET_CONTROL, zoomButton, SET_UVALUE='zoomInactiveButton', /BITMAP, $
                    SET_VALUE=filepath( 'zoom.bmp', subdirectory=subdirectory)
  end
  
  'translateActiveButton': begin
    mode = -1  
    WIDGET_CONTROL, translateButton, SET_UVALUE='translateInactiveButton', /BITMAP, $
                    SET_VALUE=filepath( 'pan.bmp', subdirectory=subdirectory)
  end
  
  'translateInactiveButton': begin
    mode = 1
    WIDGET_CONTROL, translateButton, SET_UVALUE='translateActiveButton', /BITMAP, $
                    SET_VALUE=filepath( 'pan_active.bmp', subdirectory=subdirectory)
    WIDGET_CONTROL, rotateButton, SET_UVALUE='rotateInactiveButton', /BITMAP, $
                    SET_VALUE=filepath( 'rotate.bmp', subdirectory=subdirectory)
    WIDGET_CONTROL, zoomButton, SET_UVALUE='zoomInactiveButton', /BITMAP, $
                    SET_VALUE=filepath( 'zoom.bmp', subdirectory=subdirectory)
  end
  
  'zoomActiveButton': begin
    mode = -1  
    WIDGET_CONTROL, zoomButton, SET_UVALUE='zoomInactiveButton', /BITMAP, $
                    SET_VALUE=filepath( 'zoom.bmp', subdirectory=subdirectory)
  end
  
  'zoomInactiveButton': begin
    mode = 2
    WIDGET_CONTROL, zoomButton, SET_UVALUE='zoomActiveButton', /BITMAP, $
                    SET_VALUE=filepath( 'zoom_active.bmp', subdirectory=subdirectory)
    WIDGET_CONTROL, rotateButton, SET_UVALUE='rotateInactiveButton', /BITMAP, $
                    SET_VALUE=filepath( 'rotate.bmp', subdirectory=subdirectory)
    WIDGET_CONTROL, translateButton, SET_UVALUE='translateInactiveButton', /BITMAP, $
                    SET_VALUE=filepath( 'pan.bmp', subdirectory=subdirectory)
  end
  
endcase
END


;*******************************************************************************
function TrackballMgr__handleDrawEvent, ev    ;Handle events for the draw window
;*******************************************************************************

common TrackballMgr_common, window, view, model, trackball, buttonDown, $
                            rotateButton, translateButton, zoomButton, $
                            toolTipLabel, mode
if (tag_names( ev, /STRUCTURE_NAME) ne 'WIDGET_DRAW') then return, ev

if (ev.type eq 4) then begin    ;Expose.
  WIDGET_CONTROL, ev.top, /HOURGLASS ;Redraw entire window
  window->draw, view
  return,0
endif

if ev.type ge 0 and ev.type le 2 then begin
  if mode eq -1 then return,0
  bHaveTransform = trackball->Update( ev, TRANSLATE=mode, TRANSFORM=qmat) ;trackball update
  if (bHaveTransform ne 0) then begin
    if mode eq 2 then begin  ;zoom
      ytranslation = qmat[3,1]
      model->scale, 1 + ytranslation, 1 + ytranslation, 1 + ytranslation
    endif else begin         ;rotate or translate
      model->GetProperty, TRANSFORM=t
      model->SetProperty,TRANSFORM= t # qmat
    endelse
  endif
  if (ev.type eq 0) then begin                           ;Mouse button press.
    if ev.press eq 1 then begin                          ;Left button.
      buttonDown = 1
      WIDGET_CONTROL, ev.id, /DRAW_MOTION                ;Enable mouse motion events.
    endif
  endif else if ((ev.type eq 2) and (buttonDown eq 1)) then begin ;Motion event.
    if (bHaveTransform) then window->Draw, view
  endif else if (ev.type eq 1) and buttonDown then begin ;Mouse button release.
    buttonDown = 0
    WIDGET_CONTROL, ev.top, /HOURGLASS
    window->Draw, view
    WIDGET_CONTROL, ev.id, DRAW_MOTION=0                 ;Disable motion events.
    
    ;This event tells whatever event handler catches it that the user has let up on the
    ;mouse button after dragging it around (rotate, translate or scale).
    event = {TRACKBALLDRAGFINISHED, id: ev.id, top: ev.top, handler: ev.top, value: 'TrackballDragFinished'}
    WIDGET_CONTROL, ev.id, SEND_EVENT=event
  endif
endif

if ev.type ge 5 and ev.type le 6 then begin              ;Key-press event
  ;Add the rotate/translate/zoom mode to the event structure, and send the event on
  ;for a higher event handler to catch.
  event = create_struct( ev, 'mode', mode)
  WIDGET_CONTROL, ev.id, SEND_EVENT=event
endif

return,0
END


;*******************************************************************************
pro TrackballMgr, buttonBase, window_in, view_in, model_in, xsize, ysize, $
                  toolTipFont=toolTipFont, mouseEventCallback=mouseEventCallback
;*******************************************************************************

common TrackballMgr_common, window, view, model, trackball, buttonDown, $
                            rotateButton, translateButton, zoomButton, $
                            toolTipLabel, mode

window = window_in
view  = view_in
model = model_in
buttonDown = 0
mode = 0             ;0=rotate, 1=translate, 2=zoom
subdirectory = ['resource', 'bitmaps']

rotateButton = WIDGET_BUTTON( buttonBase, uvalue='rotateActiveButton', $
                              event_pro='TrackballMgr__handleButtonEvent', $
                              value=filepath( 'rotate_active.bmp', subdirectory=subdirectory), /bitmap, $
                              /tracking_events)
translateButton = WIDGET_BUTTON( buttonBase, uvalue='translateInactiveButton', $
                                 event_pro='TrackballMgr__handleButtonEvent', $
                                 value=filepath( 'pan.bmp', subdirectory=subdirectory), /bitmap, $
                                 /tracking_events)
zoomButton = WIDGET_BUTTON( buttonBase, uvalue='zoomInactiveButton', $
                            event_pro='TrackballMgr__handleButtonEvent', $
                            value=filepath( 'zoom.bmp', subdirectory=subdirectory), /bitmap, $
                            /tracking_events)                            
toolTipLabel = WIDGET_LABEL( buttonBase, value="                    ", font=toolTipFont)
trackball = obj_new( 'Trackball', [xsize/2.0, ysize/2.0], xsize/2.0)

END
