;***************************************************************
; is_scalar.pro
;***************************************************************

; check to see if a variable is a scalar or not
;
; given  : var               an array or scalar
; return : <function>  int   1 if a scalar, 0 if not

function is_scalar, var

return, (size(var))(0) eq 0

end ; is_scalar

;*************************************************************
; check_if_set.pro
;
; check to see if the variable exists.  if it doesn't then
; set it to zero.  if it does then set it to zero unless it
; is one.  this is useful to check to see if a varible is
; set useing the slash in a procedure call.
;
;  ie     procedure, a, b, /swittch
;
; where procedure is a procedure, a and b are paramters, and
; swittch is a swittch that is set.
;
; given  : var         int    a variable that might be set
; return : <function>  int    a 1 if var is 1 zero other wise

function check_if_set, var

if not keyword_set(var) then begin  ; if not set then
  result = 0                        ;   var is zero
endif else begin                    ; else var is set
   result = 1
endelse

return, result 

end ; check_if_set

;********************************************************************
; keyword_default.pro
;
; check to see if a keyword is set then leave it alone.  if it is
; not set then let it equal the default value.
;
; given  : var           a variable
; return : <function>    if var not set, default, else var

function keyword_default,var,default

if not keyword_set(var) then begin
  var = default
endif

return, var

end ; keyword_default

;*******************************************************************
; find_string.pro
;
; search through a string to find a string embedded in it.  the
; string must be separated from the rest of the characters by
; double of single quotes.  if there is more than one string
; then an array of strings is returned.  if no string is found a
; -1 is returned.  if one string is found, it is returned
;
; given  : line         str         the line to pick the strings from
; return : <function>   str,strarr  the strings returned

function find_string,line

SINGLEQUOTE = "'"     ; a single quote
DOUBLEQUOTE = '"'     ; a double quote

result = -1   ; initalize result to no strings found
searchst = 0  ; the starting search position in the line
firstone = 1  ; this is the first string to find
done = 0      ; starting the loop so no done

while not done do begin                      ; while not done

  strst = strpos(line,SINGLEQUOTE,searchst)  ; look for starting '
  if (strst ne -1) then begin                ; if a start then
    stren = strpos(line,SINGLEQUOTE,strst+1) ; look for ending '
  endif

  if (strst eq -1) then begin                   ; if no ' found
    strst = strpos(line,DOUBLEQUOTE,searchst)   ; look for starting "
    if (strst ne -1) then begin                 ; if a start then
      stren = strpos(line,DOUBLEQUOTE,strst+1)  ; look for ending "
    endif
  endif

  if not keyword_set(stren) then stren = -1     ; if not found at all

  if (strst ne -1) and (stren ne -1) then begin  ; if string was found
    strlen = stren - strst - 1                   ; find string length
    strout = strmid(line,strst+1,strlen)         ; pick out string

    if firstone then begin           ; if this is the first string
      result = strout                ; put string in the result
      firstone = 0                   ; the first string has been found
    endif else begin
      result = [ result, strout ]    ; o' wise add string to the list
    endelse
    searchst = stren + 1             ; set begin of search for next str

  endif else begin
    done = 1                         ; o'wise no string found so done
  endelse

endwhile

return, result           ; return the result

end ; find_string

;*********************************************************************
; get_info_c.pro
;
; read information about the CRRES binary data files.  the data is
; returned in an array of structures, each element contains data about
; one of the variables in the data files.
;
; struct resstr
;   num     int   the number of the data quantity
;   name    str   the name of the data quantitiy
;   unit    str   the units for the data quantity
;   fname   str   NULL PLACEHOLDER
;   ptitle  str   the "pretty title" for plotting  
;   pmin    flt   default !Y.RANGE(0) for quantity
;   pmax    flt   default !Y.RANGE(1) for quantity
;   vecname str   the name of the data quantity to use with vector plot
;   reqs    intarr these quantities require nothing - this is just a filler
; given  : nothing
; return : result resstr  the information about the binary files

pro get_info_c,result

common data_path_list,cr_sr_pth,decdy_pth

SPACE = ' '
PATH= cr_sr_pth+'papco_data_id.txt'



datastr = { num:0, name:'', unit:'',fname:'', ptitle:'', pmin:0.,$
	   pmax:0.,vecname:'', reqs:intarr(20) }   ; a data storage structure
firstone = 1                            ; this is the first one loaded


tempjunk = findfile(PATH,count=numfiles)
if(numfiles ne 0) then begin
openr,/get_lun,lun,PATH    ; open the information file

while not eof(lun) do begin   ; while not the end of the file

  line = ''          ; line is and empty string
  readf,lun,line     ; read the next line from the file
  cmds = str_sep( strtrim( strcompress( line ) , 2 ), SPACE )

  cmd = strmid(cmds(0),0,4)   ; pick out the first 4 chars first word

  if (cmd ne '') and (strmid(cmd,0,1) ne ';') then begin 

    case cmd of
      'begi' : begin
	         qone = datastr
		 qone.num = fix(find_string(line))
                 qone.reqs(0) = qone.num
	       end
      'titl' : qone.name   = find_string(line)
      'unit' : qone.unit   = find_string(line)
      'ptit' : qone.ptitle = find_string(line) 
      'pmin' : qone.pmin   = float(find_string(line))
      'pmax' : qone.pmax   = float(find_string(line))
      'vecn' : qone.vecname= find_string(line)
      'end:' : begin


                 if firstone then begin
		   result = qone
		   firstone = 0
                 endif else begin
		   result = [result, qone]
                 endelse

               end
      else   :
    endcase

  endif

endwhile

free_lun,lun   ; close the file
endif else begin 
  result=datastr
  result.name = "Can't find papco_data_id.txt"
endelse
end ; get_info_c

;*********************************************************************
; get_calc_info.pro
;
; read information about the CRRES calculated data.  The data is
; returned in an array of structures, each element contains data about
; one of the variables in the calculated data.
;
; struct resstr
;   num     int   the number of the data quantity
;   name    str   the name of the data quantitiy
;   unit    str   the units for the data quantity
;   fname   str   the name of the function used to generate this data
;   ptitle  str   the "pretty title" for plotting  
;   pmin    flt   default !Y.RANGE(0) for quantity
;   pmax    flt   default !Y.RANGE(1) for quantity
;   reqs    intarr the other data quantities required to calculate this one
;
; given  : nothing
; return : result resstr  the information about the binary files

pro get_calc_info,result

common data_path_list,cr_sr_pth,decdy_pth

SPACE = ' '
PATH = cr_sr_pth+'papco_calc_id.txt'

datastr = { num:0, name:'', unit:'',fname:'', ptitle:'', pmin:0.,$
	   pmax:0.,vecname:'', reqs:intarr(20) }   ; a data storage structure
firstone = 1                            ; this is the first one loaded

tempjunk = findfile(PATH,count=numfiles)

if (numfiles ne 0) then begin
openr,/get_lun,lun,PATH    ; open the information file

while not eof(lun) do begin   ; while not the end of the file

  line = ''          ; line is and empty string
  readf,lun,line     ; read the next line from the file
  cmds = str_sep( strtrim( strcompress( line ) , 2 ), SPACE )

  cmd = strmid(cmds(0),0,4)   ; pick out the first 4 chars first word

  if (cmd ne '') and (strmid(cmd,0,1) ne ';') then begin 

    case cmd of
      'begi' : begin
	         qone = datastr
		 qone.num = fix(find_string(line))
	       end
      'titl' : qone.name   = find_string(line)
      'unit' : qone.unit   = find_string(line)
      'fun_' : qone.fname  = find_string(line) 
      'ptit' : qone.ptitle = find_string(line) 
      'pmin' : qone.pmin   = float(find_string(line))
      'pmax' : qone.pmax   = float(find_string(line))
      'reqs' : begin
                 tempreqstr = find_string(line)
                 tempreqarrstr = str_sep(tempreqstr,',')
		 tempreqarr = intarr(n_elements(tempreqstr)) 
		 reads,tempreqarrstr,tempreqarr 
                 if (n_elements(tempreqarr) gt 20) then begin
                    tempreqarr = tempreqarr(0:19)
                 endif 
                 for i=0,n_elements(tempreqarr)-1 do begin
                    qone.reqs(i) = tempreqarr(i)
                 endfor
               end
      'end:' : begin


                 if firstone then begin
		   result = qone
		   firstone = 0
                 endif else begin
		   result = [result, qone]
                 endelse

               end
      else   :
    endcase

  endif

endwhile

free_lun,lun   ; close the file
endif else begin
   result = datastr
   result.name = "Can't find papco_calc_id.txt"
endelse
end ; get_calc_info
;*********************************************************************
pro get_all_info,result
 
 get_info_c,result1
 get_calc_info,result2

 n1 = n_elements(result1)
 n2 = n_elements(result2)
 
 n3 = n1 + n2
 
 datastr = { num:0, name:'', unit:'',fname:'', ptitle:'', pmin:0.,$
	   pmax:0.,vecname:'', reqs:intarr(20) }   ; a data storage structure
 result = replicate(datastr,n3)

 for i=0,n1-1 do begin
    result(i).num = result1(i).num
    result(i).name = result1(i).name
    result(i).unit = result1(i).unit
    result(i).fname = result1(i).fname
    result(i).ptitle = result1(i).ptitle
    result(i).pmin = result1(i).pmin
    result(i).pmax = result1(i).pmax
    result(i).vecname = result1(i).vecname
    result(i).reqs = result1(i).reqs
 endfor
 
 for i=n1,n3-1 do begin
    j = i - n1 
    result(i).num = result2(j).num
    result(i).name = result2(j).name
    result(i).unit = result2(j).unit
    result(i).fname = result2(j).fname
    result(i).ptitle = result2(j).ptitle
    result(i).pmin = result2(j).pmin
    result(i).pmax = result2(j).pmax
    result(i).vecname = result2(j).vecname
    result(i).reqs = result2(j).reqs
 endfor

end
;*********************************************************************
; get_one_info.pro
;
; read information about a CRRES binary data file.  the data is
; returned in a structure, each element contains data about
; one of the variables in the data files.
;
; struct resstr
;   num     int   the number of the data quantity
;   name    str   the name of the data quantitiy
;   unit    str   the units for the data quantity
;   ptitle  str   the "pretty title" for plotting  
;   pmin    flt   the default !Y.RANGE(0) for the quantity
;   pmax    flt   the default !Y.RANGE(1) for the quantity
;   vecname str   the name for vector plots
;
; given  :  int   qnum   the data quantity number 
; return : result resstr  the information about the binary files

function get_one_info,qnum 

 get_info_c,result
 index = where(result.num eq qnum)
 if (not is_scalar(index)) then begin 
    return, result(index)
 endif else begin
    get_calc_info,result
    index = where(result.num eq qnum)
    if (not is_scalar(index)) then begin
       return, result(index)
    endif else return, -1
endelse
 
end ; get_one_info
;***************************************************************
;calc_crres_data.pro
;
; calculate a new quantity from a set of CRRES data files.
;
; given : int qnum  the data quantity number
;         int start
;         int stop
; return: fltarr data  the data values
;

pro calc_crres_data,qnum,data,start,stop
   common get_error,get_err_no,get_err_msg
   get_err_no = 0
   get_err_msg = 'crres_efi read successfully' 
   check_for_requested_data,qnum,missing_dataids   
   if (missing_dataids(0) eq 0) then begin
      result=get_one_info(qnum)
      fname = result.fname 
      data = call_function(fname,start,stop)
   endif else begin
     missing_names = strarr(n_elements(missing_dataids))
     result = get_one_info(qnum) 
     for i=0,n_elements(missing_dataids)-1 do begin
        tmpinfo = get_one_info(missing_dataids(i))
        missing_names(i) = tmpinfo.name
     endfor
     wm_crres_efi_missing_dataids_for_calc = widget_message( $
        ['Trying to calculate: ',result.name,$
        'Missing data quantities :',missing_names,$
        'Which correspond to data id #s :',string(missing_dataids)],/error)
     get_err_no=1
     get_err_msg='missing data quantities for calculations'
   endelse  
       
end
;***************************************************************
;is_leap.pro
;***************************************************************

;PURPOSE  :            To determine if a given year is
;                      a leap year
;
;INPUTS   :            int year (YYYY)    the year to be checked
;
;OUTPUTS  :            int result         1 if the year is leap
;                                         0 otherwise
;
;PROCEDURE:            a year is a leap year if it is divisible by
;                      4 or 400 but not 100. (i.e. 2000 is a leap
;                      year, 1900 is not)
;
;CALLING SEQUENCE:     result = is_leap(year)
;
;AUTHOR          :     Doug Rowland    University of Minnesota
;                              School of Physics and Astronomy
;
;CREATED         :     10/15/96
;
;REVISION HISTORY:     N/A

function is_leap,year

if (is_scalar(year)) then begin
   if ((year mod 400) eq 0) then return,1
   if ((year mod 100) eq 0) then return,0
   if ((year mod   4) eq 0) then return,1
   return,0
endif else begin
   yearvec = intarr(n_elements(year))
      for i=0L,n_elements(year)-1 do begin
         if ((year(i) mod 400) eq 0) then yearvec(i) = 1
         if ((year(i) mod 100) eq 0) then yearvec(i) = 0
         if ((year(i) mod   4) eq 0) then yearvec(i) = 1
      endfor	 
   return,yearvec
endelse
end ; is_leap
;***************************************************************
;doy_to_month.pro
;***************************************************************

;PURPOSE  :            find month, day of month,year from
;                      day of year
;
;INPUTS   :            int doy   (DDD)  day of year
;                                       (Jan. 1 is doy 1)
;                      int year  (YYYY) year (for leap year determination)
;KEYWORDS :            text
;                      if set, text causes mmddyyyy.month
;                      to be output as a text string
;

;
;OUTPUTS  :            structure mmddyyy whose elements are:
;		       int month (MM)   the month
;                      int   day (DD)   the   day
;                      int  year (YYYY) the year
;
;PROCEDURE:            consults look up table for days per month
;                      handles leap years using is_leap.pro
;
;CALLING SEQUENCE:     structure = doy_to_month(month,day,year)
;
;AUTHOR          :     Doug Rowland    University of Minnesota
;                              School of Physics and Astronomy
;
;CREATED         :     10/15/96
;
;REVISION HISTORY:     N/A

function doy_to_month,doy,year

days    = [0,31,59,90,120,151,181,212,243,273,304,334]
days_lp = [0,31,60,91,121,152,182,213,244,274,305,335]
monthnames = ['January','February','March','April','May','June','July','August','September','October','November','December']


if (is_scalar(year)) then begin

   if (is_leap(year) eq 1) then begin
      use_days = days_lp
   endif else use_days = days
   month = float(max(where(doy gt use_days)) + 1.)
   day   = doy - use_days(month -1.)

   name =  monthnames(month -1.) 
   
   mmddyyyy = {year:year,month:month,day:day,name:name}
   return,mmddyyyy

endif else begin

   npts = n_elements(year)
   day = fltarr(npts)
   month = fltarr(npts)
   name  = strarr(npts)

   yearvec = is_leap(year) 
   for i=0L,npts-1 do begin
      if(yearvec(i) eq 0) then begin
	 use_days = days
      endif else use_days = days_lp
      month(i) = float(max(where(doy(i) gt use_days)) + 1.)
      day(i)   = doy(i) - use_days(month(i) - 1.)
      name(i) = monthnames(month(i)-1.)
   endfor
endelse

mmddyyyy = {year:year,month:month,day:day,name:name}
return, mmddyyyy

end ; doy_to_month
;***************************************************************
;month_to_doy.pro
;***************************************************************

;PURPOSE  :            find day of year from month, day of month,
;                      year
;
;INPUTS   :            int month (MM)   the month
;                      int   day (DD)   the   day
;                      int  year (YYYY) the year
;
;OUTPUTS  :            int doy   (DDD)  day of year
;                                       (Jan. 1 is doy 1)
;
;PROCEDURE:            consults look up table for days per month
;                      handles leap years using is_leap.pro
;
;CALLING SEQUENCE:     doy = month_to_doy(month,day,year)
;
;AUTHOR          :     Doug Rowland    University of Minnesota
;                              School of Physics and Astronomy
;
;CREATED         :     10/15/96
;
;REVISION HISTORY:     N/A

function month_to_doy,month,day,year

days    = [0,31,59,90,120,151,181,212,243,273,304,334]
days_lp = [0,31,60,91,121,152,182,213,244,274,305,335]

if (is_scalar(year)) then begin
   if (is_leap(year) eq 1) then begin
      use_days = days_lp
   endif else use_days = days

   doy = use_days(month - 1) + day
  
endif else begin 
   npts = n_elements(year)
   doy = fltarr(npts)
   yearvec = is_leap(year)
   for i=0L,npts-1 do begin
      if(yearvec(i) eq 0) then begin
	 use_days = days
      endif else use_days = days_lp
      doy(i) = use_days(month(i) - 1) + day(i)
   endfor
endelse

return, doy

end ; month_to_doy
;********************************************************************
; doy_to_day.pro
;
; convert day of year to consecutive days for 1990 and 1991.
; 1990 starts at day 0.0 and 1991 starts at day 365.0
;
; given  : doy          the day of the year
;          year         then year 1990 or 1991
; return : <function>   the day in 1990 or 1991

function doy_to_day, doy, year

if (is_scalar(doy)) then begin  
   ;convert doy to 'time' day and add 365 if in 1991
   day = double(doy) - 1
   if (year eq 1991) then day = day + 365D

endif else begin

   npts = n_elements(year)
   day = fltarr(npts)
      index = where(year eq 1991)
      day = doy - 1
      if (index(0) ne -1) then day(index) = day(index) + 365.
;   for i = 0L,npts-1 do begin
;      day(i) = double(doy(i)) - 1
;      if (year(i) eq 1991) then day(i) = day(i) + 365D
;   endfor

endelse

return, day

end ; doy_to_day

;********************************************************************
; day_to_doy.pro
;
; convert consecutive days for 1990 and 1991 to day of year.
; 1990 starts at day 0.0 and 1991 starts at day 365.0
;
; given  : day          the day of the year
; retur : structure resstr
;	     int year year 1990 or 1991
;	     int doy  day of year

function day_to_doy, day

if (is_scalar(day)) then begin

   year = 1990 
   ;convert 'time' day to doy and subtract 365 if in 1991
   doy = double(day) + 1
   if (doy gt 365) then begin
      doy = doy - 365D
      year = year + 1 
   endif   
endif else begin
   
   npts = n_elements(day)
   year = fltarr(npts)
   doy  = fltarr(npts)
   year(*) = 1990

   for i=0L,npts-1 do begin
      doy(i) = double(day(i)) + 1
      if (doy(i) gt 365) then begin
	 doy(i) = doy(i) - 365D
         year(i) = year(i) + 1
      endif
   endfor

endelse

resstr = {year:year, doy:doy}
return, resstr 

end ; day_to_doy

;***************************************************************
; ut_to_day.pro
;
; calculate a time in days from a string giving a time
; in UT with the format 'HH:MM:SS'.
;
; given  :   utstring	the string containing the UT time
;            year       the year
;            doy        day of year 
; return :   day        time in days since the start of 1990 

function ut_to_day, year, doy, utstring

; initialize variables
hr = 0 & mn = 0 & sec = 0 

; read the the time from the input string
dateformat = "(i2, x, i2, x, i2)"
reads,format=dateformat,utstring, hr, mn, sec
hr = double(hr) & mn = double(mn) & sec = double(sec)


; return the time in days since the start of 1990 
time_in_days = ( (sec/60D + mn)/60D + hr )/24D
fracdoy = doy + time_in_days

day = doy_to_day(fracdoy,year)

return,day 

end ; ut_to_day
;********************************************************************
; day_to_ut.pro
;
; calculate the UT time from time in days since the start of 1990
;
; given  :   day        the time in days since the start of 1990
; 
; return :   resstr     the structure containing the UT time
;                  str  hr    UT time in hours
;                  str  min   UT time in minutes
;                  str  sec   UT time in seconds
;                  str msec   UT time in milliseconds

function day_to_ut,day

fracday = day - floor(day)
hr = floor(fracday * 24.)
min = floor((fracday - hr/24.) * 1440.)
sec = floor((fracday - hr/24. - min/1440.) * 86400.)
msec = floor((fracday - hr/24. - min/1440. - sec/86400.) * 86400000.)

shr = string(hr,format='(I2.2)')
smin = string(min,format='(I2.2)')
ssec = string(sec,format='(I2.2)')
smsec = string(msec,format='(I3.3)')

resstr = {hr:shr, min:smin, sec:ssec, msec:smsec}
return,resstr

end ; day_to_ut
;***************************************************************
; orbit_index.pro
;
; read the file which contains the orbit information, data00,
; into a structure.  this information is used as an index into
; the data files.  the structure contains the following info:
; 
;  num     :  orbit number
;  year    :  orbit year
;  doy     :  day of year
;  pts     :  number of data pts for this orbit
;  fileloc :  the index of the first point in orbit
;
; NOTE:  the path to the index file must be in the string
;        filename.
;
; given  :  nothing
; return :  a structure containing the orbit information

pro orbit_index, orbit, counter

common data_path_list,DATAPATH

; the max number of orbits and the path to the index file
NUMBER_OF_ORBIT = 1070
FILENAME = DATAPATH + 'data00'

; create a array of orbit structures to contain the index data
string = ''     ; initialize the data types
integer = 0
long = 0L
orbit = replicate({orbit, num:integer, year:integer, doy:integer,  $
	  pts:integer, fileloc:long}, NUMBER_OF_ORBIT)

tempjunk = findfile(FILENAME,count=numfiles)
if (numfiles ne 0) then begin
openr,/get_lun,lun, filename    ; open the index file


orbitnum = integer   ; create variables to read in a single orbit
year = integer
doy = integer
numpts = integer
ts = string          ; this is a string used to skip a line 

filepos = long     ; start at the beginning of the file
counter = integer  ; count the number or orbits loaded
while not eof(lun) do begin                ; while not at eof
  readf, lun, orbitnum, t1, year, doy, t2  ; read the orbit info
  readf, lun, ts
  readf, lun, numpts, t1
  readf, lun, ts
  readf, lun, ts
  orbit(counter).num = orbitnum       ; copy the orbit info
  orbit(counter).year = year          ; to the storage array
  orbit(counter).doy = doy
  orbit(counter).pts = numpts
  orbit(counter).fileloc = filepos
  filepos = filepos + numpts          ; find start of next orbit
  counter = counter + 1               ; count this orbit
endwhile

free_lun, lun        ; close the index file
endif 

end ; orbit_index

;**********************************************************************
; fileloc_doy.pro
; 
; find the index of the value in the decimal day file that is closest
; to the day, target
;
; given  :  target      the target time in decimal days
; return :  <function>  the index of the time in the data files

function fileloc_doy, target

common data_path_list,DATAPATH,FILENAME

DBLS     = 8L              ; # bytes for a double number

tempjunk = findfile(FILENAME,count=numfiles)
if (numfiles ne 0) then begin
; open the file and find the number of pts in the file
openr,/get_lun,df,FILENAME
filestat = fstat(df) & numpts = (filestat.size /  DBLS)

; do a binary search to find the time closes to target
filepos = -1                          ; return -1 if not found
left = 0L & right = numpts - 1L       ; initial left and right
time = 0D & guess = 0L & found = 0    ; initialize time, guess, found
while not found do begin              
  guess = (left + right) / 2L         ; make a new guess
  point_lun,df,guess*DBLS & readu,df,time  ; read the guess time
  case 1 of
    (guess eq left) or (guess eq right) or (time eq target) : found = 1
    (time lt target) : left = guess
    (time gt target) : right = guess
    else : print, 'ERROR in finding : ', target
  endcase
endwhile

; read the two closest values tot the target
point_lun,df,left*DBLS & readu,df,timel
point_lun,df,right*DBLS & readu,df,timer

; return the index of the closest value to the target
if abs(target - timel) lt abs(target - timer) then begin
  filepos = left
endif else begin
  filepos = right
endelse

free_lun,df    ; close the decimal time file

return,filepos    ; return the index
endif else begin
  print,"Couldn't find time file data68"  
  return,-1
endelse
end ; fileloc_doy

;********************************************************************
; get_orbit_range.pro
;
; get the starting and stopping file index values for the given
; orbit number.  If the orbit is not found then start is returned
; with a value of -1
;
; given  : orbitnum   the number of the orbit to find
;          orbit      the orbit index loaded with orbit_index
; return : start      the starting file index for the orbit
;          stop       the stopinf file index for the orbit

pro get_orbit_range, orbitnum, orbit, start, stop

; get the orbit index
oi = where( orbitnum eq orbit(*).num )

; check if the orbit exists, if does get start and stop

start = -1
stop = -1

if (oi(0) ne -1) then begin
  if (((size(oi))(0) eq 1) and (orbit(oi).pts gt 1)) then begin
     start = orbit(oi).fileloc
     stop = start + orbit(oi).pts - 1
  endif
endif

end ; get_orbit_range

;*********************************************************************
; get_time_range.pro
;
; take beginning and ending month, day and year and find the beginning
; end ending file index value corrsponding to those times.
;
; structure date_ut
;   year     : the year 1990 or 1991
;   month    : the month of the year
;   day      : the day of the month
;   ut       : the time during the day in ut
;
; given  : sdate       the starting date
;          edate       the ending date
; return : start       the starting file index for the time interval
;          stop        the ending file index for the time interval

pro get_time_range,sdate,edate,start,stop

sdoy = month_to_doy(sdate.month,sdate.day,sdate.year)
edoy = month_to_doy(edate.month,edate.day,edate.year)

print,'Converting time to ms'
sms = ut_to_day(sdate.year,sdoy,sdate.ut) ; convert to decimal days
ems = ut_to_day(edate.year,edoy,edate.ut)
print,sms,ems

start = fileloc_doy(sms)    ; find the beginning and ending index vals
stop  = fileloc_doy(ems)

;if ((edate.year eq 1990)) then begin   ;disable 1990 plotting for PAPCO
;  start = -1
;  stop  = -1
;endif

end ; get_time_range

;********************************************************************
; get_mult_orbit.pro
;
; get the starting and stoping file index values for the given
; orbit number.  If the orbit is not found then start is returned
; with a value of -1
;
; given  : onum1      the number of the start orbit to find
;          onum2      the number of the start orbit to find
;          orbit      the orbit index loaded with orbit_index
; return : start      the starting file index for the orbit
;          stop       the stopinf file index for the orbit

pro get_mult_orbit, onum1, onum2, orbit, start, stop

; get the orbit index values
oi1 = where( onum1 eq orbit(*).num )
oi2 = where( onum2 eq orbit(*).num )

; check if the orbits exist, if do, get start and stop
start = -1
if ((size(oi1))(0) eq 1) and ((size(oi2))(0) eq 1) then begin
  start = orbit(oi1).fileloc
  stop = orbit(oi2).fileloc + orbit(oi2).pts - 1
endif

end ; get_mult_orbit

;****************************************************************
; read_crres.pro
;
; read the specified quantities from the cress survey files and
; return them in a two dimensional array.  The millisecond data
; from file data01 cannot be read using this procedure  because
; it doesn't have the data type of float.
;
; given  :   start    the staring file index for the data range
;            stop     the ending file index for the data range
;            quant    the quanitities to load
; return :   data     two dim array containing the data
;            time     in days since the beginning of 1990

pro read_crres,start,stop,quant,data,time

; find the number of quantities to load
numqu = n_elements(quant)

; create storage for the data begin read
arraysize = stop - start  + 1L
data = fltarr(numqu,arraysize)

; while there are quantites left to read, read another
i = 0                          ; start with first
while (i lt numqu) do begin
  print, 'READING data for : ', quant(i)
  read_crres_data,start,stop,quant(i),temp,time  ; read the data
  data(i,*) = temp                               ; store it
  i = i + 1                                      ; move to next quant
endwhile

end ; read_crres
  
;********************************************************************
; read_crres_data.pro
;
; read the crres data from the survey files.  If the time of day,
; file 'data01,' is requested the data is returned as long integers.
; otherwise that data is returned as floating point numbers.  the
; time returned in 'time' is in days and is a double precision array.
;
; given  :  start        the starting index into the data files
;           stop         then ending index into the data files
;           filenumber   the number of the data file to read
; return :  data         the data from the requested file
;           time         the time array in days

pro read_crres_data, start, stop, filenumber, data, time

common data_path_list,PATH,TIMEFILE

PATH = papco_getenv('CRRES_EFI_DATA')   

DBLS = 8L      ; the number of bytes in a double precision number
FLTS = 4L      ; the number of bytes in a floating point number

; create a string containing the data file name
if (filenumber+1 lt 10) then begin
  snum = 'data0' + string(format = '( i1 )', filenumber+1)
endif else begin
  snum = strtrim(strcompress('data' + string(format = '( i3 )', filenumber+1),/remove_all),2)
endelse

; create array of the appropriate size and type
arraysize = stop - start + 1L
if filenumber eq 0 then begin
  data = lonarr(arraysize)
endif else begin
  data = fltarr(arraysize)
endelse

chkfile = findfile(PATH+snum,count=numfiles)
if (numfiles eq 1) then begin
   ; open the requested data file and load the data
   openr,/get_lun,lun,PATH+snum
   point_lun,lun,start*FLTS & readu,lun,data
   free_lun, lun
endif else begin
   calc_crres_data,filenumber,data,start,stop
endelse

; read the time data into a double array
time = dblarr(arraysize)
openr,/get_lun,lun,TIMEFILE
point_lun,lun,start*DBLS & readu,lun,time
free_lun,lun

end ; read_crres_data

;**********************************************************************
; read_crres_time.pro
;
; read the times in the range given by two file index numbers
;
; given  : start    the index value for the start of the interval
;          stop     the index value for the end of the interval
; return : time     the time array

pro read_crres_time,start,stop,time

common data_path_list,DATAPATH,TIMEFILE

DBLS = 8L                    ; number of bytes in a double

arraysize = stop - start + 1L   ; find the number of data points
time = dblarr(arraysize)        ; allocate data storage
openr,/get_lun,lun,TIMEFILE     ; open the decimal timefile
point_lun,lun,start*DBLS & readu,lun,time   ; read the times
free_lun,lun                    ; close the file

end ; read_crres_time

;*********************************************************************
; get_user_orbit.pro
;
; read the number of the orbit to load from the user.
;
; given  : nothing
; return : orbitnum    the orbit numbe to load

pro get_user_orbit,orbitnum

prompt = 'The orbit to load : '
orbitnum = 0
read,prompt=prompt,orbitnum

end ; get_user_orbit

;**********************************************************************
; get_user_month.pro
;
; read in starting and ending times by prompting the user.  The times
; returned are in month, day , and UT.
;
; structure date_ut
;   year     : the year 1990 or 1991
;   month    : the month of the year
;   day      : the day of the month
;   ut       : the time during the day in ut
;
; given  : nothing
; return : sdate      the starting date
;          edate      the ending date

pro get_user_month,sdate,edate

prompt1 = 'MM DD YYYY     : '    ; the prompt for the day
prompt2 = 'UT             : '    ; the prompt for the time of day in UT

smon = 0 & sday = 0 & syear = 0 & stime = ''  ; get the starting
print,'***START TIME***'                      ; time
read,prompt=prompt1,smon,sday,syear
read,prompt=prompt2,stime

emon = 0 & eday = 0 & eyear = 0 & etime = ''  ; get the ending
print,'***END TIME***'                        ; time
read,prompt=prompt1,emon,eday,eyear
read,prompt=prompt2,etime

sdate = {date_ut, year:syear, month:smon, day:sday, ut:stime}
edate = {date_ut, year:eyear, month:emon, day:eday, ut:etime}

end ; get_user_month
;*****************************************************************
; label_date90.pro
;*****************************************************************

; NAME:
;	LABEL_DATE90
;
; PURPOSE:
;	This function labels axes with dates.
;
; CATEGORY:
;	Plotting.
;
; CALLING SEQUENCE:
;	To set up:
;		dummy = LABEL_DATE90(DATE_FORMAT='string')
;	To use:
;		PLOT, x, y, XTICKFORMAT='LABEL_DATE'
;
; INPUTS:
;	No explicit user defined inputs. When called from the plotting
;	routines, the input parameters are (Axis, Index, Value)
;       ASSUMES VALUE is in days since start of 1990.
;
; KEYWORD PARAMETERS:
;	DATE_FORMAT: a format string which may contain the following:
;                      %A for day of year (3 digits)
;                      %H for hour and minute (HH:MM)
;		       %M for month (3 character abbr)
;		       %N for month (2 digit abbr)
;		       %D for day of month,
;                      %S for second (SS)
;                      %T for milliseconds (3 digits)
;		       %Y for 4 digit year.
;		       %Z for last two digits of year.
;		       %% is %.
;		     Other characters are passed directly thru.
;		     For example, '%M %D, %Y' prints DEC 11, 1993
;		       '%M %2Y' yields DEC 93
;		       '%D-%M' yields 11-DEC
;		       '%D/%N/%Y' yields 11/12/1993
;		       '%M!C%Y' yields DEC on the top line, 1993 on
;		       the bottom (!C is the new line graphic command).
;
;	MONTHS:      The names of the months, a twelve element string array.
;		     If omitted, use Jan, Feb, ..., Dec.
;
; OUTPUTS:
;	The date string to be plotted.
;
; COMMON BLOCKS:
;	LABEL_DATE_COM.
;
; RESTRICTIONS:
;	Only one date axis may be simultaneously active.
;
; PROCEDURE:
;	Straightforward.
;
; EXAMPLE:
;	For example, to plot from Jan 1, 1993, to July 12, 1994:
;	  Start_date = month_to_doy(1, 1, 1993)
;         Start_date = doy_to_day(Start_date)
;	  End_date = month_to_doy(7, 12, 1994)
;         End_date = doy_to_day(End_date)
;	  Dummy = LABEL_DATE90(DATE_FORMAT='%N/%D')  ;Simple mm/dd
;	  x = findgen(end_date+1 - start_date) + start_date ;Time axis
;	  PLOT, x, sqrt(x), XTICKFORMAT = 'LABEL_DATE', XSTYLE=1
;	  (Plot with X axis style set to exact.)
;	
; MODIFICATION HISTORY:
;	DMS, RSI.	April, 1993.	Written.
;-

function label_date90, axis, index, x, DATE_FORMAT = format, MONTHS = months

COMMON label_date_com, fmt, month_chr

if keyword_set(format) then begin	;Save format string?
	if keyword_set(months) then month_chr = months $
	else month_chr = ['Jan','Feb','Mar', 'Apr', 'May', 'Jun', 'Jul', $
		'Aug', 'Sep', 'Oct', 'Nov', 'Dec']
	fmt = format
	return, 0
	endif

if n_elements(month_chr) ne 12 or n_elements(fmt) le 0 then $
	message,' Not initialized.'

utstr  = day_to_ut(double(x))
result = day_to_doy(long(x))
resstr = doy_to_month(result.doy,result.year)
month = resstr.month
day   = resstr.day
year  = resstr.year

n = strlen(fmt)
out = ''

for i=0, n-1 do begin			;Each format character...
    c = strmid(fmt, i, 1)		;The character.
    if c eq '%' then begin
	i = i + 1
	c = strmid(fmt, i, 1)		;The function
	case c of		;format character?
        'A' : out = out + string(format='(i3.3)',result.doy)	
        'H' : out = out + utstr.hr+':'+utstr.min
	'M' : out = out + month_chr(month-1)
	'N' : out = out + string(format='(i2.2)', month)
	'D' : out = out + string(format='(i2.2)', day)
	'S' : out = out + utstr.sec
	'T' : out = out + utstr.msec
	'Y' : out = out + string(format='(i4)', year)
	'Z' : out = out + string(format='(i2.2)', year  mod 100)
	'%' : out = out + '%'
	else : message, 'Illegal character in date format string: '+fmt
	endcase
    endif else out = out + c
endfor

return, out
end
;**********************************************
;function date_to_day
;**********************************************
;takes date struct (year month day ut) and returns
;time in days since 1991 (date struct from get_user_month)

function date_to_day,date

   year = date.year
   month = date.month
   day = date.day
   ut = date.ut

doy = month_to_doy(month,day,year)
tday = doy_to_day(doy,year)
tarr = str_sep(ut,':')
hr = float(tarr(0))
min = float(tarr(1))

tday = tday + hr/24. + min/(24.*60.)

return,tday
end ;date_to_day
