;*****************************************************
; delbmag.pro
;
; calculate the difference in magnetic energy density
; between the measured value and the Olson-Pfitzer model

function delbmag,start,stop

BADVAL = 99999.0

read_crres_data,start,stop,12,bxmgse
read_crres_data,start,stop,13,bymgse
read_crres_data,start,stop,14,bzmgse
read_crres_data,start,stop,41,bxmod
read_crres_data,start,stop,42,bymod
read_crres_data,start,stop,43,bzmod

modmag = (bxmod^2 + bymod^2 + bzmod^2)
meamag = (bxmgse^2 + bymgse^2 + bzmgse^2)

const = 1./(8.*!PI*16.)     ;convert to keV/cm^3
modmag = modmag*const
meamag = meamag*const

delbmg = meamag - modmag
index = where((bxmod eq BADVAL) or (bymod eq BADVAL) or (bzmod eq BADVAL) or (bxmgse eq BADVAL)or(bymgse eq BADVAL)or(bzmgse eq BADVAL))
if (index(0) ne -1) then delbmg(index) = BADVAL
return, delbmg 

end
;*****************************************************
; BP1CALC.pro
;
; convert tmV to physical units 
; uV/ [m Hz^.5]
;
; 

function BP1CALC,start,stop

BADVAL = 99999.0
read_crres_data,start,stop,21,BP1RAW

index = where(BP1RAW eq BADVAL)

div= BP1RAW/(.07)
pten= 10.^div
a=1.8*pten

if (index(0) ne -1) then a(index) = BADVAL
return,a

end ;BP1CALC

;*****************************************************
; BP2CALC.pro
;
; convert tmV to physical units 
; uV/ [m Hz^.5]
;
; 

function BP2CALC,start,stop

BADVAL = 99999.0
read_crres_data,start,stop,22,BP2RAW

index = where(BP2RAW eq BADVAL)

div= (BP2RAW + .01)/(.07)
pten= 10.^div
a=.62*pten

if (index(0) ne -1) then a(index) = BADVAL
return,a

end ;BP2CALC

;*****************************************************
; BP3CALC.pro
;
; convert tmV to physical units 
; uV/ [m Hz^.5]
;
; 

function BP3CALC,start,stop

BADVAL = 99999.0
read_crres_data,start,stop,23,BP3RAW

index = where(BP3RAW eq BADVAL)

div= BP3RAW/(.07)
pten= 10.^div
a=.22*pten

if (index(0) ne -1) then a(index) = BADVAL
return,a

end ;BP3CALC
;*****************************************************
; EMAGCALC.pro
;
; calculate E(spinplane) sqrt(Ey^2 + Ez^2) 
;
;
; 

function EMAGCALC,start,stop

BADVAL = 99999.0
read_crres_data,start,stop,1,ey
read_crres_data,start,stop,2,ez
index = where((ey eq BADVAL) or (ez eq BADVAL))

emag = sqrt(ey^2 + ez^2)

if (index(0) ne -1) then emag(index) = BADVAL
return,emag
end ;EMAGCALC
;*****************************************************
; mag_efield.pro
;
; calculate magnitude of electric field in MGSE (total) 
;
; 

function mag_efield,start,stop

BADVAL = 99999.0
read_crres_data,start,stop,1,ey
read_crres_data,start,stop,2,ez
read_crres_data,start,stop,61,ex
index = where((ey eq BADVAL) or (ez eq BADVAL) or (ex eq BADVAL))

a= sqrt(ex^2+ey^2+ez^2)
if (index(0) ne -1) then a(index) = BADVAL
return,a

end ;mag_efield_sp.pro
;*****************************************************
; EMAGSPHCALC.pro
;
; calculate E(spinplane) sqrt(Ey^2 + Ez^2) from spheres
;
;
; 

function EMAGSPHCALC,start,stop

BADVAL = 99999.0
read_crres_data,start,stop,8,ey
read_crres_data,start,stop,9,ez
index = where((ey eq BADVAL) or (ez eq BADVAL))

emag = sqrt(ey^2 + ez^2)

if (index(0) ne -1) then emag(index) = BADVAL
return,emag
end ;EMAGSPHCALC
;*****************************************************
; lambdacalc.pro
;
; calculate angle B makes with the MGSE XY-plane (as a proxy for XY SM plane)
;
;
; 

function lambdacalc,start,stop

BADVAL = 99999.0
read_crres_data,start,stop,12,bx
read_crres_data,start,stop,13,by
read_crres_data,start,stop,14,bz

index = where((bx eq BADVAL) or (by eq BADVAL) or (bz eq BADVAL))

lambda = atan(bz,sqrt(bx^2 + by^2))*180./!PI

if (index(0) ne -1) then lambda(index) = BADVAL
return,lambda
end ;lambdacalc
;*****************************************************
; lambdamodcalc.pro
;
; calculate angle Bmodel makes with the MGSE XY-plane (as a proxy for XY SM plane)
;
;
; 

function lambdamodcalc,start,stop

BADVAL = 99999.0
read_crres_data,start,stop,12,bx
read_crres_data,start,stop,13,by
read_crres_data,start,stop,14,bz
read_crres_data,start,stop,5,bxm
read_crres_data,start,stop,6,bym
read_crres_data,start,stop,7,bzm

bxp = bx - bxm
byp = by - bym
bzp = bz - bzm

index = where((bx eq BADVAL) or (by eq BADVAL) or (bz eq BADVAL) $
	  or (bxm eq BADVAL) or (bym eq BADVAL) or (bzm eq BADVAL))

lambda = atan(bzp,sqrt(bxp^2 + byp^2))*180./!PI

if (index(0) ne -1) then lambda(index) = BADVAL
return,lambda
end ;lambdamodcalc
;*****************************************************
; lambdadevcalc.pro
;
; calculate deviation that angle B makes with the MGSE XY-plane (as a proxy for XY SM plane) relative to the model (lambda - lambdamodel)
;
;
; 

function lambdadevcalc,start,stop

BADVAL = 99999.0
read_crres_data,start,stop,117,lambda
read_crres_data,start,stop,118,modlambda

index = where((lambda eq BADVAL) or (modlambda eq BADVAL))

result = lambda-modlambda

if (index(0) ne -1) then result(index) = BADVAL
return,result
end ;lambdadevcalc
;*****************************************************
; phicalc.pro
;
; calculate phi angle of B in MGSE (clock angle with 0 degrees along the X axis)
;
;
; 

function phicalc,start,stop

BADVAL = 99999.0
read_crres_data,start,stop,12,bx
read_crres_data,start,stop,13,by

index = where((bx eq BADVAL) or (by eq BADVAL))

result = atan(by,bx)*180./!PI
in2 = where(result lt 0)
if (in2(0) ne -1) then result(in2) = 360. + result(in2)
if (index(0) ne -1) then result(index) = BADVAL
in3 = where(result ge 300)
in4 = where(result lt 300)
if ((in3(0) ne -1) and (in4(0) ne -1)) then begin
   res2 = result(in4)
   in5 = where(res2 le 100)
   if(in5(0) ne -1) then begin
      if (n_elements(in3) ge n_elements(in4)) then begin 
         result(in4)=result(in4) + 360.
      endif else result(in3) = result(in3) - 360.
   endif
endif
return,result
end ;phicalc
;*****************************************************
; phimodcalc.pro
;
; calculate phi angle of B in MGSE (clock angle with 0 degrees along the X axis) from the model
;
;
; 

function phimodcalc,start,stop

BADVAL = 99999.0
read_crres_data,start,stop,41,bxm
read_crres_data,start,stop,42,bym

index = where((bxm eq BADVAL) or (bym eq BADVAL))

result = atan(bym,bxm)*180./!PI
in2 = where(result lt 0)
if (in2(0) ne -1) then result(in2) = 360. + result(in2)
if (index(0) ne -1) then result(index) = BADVAL
in3 = where(result ge 300)
in4 = where(result lt 300)
if ((in3(0) ne -1) and (in4(0) ne -1)) then begin
   res2 = result(in4)
   in5 = where(res2 le 100)
   if(in5(0) ne -1) then begin
      if (n_elements(in3) ge n_elements(in4)) then begin 
         result(in4)=result(in4) + 360.
      endif else result(in3) = result(in3) - 360.
   endif
endif
return,result
end ;phimodcalc
;*****************************************************
; phidevcalc.pro
;
; calculate phi - phi(model)
;
;
; 

function phidevcalc,start,stop

BADVAL = 99999.0
read_crres_data,start,stop,120,phi
read_crres_data,start,stop,121,phim

index = where((phi eq BADVAL) or (phim eq BADVAL))

result = phi - phim 
in2 = where(result lt 0)
if (in2(0) ne -1) then result(in2) = 360. + result(in2)
if (index(0) ne -1) then result(index) = BADVAL
in3 = where(result ge 300)
in4 = where(result lt 300)
if ((in3(0) ne -1) and (in4(0) ne -1)) then begin
   res2 = result(in4)
   in5 = where(res2 le 100)
   if(in5(0) ne -1) then begin
      if (n_elements(in3) ge n_elements(in4)) then begin 
         result(in4)=result(in4) + 360.
      endif else result(in3) = result(in3) - 360.
   endif
endif
return,result
end ;phidevcalc
;******************************************************************
; exbhatxmgse.pro
;
; calculate the X MGSE component of ExBhat field off the equatorial
; plane (in corot. frame)
; 

function exbhatxmgse,start,stop

BADVAL = 99999.0

read_crres_data,start,stop,61,exmgse
read_crres_data,start,stop,1,eymgse
read_crres_data,start,stop,2,ezmgse
read_crres_data,start,stop,12,bxmgse
read_crres_data,start,stop,13,bymgse
read_crres_data,start,stop,14,bzmgse

bmg = sqrt(bxmgse^2 + bymgse^2 + bzmgse^2)

exbhtx = (eymgse*bzmgse - ezmgse*bymgse)/bmg

index = where((exmgse eq BADVAL)or(eymgse eq BADVAL)or(bxmgse eq BADVAL) or (ezmgse eq BADVAL)or(bzmgse eq BADVAL) or (bymgse eq BADVAL))
if (index(0) ne -1) then exbhtx(index) = BADVAL 
return, exbhtx 
end
;*****************************************************
; exbhatymgse.pro
;
; calculate the Y MGSE component of ExBhat field off the equatorial
; plane (in corot. frame)
; 

function exbhatymgse,start,stop

BADVAL = 99999.0

read_crres_data,start,stop,61,exmgse
read_crres_data,start,stop,1,eymgse
read_crres_data,start,stop,2,ezmgse
read_crres_data,start,stop,12,bxmgse
read_crres_data,start,stop,13,bymgse
read_crres_data,start,stop,14,bzmgse

bmg = sqrt(bxmgse^2 + bymgse^2 + bzmgse^2)
exbhty = (ezmgse*bxmgse - exmgse*bzmgse)/bmg
index = where((exmgse eq BADVAL)or(eymgse eq BADVAL)or(bxmgse eq BADVAL) or (ezmgse eq BADVAL)or(bzmgse eq BADVAL) or (bymgse eq BADVAL))
if (index(0) ne -1) then exbhty(index) = BADVAL
return, exbhty 
end
;*****************************************************
; exbhatzmgse.pro
;
; calculate the Z MGSE component of ExBhat field off the equatorial
; plane (in corot. frame)
; 

function exbhatzmgse,start,stop

BADVAL = 99999.0

read_crres_data,start,stop,61,exmgse
read_crres_data,start,stop,1,eymgse
read_crres_data,start,stop,2,ezmgse
read_crres_data,start,stop,12,bxmgse
read_crres_data,start,stop,13,bymgse
read_crres_data,start,stop,14,bzmgse

bmg = sqrt(bxmgse^2 + bymgse^2 + bzmgse^2)
exbhtz = (exmgse*bymgse - eymgse*bxmgse)/bmg
index = where((exmgse eq BADVAL)or(eymgse eq BADVAL)or(bxmgse eq BADVAL)$
   or(ezmgse eq BADVAL)or(bzmgse eq BADVAL) or (bymgse eq BADVAL))
if (index(0) ne -1) then exbhtz(index) = BADVAL
return, exbhtz 
end
;*****************************************************
; wavepoyntr.pro
; calculate the radially inward component of the wave Poynting flux
; (delta E x delta B) (in corot. frame)
; 


function wavepoyntr,start,stop
 
 BADVAL = 99999.0
 CONST  = .79577
 NDETRENDPOINTS = 15.*2.    ;15 minutes
 NAVGMAGFIELDPOINTS = 15.*2. ;15 minutes
  
   
   read_crres_data,start,stop,521,bxsm,time
   read_crres_data,start,stop,522,bysm
   read_crres_data,start,stop,523,bzsm
   read_crres_data,start,stop,515,exsm
   read_crres_data,start,stop,516,eysm
   read_crres_data,start,stop,517,ezsm
   read_crres_data,start,stop,529,xsm
   read_crres_data,start,stop,530,ysm
   read_crres_data,start,stop,531,zsm
    
     
      
      npts = n_elements(bxsm)
      emsm = fltarr(npts)
      easm = emsm
      epsm = emsm
      bmsm = emsm
      bpsm = emsm
      basm = emsm
       
       for i=0L,npts-1 do begin
	   bxsms = smooth(bxsm,NAVGMAGFIELDPOINTS)
	   bysms = smooth(bysm,NAVGMAGFIELDPOINTS)
	   bzsms = smooth(bzsm,NAVGMAGFIELDPOINTS)
	   bsmmag = sqrt(bxsms(i)^2+bysms(i)^2+bzsms(i)^2) 
	   bxsmun = bxsms(i)/bsmmag
	   bysmun = bysms(i)/bsmmag
	   bzsmun = bzsms(i)/bsmmag
	       zx = bxsmun
	       zy = bysmun
	       zz = bzsmun
	       yx = (ysm(i)*zz - zsm(i)*zy)
	       yy = (zsm(i)*zx - xsm(i)*zz)
	       yz = (xsm(i)*zy - ysm(i)*zx)
	     ymag = sqrt(yx^2 + yy^2 + yz^2)
               yx = yx/ymag
	       yy = yy/ymag
               yz = yz/ymag
	       xx = (yy*zz - yz*zy)
	       xy = (yz*zx - yx*zz)
	       xz = (yx*zy - yy*zx)
	   rotmat = [[xx,xy,xz],$
	             [yx,yy,yz],$
		     [zx,zy,zz]]
										  inputvect = [[exsm(i)],[eysm(i)],[ezsm(i)]]
 outputvect = rotmat##inputvect
    emsm(i) = outputvect(0,0)     ;radial inward e field
    easm(i) = outputvect(0,1)     ;westward e field
    epsm(i) = outputvect(0,2)     ;"parallel" e field
  inputvect = [[bxsm(i)],[bysm(i)],[bzsm(i)]]
 outputvect = rotmat##inputvect
    bmsm(i) = outputvect(0,0)     ;radial inward b field
    basm(i) = outputvect(0,1)     ;westward b field
    bpsm(i) = outputvect(0,2)     ;"parallel" b field

endfor

easm1 = easm - smooth(easm,NDETRENDPOINTS)
epsm1 = epsm - smooth(epsm,NDETRENDPOINTS)
emsm1 = emsm - smooth(emsm,NDETRENDPOINTS)
basm1 = basm - smooth(basm,NDETRENDPOINTS)
bpsm1 = bpsm - smooth(bpsm,NDETRENDPOINTS)
bmsm1 = bmsm - smooth(bmsm,NDETRENDPOINTS)


wpr = CONST * (easm1*bpsm1 - epsm1*basm1)
wpa = CONST * (epsm1*bmsm1 - emsm1*bpsm1)
wpp = CONST * (emsm1*basm1 - easm1*bmsm1)

bmag = sqrt(basm^2 + bpsm^2 + bmsm^2)
dvr  = 1000. * (easm1*bpsm1 - epsm1*basm1)/(bmag^2)
vr = 1000. * (easm1*(bpsm-bpsm1)  - epsm1*(basm-basm1))/(bmag^2)

index = where((exsm eq BADVAL)or(eysm eq BADVAL)or(bxsm eq BADVAL) or $
       (ezsm eq BADVAL) or (bzsm eq BADVAL) or (bysm eq BADVAL))

if (index(0) ne -1) then begin
   wpr(index) = BADVAL
   wpa(index) = BADVAL
   wpp(index) = BADVAL
endif

return,wpr
end
;*****************************************************
; wavepoyntw.pro
; calculate the westward component of the wave Poynting flux
; (delta E x delta B) (in corot. frame)
; 


function wavepoyntw,start,stop
 
 BADVAL = 99999.0
 CONST  = .79577
 NDETRENDPOINTS = 15.*2.    ;15 minutes
 NAVGMAGFIELDPOINTS = 15.*2. ;15 minutes
  
   
   read_crres_data,start,stop,521,bxsm,time
   read_crres_data,start,stop,522,bysm
   read_crres_data,start,stop,523,bzsm
   read_crres_data,start,stop,515,exsm
   read_crres_data,start,stop,516,eysm
   read_crres_data,start,stop,517,ezsm
   read_crres_data,start,stop,529,xsm
   read_crres_data,start,stop,530,ysm
   read_crres_data,start,stop,531,zsm
    
     
      
      npts = n_elements(bxsm)
      emsm = fltarr(npts)
      easm = emsm
      epsm = emsm
      bmsm = emsm
      bpsm = emsm
      basm = emsm
       
       for i=0L,npts-1 do begin
	   bxsms = smooth(bxsm,NAVGMAGFIELDPOINTS)
	   bysms = smooth(bysm,NAVGMAGFIELDPOINTS)
	   bzsms = smooth(bzsm,NAVGMAGFIELDPOINTS)
	   bsmmag = sqrt(bxsms(i)^2+bysms(i)^2+bzsms(i)^2) 
	   bxsmun = bxsms(i)/bsmmag
	   bysmun = bysms(i)/bsmmag
	   bzsmun = bzsms(i)/bsmmag
	       zx = bxsmun
	       zy = bysmun
	       zz = bzsmun
	       yx = (ysm(i)*zz - zsm(i)*zy)
	       yy = (zsm(i)*zx - xsm(i)*zz)
	       yz = (xsm(i)*zy - ysm(i)*zx)
	     ymag = sqrt(yx^2 + yy^2 + yz^2)
               yx = yx/ymag
	       yy = yy/ymag
               yz = yz/ymag
	       xx = (yy*zz - yz*zy)
	       xy = (yz*zx - yx*zz)
	       xz = (yx*zy - yy*zx)
	   rotmat = [[xx,xy,xz],$
	             [yx,yy,yz],$
		     [zx,zy,zz]]
										  inputvect = [[exsm(i)],[eysm(i)],[ezsm(i)]]
 outputvect = rotmat##inputvect
    emsm(i) = outputvect(0,0)     ;radial inward e field
    easm(i) = outputvect(0,1)     ;westward e field
    epsm(i) = outputvect(0,2)     ;"parallel" e field
  inputvect = [[bxsm(i)],[bysm(i)],[bzsm(i)]]
 outputvect = rotmat##inputvect
    bmsm(i) = outputvect(0,0)     ;radial inward b field
    basm(i) = outputvect(0,1)     ;westward b field
    bpsm(i) = outputvect(0,2)     ;"parallel" b field

endfor

easm1 = easm - smooth(easm,NDETRENDPOINTS)
epsm1 = epsm - smooth(epsm,NDETRENDPOINTS)
emsm1 = emsm - smooth(emsm,NDETRENDPOINTS)
basm1 = basm - smooth(basm,NDETRENDPOINTS)
bpsm1 = bpsm - smooth(bpsm,NDETRENDPOINTS)
bmsm1 = bmsm - smooth(bmsm,NDETRENDPOINTS)


wpr = CONST * (easm1*bpsm1 - epsm1*basm1)
wpa = CONST * (epsm1*bmsm1 - emsm1*bpsm1)
wpp = CONST * (emsm1*basm1 - easm1*bmsm1)

bmag = sqrt(basm^2 + bpsm^2 + bmsm^2)
dvr  = 1000. * (easm1*bpsm1 - epsm1*basm1)/(bmag^2)
vr = 1000. * (easm1*(bpsm-bpsm1)  - epsm1*(basm-basm1))/(bmag^2)

index = where((exsm eq BADVAL)or(eysm eq BADVAL)or(bxsm eq BADVAL) or $
       (ezsm eq BADVAL) or (bzsm eq BADVAL) or (bysm eq BADVAL))

if (index(0) ne -1) then begin
   wpr(index) = BADVAL
   wpa(index) = BADVAL
   wpp(index) = BADVAL
endif

return,wpa
end

;*****************************************************
; wavepoyntz.pro
; calculate the field-aligned component of the wave Poynting flux
; (delta E x delta B) (in corot. frame)
; 


function wavepoyntz,start,stop
 
 BADVAL = 99999.0
 CONST  = .79577
 NDETRENDPOINTS = 15.*2.    ;15 minutes
 NAVGMAGFIELDPOINTS = 15.*2. ;15 minutes
  
   
   read_crres_data,start,stop,521,bxsm,time
   read_crres_data,start,stop,522,bysm
   read_crres_data,start,stop,523,bzsm
   read_crres_data,start,stop,515,exsm
   read_crres_data,start,stop,516,eysm
   read_crres_data,start,stop,517,ezsm
   read_crres_data,start,stop,529,xsm
   read_crres_data,start,stop,530,ysm
   read_crres_data,start,stop,531,zsm
    
     
      
      npts = n_elements(bxsm)
      emsm = fltarr(npts)
      easm = emsm
      epsm = emsm
      bmsm = emsm
      bpsm = emsm
      basm = emsm
       
       for i=0L,npts-1 do begin
	   bxsms = smooth(bxsm,NAVGMAGFIELDPOINTS)
	   bysms = smooth(bysm,NAVGMAGFIELDPOINTS)
	   bzsms = smooth(bzsm,NAVGMAGFIELDPOINTS)
	   bsmmag = sqrt(bxsms(i)^2+bysms(i)^2+bzsms(i)^2) 
	   bxsmun = bxsms(i)/bsmmag
	   bysmun = bysms(i)/bsmmag
	   bzsmun = bzsms(i)/bsmmag
	       zx = bxsmun
	       zy = bysmun
	       zz = bzsmun
	       yx = (ysm(i)*zz - zsm(i)*zy)
	       yy = (zsm(i)*zx - xsm(i)*zz)
	       yz = (xsm(i)*zy - ysm(i)*zx)
	     ymag = sqrt(yx^2 + yy^2 + yz^2)
               yx = yx/ymag
	       yy = yy/ymag
               yz = yz/ymag
	       xx = (yy*zz - yz*zy)
	       xy = (yz*zx - yx*zz)
	       xz = (yx*zy - yy*zx)
	   rotmat = [[xx,xy,xz],$
	             [yx,yy,yz],$
		     [zx,zy,zz]]
										  inputvect = [[exsm(i)],[eysm(i)],[ezsm(i)]]
 outputvect = rotmat##inputvect
    emsm(i) = outputvect(0,0)     ;radial inward e field
    easm(i) = outputvect(0,1)     ;westward e field
    epsm(i) = outputvect(0,2)     ;"parallel" e field
  inputvect = [[bxsm(i)],[bysm(i)],[bzsm(i)]]
 outputvect = rotmat##inputvect
    bmsm(i) = outputvect(0,0)     ;radial inward b field
    basm(i) = outputvect(0,1)     ;westward b field
    bpsm(i) = outputvect(0,2)     ;"parallel" b field

endfor

easm1 = easm - smooth(easm,NDETRENDPOINTS)
epsm1 = epsm - smooth(epsm,NDETRENDPOINTS)
emsm1 = emsm - smooth(emsm,NDETRENDPOINTS)
basm1 = basm - smooth(basm,NDETRENDPOINTS)
bpsm1 = bpsm - smooth(bpsm,NDETRENDPOINTS)
bmsm1 = bmsm - smooth(bmsm,NDETRENDPOINTS)


wpr = CONST * (easm1*bpsm1 - epsm1*basm1)
wpa = CONST * (epsm1*bmsm1 - emsm1*bpsm1)
wpp = CONST * (emsm1*basm1 - easm1*bmsm1)

bmag = sqrt(basm^2 + bpsm^2 + bmsm^2)
dvr  = 1000. * (easm1*bpsm1 - epsm1*basm1)/(bmag^2)
vr = 1000. * (easm1*(bpsm-bpsm1)  - epsm1*(basm-basm1))/(bmag^2)

index = where((exsm eq BADVAL)or(eysm eq BADVAL)or(bxsm eq BADVAL) or $
       (ezsm eq BADVAL) or (bzsm eq BADVAL) or (bysm eq BADVAL))

if (index(0) ne -1) then begin
   wpr(index) = BADVAL
   wpa(index) = BADVAL
   wpp(index) = BADVAL
endif

return,wpp
end
;*****************************************************
; Vradialexb.pro
; calculate the radially inward component of the total ExB velocity
; (E x B) (in corot. frame)
; 


function Vradialexb,start,stop
 
 BADVAL = 99999.0
 CONST  = .79577
 NDETRENDPOINTS = 15.*2.    ;15 minutes
 NAVGMAGFIELDPOINTS = 15.*2. ;15 minutes
  
   
   read_crres_data,start,stop,521,bxsm,time
   read_crres_data,start,stop,522,bysm
   read_crres_data,start,stop,523,bzsm
   read_crres_data,start,stop,515,exsm
   read_crres_data,start,stop,516,eysm
   read_crres_data,start,stop,517,ezsm
   read_crres_data,start,stop,529,xsm
   read_crres_data,start,stop,530,ysm
   read_crres_data,start,stop,531,zsm
    
     
      
      npts = n_elements(bxsm)
      emsm = fltarr(npts)
      easm = emsm
      epsm = emsm
      bmsm = emsm
      bpsm = emsm
      basm = emsm
       
       for i=0L,npts-1 do begin
	   bxsms = smooth(bxsm,NAVGMAGFIELDPOINTS)
	   bysms = smooth(bysm,NAVGMAGFIELDPOINTS)
	   bzsms = smooth(bzsm,NAVGMAGFIELDPOINTS)
	   bsmmag = sqrt(bxsms(i)^2+bysms(i)^2+bzsms(i)^2) 
	   bxsmun = bxsms(i)/bsmmag
	   bysmun = bysms(i)/bsmmag
	   bzsmun = bzsms(i)/bsmmag
	       zx = bxsmun
	       zy = bysmun
	       zz = bzsmun
	       yx = (ysm(i)*zz - zsm(i)*zy)
	       yy = (zsm(i)*zx - xsm(i)*zz)
	       yz = (xsm(i)*zy - ysm(i)*zx)
	     ymag = sqrt(yx^2 + yy^2 + yz^2)
               yx = yx/ymag
	       yy = yy/ymag
               yz = yz/ymag
	       xx = (yy*zz - yz*zy)
	       xy = (yz*zx - yx*zz)
	       xz = (yx*zy - yy*zx)
	   rotmat = [[xx,xy,xz],$
	             [yx,yy,yz],$
		     [zx,zy,zz]]
										  inputvect = [[exsm(i)],[eysm(i)],[ezsm(i)]]
 outputvect = rotmat##inputvect
    emsm(i) = outputvect(0,0)     ;radial inward e field
    easm(i) = outputvect(0,1)     ;westward e field
    epsm(i) = outputvect(0,2)     ;"parallel" e field
  inputvect = [[bxsm(i)],[bysm(i)],[bzsm(i)]]
 outputvect = rotmat##inputvect
    bmsm(i) = outputvect(0,0)     ;radial inward b field
    basm(i) = outputvect(0,1)     ;westward b field
    bpsm(i) = outputvect(0,2)     ;"parallel" b field

endfor

easm1 = easm - smooth(easm,NDETRENDPOINTS)
epsm1 = epsm - smooth(epsm,NDETRENDPOINTS)
emsm1 = emsm - smooth(emsm,NDETRENDPOINTS)
basm1 = basm - smooth(basm,NDETRENDPOINTS)
bpsm1 = bpsm - smooth(bpsm,NDETRENDPOINTS)
bmsm1 = bmsm - smooth(bmsm,NDETRENDPOINTS)


wpr = CONST * (easm1*bpsm1 - epsm1*basm1)
wpa = CONST * (epsm1*bmsm1 - emsm1*bpsm1)
wpp = CONST * (emsm1*basm1 - easm1*bmsm1)

bmag = sqrt(basm^2 + bpsm^2 + bmsm^2)
dvr  = 1000. * (easm1*bpsm1 - epsm1*basm1)/(bmag^2)
vr = 1000. * (easm1*(bpsm-bpsm1)  - epsm1*(basm-basm1))/(bmag^2)

index = where((exsm eq BADVAL)or(eysm eq BADVAL)or(bxsm eq BADVAL) or $
       (ezsm eq BADVAL) or (bzsm eq BADVAL) or (bysm eq BADVAL))

if (index(0) ne -1) then begin
   wpr(index) = BADVAL
   wpa(index) = BADVAL
   wpp(index) = BADVAL
endif

return,vr+dvr
end
;*****************************************************
; exsmmap.pro
;
; calculate Ex in SM coordinates, mapped to the equatorial plane (dipole)
function exsmmap,start,stop

BADVAL = 99999.0

read_crres_data,start,stop,527,er
read_crres_data,start,stop,528,etheta
read_crres_data,start,stop,27,mlt

mltrad = (mlt - 12.)*!PI/12.

ex = - er * cos(mltrad) +  etheta * sin(mltrad)

index = where((er eq BADVAL) or (etheta eq BADVAL) or (mlt eq BADVAL))
if (index(0) ne -1) then ex(index) = BADVAL
return, ex 

end
;*****************************************************
; eysmmap.pro
;
; calculate Ey in SM coordinates, mapped to the equatorial plane (dipole)
function eysmmap,start,stop

BADVAL = 99999.0

read_crres_data,start,stop,527,er
read_crres_data,start,stop,528,etheta
read_crres_data,start,stop,27,mlt

mltrad = (mlt - 12.)*!PI/12.

ey = - er * sin(mltrad)  - etheta * cos(mltrad)

index = where((er eq BADVAL) or (etheta eq BADVAL) or (mlt eq BADVAL))
if (index(0) ne -1) then ey(index) = BADVAL
return, ey 
end
