;******************************************************************************
;*  NAME:        r_crres_efi
;*
;*  DESCRIPTION: Routine to read crres_efi files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       (uses common block time from papco)
;*               dataid  (UMN CRRES SURVEY data id # in papco_data_id.txt or papco_calc_id.txt)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
pro r_crres_efi, dataid,currPlotInfo,VERBOSE=VERBOSE

   common crres_efi_data, input_header, input_data
   common mjdt, mjdt_start, mjdt_end
   common get_error, get_err_no, get_err_msg
   common data_path_list,crres_survey_path,decday_path
   common PLOT_COMPOSER,widgetData

   if (widgetData.mouse_panelnr ne -1) then begin 
     currPlotInfo = widgetData.PLOTINFOS(widgetData.mouse_panelnr)
   endif
   
; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

   get_err_no=0
   get_err_msg='crres_efi read succesfully'
   
; check for env. variable pointing to data directory
   data_dir='CRRES_EFI_DATA'
   path=papco_getenv(data_dir)
   
   crres_survey_path = path
   decday_path = crres_survey_path+'data68'
   if strlen(path) eq 0 then begin
     get_err_no=1
     get_err_msg='Environmental variable '+data_dir+' not set'
     print,get_err_msg
     return
   endif
   
; check for existence of data directory
   cc=papco_finddir(path)
   if cc eq 0 then begin
     get_err_no=2
     get_err_msg='Data directory '+path+' not found'
     print,get_err_msg
     return
   endif
   
; check VERBOSE keyword and set
   if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. papco standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
   
   time=mjdt_start  &  no_files_read=0  &  day_count=0.0
  
   convert_t90_to_date,mjdt_start,year,day_oy,hour,minute,second,$
                       month,cmonth,day_om, /mjdt
   utstr = string(hour,format='(I2.2)')+':'+string(minute,format='(I2.2)')+ $
                  ':'+string(second,format='(I2.2)')
   sdate = {year:year,month:month,day:day_om,ut:utstr} 
   convert_t90_to_date,mjdt_end,year,day_oy,hour,minute,second,$
                       month,cmonth,day_om, /mjdt
   utstr = string(hour,format='(I2.2)')+':'+string(minute,format='(I2.2)')+ $
                  ':'+string(second,format='(I2.2)')
   edate = {year:year,month:month,day:day_om,ut:utstr} 
  
   orbit_index,orbit
   get_time_range,sdate,edate,start,stop
   if ((start ne -1) and (stop ne -1)) then begin
      if (currPlotInfo.typevector(1) gt 1) then begin
	 start = start - currPlotInfo.typevector(1)/2 - 1
	 stop  = stop  + currPlotInfo.typevector(1)/2 + 1
      endif
      read_crres_data,start,stop,dataid,dataarray,timeint90 
      ;index = where(timeint90 lt 365.) ;to disable 1990 plotting
      ;if (index(0) ne -1) then dataarray(index) = 99999.
      no_files_read = 1
   endif else begin
      get_err_no=1
      get_err_msg='Invalid start or stop indices into CRRES survey data, probably invalid orbit #'
      no_files_read = 0
   endelse    

; check if any files were read   
   if (no_files_read ne 0) and (get_err_no ne 1) then begin
     get_err_no=0
     get_err_msg='% r_crres_efi: '+string(no_files_read,format="(i2)")+ $
       ' files read succesfully'
     ;clear arrays and load buffer into output array
   
     npts = n_elements(dataarray)
     if (npts ge 2) then begin 
       input_data = replicate({dat:0.,time:0D},npts) 
       input_data(*).dat=dataarray  &  dataarray=0
       convert_t90_to_date,mjdt_start,year,day_oy,hour,minute,second,$
                       month,cmonth,day_om,/mjdt
       stday = doy_to_day(day_oy,year)
       input_data(*).time = (timeint90 - stday)*86400. 
     endif else begin
       input_data = {dat:dataarray(0),time:(timeint90(0) - stday)*86400.}
     endelse
   endif else begin
     get_err_no=1
     get_err_msg='% r_crres_efi: no files found for timerange'
     return
   endelse      
; now smooth data if requested
   print,currPlotInfo.typevector(1)
   if (currPlotInfo.typevector(1) gt 1) then begin
      index = where(input_data.dat eq 99999)
      indices = fltarr(n_elements(input_data.dat))
      if (index(0) ne -1) then begin 
         for i=0,n_elements(index)-1 do begin
	   if ((index(i) ge (currPlotInfo.typevector(1)/2 + 1)) $
	      and (index(i) le (n_elements(indices)-currPlotInfo.typevector(1)/2-1))) then begin
	   indices((index(i) - currPlotInfo.typevector(1)/2 - 1):(index(i) + currPlotInfo.typevector(1)/2 + 1)) = 99999
	   endif else begin
	      print,"can't smooth out ",index(i)
	      ;STOP
           endelse
         endfor
      endif
      input_data.dat = smooth(input_data.dat,currPlotInfo.typevector(1))
      ind3 = where(indices eq 99999)
      if (ind3(0) ne -1) then begin
	 input_data(ind3).dat = 99999
      endif
   endif
; restrict data to actual time range requested
   ut1=mjdt_start.t
   ut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
   index=where((input_data.Time ge ut1) and (input_data.Time le ut2),count)
   if count ne 0 then begin
      input_data=input_data(index)
   endif   
end

