;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  crres_eph_panel, plotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      plotInfo       a structure of the PAPCO_PLOTINFO-type
;*                      this structure is edited in the panel  editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*      All the functions in $PAPCO_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at last some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I 
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      swittch:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, swittch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          swittch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commincate between your
;*     panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which PAPCO supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      PAPCO internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $                       
;*                      xut2:0.0}                           
;*
;*                  
;*      For an enhanced description read 
;*                          $PAPCO_XX/PAPCO_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=PAPCO_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  crres_eph_panel, plotInfo, GROUP=group, ACTION=ACTION

COMMON crres_eph_panel_data, widgetData
COMMON crres_ephemeris

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
additionalWData={canceled:1}
;***NEW PAPCO 10.5*** 
;All the user's widget info is now stored in a structure,
;and the only thing stored in the plotinfostruct is the pointer to
;that structure. This allows unlimited freedom in variables for the
;module. The structure papco_cdf_control is defined in lanl_geo_eph_commons.pro

;test if the plotinfo pointer tag points to an 'INT' (means not
;assigned to a structure). If so, load in the control structure.
r = size(*plotinfo.USR_PTR1, /ST)
IF r.TYPE_NAME EQ 'INT' THEN plotinfo.USR_PTR1 = ptr_new(crres_eph_control)
control = *plotinfo.USR_PTR1
;control now contains the user defiend control structure for this module.

;--- create the base-window ---------------------------------------------------
title = plotinfo.panelKind + ' - ' + papco_module_version('crres_eph')
panel_base=papco_WIDGET_BASE(TITLE=title, COLUMN=1, GROUP_LEADER=group)

; All PAPCO_panel_Add* routines are PAPCO panel widgets that can be included
; in the panel editor or not. They provide functionality that PAPCO supports.
; They dynamically extend the panel editors widget structure, and all the 
; events they generate are supported by PAPCO_panel_Event, transparent to the
; user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
widgets=PAPCO_panel_AddHeightWidgets(panel_base, plotInfo)
additionalWData=create_struct(widgets, additionalWData)
   
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;   

;-- select orginal or model coords - ------------------------------------------
names = ['Original Ephemeris', 'Mag Model Coords']
orig_outer = widget_base(panel_base, COLUMN = 1, /ALIGN_CENTER, /FRAME)
orig_base = CW_BGROUP(orig_outer, names , ROW = 1, /EXCLUSIVE, $
                   LABEL_LEFT='Coord Source:', $
                   UVALUE='spb_orig', $
                   SET_VALUE=control.orig)

;;--- select orig coordinates -------------------------------------------------
eph_outer = widget_base(panel_base, ROW=1, /ALIGN_CENTER, /FRAME)
eph_names = orig_names
eph_base=CW_BGROUP(eph_outer, eph_names, ROW=2, /EXCLUSIVE, $
                   LABEL_TOP='Original Ephemeris Coordinates', $
                   UVALUE='spb_eph', $
                   SET_VALUE=control.product)

;--- add Mag Field model choices ----------------------------------------------
widgets=papco_panel_Add_MagMod_Widgets(panel_base, control)
additionalWData=create_struct(widgets, additionalWData)
   
;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;  
   
; for examples of standard buttons or sliders look at other panel editors.
; some standard buttons which have been recycled before can be found in
; $PAPCO_MODULES/panels&slice/more_panel_widgets

; As the same panel is called when you edit a plot, interrogate the tags in the
; plotInfoStruct (aplotInfo) passed in and set your widgets accordingly. 
; In this way the widget is always reinitialized to its last state.

; The next four PAPCO panel widgets will be added only if you want to
; support that functionality, and if they make sense for your crres_eph.

;--- add the PAPCO panel widget for choosing the color bar for color plots ----
;   widgets=PAPCO_panel_AddColorBar_Widgets(panel_base, plotInfo)
;   additionalWData=create_struct(widgets, additionalWData)
;--- add the  PAPCO panel widget for choosing the plot-style for line plots ---
widgets=PAPCO_panel_add_PlotStyleButtons(panel_base, plotInfo, 0)
additionalWData=create_struct(widgets, additionalWData)
;--- add the PAPCO panel widget for choosing the y-axis-ranges ----------------
widgets=PAPCO_PANEL_AddYScale_Widgets(panel_base, plotInfo)
additionalWData=create_struct(widgets, additionalWData)
;--- add the PAPCO panel widgets for choosing the z-axis-ranges ---------------
;   widgets=PAPCO_PANEL_AddZScale_Widgets(panel_base, plotInfo)
;   additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.
;    if you want to make use of the configure feature, then add the
;    keyword CONFIG
widgets=PAPCO_panel_AddActionButtons(panel_base, Action, plotInfo, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create 
;    that helpfile in the papco_interface directory of your module (a template
;    for this help file exist in the crres_eph example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
thisWidgetsData={helpfile:'crres_eph_panel.help', $
                 eph_outer:eph_outer}
widgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)  
widgetData=papco_panel_CreateWidgetData(widgetData, control)


;--- realize the window
PAPCO_Panel_Realize, panel_base, widgetData
crres_eph_refresh, /ALL

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reported but the 
; data contained in plotInfo would be returned.
papco_xmanager, 'crres_eph_panel', panel_base

;now return data. Since all the widget's info in in a control structure,
;only load it's pointer into USR_PTR1 and return that.

;control is part of the widgetData structure. Extract it
struct_assign, widgetData, *plotinfo.USR_PTR1

;work out if there has been a change in the control structure
plotinfo.modified = plotinfo.modified + $
                    papco_compare_struct(control, *plotinfo.USR_PTR1)

returnData={USR_PTR1:plotinfo.USR_PTR1, $
            MODIFIED:plotinfo.modified }
additionalRData=PAPCO_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)

END

  
;******************************************************************************
;* PROCEDURE:     
;*      PRO crres_eph_refresh
;* 
;* DESCRIPTION:  
;*      refresh rotuine for crres_eph_panel, called by event handler
;*
;* INPUTS:       
;*      none
;*                  
;* KEYWORDS:
;*      
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO crres_eph_refresh, ALL = ALL, COORDS = COORDS

COMMON crres_eph_panel_data, widgetData

IF keyword_set(ALL) THEN BEGIN
    COORDS = 1
ENDIF

IF keyword_set(COORDS) THEN BEGIN
    IF widgetData.orig EQ 0 THEN BEGIN 
        widget_control, widgetData.eph_outer,  SENSITIVE = 1
        widget_control, widgetData.mag_model_outer_base,  SENSITIVE = 0
    ENDIF ELSE BEGIN
        widget_control, widgetData.eph_outer,  SENSITIVE = 0
        widget_control, widgetData.mag_model_outer_base,  SENSITIVE = 1
    ENDELSE 
END

END 

;******************************************************************************
;* PROCEDURE:     
;*      PRO crres_eph_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO crres_eph_panel_event, event

COMMON crres_eph_panel_data, widgetData

  WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to 
; 'PAPCO_panel_Event', which handles the events generated
; by the PAPCO-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'PAPCO_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.
   
  CASE uval OF

    'spb_orig': BEGIN
        widgetData.orig = event.value
        crres_eph_refresh, /COORDS
     END

    'spb_eph' : widgetData.product = event.value

    ELSE: BEGIN
      widgetData=papco_panel_Event(event, widgetData)
    end

ENDCASE
 
END  


