;******************************************************************************
;* file papcoadd_crres_eph
;*      papcoadd_crres_eph
;*      papcoadd_crres_eph_ticks
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for a crres_eph. 
;*
;*    These procedures are contained:
;*
;*    PRO addplottype_crres_eph,plottypes_base,pb_plottype
;*        ->  adds the button for the new plot type to the main window.
;*
;*    PRO plottable_crres_eph, plottable
;*        -> Sets a variable to indicate plottable without orbit.
;*
;*    PRO descriptor_crres_eph, plotInfo, descriptStr
;*        -> Constructs the string used to label plots in the panels window
;*
;*    PRO addpanel_crres_eph
;*        -> Calls the panel editor for this plottype and adds the plot to the
;*            panel window.
;*
;*    PRO editor_crres_eph, plotNumber, newPlotData
;*        -> This procedure calls the panel editor when called from 
;*           PAPCO_EditPlot (papco_paneledit.pro).
;*
;*    PRO  draw_crres_eph, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_crres_eph,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*      This procedure adds a button for this crres_eph to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*      package, so that they can be automatically called when needed.
;*      The button itself can contain access to a sub-menu for further
;*      crres_ephs.
;*
;* INPUTS:       
;*      plottypes_base  the widget ID of the parent widget to contain the
;*                      plottype buttons.
;*
;* OUTPUTS: 
;*      pb_plottype     the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for sub-types.
;*
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      addplottype_crres_eph,plottypes_base,buttonGeometry,pb_plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro addplottype_crres_eph,plottypes_base,buttonGeometry,pdm_crres_eph

  MenuDesc = [ $
        { CW_PDMENU_S, 3, strcenter('CRRES-EPH',10) }, $ 
        { CW_PDMENU_S, 0, ' configure crres_eph ' }, $
        { CW_PDMENU_S, 0, ' crres_eph      ' }, $ 
        { CW_PDMENU_S, 2, ' crres_eph_ticks' } ]  
  
  pdm_crres_eph = CW_PDMENU(plottypes_base, MenuDesc, $
                            UVALUE='pdm_crres_eph')  
  
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_crres_eph
;*
;* DESCRIPTION:  
;*      Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      Using an instance of the PAPCO_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled. The
;*      desired default values for the plotinfo tags need to be edited by
;*      the user. This plotinfo structure will be passed to the panel editor
;*      When it is first called for the crres_eph.
;*      "panelEditor" must be set to the name of the panel editor routine.
;*
;* INPUTS:       
;*      event           event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      addpanel_crres_eph
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;***************************************************************************
pro addpanel_crres_eph,event
  
  common get_error, get_err_no, get_err_msg
  
  r_module_config, strlowcase('crres_eph'), config_data, msg
  if (get_err_no ne 0) and (event.value ne 1) then begin
    message,'You need to configure the module first', /cont
    papco_module_configure,'crres_eph'
    return
  endif  
 
  CASE event.value of
    1: BEGIN
      papco_module_configure,'crres_eph'
    end
    2: BEGIN
      panelKind='crres_eph'
      panelEditor=panelkind+'_panel'
      defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
      defaults.typeVector=[0,0,0,0]      ;set some tags to desired defaults
      defaults.channel=0            
      defaults.ylinlog=0
      defaults.options(2:3)=[2,10]
      papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
    end
    3: BEGIN
      panelKind='crres_eph_ticks'
      panelEditor=panelkind+'_panel'
      defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
      papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults
    end  
    else: begin
      message,'Nothing defined for this submenu entry', /cont
      return
    end  
  endcase
      
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO plottable_crres_eph, plottable
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate plottable without orbit.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      plottable        a boolean 0 if plottable only by crres orbit
;*                                1 if plottable by time also.
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      plottable_crres_eph, plottable
;*
;* MODIFICATION HISTORY:       
;*      written November 1995, Reiner Friedel
;******************************************************************************
pro plottable_crres_eph, plottable
   plottable=1     ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 
END
 
pro plottable_crres_eph_ticks, plottable
   plottable=1     ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 
END
 
 
;******************************************************************************
;* PROCEDURE:     
;*      PRO canget_crres_eph, canget
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate that crres_meb support the OUTPUT
;*      functionality. 
;*      
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "canget_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_crres_meb.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      canget         a boolean 0 if this feature is not available
;*                               1 if this feature is available
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      canget_crres_meb, canget
;******************************************************************************
pro canget_crres_eph, canget
  canget=  1      ; 1 -> plot routine can return just data to file
                  ; 0 -> feature disabled
 END 
 
pro canget_crres__eph_ticks, canget
  canget=  0      ; 1 -> plot routine can return just data to file
                  ; 0 -> feature disabled
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_crres_eph, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*      Constructs the string used to label plots in the panel window.
;*      The plotinfo structure passed in here is the panel's plotinfo
;*      as returned by the panel editor for this crres_eph. You can use
;*      the information in that structure to construct some informative
;*      label for the panel which then appears in the Panel Compose list.
;*
;* INPUTS:       
;*      plotInfo        structure containing all data for this plot.
;*                      (structure type: PAPCO_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*      descriptStr     a string containing the label
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      descriptor_crres_eph, plotInfo, descriptStr
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
PRO descriptor_crres_eph, plotInfo, descriptStr
  
COMMON crres_ephemeris

control = *plotinfo.USR_PTR1
IF control.orig EQ 0 THEN out_str = orig_names(control.product) ELSE $
  out_str = papco_panel_Add_MagMod_Widgets(0, control,  /LABEL)
descriptStr=descriptStr+out_str
   
END

PRO descriptor_crres_eph_ticks, plotInfo, descriptStr
   
out_str = ''
descriptStr=descriptStr+out_str
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO editor_crres_eph, plotNumber, newPlotData
;* 
;* DESCRIPTION:  
;*      This procedure calls the panel editor when called from 
;*      PAPCO_EditPlot (papco_paneledit.pro). Provided here to allow
;*      several crres_ephs to share a common panel editor. Normally this
;*      procedure does not need to be edited beyond the required name
;*      change for crres_eph.
;*
;* INPUTS:       
;*      plotNumber      anInteger. 0<=plotNumber<=widgetData.numberOfPlots
;*                      the index of the plot in widgetData.plotInfos.
;*            
;* OUTPUTS: 
;*      newPlotData     the new structure returned by the panel editor
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      editor_crres_eph, plotNumber, newPlotData
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
PRO editor_crres_eph, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

  plotInfo=widgetData.plotInfos(plotNumber)
  newPlotData=crres_eph_panel(plotInfo, $
                              ACTION='Update '+plotInfo.panelKind, $
                              GROUP=widgetData.main_base)

END

PRO editor_crres_eph_ticks, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

  plotInfo=widgetData.plotInfos(plotNumber)
  newPlotData=crres_eph_ticks_panel(plotInfo, $
                              ACTION='Update '+plotInfo.panelKind, $
                              GROUP=widgetData.main_base)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO draw_crres_eph, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:  
;*      This procedure performs the actual data read call and the plot call
;*      for this plottype. The user need to pass to his plot routine all the
;*      parameters needed, typically these parameters are stored in the 
;*      currplotinfo structure or constructed from them. The currplotinfo
;*      structure tags normally used for passing information from the panel
;*      editor to the plot routine are:
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          swittch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      
;*      These tags have grown out of traditional usage and can be used for
;*      anything the user wants to use them for. There are plenty variables
;*      for most applications: Adding a new tag to the plotinfo structure
;*      is not user-definable but need to be done at the PAPCo core level.
;* 
;*      The panelVector MUST be passed to your plotroutine as it is used
;*      to set the panel position.
;*      The _EXTRA=extra_PlotInfo should be passed if you intend to make use
;*      of PAPCO functionality such as overplotting, linestyle choice and
;*      color bar choice.
;*
;* INPUTS:       
;*      panelVector    position of the plot (see papco_conventions.text, 6)
;*      currPlotInfo   structure containing all data for this plot.
;*                     (structure PAPCO_PLOTINFO, see papco_variables.text) 
;*
;* OUTPUTS: 
;*      get_data_call  a string containing the data call. Used by other
;*                     routines that need to know how to get the data
;*                     in a panel, e.g. to draw spectra, distribution functions
;*      IsRead         a boolean which is set when the the data is read the
;*                     first time. Useful if a given data-read puts data
;*                     into commons which can be plotted in many ways.
;*                     When set, data does not have to be read again.
;   slice_type:   Selects which slices are supported and for which slice
;                 routines have been written.
;                 This is an integer logical with 16 swittches. Currently
;                 there are 5 papco slices. You may extend this by
;                 further USER slices up to a total of 16 slices.
;                     0 = disable all slices
;                     1 = papco Zoom                 (2^0)
;                     2 = papco View Slice           (2^1)
;                     4 = papco Access level Zero    (2^2)
;                     8 = papco Write Paneldata      (2^3)
;                    16 = papco Hair Cursor          (2^4)
;                    32 = USER slice A or not used   (2^5)
;                    64 = USER slice B or not used   (2^6)
;                    "     "     "   "  "  "    "      "    
;                 32768 = USER slice Z or not used  (2^15)
;                        
;   You can support more than one type of slice by arithmetically adding the
;   options, eg slice_type=1+4  support Zoom and Level Zero
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      draw_crres_eph, panelVector, currPlotInfo, orbitNo, $
;*                                 get_data_call, slice_type, IsRead
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro draw_crres_eph, panelVector, PlotInfo, orbit, $
                    get_data_call, slice_type, IsRead, OUTPUT=OUTPUT

COMMON extra, extra_PlotInfo  ; passes in PAPCO-set keywords structure

get_data_call = ''

; as papco dynamically allocates slice values for the user defined
; slices, one needs to be able to find out this assignment! Use the
; following function to find out the slice values for any slices you
; might have added for your module, so that you know which values to enable!
slice_assigned_value=papco_slice_value('USER Defined Slice A')
   
; set up which mouse functions to allow
slice_type=1+16

;always re-read
IsRead = 0

IF NOT IsRead THEN BEGIN            ; check if data has already been read
    r_crres_eph, PlotInfo, orbit    ; read data
    ; check the read for success.
    IF PAPCO_Draw_CheckResult(PlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              PlotInfo.typeVector, $
                              PlotInfo.channel)  THEN IsRead=1
ENDIF

IF IsRead THEN p_crres_eph, panelVector, PlotInfo, orbit, $
                            OUTPUT=OUTPUT, _EXTRA=extra_PlotInfo 

; Note: The structure extra_PlotInfo passed to the plot routine contains the
;       following tags: 
;                  overPlot               - overplot swittch
;                  plots_AtThisPosition   - no. of overplots here
;                  subtable               - index of PAPCO colortable 
;                  PSYM                   - linestyle keyword for plot
;                  COLOR                  - linecolor keyword for plot
;               
;       They are set by PAPCO and your plot routine can use them if you
;       want to support that functionality.

END

;******************************************************************************
pro draw_crres_eph_ticks, panelVector, currPlotInfo, orbitNo, $
                          get_data_call, slice_type, IsRead

  COMMON extra, extra_PlotInfo  ; passes in PAPCO-set keywords structure

  get_data_call=''              ; data read call used by slices
  pt=currPlotInfo.typeVector(0) ; set parameter for plot routine. 
  slice_type=0                  ; set up which mouse functions to allow: none
  
  IF NOT IsRead THEN BEGIN      ; check if data has already been read
    r_crres_eph, orbitNo        ; read data
    ; check the read for success.
    IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                               panelVector, orbitNr, $
                               currPlotInfo.typeVector, $
                               currPlotInfo.channel)  THEN IsRead=1
   ENDIF
    
  IF IsRead THEN $
    p_crres_eph_ticks, panelVector, pt, orbitNo, _EXTRA=extra_PlotInfo 

; Note: The structure extra_PlotInfo passed to the plot routine contains the
;       following tags: 
;                  overPlot               - overplot swittch
;                  plots_AtThisPosition   - no. of overplots here
;                  subtable               - index of PAPCO colortable 
;                  PSYM                   - linestyle keyword for plot
;                  COLOR                  - linecolor keyword for plot
;               
;       They are set by PAPCO and your plot routine can use them if you
;       want to support that functionality.

END
