;******************************************************************************
;* file p_crres_eph_line.pro
;*
;* this file is a template for a plot routine which can be used by papco.
;* it includes all the periphery needed, the user simply has to add in
;* the actual plot statement for his/her kind of data.
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_crres_eph_line, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_crres_eph_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_crres_eph_line, panel, plotinfo, orbit, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_crres_eph_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags which are meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type, swittch. Replace type in the call with whatever variables
; you need, or add more

; The following common blocks are needed:
  common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
  common yscale, yscl		       ;man/auto yscaling
  common shift_label, down             ;common for x-axis label shifting
  common coordinateSystems, plotted_x, plotted_y  
                                       ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_crres_eph. 
  common crres_eph_data, input_header, input_data
  
  pt=plotinfo.typeVector(0)
  l_low=plotinfo.options(2)
  l_high=plotinfo.options(3)
  panelset,panel		;sets the panel position viewport
  
; restrict data to actual time range requested - this makes for faster zooming
; remember input data is in TAI. For plotting, always start at zero
; to get maximum resolution (plot works in float only, large values
; get rounded!)
  convert_t90_to_Date, mjdt_start, year,doy,hr,min,sec,mon,cmon,doym, /mjdt 
  start_tai=UTC2TAI({YEAR:year, MONTH:mon, DAY:doym, HOUR:hr, MINUTE:min, $
                     SECOND:sec, MILLISECOND:0})
  convert_t90_to_Date, mjdt_end, year,doy,hr,min,sec,mon,cmon,doym, /mjdt 
  end_tai=UTC2TAI({YEAR:year, MONTH:mon, DAY:doym, HOUR:hr, MINUTE:min, $
                   SECOND:sec, MILLISECOND:0})
  
  index=where((input_data.tai ge start_tai) AND $
              (input_data.tai le end_tai),count)
  if count ne 0 then data=input_data(index) else data=input_data
  
  xut1=0  &  xut2=end_tai-start_tai  
  
; set time array
  time=data.tai-start_tai
  yray=data.(pt(0)+1)
  
; limit data in L - set unwanted data to nodata flag.
  nodata=1e-20 
  index=where( (data.XLSHEL le l_low) OR (data.XLSHEL ge l_high), c )
  if c ne 0 then begin
    yray(index)=nodata
    str_lrange=string(l_low,l_high,$
                       format="('L-range ',f4.1,'-',f4.1)")
  endif else begin
    message, 'L-range not found, plotting all', /cont 
    str_lrange=''
  endelse  
  
;from the tags in the data structure construct a list of choices. 
  tNames=TAG_NAMES(data)
  tNames=tNames(1:n_elements(tNames)-1)
  tunits=['km','km','km','km','km','deg','deg','hours','hours','deg',$
             'L-value','nT','nT','nT','km/s','km/s','km/s']
  tlim=fltarr(17,2)
  index=where(tunits eq 'km')    & tlim(index,0)=0     & tlim(index,1)=42000
  index=where(tunits eq 'hours') & tlim(index,0)=0     & tlim(index,1)=24
  index=where(tunits eq 'nT')    & tlim(index,0)=-12e3 & tlim(index,1)=12e3
  index=where(tunits eq 'km/s')  & tlim(index,0)=-10   & tlim(index,1)=10
  tlim(5,0)=-90 & tlim(5,1)=90
  tlim(6,0)=-180   & tlim(6,1)=180
  tlim(9,0)=-90 & tlim(9,1)=90
  tlim(10,0)=1  & tlim(10,1)=9
  
  if orbit eq 0 then orb_str='' else orb_str=string(orbit,format="(i4.4)")
  utitle='CRRES '+orb_str+'!C'+tNames(pt(0))+ '!C'+str_lrange
  uytitle=tunits(pt(0))
  ylim3,time,yray,yst,yen
  
  if tlim(pt(0),0) ne tlim(pt(0),1) then begin
    yst=tlim(pt(0),0)  &  yen=tlim(pt(0),1)
  endif  
     
; the variables used above have the following meaning:
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;
; the procedure YLIM3 is optional and sets the ylimit automatically for the
; data leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data. 
; (in $papco/papco_Lib/ylim.pro)

; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for "linear" and 1 for "log"  scale
; when scaling is set to automatic, the automatic y-limits are loaded in.
  if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
  endif else begin
    yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
  endelse
  
; use auto y-labeling routine to get "good" y-lables
  papco_y_label, yst, yen, LOG=yscl(panel(0),3)  

; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO:
;
;    !P.CHARSIZE  = widgetData.default_charsize    
;    !P.CHARTHICK = widgetData.default_charthick
;    !P.FONT      = -1
;    !P.NOERASE   = 1
;    !P.TICKLEN   = widgetData.default_TICKLEN
;    !X.TICKS     = widgetData.default_XTICKS
;    !X.STYLE     = widgetData.default_XSTYLE
;    !Y.STYLE     = 1
;    !X.MINOR     = widgetData.default_XMINOR
;
; Overriding these values by setting keywords is of course possible
; and allowed but discouraged, since 
; then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure.
   extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen],  $
                  ylog:yscl(panel(0),3), xtickformat:'noticks'}
   
; add keyword structure set here with the one passed in
   extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; use papco rotuine to draw time axis. This checks for bottom plot
; and uses the user's  xtickformat if it is something other than 'noticks'.
  down=0
  papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

; now do the actual data plot, data only, no axis
  papco_gap_plot,time, yray,nodata, _extra=extra_plotPar, xstyle=5, ystyle=5
  ;plot, time, yray, _extra=extra_plotPar, xstyle=5, ystyle=5
  
; draw y-axis label
  left_side_label, panel, uytitle, /ROT90

; store the coordinate information into common block for slices
  plotted_x = !x  &   plotted_y = !y  

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; down sets the axis labeling level
  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin

  endif

; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION
  if not keyword_set(overplot) then begin
    right_side_label,panel,utitle,_extra=extra_Par
  endif else begin
    for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C'+utitle
    right_side_label,panel,utitle,_extra=extra_Par
  endelse
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
  if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file',/cont
    description='crres_eph data '
    time_label='Format is TAI'
    ;channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, $
                      RS_LABEL=RS_LABEL
  endif  
      
end 
