;******************************************************************************
;* file p_crres_eph_ticks_line.pro
;*
;* this file is a template for a plot routine which can be used by papco.
;* it includes all the periphery needed, the user simply has to add in
;* the actual plot statement for his/her kind of data.
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_crres_eph_ticks_line, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_crres_eph_ticks_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_crres_eph_ticks, panel, pt, orbit, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                      SUBTABLE=SUBTABLE, _EXTRA=extra_par

common plot_composer, widgetData
  
; The following common blocks are needed:
common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
common yscale, yscl			;man/auto yscaling
common shift_label, down             ;common for x-axis label shifting
common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse
  
; One of the common blocks should contain the plot data as returned by the
; data read procedure r_crres_eph. 
common crres_eph_data, input_header, input_data
common crres_eph_tick1, time, data, tick_idx
  
panelset,panel			;sets the panel position viewport

;restrict data to actual time range requested - this makes for faster zooming
;input data is in TAI. For plotting, always start at zero to get maximum
;resolution (IDL plot works in float only, large values get rounded!) 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
if c ne 0 then data=input_data(index) ELSE return
xut1=0  &  xut2=tai_2-tai_1 & time = data.tai-tai_1
  
;from the tags in the input_data structure construct a list of choices. 
names=TAG_NAMES(input_data)
names=names(1:n_elements(names)-1)
  
names=['X', 'Y', 'Z', 'RAD', 'ALT','LAT', 'LON', 'LT', 'MLT', 'ILAT', $
         'L', 'BTEPF', 'BYEPF', 'BZEPF', 'VXEPF', 'VYEPF', 'VZEPF', 'UNI_L']
  
tunits=['(km)','(km)','(km)','(km)','(Re)','(deg)','(deg)','(hr)','(hr)', $
          '(deg)','','(nT)','(nT)','(nT)','(km/s)','(km/s)','(km/s)','']
    
extra_plotPar={xrange:[xut1,xut2], yrange:[0,1], ystyle:4, xstyle:9, $
                 xtickformat:'noticks'}
   
; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)
  
; use papco rotuine to draw time axis. This checks for bottom plot
; and uses the user's  xtickformat if it is something other than 'noticks'.
down=0
p=[0,panel(1),panel(2)]
papco_draw_time_axis, p, OVERPLOT=0, _extra=extra_plotPar
  
; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y
    
; plot extra x-axis labels
extra_plotPar.xtickformat='crres_ticks'
label=''
for i=0,down-1 do label=label+'!C'
  for i=0l,n_elements(names)-1 do begin
    if (pt and 2l^i) eq 2l^i then begin
      tick_idx=i+1
      papco_draw_time_axis, p, OVERPLOT=0, /FORCE, _extra=extra_plotPar
      if widgetData.default_right_label eq 0 then $
        label=label+names(i)+' '+tunits(i) +'!C' else $
        label=label+names(i)+'!C'
      down=down+1
    endif  
endfor
  
; The routine x_side_label puts x-axis label to the right of the axis.
x_side_label,panel, label
 
end 

;-----------------------------------------------------------------------------
; ONE function for ALL crres eph ticks.
FUNCTION crres_ticks, axis, index, t

COMMON mjdt, mjdt_start, mjdt_end
COMMON crres_eph_data, input_header, input_data
common crres_eph_tick1, time, data, tick_idx

COMMON shift_label, down             ;common for x-axis label shifting
  
down_str=''  &  for i=0,down-1 do down_str=down_str+'!C'
  
yray = data.(tick_idx)

;time starts at zero, papco_draw_time_axis at startsec of day. Need to add in. 
tt = time + double(mjdt_start.t)

difft=abs(tt-t) & result=min(difft,minindex) & tick=yray(minindex)
  
case tick_idx of
    1: out_str=string(tick,format="(i6)")
    2: out_str=string(tick,format="(i6)")
    3: out_str=string(tick,format="(i6)")
    4: out_str=string(tick,format="(i6)")
    5: out_str=string(tick,format="(f4.1)")
    9: out_str=string(tick,format="(f4.1)") ;XMLT
   10: out_str=string(tick,format="(f6.1)") ;XILAT
   11: out_str=string(tick,format="(f4.1)") ;XLSHEL
    else: out_str=string(tick)
endcase
  
return,down_str+out_str
  
END 

;tunits=['km','km','km','km','km','deg','deg','hours','hours','deg',$
;          'L-value','nT','nT','nT','km/s','km/s','km/s']
