;******************************************************************************
;*  NAME:        r_crres_eph
;*
;*  DESCRIPTION: Routine to read crres_eph files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*                                  Reads data by orbit only!
;*
;*  INPUT:       orbit - orbit number of crres orbit
;*               plotinfo - panel's plotinfo struct
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        January 2001
;*
;*  AUTOR:       R. Friedel - now abale to read across multiple orbit files
;******************************************************************************
PRO r_crres_eph, plotinfo, orbit, VERBOSE=VERBOSE

COMMON crres_eph_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg

; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.
get_err_no=0  &  get_err_msg='crres_eph read succesfully'
   
; check VERBOSE keyword and set
IF keyword_set(VERBOSE) THEN verbose=VERBOSE ELSE verbose=0     
   
; clear data arrays for reuse
input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here data are stored by orbit, so we need to find the orbit
; range corresponding to the start and end times.
time=0  &  no_files_read=0
   
; if orbit is non-zero -> read that orbit - otherwise get list of orbits
IF orbit EQ 0 THEN orbit_list = papco_getOrbitList('CRRES') $
              ELSE orbit_list = [orbit]
      
FOR i=0,n_elements(orbit_list)-1 DO BEGIN 

    ; read one orbit of data
    r_crres_eph_oneorbit, plotinfo, orbit_list(i), /VERBOSE
     
    ; check for an error in get_error - and then get the file remotely!
    ; comment this section out if you do not want this feature.
    ;if get_err_no ne 0 then begin      
    ;  c_url=papco_getenv('CRRES_EPH_FTP_ARCHIVE')
    ; assemble the data file from the orbit requested
    ;  orbitstr=string(format='(i4.4)',orbit)
    ;  fln='bin'+orbitstr
    ;  c_file=fln+'.idl'
    ;  c_download_dir=path
    ;  papco_get_remote_data, c_url, c_file, c_download_dir
    ;  ;check if file "arrived" and then read it
    ;  if get_err_no eq 0 then $
    ;    r_crres_eph_oneorbit_idlsunbin, path+fln, /VERBOSE $
    ;  else begin
    ;    get_err_msg='% r_crres_eph: No data locally, remote get failed!C  ' $
    ;      + get_err_msg
    ;    return
    ;  endelse  
    ;endif 
     
    ;if a file was succesfully read, concatonate!
    IF get_err_no EQ 0 THEN BEGIN 
        IF no_files_read EQ 0 THEN BEGIN   ;start the array buffer
            input_data_buf=temporary(input_data)
            ndat=input_header.ndat
        ENDIF ELSE BEGIN       ;otherwise concatonate arrays
            input_data_buf=[temporary(input_data_buf),temporary(input_data)]
            ndat=ndat+input_header.ndat
            input_header.orbit=0000
        ENDELSE    
        input_data=0                     ;clear array for reuse
        no_files_read=no_files_read+1
    ENDIF 
ENDFOR          
   
; check if any files were read   
IF (no_files_read NE 0) AND (get_err_no EQ 0) THEN BEGIN
    get_err_no = 0
    get_err_msg = string(no_files_read,format="(i2)")+ ' files read'
    ;clear arrays and load buffer into output array
    input_data = 0 & input_data = input_data_buf & input_data_buf = 0
    input_header.ndat = ndat
ENDIF ELSE BEGIN 
    get_err_no=1
    get_err_msg='no files found for timerange'
    return
ENDELSE 
message, get_err_msg, /cont
   
END 

;******************************************************************************
;*  NAME:        r_crres_eph_oneorbit
;*
;*  DESCRIPTION: Routine to read one crres_eph orbit file. 
;*
;*  INPUT:       orbit  - CRRES orbit number
;*               plotinfo - panel's plotinfo struct
;*
;*  KEYWORDS:    VERBOSE - if set, give plenty diagnostics. Default is off.
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        07 October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_crres_eph_oneorbit, plotinfo, orbit, VERBOSE=VERBOSE

COMMON crres_eph_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg
      
; check env. variable pointing to data directory
IF NOT (papco_check_data_env('CRRES_EPH_DATA', path=PATH)) THEN return 

; check VERBOSE keyword and set
IF keyword_set('VERBOSE') THEN verbose=VERBOSE ELSE verbose=0

get_err_no=0 &  get_err_msg='crres_eph_oneorbit read succesfully'
  
control = *plotinfo.USR_PTR1

fln = path+'savesets/crres_eph_'+string(orbit,format="(i4.4)")+'.idl'
eph_struct_flname = path+'savesets/eph_struct.idl'

on_ioerror, read_orig
restore, fln
message, fln ,/INFO
GOTO, check_mag

read_orig:

r_crres_eph_ascii, path+'archive/'+fln
IF get_err_no NE 0 THEN GOTO, bad_restore
   
check_mag:
;now make/add/check mag coords in input_data.
;need to "hand over" time in TAI,coord system info, actual eph coords. 

;CRRES eph is in geographic lat/long
tai = input_data.tai
ndat =  n_elements(input_data)
pos = dblarr(3, ndat)

rad = input_data.XRAD
lon = input_data.XLONG
colat = 90.0d - input_data.XLAT

cor_type='CAR' & cor_sys='GEO'
pos(0, *) = rad*sin(colat/!radeg)*cos(lon/!radeg)
pos(1, *) = rad*sin(colat/!radeg)*sin(lon/!radeg)
pos(2, *) = rad*cos(colat/!radeg)

papco_add_mag, control, cor_type, cor_sys, $
               eph_struct_flname, new_struct_name, tai, pos, input_data, error

IF error EQ 2 THEN return ;no new structures added to input_data

;save the new eph data to file
message, 'Writing new', /cont
print, '  ', fln
save, file = fln, input_header, input_data, /COMPRESS

;save the new eph structure to file
message, 'Writing new structure contents to file', /cont
eph_struct = input_data(0)
save, file = eph_struct_flname, eph_struct

return

bad_restore: 
; if the orignal data can't be found either, return with an error message.
print, !err_string
get_err_no=1  &  get_err_msg=!ERR_STRING

END 
 

;******************************************************************************
PRO r_crres_eph_ascii, fln
  
common crres_eph_data, input_header, input_data
common get_error, get_err_no, get_err_msg
      
get_err_no=0 & et_err_msg='crres_eph_ascii read succesfully'

message,'reading original ASCII format '+fln+'.txt',/cont
   
   input_header={crres_eph_data_header, orbit:0, doy:0, year:0, ndat:0} 
   
   dummy={crres_ephemeris, time:0.0d, xsat:0.0, ysat:0.0, zsat:0.0, $
          xrad:0.0, xalt:0.0, xlat:0.0, xlong:0.0, xlt:0.0, xmlt:0.0, $
          xilat:0.0, xlshel:0.0, btepf:0.0, byepf:0.0, bzepf:0.0, $
          vxepf:0.0, vyepf:0.0, vzepf:0.0}
   
   on_ioerror, goreport
   openr,unit,fln+'.txt',/get_lun
   
   readf,unit,input_header
   input_data=replicate(dummy,input_header.ndat)
   
   i=-1
   while not eof(unit) do begin
     i=i+1
     readf,unit,dummy
     input_data(i)=dummy
   endwhile   
   
   close,unit
   free_lun,unit
   
save,filename=c_path+fln+'.idl',input_header,input_data


   return
   
   goreport:
   get_err_no=3
      
END  


;******************************************************************************
;program to make new eph for CRRES with TAI time in it
pro crres_tai_eph
  
  eph_path='/data1/tmp/CRRES/EPH/ascii_complete/'  
  new_eph_path='/data1/tmp/CRRES/EPH/savesets/'
  
  spawn, 'pwd', result  &  currdir=result(0)
  
  cd, eph_path
  files=findfile('*.idl',count=c)
  
  for i=0,c-1 do begin
    print,''
    message,'Doing '+files(i),/cont
    restore, files(i)
    
    dummy=create_struct(name='CRRES_EPH_TAI', input_data(0), 'TAI',0.0d)
    new_input_data=replicate(dummy,input_header.ndat)
    
    message,'converting to TAI',/cont
    utc=doy2utc(input_header.DOY,1900+input_header.YEAR)
    for j=0, input_header.ndat-1 do begin
      for k=0,17 do new_input_data(j).(k)=input_data(j).(k)
      utc.time=new_input_data(j).time*1000
      new_input_data(j).tai=utc2tai(utc)
    endfor
    
    input_data=temporary(new_input_data)
    fln='crres_eph_'+strmid(files(i),1,4)+'.idl'
    message, 'Saving to' + new_eph_path+fln, /cont
    save, filename=new_eph_path+fln, input_header, input_data
    
  endfor
  
  cd, currdir
  
END 
