;******************************************************************************
;*  NAME:        r_crres_ephem_uni
;*
;*  DESCRIPTION: Routine to read crres_ephem_uni files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
pro r_crres_ephem_uni, orbit, VERBOSE=VERBOSE

   common crres_ephem_uni_data,crres_ephem_uni  
   common mjdt, mjdt_start, mjdt_end
   common get_error, get_err_no, get_err_msg

; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

   get_err_no=0
   get_err_msg='crres_ephem_uni read succesfully'
   
; check for env. variable pointing to data directory
   data_dir='crres_ephem_uni_DATA'
   ;path=papco_getenv(data_dir)
   path='/net/hydra/data/CRRES/EPHEMERIS/CIF/unilib_bla/'
   if strlen(path) eq 0 then begin
     get_err_no=1
     get_err_msg='Environmental variable '+data_dir+' not set'
     print,get_err_msg
     return
   endif
   
; check for existence of data directory
   cc=papco_finddir(path)
   if cc eq 0 then begin
     get_err_no=2
     get_err_msg='Data directory '+path+' not found'
     print,get_err_msg
     return
   endif
   
; check VERBOSE keyword and set
   if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. papco standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
   
;   time=mjdt_start  &  no_files_read=0  &  day_count=0.0
   
   orbitstr=string(orbit,format="(i4.4)")
   fln='CRRES_EPH_'+orbitstr
   
   r_crres_ephem_uni_oneorbit, path+fln, /VERBOSE

end

;******************************************************************************
;*  NAME:        r_crres_ephem_uni_oneorbit,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_crres_ephem_uni_oneorbit, fln, VERBOSE=VERBOSE
  
  common crres_ephem_uni_data, crres_ephem_uni
  common get_error, get_err_no, get_err_msg  
  
  get_err_no=0
  get_err_msg='crres_ephem_uni_onefile_orignal read succesfully'
  
; try to read idl binary first
  filethere=findfile(fln+'.blaidl')
  if filethere(0) ne '' then begin
    restore,fln+'.blaidl'
    print, '% r_crres_ephem_uni: '+fln+'.blaidl' 
    return
  endif
    
  bad_restore:
  
  print, '% r_crres_ephem_uni: ASCII '+fln+'.blaidl (slow)'
  
  dummy={POLAR_EPHEM_UNI, $
         Time:0.0d,       $ 
         MAG_LONG:0.0,    $
         MAG_LAT:0.0,     $
         ALT:0.0,         $
         P_ANGLE:0.0,     $
         Br:0.0,          $
         Btheta:0.0,      $
         Bphi:0.0,        $
         B:0.0,           $
         B_eq:0.0,        $
         L_SHELL:0.0   }
  
  crres_ephem_uni=replicate(dummy,3000)

  dummystr=''
  openr, unit, fln+'.bla', /get_lun, ERROR=err
  IF err NE 0 THEN BEGIN
    print, '% r_ploar_ephem_uni: '+fln+' : not available.'
    get_err_no=2
    get_err_msg=!ERR_STRING
    print, !ERR_STRING
    RETURN
  ENDIF
  
  readf,unit,dummystr
  readf,unit,dummystr
  readf,unit,dummystr
  readf,unit,dummystr
  i=-1
  while not eof(unit) do begin
    i=i+1
    readf,unit,dummystr
    reads,dummystr,dummy
    crres_ephem_uni(i)=dummy
  endwhile
  
  crres_ephem_uni = crres_ephem_uni(0:i-1)

  ; convert time from decimal hours to seconds since midnight
  crres_ephem_uni.Time=(crres_ephem_uni.Time-floor(crres_ephem_uni.Time)) * $
    (3600.*24.)+(floor(crres_ephem_uni.time)- $
                 floor(crres_ephem_uni(0).time)) * 86400.
  close,/file
   
; save read data with correct time. Rename data to *.idl
  save,filename=fln+'.blaidl',crres_ephem_uni,/verbose  
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return
  
  out:
  ; if there was an error, report it. Use the IDL error variables. Error
  ; 171 is for "file not fond" which is used to invoke the remote get.  
  get_err_no=!ERROR
  get_err_msg='r_crres_ephem_uni_onefile:!C '+!ERR_STRING
   
end
