;******************************************************************************
;* file papcoadd_crres_lepa
;*      papcoadd_crres_lepa_flux
;*      papcoadd_crres_lepa_spectra
;*      papcoadd_crres_lepa_moments
;*      papcoadd_crres_lepa_pitch
;*      papcoadd_crres_lepa_overview
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for a crres_lepa. 
;*

;* Whenever papco has to undertake some action for a
;* crres_lepa, the field plotInfo.panelkind is compared to the entries in the
;* (COMMON ADDINS) crres_lepaS array. If a match is found, the required
;* procedure name is constructed and called If not, an error occurs.
;*
;*    These procedures are contained:
;*
;*    PRO addplottype_crres_lepa,plottypes_base,pb_plottype
;*        ->  adds the button for the new plot type to the main window.
;*
;*    PRO plottable_crres_lepa, plottable
;*        -> Sets a variable to indicate plottable without orbit.
;*
;*    PRO canget_crres_lepa, canget
;*        -> Sets a variable to indicate output to file is supported.
;*
;*    PRO descriptor_crres_lepa, plotInfo, descriptStr
;*        -> Constructs the string used to label plots in the panels window
;*
;*    PRO addpanel_crres_lepa
;*        -> Calls the panel editor for this plottype and adds the plot to the
;*            panel window.
;*
;*    PRO editor_crres_lepa, plotNumber, newPlotData
;*        -> This procedure calls the panel editor when called from 
;*           papco_EditPlot (papco_paneledit.pro).
;*
;*    PRO  draw_crres_lepa, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_crres_lepa,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*      This procedure adds a button for this crres_lepa to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*      package, so that they can be automatically called when needed.
;*      The button itself can contain access to a sub-menu for further
;*      crres_lepas.
;*
;* INPUTS:       
;*      plottypes_base  the widget ID of the parent widget to contain the
;*                      plottype buttons.
;*
;* OUTPUTS: 
;*      pb_plottype     the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for sub-types.
;*
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      addplottype_crres_lepa,plottypes_base,buttonGeometry,pb_plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro addplottype_crres_lepa,plottypes_base,buttonGeometry,pdm_crres_lepa

  MenuDesc = [ $
               { CW_PDMENU_S, 3, strcenter('crres_lepa',10) }, $ 
               { CW_PDMENU_S, 0, ' configure crres_lepa ' }, $
               { CW_PDMENU_S, 0, ' lepa flux      ' }, $  
               { CW_PDMENU_S, 0, ' lepa spectra   ' }, $ 
               { CW_PDMENU_S, 0, ' lepa moments   ' }, $ 
               { CW_PDMENU_S, 0, ' lepa pitch     ' }, $ 
               { CW_PDMENU_S, 2, ' lepa overview  ' } ]  
  
  pdm_crres_lepa = CW_PDMENU(plottypes_base, MenuDesc, $
                            UVALUE='pdm_crres_lepa')
  
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_crres_lepa
;*
;* DESCRIPTION:  
;*      Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      Using an instance of the papco_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled. The
;*      desired default values for the plotinfo tags need to be edited by
;*      the user. This plotinfo structure will be passed to the panel editor
;*      When it is first called for the crres_lepa.
;*      "panelEditor" must be set to the name of the panel editor routine.
;*
;* INPUTS:       
;*      event           event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      addpanel_crres_lepa
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;***************************************************************************
pro addpanel_crres_lepa,event
  
  common get_error, get_err_no, get_err_msg
  
  r_module_config, strlowcase('crres_lepa'), config_data, msg
  if (get_err_no ne 0) and (event.value ne 1) then begin
    print,'%addpanel_crres_lepa: You need to configure the module first'
    papco_module_configure,'crres_lepa'
    return
  endif  
 
  CASE event.value of
    1: BEGIN
        papco_module_configure,'crres_lepa'
    end
    2: BEGIN
        panelKind='crres_lepa_flux'
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
        defaults.typeVector=[0,0,0,0]      ;set some tags to desired defaults
        defaults.channel=0            
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
    end
    3: BEGIN
        panelKind='crres_lepa_spectra'
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
        defaults.typeVector=[0,2,0,0]      ;set some tags to desired defaults
        defaults.channel=0            
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
    end
    4: BEGIN
        panelKind='crres_lepa_moments'
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
        defaults.typeVector=[0,0,0,0]      ;set some tags to desired defaults
        defaults.channel=0            
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
    end
    5: BEGIN
        panelKind='crres_lepa_pitch'
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
        defaults.typeVector=[0,0,0,0]      ;set some tags to desired defaults
        defaults.channel=0            
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
    end
    6: BEGIN
        panelKind='crres_lepa_overview'
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
        defaults.typeVector=[0,0,0,0]      ;set some tags to desired defaults
        defaults.channel=0            
        defaults.ylinlog=0
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
    end
    else: begin
      print,'%addpanel_crres_lepa: Nothing defined for this submenu entry'
      return
    end  
  endcase
      
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO plottable_crres_lepa, plottable
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate plottable with / without orbit.
;*      PAPCO provides "set time by orbit" feature for CRRES and POLAR
;*      so far. Included for compatibility with older CRRES modules
;*      which ONLY use orbit as input to their read/plot routines
;*      (plotting over orbit boundaries is not enabled for CRRES)
;* 
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "plottable_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_crres_lepa.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      plottable        a boolean 0 if plottable only by orbit
;*                                 1 if plottable by time also.
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      plottable_crres_lepa, plottable
;*
;* MODIFICATION HISTORY:       
;*      written November 1995, Reiner Friedel
;******************************************************************************
pro plottable_crres_lepa_flux, plottable

   plottable=0     ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 
END
pro plottable_crres_lepa_spectra, plottable

   plottable=0     ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 
END
pro plottable_crres_lepa_moments, plottable

   plottable=0     ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 
END
pro plottable_crres_lepa_pitch, plottable

   plottable=0     ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 
END
pro plottable_crres_lepa_overview, plottable

   plottable=1     ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO canget_crres_lepa, canget
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate that crres_lepa is capable of
;*      returning data from it's plot routine into a standard PAPCO
;*      array structure. Used for the scratch pad utility module.
;*      
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "canget_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_crres_lepa.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      canget         a boolean 0 if this feature is not available
;*                                1 if this feature is available
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      canget_crres_lepa, canget
;*
;* MODIFICATION HISTORY:       
;*      written November 1997, Reiner Friedel
;******************************************************************************
pro canget_crres_lepa_flux, canget

   canget=  1      ; 1 -> plot routine can return just data to file
                   ; 0 -> feature disabled
END
pro canget_crres_lepa_spectra, canget

   canget=  1      ; 1 -> plot routine can return just data to file
                   ; 0 -> feature disabled
END
pro canget_crres_lepa_moments, canget

   canget=  1      ; 1 -> plot routine can return just data to file
                   ; 0 -> feature disabled
END
pro canget_crres_lepa_pitch, canget

   canget=  1      ; 1 -> plot routine can return just data to file
                   ; 0 -> feature disabled
END
pro canget_crres_lepa_overview, canget

   canget=  1      ; 1 -> plot routine can return just data to file
                   ; 0 -> feature disabled
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_crres_lepa, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*      Constructs the string used to label plots in the panel window.
;*      The plotinfo structure passed in here is the panel's plotinfo
;*      as returned by the panel editor for this crres_lepa. You can use
;*      the information in that structure to construct some informative
;*      label for the panel which then appears in the Panel Compose list.
;*
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "descriptor_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_crres_lepa.
;*
;* INPUTS:       
;*      plotInfo        structure containing all data for this plot.
;*                      (structure type: papco_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*      descriptStr     a string containing the label
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      descriptor_crres_lepa, plotInfo, descriptStr
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro descriptor_crres_lepa_flux, plotInfo, descriptStr

   
END
pro descriptor_crres_lepa_spectra, plotInfo, descriptStr

   
END
pro descriptor_crres_lepa_moments, plotInfo, descriptStr

   
END
pro descriptor_crres_lepa_pitch, plotInfo, descriptStr

   
END
pro descriptor_crres_lepa_overview, plotInfo, descriptStr

   
END
;******************************************************************************
;* PROCEDURE:     
;*      PRO editor_crres_lepa, plotNumber, newPlotData
;* 
;* DESCRIPTION:  
;*      This procedure calls the panel editor when called from 
;*      papco_EditPlot (papco_paneledit.pro). Provided here to allow
;*      several crres_lepas to share a common panel editor. Normally this
;*      procedure does not need to be edited beyond the required name
;*      change for crres_lepa.
;*
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "editor_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_crres_lepa.
;*
;* INPUTS:       
;*      plotNumber      anInteger. 0<=plotNumber<=widgetData.numberOfPlots
;*                      the index of the plot in widgetData.plotInfos.
;*            
;* OUTPUTS: 
;*      newPlotData     the new structure returned by the panel editor
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      editor_crres_lepa, plotNumber, newPlotData
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
PRO editor_crres_lepa_flux, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=crres_lepa_flux_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END
PRO editor_crres_lepa_spectra, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=crres_lepa_spectra_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END
PRO editor_crres_lepa_moments, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=crres_lepa_moments_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END
PRO editor_crres_lepa_pitch, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=crres_lepa_pitch_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END
PRO editor_crres_lepa_overview, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=crres_lepa_overview_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO draw_crres_lepa, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:  
;*      This procedure performs the actual data read call and the plot call
;*      for this plottype. The user need to pass to his plot routine all the
;*      parameters needed, typically these parameters are stored in the 
;*      currplotinfo structure or constructed from them. The currplotinfo
;*      structure tags normally used for passing information from the panel
;*      editor to the plot routine are:
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          swittch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      
;*      These tags have grown out of traditional usage and can be used for
;*      anything the user wants to use them for. There are plenty variables
;*      for most applications: Adding a new tag to the plotinfo structure
;*      is not user-definable but need to be done at the papco core level.
;* 
;*      The panelVector MUST be passed to your plotroutine as it is used
;*      to set the panel position.
;*      The _EXTRA=extra_PlotInfo should be passed if you intend to make use
;*      of papco functionality such as overplotting, linestyle choice and
;*      color bar choice.
;*
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "draw_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_crres_lepa.
;*
;* INPUTS:       
;*      panelVector    position of the plot (see papco_conventions.text, 6)
;*      currPlotInfo   structure containing all data for this plot.
;*                     (structure papco_PLOTINFO, see papco_variables.text) 
;*
;* OUTPUTS: 
;*      get_data_call  a string containing the data call. Used by other
;*                     routines that need to know how to get the data
;*                     in a panel, e.g. to draw spectra, distribution functions
;*      IsRead         a boolean which is set when the the data is read the
;*                     first time. Useful if a given data-read puts data
;*                     into commons which can be plotted in many ways.
;*                     When set, data does not have to be read again.
;   slice_type:   Selects which slices are supported and for which slice
;                 routines have been written.
;                 This is an integer logical with 16 swittches. Currently
;                 there are 5 papco slices. You may extend this by
;                 further USER slices up to a total of 16 slices.
;                     0 = disable all slices
;                     1 = papco Zoom                 (2^0)
;                     2 = papco View Slice           (2^1)
;                     4 = papco Access level Zero    (2^2)
;                     8 = papco Write Paneldata      (2^3)
;                    16 = papco Hair Cursor          (2^4)
;                    32 = USER slice A or not used   (2^5)
;                    64 = USER slice B or not used   (2^6)
;                    "     "     "   "  "  "    "      "    
;                 32768 = USER slice Z or not used  (2^15)
;                        
;   You can support more than one type of slice by arithmetically adding the
;   options, eg slice_type=1+4  support Zoom and Level Zero
;*
;* KEYWORDS:
;*      OUTPUT        if set then plot routine should output data to file,
;*                    as well as to screen.
;*
;* 
;* CALLING SEQUENCE:
;*      draw_crres_lepa, panelVector, currPlotInfo, orbitNo, $
;*                      get_data_call, slice_type, IsRead, NODATA=NODATA
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro draw_crres_lepa_flux, panelVector, currPlotInfo, orbitNr, $
                    get_data_call, slice_type, IsRead, $
                    OUTPUT=OUTPUT

  COMMON extra, extra_PlotInfo  ; passes in papco-set keywords structure
    
  get_data_call="r_crres_lepa"   ; data read call used by slices

; set up which mouse functions to allow
  slice_type=1+2+4+8+16

  IF NOT IsRead THEN BEGIN      ; check if data has already been read
    r_crres_lepa                ; read data
    ; check the read for success.
    IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)       THEN IsRead=1
  ENDIF
  
  IF IsRead THEN BEGIN
    if channel ne 0 then $
      p_crres_lepa_line, panelVector, channel, OUTPUT=OUTPUT, $
      _EXTRA=extra_PlotInfo $
    else $
      p_crres_lepa_spec, panelVector, channel, OUTPUT=OUTPUT, $
      _EXTRA=extra_PlotInfo
  ENDIF

END
;------------------------------------------------------------------------------
pro draw_crres_lepa_spectra, panelVector, currPlotInfo, orbitNr, $
                    get_data_call, slice_type, IsRead, $
                    OUTPUT=OUTPUT

  COMMON extra, extra_PlotInfo  ; passes in papco-set keywords structure
  
  str_orbitNr=strtrim(string(orbitNr), 2)
  str_typeVector=string(currPlotInfo.typeVector, $
                        FORMAT='("[", i2, ",", i2, ",", i2, ",", i2, "]")')
  
  get_data_call='r_crres_meb, '+str_orbitNr+', '+str_typeVector+$
    '& r_crres_lepa_spectra'+str_orbitNr ; data read call used by slices
  
  r_crres_eph, orbit=orbitNr    ; read ephemeris information to be used for
                                ; tick marks
  
; set up which mouse functions to allow
  slice_type=1+2+4+8+16

  IF NOT IsRead THEN BEGIN      ; check if data has already been read
    r_crres_lepa_spectra, orbitNr       ; read data
    ; check the read for success.
    IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)       THEN IsRead=1
  ENDIF
  
  IF IsRead THEN BEGIN
    p_crres_lepa_spectra, panelVector, currPlotInfo.typeVector, OUTPUT=OUTPUT, $
      _EXTRA=extra_PlotInfo
  ENDIF

END
;------------------------------------------------------------------------------
pro draw_crres_lepa_moments, panelVector, currPlotInfo, orbitNr, $
                    get_data_call, slice_type, IsRead, $
                    OUTPUT=OUTPUT

  COMMON extra, extra_PlotInfo  ; passes in papco-set keywords structure
    
  get_data_call="r_crres_lepa"   ; data read call used by slices

; set up which mouse functions to allow
  slice_type=1+2+4+8+16

  IF NOT IsRead THEN BEGIN      ; check if data has already been read
    r_crres_lepa                ; read data
    ; check the read for success.
    IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)       THEN IsRead=1
  ENDIF
  
  IF IsRead THEN BEGIN
    if channel ne 0 then $
      p_crres_lepa_line, panelVector, channel, OUTPUT=OUTPUT, $
      _EXTRA=extra_PlotInfo $
    else $
      p_crres_lepa_spec, panelVector, channel, OUTPUT=OUTPUT, $
      _EXTRA=extra_PlotInfo
  ENDIF

END
;------------------------------------------------------------------------------
pro draw_crres_lepa_pitch, panelVector, currPlotInfo, orbitNr, $
                    get_data_call, slice_type, IsRead, $
                    OUTPUT=OUTPUT

  COMMON extra, extra_PlotInfo  ; passes in papco-set keywords structure
    
  get_data_call="r_crres_lepa"   ; data read call used by slices

; set up which mouse functions to allow
  slice_type=1+2+4+8+16

  IF NOT IsRead THEN BEGIN      ; check if data has already been read
    r_crres_lepa                ; read data
    ; check the read for success.
    IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)       THEN IsRead=1
  ENDIF
  
  IF IsRead THEN BEGIN
    if channel ne 0 then $
      p_crres_lepa_line, panelVector, channel, OUTPUT=OUTPUT, $
      _EXTRA=extra_PlotInfo $
    else $
      p_crres_lepa_spec, panelVector, channel, OUTPUT=OUTPUT, $
      _EXTRA=extra_PlotInfo
  ENDIF

END
;------------------------------------------------------------------------------
pro draw_crres_lepa_overview, panelVector, currPlotInfo, orbitNr, $
                    get_data_call, slice_type, IsRead, $
                    OUTPUT=OUTPUT

  COMMON extra, extra_PlotInfo  ; passes in papco-set keywords structure
    
  str_typeVector=string(currPlotInfo.typeVector, $
                           FORMAT='("[", i2, ",", i2, ",", i2, ",", i2, "]")')
  get_data_call='r_lepa_overview'+str_typeVector
  
; set up which mouse functions to allow
  slice_type=1+2+4+8+16
  
  IsRead=0
  IF NOT IsRead THEN BEGIN      ; check if data has already been read
    r_lepa_overview, currPlotInfo.typeVector ; read data
    ; check the read for success.
    IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)       THEN IsRead=1
  ENDIF
  
  IF IsRead THEN BEGIN
    p_crres_lepa_overview, panelVector, currPlotInfo.typeVector, $ 
      _EXTRA=extra_PlotInfo
  ENDIF

END
