;****************************************************************************
;*  NAME:        g_lepa
;*
;*  DESCRIPTION: Routine to convert LEPA "LSD" files to binary
;*		 idl files. It calculates number fluxes only.
;*
;*  INPUT:       CRRES orbit number
;*
;*  OUTPUT:      An idl binary file "*.lsdbin"
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     January 1997 by C. Mouikis: Background extraction was
;*		 corrected, only counts are culculated.
;****************************************************************************
pro g_lepa,orbit

   common rawdata, elflux,pflux
   
   
   junk = ''
   buf  = 0.0
   fe0  = fltarr(15)		;electron efficiencies mode 0
   feA  = fltarr(8)		;electron efficiencies mode 10
   fi   = fltarr(8)		;ion efficiencies
   be0  = fltarr(15)		;electron background mode 0
   beA  = fltarr(8)		;electron background mode 10
   bi   = fltarr(8)		;ion background
   gel  = fltarr(20)		;electron low voltage geometric factors
   geh  = fltarr(20)		;electron high voltage geometric factors
   gil  = fltarr(29)		;ion low voltage geometric factors
   gih  = fltarr(29)		;ion high voltage geometric factors
   
;----------------------------------------------------------------------------
; read the survey.cal file first to get all calibration data.
; Latest version of "survey.cal" is #4
;----------------------------------------------------------------------------

;   path=getenv('PAPCO_MODULES')
   path='/home/mouikis/mypapco/crres_lepa/plot_read/'

   openr,1,path+'survey.cal', ERROR=err
   readf,1,junk 
   readf,1,junk                   ;electron efficiencies mode 0
   print,junk
   for i=0,14 do begin 
      readf,1,buf
      fe0(i)=1./buf
   endfor
   readf,1,junk                   ;electron efficiencies mode 10
   print,junk
   for i=0,7 do begin 
      readf,1,buf 
      feA(i)=1./buf 
   endfor
   readf,1,junk                   ;ion efficiencies
   print,junk
   for i=0,7 do begin 
      readf,1,buf 
      fi(i)=1./buf 
   endfor
   readf,1,junk                   ;electron background mode 0
   print,junk
   for i=0,14 do begin  
      readf,1,buf 
      be0(i)=buf 
   endfor
   readf,1,junk                   ;electron background mode 10
   print,junk
   for i=0,7 do begin 
      readf,1,buf  
      beA(i)=buf 
   endfor
   readf,1,junk                   ;ion background
   print,junk
   for i=0,7 do begin 
      readf,1,buf 
      bi(i)=buf 
   endfor
   readf,1,junk                   ;electron low voltage geometric factors
   print,junk
   for i=0,19 do begin 
      readf,1,buf,format='(E10.4)' 
      gel(i)=1./buf 
   endfor
   readf,1,junk                   ;electron high voltage geometric factors
   print,junk
   for i=0,19 do begin 
      readf,1,buf,format='(E10.4)' 
      geh(i)=1./buf 
   endfor
   readf,1,junk                   ;ion low voltage geometric factors
   print,junk
   for i=0,28 do begin 
      readf,1,buf,format='(E10.4)' 
      gil(i)=1./buf 
   endfor
   readf,1,junk                   ;ion high voltage geometric factors
   print,junk
   for i=0,28 do begin 
      readf,1,buf,format='(E10.4)' 
      gih(i)=1./buf 
   endfor
   readf,1,junk 
   print,'survey.cal: '+junk
   close,1
   
;----------------------------------------------------------------------------
; Calculate bi9, be09, bea9
;----------------------------------------------------------------------------

   bi9 = (bi(3) + bi(4)) * 0.5
   bea9 = (beA(3) + beA(4)) *0.5
   be09 = (be0(6) + be0(7) + be0(8) + be0(9)) * 0.25
   
;----------------------------------------------------------------------------
; Now read the LEPA file
;----------------------------------------------------------------------------

   buffer=bytarr(3000,595)
   inst=replicate(0B,595)
;   path=getenv('LEPA_LSD_PATH')
   path='/cdrom/lsd'
   fln=path+'/'+string(orbit,format="(i8.8)")
   openr,1,fln+'.lsd'
   readu,1, inst 

   id= inst(0)+inst(1)*256
   iyear= inst(2)+inst(3)*256
   iday= inst(4)+inst(5)*256
   iorbit= inst(6)+ inst(7)*256
   tb=long(inst(8)+inst(9)*256+inst(10)*65536l+inst(11)*16777216l)
   te=long(inst(12)+inst(13)*256+inst(14)*65536l+inst(15)*16777216l)

   ii=-1
   while  not eof(1) do begin
      ii=ii+1 
      readu,1, inst 
      buffer(ii,*)=inst
   endwhile
   recs=ii
   close,1

   xut=fltarr(recs+1)            ; time array
   
   e_cps=fltarr(recs+1,20,15)    ; counts / el   20 energies 15 zones
   p_cps=fltarr(recs+1,29,8)     ; counts / ions 29 energies 8 zones
   e_cps_perp=fltarr(recs+1,20)  ; counts / perp el   20 energies
   p_cps_perp=fltarr(recs+1,29)  ; counts / perp ions 29 energies
   
   lc_arr=byte(intarr(recs+1))   ; loss cone array
   mo_arr=byte(intarr(recs+1))   ; mode array
   an_arr=byte(intarr(recs+1))   ; angle array
   tc_arr=fltarr(recs+1)         ; sampling frequency array
   
   rec_arr=fltarr(recs+1)        ; electron coarse zone background 
   re9_arr=fltarr(recs+1)        ; electron 90 degree snapshot bkg
   ric_arr=fltarr(recs+1)        ; ion coarse zone bkg
   ri9_arr=fltarr(recs+1)        ; ion 90 degree snapshot bkg
   
;----------------------------------------------------------------------------
;now untangle the bytes into arrays and decompress
;----------------------------------------------------------------------------

   print,'read whole file, processing....>'
   i=-1
   for ii=0,recs do begin
      i=i+1
      inst=buffer(ii,*)
      ut=long(( inst(0)+ inst(1)*256l ) * 3l) ; Time of half spin (sec div 3)
      xut(i)=float(ut)
      
      tc=long(inst(2)+ inst(3)*256l) 
      tc_arr(i)=tc
      tc1 = tc / 500	        ; Sampling frequency for electrons
      tc8 = tc1 * 0.125		; Sampling frequency for ions
      
      bm= inst(4)       ; Bmin/Bloc*100 (i.e. 0-100%)
      lc= inst(5)       ; width of the loss cone in degrees
      ltt= inst(6)      ; hours*10 (i.e. mlt = LT / 10 hours)(GSM coord)
      l= inst(7) / 20.0 ; L shell in EMR * 20 (i.e. L shell in Re = L /20)
      ebgc= inst(8)     ; compressed el bkg coarse counts
      ebg9= inst(9)     ; compressed el bkg 90 deg snapshot counts
      mo= inst(10)      ; mode with 
      an= inst(11)      ; angle in degrees between B field and +z S/C axis
      ibgc= inst(12)    ; compressed ion bkg coarse counts
      ibg9= inst(13)    ; compressed ion bkg 90 deg snapshot counts
      
      lc_arr(i)=lc      ; save in loss cone array (byte)
      mo_arr(i)=mo      ; save in mode array      (byte)
      an_arr(i)=an      ; save in angle array     (byte)

;----------------------------------------------------------------------------
; uncompress ebgc, ebg9, ibgc, ibg9:
; Compressed counts * 8 for electrons and ions,
; coarse zone and 90 degree snapshots
;----------------------------------------------------------------------------

      rec = 0.0
      if (ebgc lt 255) then begin
	rec = lepa_uc(ebgc) ;uncomressed value
        rec = rec * tc8
        rec_arr(i)=rec
      endif
      re9 = 0.0
      if (ebg9 lt 255) then begin
	re9 = lepa_uc(ebg9) ;uncomressed value
        re9 = re9 * tc8
        re9_arr(i)=re9
      endif
      ric = 0.0
      if (ibgc lt 255) then begin
	ric = lepa_uc(ibgc) ;uncomressed value
        ric = ric * tc8
        ric_arr(i)=ric
      endif
      ri9 = 0.0
      if (ibg9 lt 255) then begin
	ri9 = lepa_uc(ibg9) ;uncomressed value
        ri9 = ri9 * tc8
        ri9_arr(i)=ri9
      endif
      


;----------------------------------------------------------------------------
;   get array of electron spectra
;   compressed counts 20 highest energy
;   channels, 15 zones (coarse zone)
;----------------------------------------------------------------------------

      for c=0,19 do begin
         for z=0,14 do begin
            ebyte=inst(14+c*15+z) ;raw byte
            if ebyte eq 255 then ebyte=0
            euncomp=lepa_uc(ebyte) ;uncompressed value

	    e_cps(i,c,z) = tc1 * float(euncomp)
         endfor 
      endfor

;----------------------------------------------------------------------------
; get array of ion spectra
; compressed counts * 8, 29 highest
; energy channels, zones (0..7)
;----------------------------------------------------------------------------

      for c=0,28 do begin
         for z=0,7 do begin
            pbyte=inst(314+c*8+z) ;raw byte
            if pbyte eq 255 then pbyte=0
            puncomp=lepa_uc(pbyte) ;uncompressed value

	    p_cps(i,c,z) = tc8 * float(puncomp)
         endfor 
      endfor

;----------------------------------------------------------------------------
; get array of electron spectra near 90 deg pitch angle
; compressed counts, 20 highest energy channels,
; average of zones 6-9 in 90 degree snapshot
;----------------------------------------------------------------------------

      for c=0,19 do begin
         ebyte=inst(546+c)      ;raw bytes
         if ebyte eq 255 then ebyte=0
         euncomp=lepa_uc(ebyte) ;uncompressed value

         e_cps_perp(i,c) = tc1 * float(euncomp)
      endfor

;----------------------------------------------------------------------------
; get array of ion spectra near 90 deg pitch angle
; cm compressed counts * 8 , 29 highest energy channels,
; cm average of zones 3-4 in 90 degree snapshot
;----------------------------------------------------------------------------

      for c=0,28 do begin
         pbyte=inst(566+c)      ;raw bytes
         if pbyte eq 255 then pbyte=0
         puncomp=lepa_uc(pbyte) ;uncompressed value

	 p_cps_perp(i,c) = tc8 * float(puncomp)
      endfor

   endfor

   print,'no of records=',i

   xut=xut(0:i)
   
   e_cps=e_cps(0:i,*,*)
   p_cps=p_cps(0:i,*,*)
   e_cps_perp=e_cps_perp(0:i,*)
   p_cps_perp=p_cps_perp(0:i,*)
   
   lc_arr=lc_arr(0:i)
   mo_arr=mo_arr(0:i)
   an_arr=an_arr(0:i)
   tc_arr=tc_arr(0:i)
   
   rec_arr=rec_arr(0:i)
   re9_arr=re9_arr(0:i)
   ric_arr=ric_arr(0:i)
   ri9_arr=ri9_arr(0:i)
   
;----------------------------------------------------------------------------
;write data arrays to binary file for speed
;----------------------------------------------------------------------------

   fln1='/net/hydra/data/CRRES/LEPA'+'/'+string(orbit,format="(i8.8)")
   openw,2,fln1+'.lsdbin'
   writeu,2,iorbit,iday,iyear,i+1
   writeu,2, xut,                                 $
             e_cps,p_cps,e_cps_perp,p_cps_perp,   $
             lc_arr,mo_arr,an_arr, tc_arr,      $
             rec_arr,re9_arr,ric_arr,ri9_arr,tc_arr
   close,2
   ;spawn, 'gzip -9 '+fln1+'.lsdbin'

end

;****************************************************************************
;*  NAME:        lepa_uc
;*
;*  DESCRIPTION: Function to uncompress data
;*
;*  INPUT:       byte to be uncompressed
;*
;*  OUTPUT:      uncompressed value
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;****************************************************************************

function lepa_uc,in
   inbyte=byte(in)
   uc_value=(((inbyte AND 31b)+32b)*(2^(inbyte/32b)))-32b
   return,uc_value
end
