;******************************************************************************
;* file p_crres_lepa_spectra.pro
;*
;* This file is a template for a plot routine which can be used by PAPCO.
;* It includes all the periphery needed. This routine makes use of a
;* color spectrogram routine that is supplied with PAPCO:
;* papco_XX/papco_lib/papco_plot_colorspec.pro. This routine should
;* meet all the requirements for the display of 3-D data (tyhe third
;* dimension being color). This routine also adheres to the PAPCO color scheme.
;* You of course can use your own color plot rotuine if desired!
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_crres_lepa_spectra, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*      This procedure plots LEPA spectra. Data that are passed through the 
;*      common blocks are:
;*      xut--------------------> time in seconds from the beggining of the day
;*      e_cps, p_cps-----------> elec/ion counts
;*      e_cps_perp, p_cps_perp-> elec/ion counts for the 90 degree snapshot
;*      rec, re9, 
;*      ric, ri9---------------> background factors
;*      lc, an, 
;*      mo, tc-----------------> arrays for the status of LEPA
;*      fe0,feA,fi,
;*      be0,beA,bi,
;*      gel,geh,gil,gih,
;*      bi9,bea9,be09----------> calibration factors from survay.cal
;*
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    type(0)=0 --> Electrons
;*              type(0)=1 --> Ions
;*              type(1)=0 --> CPS
;*              type(1)=1 --> CPS-bkg
;*              type(1)=2 --> (CPS-bkg)/f
;*              type(1)=3 --> Diff. Number Flux
;*              type(1)=4 --> Diff. Energy Flux
;*              type(2)=0 --> Filter (only for type(1)=2,3,4)
;*              type(2)=1 --> Non-Filter
;*              type(3)=0 --> Perp
;*              type(3)=1 --> LC-Earthwards
;*              type(3)=2 --> LC-Equatorwards
;*              type(3)=3 --> LC-Average
;*              type(3)=4 --> LC-Difference
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_plot_type, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* COMMENTS:
;*       The processing part of this module is based on the pascal
;*       survey.pas written by Kevin Cairns at AFGL, the only main
;*       difference being that the structure of this program is not
;*       as criptic (nightmare to decode but also briliantly done for 
;*       PC's) as Kevin's pascal code.
;*
;* MODIFICATION HISTORY:       
;*     written december 1997, Christopher Mouikis
;******************************************************************************
pro p_crres_lepa_spectra, panel, type, $
                          OUTPUT=OUTPUT, $
                          OVERPLOT=OVERPLOT, $
                          PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                          SUBTABLE=SUBTABLE, $
                          _EXTRA=extra_par

; The following common blocks are needed for PAPCO:

   common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
   common yscale, yscl			;man/auto yscaling
   common zscale, zscl			;man/auto zscaling
   common shift_label, down             ;common for x-axis label shifting
   common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse
   ;--- lepa common blocks ------------------------------------------------
   common ephem, xutepf,xmlt,xilat,xlshel,xrad,ndat
   common lepa_time, xut
   common counts, e_cps, p_cps, e_cps_perp, p_cps_perp
   common bkg_factors, rec, re9, ric, ri9
   common special_arrays, lc, an, mo, tc
   common calibration, fe0,feA,fi,be0,beA,bi,gel,geh,gil,gih,bi9,bea9,be09
   ;--- lepa common blocks ------------------------------------------------

   papco_Set_SubTable, SUBTABLE         ;sets the papco color table index
   panelset,panel			;sets the panel position viewport

; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.

   ytickno=50*(!p.position(3)-!p.position(1))
   if ytickno gt 10 then ytickno=10

;   xtickno=6

    linlogbit=1
;***************************************************************************
;* START PROCESSING
;*   
;* ELECTRONS   
;***************************************************************************
   
   
   if type(0) eq 0 then begin   ; ELECTRONS
     
     raws=size(xut)
     it=raws(1)                 ; NUMBER OF TIME SLICES
     en_ch=20                   ; NUMBER OF ENERGY CHANNELS
     Erange=fltarr(en_ch+1)     ; ENERGY RANGE ARRAY
     ymat=fltarr(it,en_ch)      ; PLOTTING ARRAY
     yarr=fltarr(en_ch,2)
     
     ml=byte(mo and 128) / 128b ; ml:   0 FOR NORTH 1 FOR SOUTH ml
     ls=byte(mo and 64) / 64b   ; ls:   LOOK DIRECTION
     hv=byte(mo and 48) / 48b   ; hv:   0 FOR LOW VOLT. 1 FOR HIGH VOLT.
     mode=byte(mo and 15)       ; mode: OPERATING MODES, MAINLY 0 AND 10
     
     ; I have to make sure that the voltage does not change during an orbit
     ; and that the last value of hv always represents the voltage of the
     ; orbit
     if hv(it-1) eq 0 then begin ; ENERGY RANGE FOR LOW AND HIGH VOLTAGE
       for n=80,0,-4 do Erange((80-n)/4)=19.68*0.9349^n
     endif else begin
       for n=80,0,-4 do Erange((80-n)/4)=28.47*0.9343^n
     endelse
     
     
;---------------------------------------------------------------------------
;--- CPS  and (CPS-bkg)
;---------------------------------------------------------------------------
        
     
     if (type(1) eq 0) or (type(1) eq 1) then begin ; CPS or CPS-bkg
       
       if type(3) eq 0 then begin ; PERP SNAPSHOT --------------------------
         bkg_flag=0.
         if type(1) eq 1 then bkg_flag=1.
         for i=0,it-1 do begin  ; OVER ALL TIME
           if (mode(i) eq 10) then bkg_counts=bkg_flag*(re9(i)*bea9) $
           else $
             bkg_counts=bkg_flag*(re9(i)*be09)
           for e=0,en_ch-1 do begin ; OVER ALL ENERGIES
             ymat(i,en_ch-1-e)=(e_cps_perp(i,e)-bkg_counts) > 0
           endfor
         endfor
         time=xut
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='Counts/Sec'
         uytitle='Energy (keV)'
         if type(1) eq 0 then $
           utitle='LEPA Perp!Ce CPS' $
         else $
           utitle='LEPA Perp!Ce CPS-bkg'
         goto, next
       endif
       
       if type(3) eq 1 then begin ; LOSS-CONE EARTH-WARDS -------------------
         
         j=-1
         bkg_flag=0.
         xuter=fltarr(it)
         if type(1) eq 1 then bkg_flag=1.
         for i=0, it-1 do begin ; OVER ALL TIME
           if (an(i) lt 255) then begin
             if (mode(i) eq 10) then begin ; SELECT ZONE FOR MODE 10
               if (ls(i) ne 0) then $ ; FOR NORTH LOOK DIRECTION
                 zone=((153-an(i))/16) $
               else $                      ; FOR SOUTH LOOK DIRECTION
                 zone=((an(i)-26)/16)
               if (zone lt 0) then zone=0
               if (zone gt 7) then zone=7
               if (ml(i) eq ls(i)) then begin ; EARTHWARDS
                 j=j+1
                 xuter(j)=xut(i)
                 for e=0,en_ch-1 do $ ; OVER ALL ENERGIES
                   ymat(j,en_ch-1-e)=(e_cps(i,e,zone) $
                                      -bkg_flag*(rec(i)*beA(zone)))>0
               endif
             endif else begin   ; SELECT ZONE FOR MODE 0 (NON-10)
               if (ls(i) eq 0) then $  ; FOR SOUTH LOOK DIRECTION
                 zone=((153-an(i))/8) $
               else $                  ; FOR NORTH LOOK DIRECTION
                 zone=((an(i)-26)/8)
               if (zone lt 0) then zone=0
               if (zone gt 14) then zone=14
               if (ml(i) eq ls(i)) then begin ; EARTHWARDS
                 j=j+1
                 xuter(j)=xut(i)
                 for e=0,en_ch-1 do $ ; OVER ALL ENERGIES
                   ymat(j,en_ch-1-e)=(e_cps(i,e,zone) $
                                      -bkg_flag*(rec(i)*be0(zone)))>0
               endif
             endelse
           endif
         endfor
         time=xuter(0:j)
         ymat=ymat(0:j,*)
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='Counts/Sec'
         uytitle='Energy (keV)'
         if type(1) eq 0 then $
           utitle='LEPA LCErth!Ce CPS' $
         else $
           utitle='LEPA LCErth!Ce CPS-Bkg'
         goto, next
         
       endif
       
       if type(3) eq 2 then begin ; LOSS-CONE EQUATOR-WARDS -----------------
         
         j=-1
         bkg_flag=0.
         xuteq=fltarr(it)
         if type(1) eq 1 then bkg_flag=1.
         for i=0, it-1 do begin ; OVER ALL TIME
           if (an(i) lt 255) then begin
             if (mode(i) eq 10) then begin ; SELECT ZONE FOR MODE 10
               if (ls(i) ne 0) then $      ; FOR NORTH LOOK DIRECTION
                 zone=((153-an(i))/16) $
               else $                      ; FOR SOUTH LOOK DIRECTION
                 zone=((an(i)-26)/16)
               if (zone lt 0) then zone=0
               if (zone gt 7) then zone=7
               if (ml(i) ne ls(i)) then begin ; EQUATORWARDS
                 j=j+1
                 xuteq(j)=xut(i)
                 for e=0,en_ch-1 do $ ; OVER ALL ENERGIES
                   ymat(j,en_ch-1-e)=(e_cps(i,e,zone) $
                                      -bkg_flag*(rec(i)*beA(zone)))>0
               endif  
             endif else begin   ; SELECT ZONE FOR MODE 0 (NON-10)
               if (ls(i) eq 0) then $ ; FOR SOUTH LOOK DIRECTION
                 zone=((153-an(i))/8) $
               else $                 ; FOR NORTH LOOK DIRECTION
                 zone=((an(i)-26)/8)
               if (zone lt 0) then zone=0
               if (zone gt 14) then zone=14
               if (ml(i) ne ls(i)) then begin ; EQUATORWARDS
                 j=j+1
                 xuteq(j)=xut(i)
                 for e=0,en_ch-1 do $ ; OVER ALL ENERGIES
                   ymat(j,en_ch-1-e)=(e_cps(i,e,zone) $
                                      -bkg_flag*(rec(i)*be0(zone)))>0
               endif
             endelse
           endif
         endfor
         time=xuteq(0:j)
         ymat=ymat(0:j,*)
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='Counts/Sec'
         uytitle='Energy (keV)'
         if type(1) eq 0 then $
           utitle='LEPA LCEquat!Ce CPS' $
         else $
           utitle='LEPA LCEquat!Ce CPS-bkg'
         goto, next
         
       endif
       
       if type(3) eq 3 then begin ; LOSS-CONE AVERAGE ---------------------
         
         bkg_flag=0.
         if type(1) eq 1 then bkg_flag=1.
         for i=0, it-1 do begin ; OVER ALL TIME
           if (an(i) lt 255) then begin
             if (mode(i) eq 10) then begin ; SELECT ZONE FOR MODE 10
               if (ls(i) ne 0) then $ ; FOR NORTH LOOK DIRECTION
                 zone=((153-an(i))/16) $
               else $                 ; FOR SOUTH LOOK DIRECTION
                 zone=((an(i)-26)/16)
               if (zone lt 0) then zone=0
               if (zone gt 7) then zone=7
               for e=0,en_ch-1 do $ ; OVER ALL ENERGIES
                 ymat(i,en_ch-1-e)=(e_cps(i,e,zone) $
                                    -bkg_flag*(rec(i)*beA(zone)))>0
             endif else begin ; SELECT ZONE FOR MODE 0 (NON-10)
               if (ls(i) eq 0) then $ ; FOR SOUTH LOOK DIRECTION
                 zone=((153-an(i))/8) $
               else $                 ; FOR NORTH LOOK DIRECTION
                 zone=((an(i)-26)/8)
               if (zone lt 0) then zone=0
               if (zone gt 14) then zone=14
               for e=0,en_ch-1 do $ ; OVER ALL ENERGIES
                 ymat(i,en_ch-1-e)=(e_cps(i,e,zone) $
                                    -bkg_flag*(rec(i)*be0(zone)))>0
             endelse
           endif
         endfor
         time=xut
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='Counts/Sec'
         uytitle='Energy (keV)'
         if type(1) eq 0 then $
           utitle='LEPA LsCn!Ce CPS' $
         else $
           utitle='LEPA LsCn!Ce CPS-bkg'
         goto, next         
         
       endif
       
     endif
     
;---------------------------------------------------------------------------
;--- Diff. Num. Flux and Diff. En. Flux
;---------------------------------------------------------------------------
        
     if (type(1) eq 3) or (type(1) eq 4) then begin ; Diff. Num. Flux or 
                                                    ; Diff En. Flux
       
       elv=0.
       ehv=0.
       diff_en_flag=0.
       diff_en_factor=1.
       errlimit_flag=0
       tc1=(tc/500.)
       if type(1) eq 4 then diff_en_flag=1.
       if type(2) eq 0 then errlimit_flag=1.
       
       if type(3) eq 0 then begin ; PERP SNAPSHOT --------------------------
         ;errlimit=tc1*re9*0.01
         for i=0,it-1 do begin  ; over all time
           ;if errlimit(i) lt tc1(i) then errlimit(i)=tc1(i)
           for e=0,en_ch-1 do begin ; over all energies
             if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
             if (hv(i) eq 0) then elv=1. else ehv=1.
             if (mode(i) eq 10) then begin
               errlimit=(re9(i)*bea9) > tc1(i)
               counts=((e_cps_perp(i,e)-(re9(i)*bea9))>0)
               ;if (counts lt errlimit(i)) and errlimit_flag then counts=0.
               if (counts lt errlimit) and errlimit_flag then counts=0.
               ymat(i,en_ch-1-e)=counts $
                 * (elv*gel(e)+ehv*geh(e))*diff_en_factor
             endif else begin
               errlimit=(re9(i)*be09) > tc1(i)
               counts=((e_cps_perp(i,e)-(re9(i)*be09))>0)
               ;if (counts lt errlimit(i)) and errlimit_flag then counts=0.
               if (counts lt errlimit) and errlimit_flag then counts=0.
               ymat(i,en_ch-1-e)=counts $
                 * (elv*gel(e)+ehv*geh(e))*diff_en_factor
             endelse
           endfor
         endfor
         time=xut
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='[Cm!E-2!Ns!E-1!Nsr!E-1!NkeV!E-1!n]'
         uytitle='Energy (keV)'
         if type(1) eq 3 then $
           ;utitle='LEPA Perp!Ce CDiff.# Flux' $
         utitle='LEPA !C e Perp' $
         else $
           utitle='LEPA Perp!Ce CDiff.En.Flux'
         goto, next
       endif
       
       if type(3) eq 1 then begin ; LOSS-CONE EARTH-WARDS -------------------
         j=-1
         xuter=fltarr(it)
         elv=0.
         ehv=0.
         ;errlimit=tc1*rec*0.01
         for i=0, it-1 do begin ; over all time
           ;if errlimit(i) lt tc1(i) then errlimit(i)=tc1(i)
           if (hv(i) eq 0) then elv=1. else ehv=1.           
           if (an(i) lt 255) then begin
             if (mode(i) eq 10) then begin ; select zone for mode 10
               if (ls(i) ne 0) then $
                 zone=((153-an(i))/16) $
               else $
                 zone=((an(i)-26)/16)
               if (zone lt 0) then zone=0
               if (zone gt 7) then zone=7
               if (ml(i) eq ls(i)) then begin
                 j=j+1
                 xuter(j)=xut(i)
                 for e=0,en_ch-1 do begin  ; over all energies
                   if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                   errlimit=(rec(i)*beA(zone)) > tc1(i)
                   counts=((e_cps(i,e,zone)-(rec(i)*beA(zone)))>0)*feA(zone)
                   if (counts lt errlimit) and errlimit_flag then counts=0.
                   ymat(j,en_ch-1-e)=counts $
                                      * (elv*gel(e)+ehv*geh(e)) $
                                      * diff_en_factor
                 endfor  
               endif
             endif else begin   ; select zone for mode 0 (non-10)
               if (ls(i) eq 0) then zone=((153-an(i))/8) $
               else $
                 zone=((an(i)-26)/8)
               if (zone lt 0) then zone=0
               if (zone gt 14) then zone=14
               if (ml(i) eq ls(i)) then begin
                 j=j+1
                 xuter(j)=xut(i)
                 for e=0,en_ch-1 do begin ; over all energies
                   if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                   errlimit=(rec(i)*be0(zone)) > tc1(i)
                   counts=((e_cps(i,e,zone)-(rec(i)*be0(zone)))>0)*fe0(zone)
                   if (counts lt errlimit) and errlimit_flag then counts=0.
                   ymat(j,en_ch-1-e)=counts $
                                      * (elv*gel(e)+ehv*geh(e)) $
                                      * diff_en_factor
                 endfor
               endif
             endelse
           endif
         endfor
         time=xuter(0:j)
         ymat=ymat(0:j,*)
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='[Cm!E-2!Ns!E-1!Nsr!E-1!NkeV!E-1!n]'
         uytitle='Energy (keV)'
         if type(1) eq 3 then $
           ;utitle='LEPA LCErth!Ce CDiff.# Flux' $
         utitle='LEPA!Ce LCErth' $
         else $
           utitle='LEPA LCErth!Ce CDiff.En.Flux'
         goto, next
       endif
       
       if type(3) eq 2 then begin ; LOSS-CONE EQUATOR-WARDS -----------------
         j=-1
         xuteq=fltarr(it)
         elv=0.
         ehv=0.
         ;errlimit=tc1*rec*0.01
         for i=0, it-1 do begin ; over all time
           ;if errlimit(i) lt tc1(i) then errlimit(i)=tc1(i)
           if (hv(i) eq 0) then elv=1. else ehv=1.           
           if (an(i) lt 255) then begin
             if (mode(i) eq 10) then begin
               if (ls(i) ne 0) then $
                 zone=((153-an(i))/16) $
               else $
                 zone=((an(i)-26)/16)
               if (zone lt 0) then zone=0
               if (zone gt 7) then zone=7
               if (ml(i) ne ls(i)) then begin
                 j=j+1
                 xuteq(j)=xut(i)
                 for e=0,en_ch-1 do begin ; over all energies
                   if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                   errlimit=(rec(i)*beA(zone)) > tc1(i)
                   counts=((e_cps(i,e,zone)-(rec(i)*beA(zone)))>0)*feA(zone)
                   if (counts lt errlimit) and errlimit_flag then counts=0.
                   ymat(j,en_ch-1-e)=counts $
                                      * (elv*gel(e)+ehv*geh(e)) $
                                      * diff_en_factor  
                 endfor  
               endif  
             endif else begin
               if (ls(i) eq 0) then $
                 zone=((153-an(i))/8) $
               else $
                 zone=((an(i)-26)/8)
               if (zone lt 0) then zone=0
               if (zone gt 14) then zone=14
               if (ml(i) ne ls(i)) then begin
                 j=j+1
                 xuteq(j)=xut(i)
                 for e=0,en_ch-1 do begin ; over all energies
                   if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                   errlimit=(rec(i)*be0(zone)) > tc1(i)
                   counts=((e_cps(i,e,zone)-(rec(i)*be0(zone)))>0)*fe0(zone)
                   if (counts lt errlimit) and errlimit_flag then counts=0.
                   ymat(j,en_ch-1-e)=counts $
                                       * (elv*gel(e)+ehv*geh(e)) $
                                       * diff_en_factor  
                 endfor  
               endif
             endelse
           endif
         endfor
         time=xuteq(0:j)
         ymat=ymat(0:j,*)
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='[Cm!E-2!Ns!E-1!Nsr!E-1!NkeV!E-1!n]'
         uytitle='Energy (keV)'
         if type(1) eq 3 then $
           ;utitle='LEPA LCEquat!Ce CDiff.# Flux' $
         utitle='LEPA!Ce LCEquat' $
         else $
           utitle='LEPA LCEquat!Ce CDiff.En.Flux'
         goto, next
       endif
       
       if type(3) eq 3 then begin ; LOSS-CONE AVERAGE ---------------------
         elv=0.
         ehv=0.
         ;errlimit=tc1*rec*0.01
         for i=0, it-1 do begin ; over all time
           ;if errlimit(i) lt tc1(i) then errlimit(i)=tc1(i)
           if (hv(i) eq 0) then elv=1. else ehv=1.
           if (an(i) lt 255) then begin
             if (mode(i) eq 10) then begin
               if (ls(i) ne 0) then $
                 zone=((153-an(i))/16) $
               else $
                 zone=((an(i)-26)/16)
               if (zone lt 0) then zone=0
               if (zone gt 7) then zone=7
               for e=0,en_ch-1 do begin ; over all energies
                 if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                 errlimit=(rec(i)*beA(zone)) > tc1(i)
                 counts=((e_cps(i,e,zone)-(rec(i)*beA(zone)))>0)*feA(zone)
                 if (counts lt errlimit) and errlimit_flag then counts=0.
                 ymat(i,en_ch-1-e)=counts $
                                    * (elv*gel(e)+ehv*geh(e)) $
                                    * diff_en_factor  
               endfor  
             endif else begin
               if (ls(i) eq 0) then $
                 zone=((153-an(i))/8) $
               else $
                 zone=((an(i)-26)/8)
               if (zone lt 0) then zone=0
               if (zone gt 14) then zone=14
               for e=0,en_ch-1 do begin ; over all energies
                 if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                 errlimit=(rec(i)*be0(zone)) > tc1(i)
                 counts=((e_cps(i,e,zone)-(rec(i)*be0(zone)))>0)*fe0(zone)
                 if (counts lt errlimit) and errlimit_flag then counts=0.
                 ymat(i,en_ch-1-e)=counts $
                                    * (elv*gel(e)+ehv*geh(e)) $
                                    * diff_en_factor    
               endfor  
             endelse
           endif
         endfor
         time=xut
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='[Cm!E-2!Ns!E-1!Nsr!E-1!NkeV!E-1!n]'
         uytitle='Energy (keV)'
         if type(1) eq 3 then $
           utitle='LEPA LsCn!Ce CDiff.# Flux' $
         else $
           utitle='LEPA LsCn!Ce CDiff.En.Flux'
         
         goto, next         
       endif
       
       if type(3) eq 4 then begin ; LOSS-CONE DIFFERENCE ------------------
         j=-1
         xutdiff=fltarr(it)
         ymateq=fltarr(en_ch)
         ymater=fltarr(en_ch)
         ymat=fltarr(it,en_ch)
         elv=0.
         ehv=0.
         ;errlimit=tc1*rec*0.01
         
         event=-1
         for i=0, it-1 do begin ; over all time
           
           ;if errlimit(i) lt tc1(i) then errlimit(i)=tc1(i)
           if (hv(i) eq 0) then elv=1. else ehv=1.           
           if (an(i) lt 255) then begin
             
             if (mode(i) eq 10) then begin ; select zone for mode 10
               if (ls(i) ne 0) then $
                 zone=((153-an(i))/16) $
               else $
                 zone=((an(i)-26)/16)
               if (zone lt 0) then zone=0
               if (zone gt 7) then zone=7
               
               if (ml(i) eq ls(i)) then begin
                 event=i
                 for e=0,en_ch-1 do begin ; over all energies
                   if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                   errlimit=(rec(i)*beA(zone)) > tc1(i)
                   counts=((e_cps(i,e,zone)-(rec(i)*beA(zone)))>0)*feA(zone)
                   if (counts lt errlimit) and errlimit_flag then counts=0.
                   ymater(en_ch-1-e)=counts $
                     * (elv*gel(e)+ehv*geh(e)) $
                     * diff_en_factor
                 endfor  
               endif else begin
                 if i eq (event+1) and i gt 0 then begin
                   j=j+1
                   xutdiff(j)=(xut(i)+xut(i-1))/2.
                   for e=0,en_ch-1 do begin ; over all energies
                     if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                     errlimit=(rec(i)*beA(zone)) > tc1(i)
                     counts=((e_cps(i,e,zone)-(rec(i)*beA(zone)))>0)*feA(zone)
                     if (counts lt errlimit) and errlimit_flag then counts=0.
                     ymateq(en_ch-1-e)=counts $
                       * (elv*gel(e)+ehv*geh(e)) $
                       * diff_en_factor
                   endfor
                   ymat(j,0:en_ch-1)=ymateq(0:en_ch-1)-ymater(0:en_ch-1)
                 endif
               endelse               
               
             endif else begin   ; select zone for mode 0 (non-10)
               if (ls(i) eq 0) then $
                 zone=((153-an(i))/8) $
               else $
                 zone=((an(i)-26)/8)
               if (zone lt 0) then zone=0
               if (zone gt 14) then zone=14               
               
               if (ml(i) eq ls(i)) then begin
                 event=i
                 for e=0,en_ch-1 do begin ; over all energies
                   if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                   errlimit=(rec(i)*be0(zone)) > tc1(i)
                   counts=((e_cps(i,e,zone)-(rec(i)*be0(zone)))>0)*fe0(zone)
                   if (counts lt errlimit) and errlimit_flag then counts=0.
                   ymater(en_ch-1-e)=counts $
                     * (elv*gel(e)+ehv*geh(e)) $
                     * diff_en_factor
                 endfor  
               endif else begin
                 if i eq (event+1) and i gt 0 then begin
                   j=j+1
                   xutdiff(j)=(xut(i)+xut(i-1))/2.
                   for e=0,en_ch-1 do begin ; over all energies
                     if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                     errlimit=(rec(i)*be0(zone)) > tc1(i)
                     counts=((e_cps(i,e,zone)-(rec(i)*be0(zone)))>0)*fe0(zone)
                     if (counts lt errlimit) and errlimit_flag then counts=0.
                     ymateq(en_ch-1-e)=counts $
                       * (elv*gel(e)+ehv*geh(e)) $
                       * diff_en_factor
                   endfor
                   ymat(j,0:en_ch-1)=ymateq(0:en_ch-1)-ymater(0:en_ch-1)
                 endif
               endelse
             endelse
           endif
         endfor
         
         time=xutdiff(0:j)
         ymat=ymat(0:j,*)
         ymat=100.*ymat/(0.8 * max(abs(ymat)))
         ymat=ymat+100.
         linlogbit=0
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='[Cm!E-2!Ns!E-1!Nsr!E-1!NkeV!E-1!n]'
         uytitle='Energy (keV)'
         if type(1) eq 3 then $
           utitle='LEPA LCDiff!Ce CDiff.# Flux' $
         else $
           utitle='LEPA LCDiff!Ce CDiff.En.Flux'         
         goto, next  
       endif
       
     endif
     
   endif     
   
;***************************************************************************   
; IONS   
;***************************************************************************   
   
   if type(0) eq 1 then begin   ; IONS
     
     raws=size(p_cps_perp)
     it=raws(1)                 ; NUMBER OF TIME SLICES
     en_ch=29                   ; NUMBER OF ENERGY CHANNELS
     Erange=fltarr(en_ch+1)     ; ENERGY RANGE ARRAY
     ymat=fltarr(it,en_ch)      ; PLOTTING ARRAY
     yarr=fltarr(en_ch,2)
     
     ml=byte(mo and 128) / 128b ; ml:   0 FOR NORTH 1 FOR SOUTH ml
     ls=byte(mo and 64) / 64b   ; ls:   LOOK DIRECTION
     hv=byte(mo and 48) / 48b   ; hv:   0 FOR LOW VOLT. 1 FOR HIGH VOLT.
     mode=byte(mo and 15)       ; mode: OPERATING MODES, MAINLY 0 AND 10
     
     if hv(it-1) eq 0 then begin ; ENERGY RANGE FOR LOW AND HIGH VOLTAGE
       for n=116,0,-4 do Erange((116-n)/4)=19.68*0.9349^n
     endif else begin
       for n=116,0,-4 do Erange((116-n)/4)=28.47*0.9343^n
     endelse
     
     
;---------------------------------------------------------------------------
;--- CPS  and (CPS-bkg)
;---------------------------------------------------------------------------
        
     
     if (type(1) eq 0) or (type(1) eq 1) then begin ; CPS or CPS-bkg
       
       if type(3) eq 0 then begin ; PERP SNAPSHOT --------------------------
         bkg_flag=0.
         if type(1) eq 1 then bkg_flag=1.
         for i=0,it-1 do begin  ; OVER ALL TIME
           bkg_counts=bkg_flag*(ri9(i)*bi9)
           for e=0,en_ch-1 do begin ; OVER ALL ENERGIES
             ymat(i,en_ch-1-e)=(p_cps_perp(i,e)-bkg_counts) > 0
           endfor
         endfor
         time=xut
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='Counts/Sec'
         uytitle='Energy (keV)'
         if type(1) eq 0 then $
           utitle='LEPA Perp!Ci CPS' $
         else $
           utitle='LEPA Perp!Ci CPS-bkg'
         goto, next
       endif
       
       if type(3) eq 1 then begin ; LOSS-CONE EARTH-WARDS -------------------
         
         j=-1
         bkg_flag=0.
         xuter=fltarr(it)
         if type(1) eq 1 then bkg_flag=1.
         for i=0, it-1 do begin ; OVER ALL TIME
           if (an(i) lt 255) then begin
             if (ls(i) ne 0) then $ ; FOR NORTH LOOK DIRECTION
               zone=((153-an(i))/16) $
             else $             ; FOR SOUTH LOOK DIRECTION
               zone=((an(i)-26)/16)
             if (zone lt 0) then zone=0
             if (zone gt 7) then zone=7
             if (ml(i) eq ls(i)) then begin ; EARTHWARDS
               j=j+1
               xuter(j)=xut(i)
               for e=0,en_ch-1 do $ ; OVER ALL ENERGIES
                 ymat(j,en_ch-1-e)=(p_cps(i,e,zone) $
                                    -bkg_flag*(ric(i)*bi(zone)))>0
             endif
           endif
         endfor
         time=xuter(0:j)
         ymat=ymat(0:j,*)
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='Counts/Sec'
         uytitle='Energy (keV)'
         if type(1) eq 0 then $
           utitle='LEPA LCErth!Ci CPS' $
         else $
           utitle='LEPA LCErth!Ci CPS-Bkg'
         goto, next
         
       endif
       
       if type(3) eq 2 then begin ; LOSS-CONE EQUATOR-WARDS -----------------
         
         j=-1
         bkg_flag=0.
         xuteq=fltarr(it)
         if type(1) eq 1 then bkg_flag=1.
         for i=0, it-1 do begin ; OVER ALL TIME
           if (an(i) lt 255) then begin
             if (ls(i) ne 0) then $ ; FOR NORTH LOOK DIRECTION
               zone=((153-an(i))/16) $
             else $             ; FOR SOUTH LOOK DIRECTION
               zone=((an(i)-26)/16)
             if (zone lt 0) then zone=0
             if (zone gt 7) then zone=7
             if (ml(i) ne ls(i)) then begin ; EQUATORWARDS
               j=j+1
               xuteq(j)=xut(i)
               for e=0,en_ch-1 do $ ; OVER ALL ENERGIES
                 ymat(j,en_ch-1-e)=(p_cps(i,e,zone) $
                                    -bkg_flag*(ric(i)*bi(zone)))>0
             endif  
           endif
         endfor
         time=xuteq(0:j)
         ymat=ymat(0:j,*)
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='Counts/Sec'
         uytitle='Energy (keV)'
         if type(1) eq 0 then $
           utitle='LEPA LCEquat!Ci CPS' $
         else $
           utitle='LEPA LCEquat!Ci CPS-bkg'
         goto, next
         
       endif
       
       if type(3) eq 3 then begin ; LOSS-CONE AVERAGE ---------------------
         
         bkg_flag=0.
         if type(1) eq 1 then bkg_flag=1.
         for i=0, it-1 do begin ; OVER ALL TIME
           if (an(i) lt 255) then begin
             if (ls(i) ne 0) then $ ; FOR NORTH LOOK DIRECTION
               zone=((153-an(i))/16) $
             else $             ; FOR SOUTH LOOK DIRECTION
               zone=((an(i)-26)/16)
             if (zone lt 0) then zone=0
             if (zone gt 7) then zone=7
             for e=0,en_ch-1 do $ ; OVER ALL ENERGIES
               ymat(i,en_ch-1-e)=(p_cps(i,e,zone) $
                                  -bkg_flag*(ric(i)*bi(zone)))>0
           endif
         endfor
         time=xut
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='Counts/Sec'
         uytitle='Energy (keV)'
         if type(1) eq 0 then $
           utitle='LEPA LsCn!Ci CPS' $
         else $
           utitle='LEPA LsCn!Ci CPS-bkg'
         goto, next         
         
       endif
       
     endif
     
;---------------------------------------------------------------------------
;--- Diff. Num. Flux and Diff. En. Flux
;---------------------------------------------------------------------------
        
     if (type(1) eq 3) or (type(1) eq 4) then begin ; Diff. Num. Flux or 
                                                    ; Diff En. Flux
       
       ilv=0.
       ihv=0.
       diff_en_flag=0.
       diff_en_factor=1.
       errlimit_flag=0
       tc1=(tc/500.)*0.125
       if type(1) eq 4 then diff_en_flag=1.
       if type(2) eq 0 then errlimit_flag=1.
       
       if type(3) eq 0 then begin ; PERP SNAPSHOT --------------------------
         ;errlimit=tc1*ri9*0.01
         for i=0,it-1 do begin  ; over all time
           ;if errlimit(i) lt tc1(i) then errlimit(i)=tc1(i)
           for e=0,en_ch-1 do begin ; over all energies
             if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
             if (hv(i) eq 0) then ilv=1. else ihv=1.
             errlimit=ri9(i)*bi9*0.01 > tc1(i)
             counts=((p_cps_perp(i,e)-(ri9(i)*bi9))>0)
             if (counts lt errlimit) and errlimit_flag then counts=0.
             ymat(i,en_ch-1-e)=counts $
               * (ilv*gil(e)+ihv*gih(e))*diff_en_factor
           endfor
         endfor
         time=xut
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='[Cm!E-2!Ns!E-1!Nsr!E-1!NkeV!E-1!n]'
         uytitle='Energy (keV)'
         if type(1) eq 3 then $
           ;utitle='LEPA Perp!Ci Diff.# Flux' $
         utitle='LEPA!Ci Perp' $
         else $
           utitle='LEPA Perp!Ci Diff.En.Flux'
         goto, next
       endif
       
       if type(3) eq 1 then begin ; LOSS-CONE EARTH-WARDS -------------------
         j=-1
         xuter=fltarr(it)
         ilv=0.
         ihv=0.
         ;errlimit=tc1*ric*0.01
         for i=0, it-1 do begin ; over all time
           ;if errlimit(i) lt tc1(i) then errlimit(i)=tc1(i)
           if (hv(i) eq 0) then ilv=1. else ihv=1.           
           if (an(i) lt 255) then begin
             if (ls(i) ne 0) then $
               zone=((153-an(i))/16) $
             else $
               zone=((an(i)-26)/16)
             if (zone lt 0) then zone=0
             if (zone gt 7) then zone=7
             if (ml(i) eq ls(i)) then begin
               j=j+1
               xuter(j)=xut(i)
               for e=0,en_ch-1 do begin ; over all energies
                 if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                 errlimit=ric(i)*bi(zone)> tc1(i)
                 counts=((p_cps(i,e,zone)-(ric(i)*bi(zone)))>0)*fi(zone)
                 if (counts lt errlimit) and errlimit_flag then counts=0.
                 ymat(j,en_ch-1-e)=counts $
                   * (ilv*gil(e)+ihv*gih(e)) $
                   * diff_en_factor
               endfor  
             endif
           endif
         endfor
         time=xuter(0:j)
         ymat=ymat(0:j,*)
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='[Cm!E-2!Ns!E-1!Nsr!E-1!NkeV!E-1!n]'
         uytitle='Energy (keV)'
         if type(1) eq 3 then $
           utitle='LEPA !Ci LCErth' $
         else $
           utitle='LEPA LCErth!Ci CDiff.En.Flux'
         goto, next
       endif
       
       if type(3) eq 2 then begin ; LOSS-CONE EQUATOR-WARDS -----------------
         l=-1
         xuteq=fltarr(it)
         ilv=0.
         ihv=0.
         ;errlimit=tc1*ric*0.01
         for i=0, it-1 do begin ; over all time
           ;if errlimit(i) lt tc1(i) then errlimit(i)=tc1(i)
           if (hv(i) eq 0) then ilv=1. else ihv=1.           
           if (an(i) lt 255) then begin
             if (ls(i) ne 0) then $
               zone=((153-an(i))/16) $
             else $
               zone=((an(i)-26)/16)
             if (zone lt 0) then zone=0
             if (zone gt 7) then zone=7
             if (ml(i) ne ls(i)) then begin
               l=l+1
               xuteq(l)=xut(i)
               for e=0,en_ch-1 do begin ; over all energies
                 if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                 errlimit=ric(i)*bi(zone) > tc1(i)
                 counts=((p_cps(i,e,zone)-(ric(i)*bi(zone)))>0)*fi(zone)
                 if (counts lt errlimit) and errlimit_flag then counts=0.
                 ymat(l,en_ch-1-e)=counts $
                   * (ilv*gil(e)+ihv*gih(e)) $
                   * diff_en_factor
               endfor  
             endif  
           endif
         endfor
         time=xuteq(0:l)
         ymat=ymat(0:l,*)
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='[Cm!E-2!Ns!E-1!Nsr!E-1!NkeV!E-1!n]'
         uytitle='Energy (keV)'
         if type(1) eq 3 then $
           utitle='LEPA !Ci LCEquat' $
         else $
           utitle='LEPA LCEquat!Ci CDiff.En.Flux'
         goto, next
       endif
       
       if type(3) eq 3 then begin ; LOSS-CONE AVERAGE ---------------------
         ilv=0.
         ihv=0.
         ;errlimit=tc1*ric*0.01
         for i=0, it-1 do begin ; over all time
           ;if errlimit(i) lt tc1(i) then errlimit(i)=tc1(i)
           if (hv(i) eq 0) then ilv=1. else ihv=1.
           if (an(i) lt 255) then begin
             if (ls(i) ne 0) then $
               zone=((153-an(i))/16) $
             else $
               zone=((an(i)-26)/16)
             if (zone lt 0) then zone=0
             if (zone gt 7) then zone=7
             for e=0,en_ch-1 do begin ; over all energies
               if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
               counts=((p_cps(i,e,zone)-(ric(i)*bi(zone)))>0)*fi(zone)
               if (counts lt errlimit) and errlimit_flag then counts=0.
               ymat(i,en_ch-1-e)=counts $
                 * (ilv*gil(e)+ihv*gih(e)) $
                 * diff_en_factor
             endfor  
           endif
         endfor
         time=xut
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='[Cm!E-2!Ns!E-1!Nsr!E-1!NkeV!E-1!n]'
         uytitle='Energy (keV)'
         if type(1) eq 3 then $
           utitle='LEPA !Ci LsCn' $
         else $
           utitle='LEPA LsCn!Ci CDiff.En.Flux'
         
         goto, next         
       endif
       
       if type(3) eq 4 then begin ; LOSS-CONE DIFFERENCE ------------------
         j=-1
         xutdiff=fltarr(it)
         ymateq=fltarr(en_ch)
         ymater=fltarr(en_ch)
         ymat=fltarr(it,en_ch)
         ilv=0.
         ihv=0.
         ;errlimit=tc1*ric*0.01
         
         event=-1
         for i=0, it-1 do begin ; over all time
           ;if errlimit(i) lt tc1(i) then errlimit(i)=tc1(i)
           if (hv(i) eq 0) then ilv=1. else ihv=1.
           if (an(i) lt 255) then begin
             if (ls(i) ne 0) then $
               zone=((153-an(i))/16) $
             else $
               zone=((an(i)-26)/16)
             if (zone lt 0) then zone=0
             if (zone gt 7) then zone=7
             if ml(i) eq ls(i) then begin
               event=i
               for e=0,en_ch-1 do begin ; over all energies
                 if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                 errlimit=ric(i)*bi(zone)>tc1(i)
                 counts=((p_cps(i,e,zone)-(ric(i)*bi(zone)))>0)*fi(zone)
                 if (counts lt errlimit) and errlimit_flag then counts=0.
                 ymater(en_ch-1-e)=counts $
                   * (ilv*gil(e)+ihv*gih(e)) $
                   * diff_en_factor
               endfor               
             endif else begin
               if i eq (event+1) and i gt 0 then begin
                 j=j+1
                 xutdiff(j)=(xut(i)+xut(i-1))/2.
                 for e=0,en_ch-1 do begin ; over all energies
                   if diff_en_flag then diff_en_factor=Erange(en_ch-1-e)
                   errlimit=ric(i)*bi(zone)>tc1(i)
                   counts=((p_cps(i,e,zone)-(ric(i)*bi(zone)))>0)*fi(zone)
                   if (counts lt errlimit) and errlimit_flag then counts=0.
                   ymateq(en_ch-1-e)=counts $
                     * (ilv*gil(e)+ihv*gih(e)) $
                     * diff_en_factor
                 endfor
                 ymat(j,0:en_ch-1)=ymateq(0:en_ch-1)-ymater(0:en_ch-1)
               endif
             endelse
           endif
         endfor
         
         time=xutdiff(0:j)
         ymat=ymat(0:j,*)
         ;ymat=100.*ymat/1.e7
         ymat=100.*ymat/(0.8 * max(abs(ymat)))
         ymat=ymat+100.
         linlogbit=0
         yarr(*,0)=Erange(0:en_ch-1)
         yarr(*,1)=Erange(1:en_ch)
         uztit='[Cm!E-2!Ns!E-1!Nsr!E-1!NkeV!E-1!n]'
         uytitle='Energy (keV)'
         if type(1) eq 3 then $
           utitle='LEPA !Ci LCDiff' $
         else $
           utitle='LEPA LCDiff!Ci CDiff.En.Flux'         
         goto, next  
       endif
                
       goto, next  
     
     endif
          
     
   endif   
      
   next:
   
;***************************************************************************   
; END OF PROCESSING   
;***************************************************************************
   
; the plot-calls normally use the arrays time for the time axis and yarr (zmat 
; for color plots) for the y or z axis.
; depending on your plot options different data may be selected and different
; axis-labeling may be needed. This is normally done using a case statement.

; in this example we have a single test array, so no case statement is needed.

; set the begin end end times. PAPCO convention is to use running secconds
; starting with the no of seconds since the beginning of the start
; day.
   
   xut1=mjdt_start.t                       
   xut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t

;   if xut1 lt xut(0) then begin
;     xut1=86400l+mjdt_start.t
;     xut2=long(mjdt_end.mjd-mjdt_start.mjd+1.)*86400l+mjdt_end.t
;   endif

   yst=yarr(0,0)
   yen=yarr(en_ch-1,1)

; restrict data to actual time range requested - this makes for faster zooming

   index=where((time ge xut1) and (time le xut2),count)
   if count ne 0 then begin
      ymat=ymat(index,*)
      time=time(index)
   endif
   
; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data

; the procedure papco_autorange (in papco_XX/papco_lib/autoscaling is
; optional and sets the zlimits automatically for the data, attempting
; to use the full dynamic range of the color table. you may need to
; supply your own procedure for your own data.

; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling
   
   if (zscl(panel(0),0) eq 1) then begin
     zmin=zscl(panel(0),1)
     zmax=zscl(panel(0),2)
   endif else begin
     papco_autorange,ymat,zmin,zmax, $
       log=zscl(panel(0),3), exclude=0, nodata=1e-20
     zscl(panel(0),1)=zmin
     zscl(panel(0),2)=zmax
   endelse

   if (yscl(panel(0),0) eq 1) then begin
     yst=yscl(panel(0),1)
     yen=yscl(panel(0),2)
   endif else begin
     yscl(panel(0),1)=yst
     yscl(panel(0),2)=yen
   endelse
   
; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO:
;
;    !P.CHARSIZE  = widgetData.default_charsize    
;    !P.CHARTHICK = widgetData.default_charthick
;    !P.FONT      = -1
;    !P.NOERASE   = 1
;    !P.TICKLEN   = widgetData.default_TICKLEN
;    !X.TICKS     = widgetData.default_XTICKS
;    !X.STYLE     = widgetData.default_XSTYLE
;    !Y.STYLE     = 1
;    !X.MINOR     = widgetData.default_XMINOR
;
; Overriding these values by setting keywords is of course possible
; and allowed but discouraged, since 
; then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure.

   extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zmin,zmax], $
                  ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
                  ytitle:uytitle, xtickformat:'noticks'}

; check if the passed-in structure extra_par was set. If not, create
; it and set color to default black; which for the color table used by
; papco is 1. If extra_par was set, see if the tag color exists. If it
; does, leave it unchanged, if not, add it and set color to default
; black. 

  if n_elements(extra_par) EQ 0 then extra_par={color:1} else begin
    names=tag_names(extra_par)
    res=where(names eq 'COLOR',count)
    if count eq 0 then extra_par=create_struct({color:1}, extra_par)
  endelse  

; add keyword structure set locally and the one passed in

  extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; change the xtickformat if the panel is the bottom one and needs to
; plot a time axis   
   
  if panel(0) eq 0 then extra_plotPar.xtickformat='MJDT_DATE_4'
   
; You might have to add your own xtickformat for time axis
; labeling. If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; You can at this point also define a custom ytickformat, or leave it out for
; IDL default plot behaviour.
   
; "down" sets the axis labeling level. 
  
  down=0

;do color plot and color bar plot. nodata specifies the "nodata" flag
;in your data. Here we do not plot axis and data seperate, as
;papco_plot_colorspec cannot be used as an overplot!

  papco_plot_colorspec,ymat,time,yarr, $
    nodata=1e-20, /xcenter, _extra=extra_plotPar
  papco_colorbar,zmin,zmax,not(linlogbit),0,uztit
  
; store the coordinate information into common block

  plotted_x = !x
  plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label - 0r use any other function you define yourself.
; The routine x_side_label puts x-axis label to the right of the axis.
  
  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
    extra_plotPar.xtickformat='n_mltticks'
    extra_plotPar.color=1
    down=1
    plot,time,time, _extra=extra_plotPar, /NODATA
    extra_plotPar.xtickformat='n_ilatticks'
    extra_plotPar.color=1
    down=1
    plot,time,time, _extra=extra_plotPar, /NODATA
    extra_plotPar.xtickformat='n_lshelticks'
    extra_plotPar.color=1
    down=1
    plot,time,time, _extra=extra_plotPar, /NODATA
    extra_plotPar.xtickformat='n_radticks'
    extra_plotPar.color=1
    down=1
    plot,time,time, _extra=extra_plotPar, /NODATA    
    x_side_label,panel,'time (UT)!CMLT!CILAT!CLSHELL!CRADIUS'
  endif
      
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.

  right_side_label,panel,utitle,/rot90
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  
  if KEYWORD_SET(OUTPUT) then begin
    print,'% p_crres_lepa_spectra: Writing plot data out to file'
    description='plot_type template sample data - Energy spectra, 10 channels'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, time, ymat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
  endif  
   
end 

;-----------------------------------------------------------------------------
function n_mltticks,axis,index,t

  common crres_eph_data, input_header, input_data
  
  xutepf = input_data.time
  xmlt   = input_data.xmlt
  
  difft=abs(xutepf-t)
  result=min(difft,minindex)
  
  mlt=xmlt(minindex)
  
  return,string(mlt,format="('!C',f4.1)")
  
end

;-----------------------------------------------------------------------------
function n_ilatticks,axis,index,t

  common crres_eph_data, input_header, input_data
  
  xutepf = input_data.time
  xilat  = input_data.xilat
  
  difft=abs(xutepf-t)
  result=min(difft,minindex)
  
  ilat=xilat(minindex)
  
  return,string(ilat,format="('!C!C',f5.1)")
  
end 

;-----------------------------------------------------------------------------
function n_lshelticks,axis,index,t
  
  common crres_eph_data, input_header, input_data
  
  xutepf = input_data.time
  xlshel = input_data.xlshel 
  
  difft=abs(xutepf-t)
  result=min(difft,minindex)
  
  lshel=xlshel(minindex)
  
  return,string(lshel,format="('!C!C!C',f3.1)")
  
end 

;-----------------------------------------------------------------------------
function n_radticks,axis,index,t
  
  common crres_eph_data, input_header, input_data
  
  xutepf = input_data.time
  xrad   = input_data.xrad
  
  difft=abs(xutepf-t)
  result=min(difft,minindex)
  
  rad=xrad(minindex)
  
  return,string(rad,format="('!C!C!C!C',i5)")
  
end 






