;******************************************************************************
;* file p_crres_mag_line.pro
;*
;* this file is a template for a plot routine which can be used by papco.
;* it includes all the periphery needed, the user simply has to add in
;* the actual plot statement for his/her kind of data.
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_crres_mag_line, panel, type, $
;*           OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_crres_mag_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_crres_mag_line, panel, type, $
                      OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_crres_mag_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags which are meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type, swittch. Replace type in the call with whatever variables
; you need, or add more

; The following common blocks are needed:
  common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
  common yscale, yscl			;man/auto yscaling
  common shift_label, down             ;common for x-axis label shifting
  common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_crres_mag. 
  common crres_mag_data, input_header, input_data
  
  orbit=input_header.orbit
  panelset,panel			;sets the panel position viewport
   
; if yaxis is logarithmic, allow automatic yticks.
  if yscl(panel(0),3) eq 1 then ytickno=0

; since there is a common time axis plots look better if they all use the same
; number of xticks. In papco, this number is 6 by default. This is set by papco
; using the IDL graphics system variable !X.TICKS.
; You can override this for your own plot by setting the variable
; below. However, for plotting panels together, it makes sense to stick to the
; default. If you do set this variable, it needs to be included in the 
; extra_plotPar_common definition below.

;   xtickno=6

; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 
; Avoid setting the global IDL graphics system variables in your plot routine,
; as they may affect other modules' plot routines. The IDL graphics system 
; variables are set by papco to default values that guarantee some uniformity
; of plot appearance and should not be tampered with - unless you don't
; care if your plot "looks good" when combined with the other modules.
; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 

; the plot-calls normally use the arrays time for the time axis and yarr (ymat 
; for color plots) for the y or z axis.
; depending on your plot options, different data may be selected and different
; axis-labeling may be needed. This is normally done using a case statement.
; In this example the data array consist of flux values in ten channels 
; at different energies, typical of particle spectrometer data. Here I use the
; channel information passed in the plotinfo structure from the panel editor
; to select the channel to be drawn.

; set the begin end end times. papco convention is to use running seconds
; starting with the no of seconds since the beginning of the start day.
   time=input_data.time
   xut1=mjdt_start.t                       
   xut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
   index=where((input_data.time ge xut1) and (input_data.time le xut2),count)
   if count eq 0 then begin
     xut1=xut1+86400.0 & xut2=xut2+86400.0   
   endif
   
; restrict data to actual time range requested - this makes for faster zooming
   index=where((input_data.time ge xut1) and (input_data.time le xut2),count)
   if count ne 0 then data=input_data(index)
   
   time=data.time
   bdat=transpose(data.bdat)
   bmod=transpose(data.bmod)
   uytitle='nT'     

   case type of
      1:begin
         yray=bdat(*,0)
         index1=where(yray gt -500)
         index2=where(yray(index1) lt 500)
         ylim,time(index1(index2)),yray(index1(index2)),yst,yen
         utitle='B!DX!N:V (VDH)'
      end
      2:begin
         yray=bdat(*,1)
         index1=where(yray gt -500)
         index2=where(yray(index1) lt 500)
         ylim,time(index1(index2)),yray(index1(index2)),yst,yen
         utitle='B!DY!N:D (VDH)'
      end
      3:begin
         yray=bdat(*,2)
         index1=where(yray gt -500)
         index2=where(yray(index1) lt 500)
         ylim,time(index1(index2)),yray(index1(index2)),yst,yen
         utitle='B!DZ!N:H (VDH)'
      end
      4:begin
         yray=bmod(*,0)
         ylim,time,yray,yst,yen
         utitle='B!DX!N:V (VDH) OP77'
      end
      5:begin
         yray=bmod(*,1)
         ylim,time,yray,yst,yen
         utitle='B!DY!N:D (VDH) OP77'
      end
      6:begin
         yray=bmod(*,2)
         ylim,time,yray,yst,yen
         utitle='B!DZ!N:H (VDH) OP7'
      end
      7:begin
         yray=bdat(*,0)-bmod(*,0)
         ylim,time,yray,yst,yen
         utitle='B!DX!N:V (VDH) - OP77'
      end
      8:begin
         yray=bdat(*,1)-bmod(*,1)
         ylim,time,yray,yst,yen
         utitle='B!DY!N:D (VDH) - OP77'
      end
      9:begin
         yray=bdat(*,2)-bmod(*,2)
         ylim,time,yray,yst,yen
         utitle='B!DZ!N:H (VDH) - OP77'
      end
      10:begin
         yray=sqrt(bdat(*,0)^2+bdat(*,1)^2+bdat(*,2)^2)
         index1=where(yray lt 500)         
         ylim,time(index1),yray(index1),yst,yen
         utitle='B!DTOT'
      end
      11:begin
         yray=sqrt(bmod(*,0)^2+bmod(*,1)^2+bmod(*,2)^2)
         ylim,time,yray,yst,yen
         utitle='B!DTOT!N OP77'
      end 
      12:begin
         yray=sqrt(bdat(*,0)^2+bdat(*,1)^2+bdat(*,2)^2)-$
                   sqrt(bmod(*,0)^2+bmod(*,1)^2+bmod(*,2)^2)
         ylim,time,yray,yst,yen
         utitle='B!DTOT - OP77'
      end
      13:begin
         yray=atan(bdat(*,2),bdat(*,0))*57.296
         for j=0,nd-1 do if (yray(j) lt 0.0) then yray(j)=yray(j)+360
         ylim,time,yray,yst,yen
         utitle='angle H/V'
         uytitle='deg'
      end 
      14:begin
         yray=atan(bmod(*,2),bmod(*,0))*57.296
         for j=0,nd-1 do if (yray(j) lt 0.0) then yray(j)=yray(j)+360 
         ylim,time,yray,yst,yen
         utitle='angle H/V OP 77'
         uytitle='deg'
      end 
      15:begin
         yray=atan(bdat(*,2),bdat(*,1))*57.296
         for j=0,nd-1 do if (yray(j) lt 0.0) then yray(j)=yray(j)+360
         ylim,time,yray,yst,yen
         utitle='angle H/D'
         uytitle='deg'
      end 
      16:begin
         yray=atan(bmod(*,2),bmod(*,1))*57.296
         for j=0,nd-1 do if (yray(j) lt 0.0) then yray(j)=yray(j)+360    
         ylim,time,yray,yst,yen
         utitle='angle H/D OP 77'
         uytitle='deg'
      end 
      17:begin
         yray=atan(bdat(*,0),bdat(*,1))*57.296
         for j=0,nd-1 do if (yray(j) lt 0.0) then yray(j)=yray(j)+360
         ylim,time,yray,yst,yen
         utitle='angle V/D'
         uytitle='deg'
      end 
      18:begin
         yray=atan(bmod(*,0),bmod(*,1))*57.296
         for j=0,nd-1 do if (yray(j) lt 0.0) then yray(j)=yray(j)+360  
         ylim,time,yray,yst,yen
         utitle='angle V/D OP 77'
         uytitle='deg'
      end 
      19:begin
         bmag=sqrt(bdat(*,0)^2+bdat(*,1)^2+bdat(*,2)^2)
         nn=n_elements(bmag)
         bmagdiff=bmag(0:nn-2) - bmag(1:nn-1)
         tt=time(0:nn-2) - time(1:nn-1)
         yray = bmagdiff/tt
         yray = 100 * yray / bmag(0:nn-2)
         ylim,time,yray,yst,yen
         yen=6
         yst=-6
         utitle='dB/dt !DTOT!C in % var!C of total'
         uytitle='percent/sec'
      end 
    endcase
   
; the variables used above have the following meaning:
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;
; the procedure YLIM3 is optional and sets the ylimit automatically for the
; data leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data. 
; (in $papco/papco_Lib/ylim.pro)

; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for "linear" and 1 for "log"  scale
; when scaling is set to automatic, the automatic y-limits are loaded in.
   if (yscl(panel(0),0) eq 1) then begin
     yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
   endif else begin
     yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
   endelse

; use auto y-labeling routine to get "good" y-lables
   papco_y_label, yst, yen
   
; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO:
;
;    !P.CHARSIZE  = widgetData.default_charsize    
;    !P.CHARTHICK = widgetData.default_charthick
;    !P.FONT      = -1
;    !P.NOERASE   = 1
;    !P.TICKLEN   = widgetData.default_TICKLEN
;    !X.TICKS     = widgetData.default_XTICKS
;    !X.STYLE     = widgetData.default_XSTYLE
;    !Y.STYLE     = 1
;    !X.MINOR     = widgetData.default_XMINOR
;
; Overriding these values by setting keywords is of course possible
; and allowed but discouraged, since 
; then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure.
   extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen],  $
                  ylog:yscl(panel(0),3), $
                  ytitle:uytitle, xtickformat:'noticks'}
   
; check if the passed-in structure extra_par was set. If not, create
; it and set color to default black; which for the color table used by
; papco is 1. If extra_par was set, see if the tag color exists. If it
; does, leave it unchanged, if not, add it and set color to default
; black. 
  if n_elements(extra_par) EQ 0 then extra_par={color:1} else begin
    names=tag_names(extra_par)
    res=where(names eq 'COLOR',count)
    if count eq 0 then extra_par=create_struct({color:1}, extra_par)
  endelse     
   
; add keyword structure set here with the one passed in
  extra_plotPar=create_struct(extra_plotPar, extra_par)
  
; use papco routine to draw time axis. This checks for bottom plot
; and uses the user's xtickformat if it is something other than 'noticks'.
  down=0
  papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar  

; To keep things general, plotting of the data is split up into plotting the 
; axis with no data, and then plotting the data with no axis. This is done to 
; enable the feature of overplots, which just plot data into an existing axis.
; plot the axis in color=1, and only plot axis if if you're NOT over-plotting!
; down sets the axis labeling level (adds carriage returns [!C] to labels).

; now do the actual data plot, data only, no axis
   plot,time,yray, _extra=extra_plotPar, xstyle=5, ystyle=5

; store the coordinate information into common block for slices
   plotted_x = !x  &   plotted_y = !y

; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION
  if not keyword_set(overplot) then begin
    utitle='CRRES '+string(orbit,format="(i4.4)")+'!C'+utitle
    right_side_label,panel,utitle,_extra=extra_Par
  endif else begin
    for i=0, PLOTS_ATTHISPOSITION-1 do $
      if i eq 0 then utitle='!C!C'+utitle else utitle='!C'+utitle
    right_side_label,panel,utitle,_extra=extra_Par
  endelse
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
  
  if KEYWORD_SET(OUTPUT) then begin
    print,'% p_crres_mag_line: Writing plot data out to file'
    description='crres_mag data - single energy channel line'
    time_label='Seconds since start of day'
    ;channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, $
                      RS_LABEL=RS_LABEL
  endif  
      
  xut1=mjdt_start.t                       
  xut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
   
end 
