;******************************************************************************
;* file p_CRRES_MAG_overview.pro
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_CRRES_MAG_overview, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vectorhomepage.html
;*              Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_plot_type, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written September 1997, Reiner Friedel
;******************************************************************************
pro p_CRRES_MAG_overview, panel, $
                          type, $
                          OVERPLOT=OVERPLOT, $
                          PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                          SUBTABLE=SUBTABLE, $
                          _EXTRA=extra_par

; The following common blocks are needed:

   common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
   common yscale, yscl			;man/auto yscaling
   common zscale, zscl			;man/auto zscaling
   common shift_label, down             ;common for x-axis label shifting
   common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure.
   
   common crres_mag_overview, info, lsum, valdat
   common crres_mag_overview_extra, intime, sp, lbin, lrange
   
   papco_Set_SubTable, SUBTABLE         ;sets the papco color table index
   panelset,panel			;sets the panel position viewport

; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.

   ytickno=50*(!p.position(3)-!p.position(1))
   if ytickno gt 7 then ytickno=7
   
;   xtickno=6

   nobins=(lrange-1)/float(lbin)+1
   yarr=fltarr(nobins,2)
   for i=0,nobins-1 do begin
     yarr(i,0)=1+i*lbin
     yarr(i,1)=1+(i+1)*lbin
   endfor
   
   yst=yarr(0,0) ;yst=1.0 
   yen=yarr(nobins-1,1) ;yen=7.5
   
; for color plots, the y-axis normally refers to a channel. You have to 
; re-scale the data array to the range required before passing it to the
; color plot routine, also the ymin and ymax arrays.

; the plot-calls normally use the arrays time for the time axis and yarr (ymat 
; for color plots) for the y or z axis.
; depending on your plot options different data may be selected and different
; axis-labeling may be needed. This is normally done using a case statement.
   
   ymat=fltarr(sp,nobins)       ; make the plot arrays needed - total counts
   yval=fltarr(sp,nobins)       ; no of samples in bin 

   time=(intime(*,0)+intime(*,1))/2.
   xarr=intime
   
   xut1=mjdt_start.mjd+mjdt_start.t / 86400.0                     
   xut2=mjdt_end.mjd  +mjdt_end.t   / 86400.0  
   
   for i=0,sp-1 do begin        ; fill the plot arrays
      ymat(i,*)=  lsum(i,type-1,*)
      yval(i,*)=valdat(i,type-1,*)
   endfor
   index=where(yval ne 0, count)
   if count gt 0 then $
     ymat(index)=ymat(index)/yval(index) ; get average counts in each bin 
   case type of
     1:begin         
        utitle='CRRES B!DX!N=V (VDH)'
        uztit='[nT]'
        autorange,panel(0),time,ymat,zmax,zmin,1
         zmaxc=zmax & zminc=zmin
      end
      2:begin
        utitle='CRRES B!DY!N=D (VDH)'
        uztit='[nT]'
        autorange,panel(0),time,ymat,zmax,zmin,1
         zmaxc=zmax & zminc=zmin
      end
      3:begin
         utitle='CRRES B!DZ!N=H (VDH)'
         uztit='[nT]'
        autorange,panel(0),time,ymat,zmax,zmin,1
         zmaxc=zmax & zminc=zmin
      end
      4:begin
         utitle='CRRES B!DX!N=V (VDH)!COls.Pf.77'
         uztit='[nT]'
         autorange,panel(0),time,ymat,zmax,zmin,1
         zmaxc=zmax & zminc=zmin
      end
      5:begin
         utitle='CRRES B!DY!N=D (VDH)!COls.Pf.77'
         uztit='[nT]'
         autorange,panel(0),time,ymat,zmax,zmin,1
         zmaxc=zmax & zminc=zmin
      end
      6:begin
         utitle='CRRES B!DZ!N=H (VDH)!COls.Pf.77'
         uztit='[nT]'
         autorange,panel(0),time,ymat,zmax,zmin,1
         zmaxc=zmax & zminc=zmin
      end
      7:begin
        zmin=0
        zmax=50
        if (zscl(panel(0),0) eq 1) then begin
          zmin=zscl(panel(0),1)
          zmax=zscl(panel(0),2)
        endif else begin
          zscl(panel(0),1)=zmin
          zscl(panel(0),2)=zmax
        endelse           
        utitle='B!DX!N=V !Cminus!COls.Pf.77'
        uztit='[nT]'
        ; calculate the absolute difference in V
        for i=0, sp-1 do begin
          ymat(i,*)=  abs(lsum(i,0,*)*valdat(i,3,*)- $
                          lsum(i,3,*)*valdat(i,0,*))
          yval(i,*)=valdat(i,0,*)*valdat(i,3,*)
        endfor
        index=where(yval eq 0,count)
        if count gt 0 then $
          ymat(index)=1000.0+(zmax/2.)
        index=where(yval ne 0,count)
        if count gt 0 then $
          ymat(index)=ymat(index)/yval(index)
        index=where(abs(ymat) gt 200., count)
        if count gt 0 then $
          ymat(index)=1000.0+(zmax/2.)
        zmaxc=zmax & zminc=zmin
      end
      8:begin
        zmin=0
        zmax=100
        if (zscl(panel(0),0) eq 1) then begin
          zmin=zscl(panel(0),1)
          zmax=zscl(panel(0),2)
        endif else begin
          zscl(panel(0),1)=zmin
          zscl(panel(0),2)=zmax
        endelse         
        utitle='B!DY!N=D !Cminus!COls.Pf.77'
        uztit='[nT]'
        zmaxc=zmax/2
        zminc=-zmax/2
        index=where(yval eq 0,count)
        if count gt 0 then $
          ymat(index)=1000.0
        index=where(abs(ymat) gt 200., count)
        if count gt 0 then $
          ymat(index)=1000.0
        index=where(ymat le -(zmax/2))
        ymat(index)=-(zmax/2)
        ymat=ymat+(zmax/2)
      end
      9:begin
        zmin=0
        zmax=100
        if (zscl(panel(0),0) eq 1) then begin
          zmin=zscl(panel(0),1)
          zmax=zscl(panel(0),2)
        endif else begin
          zscl(panel(0),1)=zmin
          zscl(panel(0),2)=zmax
        endelse         
        utitle='B!DZ!N=H !Cminus!COls.Pf.77'
        uztit='[nT]'
        zmaxc=(zmax/2)
        zminc=-(zmax/2)
        index=where(yval eq 0,count)
        if count gt 0 then $
          ymat(index)=1000.0
        index=where(abs(ymat) gt 200, count)
        if count gt 0 then $
          ymat(index)=1000.0
        index=where(ymat le -(zmax/2),count)
        if count gt 0 then $
          ymat(index)=-(zmax/2)
        ymat=ymat+(zmax/2)
      end
      10:begin
         utitle='CRRES B!DTOT'
         uztit='[nT]'
         index=where((ymat gt 103.415) and (ymat lt 103.417),count)
         if count ne 0 then ymat(index)=0
         index=where((ymat gt 100.730) and (ymat lt 100.740),count)
         if count ne 0 then ymat(index)=0         
         autorange,panel(0),time,ymat,zmax,zmin,1
         zmaxc=zmax & zminc=zmin
      end
      11:begin
         utitle='CRRES B!DTOT!COls.Pf.77'
         uztit='[nT]'
         autorange,panel(0),time,ymat,zmax,zmin,1
         zmaxc=zmax & zminc=zmin
      end 
      12:begin
        zmin=0
        zmax=100
        if (zscl(panel(0),0) eq 1) then begin
          zmin=zscl(panel(0),1)
          zmax=zscl(panel(0),2)
        endif else begin
          zscl(panel(0),1)=zmin
          zscl(panel(0),2)=zmax
        endelse
        utitle='B!DTOT !N minus!COls.Pf.77'
        uztit='[nT]'
        zmaxc=zmax/2
        zminc=-zmax/2
        btot=fltarr(sp,nobins)
        for i=0,sp-1 do btot(i,*)=lsum(i,9,*)
        index=where(btot eq 0)
        ymat(index)=1000.0
        index=where(abs(ymat) gt 200., count)
        if count gt 0 then $
          ymat(index)=1000.0        
        index=where(ymat le -(zmax/2),count)
        if count gt 0 then ymat(index)=-(zmax/2)
        ymat=ymat+(zmax/2)
      end
      13:begin
        zmin=0.
        zmax=360.
        if (zscl(panel(0),0) eq 1) then begin
          zmin=zscl(panel(0),1)
          zmax=zscl(panel(0),2)
        endif else begin
          zscl(panel(0),1)=zmin
          zscl(panel(0),2)=zmax
        endelse         
        utitle='angle H/V !Cminus!COls.Pf.77 '
        uztit='[deg]'
        for i=0, sp-1 do begin
          ymat(i,*)= abs(lsum(i,12,*)*valdat(i,13,*)- $
                      lsum(i,13,*)*valdat(i,12,*))
          yval(i,*)=valdat(i,12,*)*valdat(i,13,*)
        endfor
        ;set ymat to 1000. if yval is zero
        index=where(yval eq 0,count)
        if count gt 0 then $
          ymat(index)=1000.0
        ;normalise if yval not zero
        index=where(yval ne 0,count)
        if count gt 0 then $
          ymat(index)=ymat(index)/yval(index)
        ;negative H values give large H/V
        index=where(ymat gt 210 and ymat lt 360, count)
        if count gt 0 then $
          ymat(index)=abs(ymat(index)-360.)
        ymat=ymat+zmax/2.
        zmaxc= zmax/2.
        zminc=-zmax/2.
      end 
      14:begin
         utitle='Ols.Pf.77'
         uztit='[deg]'
         autorange,panel(0),time,ymat,zmax,zmin,1
         zmaxc=zmax & zminc=zmin
      end 
      15:begin
         utitle='angle H/D'
         uztit='[deg]'
         autorange,panel(0),time,ymat,zmax,zmin,1
         zmaxc=zmax & zminc=zmin
      end 
      16:begin
         utitle='Ols.Pf.77'
         uztit='[deg]'
         autorange,panel(0),time,ymat,zmax,zmin,1
         zmaxc=zmax & zminc=zmin
      end 
      17:begin
         utitle='CRRES angle V/D'
         uztit='[deg]'
         autorange,panel(0),time,ymat,zmax,zmin,1
         zmaxc=zmax & zminc=zmin
      end 
      18:begin
         utitle='Ols.Pf.77'
         uztit='[deg]'
         autorange,panel(0),time,ymat,zmax,zmin,1
         zmaxc=zmax & zminc=zmin
      end
   endcase   
;   uytitle='Ols.Pf.!CL-value'
   uytitle='L-value'
   
; set the begin end end times. papco convention is to use running secconds
; starting with the no of seconds since the begiing of the start day.
   

; restrict data to actual time range requested - this makes for faster zooming

   index=where((xarr(*,0) ge xut1) and (xarr(*,1) le xut2),count)
   if count ne 0 then begin
      time=time(index)
      xarr=xarr(index,*)
      ymat=ymat(index,*)      
    endif
    
    index=where(xarr(1:*,0)-xarr(0:*,0) gt 0.,count)+1
    if count ne 0 then begin
      time=time(index)
      xarr=xarr(index,*)
      ymat=ymat(index,*)      
    endif
    
; set a cut off at L=2.5   
    cut_index=where(yarr(*,1) lt 2.5)
    ymat(*,cut_index)=1000.+(zmax/2.)
    
; the variables used above have the following meaning:
;       uztit   : the z-axis (colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data

; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling
   
   if (yscl(panel(0),0) eq 1) then begin
      yst=yscl(panel(0),1)
      yen=yscl(panel(0),2)
   endif else begin
      yscl(panel(0),1)=yst
      yscl(panel(0),2)=yen
   endelse
   
; set up extra plot keywords, common ones first. 
; yrange, xrange  are always needed. 
; the rest you can set if you want to override papco defaults.
; See $papco_BASE/papco/papco_draw for the definition of the papco default
; graphic sytem variable settings.

   extra_plotPar={yrange:[yst, yen], xrange:[xut1, xut2], $
                         zrange:[zmin,zmax], ylog:yscl(panel(0),3) , $
                         zlog:zscl(panel(0),3), $
                         ytitle:uytitle, xtickformat:'noticks'}

; check if the passed-in structure extra_par was set. If not, set color to
; default black; which for the color table used by papco is 1.

   if n_elements(extra_par) EQ 0 then extra_par={color:1} else begin
     names=tag_names(extra_par)
     res=where(names eq 'COLOR',count)
     if count eq 0 then extra_par=create_struct({color:1}, extra_par)
   endelse
   
; add keyword structure set here with the one passed in

   extra_plotPar=create_struct(extra_plotPar, extra_par)

; change the xtickformat if the panel is the bottom one and needs to
; plot a time axis   
   
   if panel(0) eq 0 then extra_plotPar.xtickformat='mjd_dec_2'

; You might have to add your own xtickformat for time axis
; labeling. If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; You can at this point also define a custom ytickformat, or leave it out for
; IDL default plot behaviour.
   
; "down" sets the axis labeling level. 
  
  down=0
  
;do color plot and color bar plot. nodata specifies the "nodata" flag
;in your data. Here we do not plot axis and data seperate, as
;papco_plot_colorspec cannot be used as an overplot!
  
  linlogbit=zscl(panel(0),3)    ; set linear/log z-scaling
  smbit=0                       ; set if you want to smooth in y-direction
  
  papco_plot_colorspec,ymat,xarr,yarr, $
    nodata=1000.+(zmax/2.),_extra=extra_plotPar
  papco_colorbar,zminc,zmaxc,not(linlogbit),0,uztit
  
; store the coordinate information into common block
  
  plotted_x = !x
  plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; The routine x_side_label puts x-axis label to the right of the axis.
; If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; Or use any other function you define yourself.
; down sets the axis labeling level
   
   
   if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
     extra_plotPar.xtickformat='mjd_dec_1'
     extra_plotPar.color=1
     down=1
     plot,time,time, _extra=extra_plotPar, /NODATA
     extra_plotPar.xtickformat='crres_mag_orb_ticks'
     extra_plotPar.color=1
     down=2
     plot,time,time, _extra=extra_plotPar, /NODATA
     x_side_label,panel,'DOY/Time!CDate!COrbit'
   endif  
   
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.

  right_side_label,panel,utitle,/rot90
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  
  if KEYWORD_SET(OUTPUT) then begin
    print,'% p_mea_overview: Writing plot data out to file'
    description='plot_type template sample data - Energy spectra, 10 channels'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, time, zarr, $
      DESCRIPTION=description, TIME_LABEL=time_label, $
      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
      RS_LABEL=rs_label, NODATA=nodata
  endif  
  
end 

;------------------------------------------------------------------------------
; This function labels orbits given that a mag-lav file was read

function crres_mag_orb_ticks,axis,index,t

  common crres_mag_overview, info, lsum, valdat
  common crres_mag_overview_extra, intime, sp, lbin, lrange
  
  time=(intime(*,0)+intime(*,1))/2.
  res=min(abs(time-t),mindex)
  orbit=info(mindex,3)
  if res lt 1.0 then $
    return,string(orbit,format="('!C!C',i4.4)") $
  else $
    return,''
  
end  
