;******************************************************************************
;* FUNCTION:     
;*      FUNCTION crres_mea_panel, PlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	PlotInfo	a structure of the papco_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $papco_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{       canceled:1, $
;*	    		panelHeight:1, $
;*	    		channel:0, $
;*	    		typeVector:[0,0,0,0], $ 
;*	    		manualYScaling:0 $
;*	    		yScaleMin:LONG(1), $
;*	    		yScaleMax:LONG(1), $
;*	    		panelKind:'some string' }
;*
;*      Additionally, some Panel-editors may return these fields in
;*      structure:
;*               {  color:1, $
;*		    plotStyle:1 }
;*			
;*      For an enhanced description read $papco_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	plotinfoStruct = papco_getPlotInfoStruct()
;*
;* MODIFICATION HISTORY:       August 2004, updated for papco 10 by R.Friedel
;******************************************************************************
FUNCTION  crres_mea_panel, PlotInfo, GROUP=group, ACTION=ACTION

COMMON crres_mea_panel_data, widgetData
COMMON crres_mea ;see crres_mea_commons.pro for contents of this block

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
additionalWData={canceled:1}

;***NEW*** All the user's widget info is now stored in a structure,
;and the only thing stored in the plotinfostruct is the pointer to
;that structure. This allows unlimited freedom in variables for the
;module.

;test if the pointer already points to the modules control
;structure. If so, load it in, if not, load in the default.
r = size(*plotinfo.USR_PTR1, /st)
IF r.TYPE_NAME EQ 'INT' THEN plotinfo.USR_PTR1 = ptr_new(crres_mea_control)
control = *plotinfo.USR_PTR1

;--- create the base-window ---------------------------------------------------
title = plotinfo.panelKind + ' - ' + papco_module_version('crres_mea')
panel_base=papco_WIDGET_BASE(TITLE=title, ROW=1, GROUP_LEADER=group)
panel_left=widget_base(panel_base,COLUMN=1, /FRAME)
panel_right=widget_base(panel_base,COLUMN=1,/FRAME)

;--- the first part of the window will consist in the widgets -----------------
;    that control the height of that plot
widgets=papco_panel_AddHeightWidgets(panel_left, plotinfo)
additionalWData=create_struct(widgets, additionalWData)

;--- Plot type selection buttons ----------------------------------------------
plot_base=WIDGET_BASE(panel_left, ROW = 2, /ALIGN_CENTER)

names = papco_make_choice_names(control.pl_type_info)
pl_type_base = papco_CW_BGROUP(plot_base, names, ROW = 1, /FRAME, /EXCLUSIVE, $
                   UVALUE='spb_pl_type', SET_VALUE=control.pl_type)

;--- reload control buttons --------------------------------------------------
names = papco_make_choice_names(control.reload_info)
reload_base = papco_CW_BGROUP(plot_base, names, ROW = 1, /EXCLUSIVE, /FRAME, $
                   UVALUE='spb_reload', SET_VALUE=control.reload)

;--- spec or line plot -------------------------------------------------------
names = papco_make_choice_names(control.spec_info)
spec_base = papco_CW_BGROUP(plot_base, names, ROW = 1, /EXCLUSIVE, /FRAME, $
                   UVALUE='spb_spec', SET_VALUE=control.spec)

;--- Slice averaging entry field ----------------------------------------------
entry_base=WIDGET_BASE(panel_left, ROW=2, /FRAME)

fld_time_av = CW_FIELD(entry_base, /ROW,  XSIZE=3, /INTEGER, $
                        title='Time av.', $
                        value=control.time_av > 30, UVALUE='fld_time_bin')

fld_slice_av=CW_FIELD(entry_base, /ROW,  XSIZE=2, /INTEGER, $
                        title='Slice av. +/-', $
                        value=control.slice_av, UVALUE='fld_slice_av')

fld_pa_bin = CW_FIELD(entry_base, /ROW,  XSIZE=3, $
                   TITLE = 'PA bin:', $
                   UVALUE= 'fld_pa_bin', VALUE = control.pa_bin)

fld_mu = CW_FIELD(entry_base, /ROW,  XSIZE=47, $
                   TITLE = 'Enter mu1, mu2, ... (MeV/G):', $
                   UVALUE= 'fld_mu', VALUE = control.mu)

; --- Outer Slider Base -------------------------------------------------------
average_Base=WIDGET_BASE(panel_left, COLUMN=1, /ALIGN_CENTER)

;--- make select widgets for all variables in "average_names" -----------------
;    use papco_panel compound widget for this!
FOR i = 0, n_elements(average_names)-1 DO BEGIN

   r = execute('FROM_VAL = control.'+average_names(i)+'_AVERAGE_FROM_VAL')
   r = execute('TO_VAL = control.'+average_names(i)+'_AVERAGE_TO_VAL')

   FROM_VAL = FROM_VAL > average_range(0, i) < average_range(1, i)
   TO_VAL = TO_VAL > average_range(0, i) < average_range(1, i)

   widgets = papco_panel_AddVar_Ave_Widgets(average_base, average_names(i), $
               FROM_VAL, TO_VAL, $
               average_range(0, i), average_range(1, i), average_range(2, i), $
               average_frmts(i))
   additionalWData=create_struct(widgets, additionalWData)

END
  
;--- add the papco panel widget for specifying own plot description label -----
;widgets=papco_panel_AddDescr_Widgets(panel_right, plotinfo, /VERTICAL)
;additionalWData=create_struct(widgets, additionalWData)

;--- make base for color bar OR plotstyle widget. One only is mapped ----------
line_col_outer_base=WIDGET_BASE(panel_left, COLUMN=1)
line_col_base=WIDGET_BASE(line_col_outer_base)

plot_style_base=WIDGET_BASE(line_col_base, COLUMN = 1, MAP=0)
widgets=papco_panel_add_PlotStyleButtons(plot_style_base, plotinfo, 0)
additionalWData=create_struct(widgets, additionalWData)

colorbar_base=WIDGET_BASE(line_col_base, COLUMN = 1, MAP=0)
widgets=papco_panel_Addcolorbar_Widgets(colorbar_base, plotinfo)  
additionalWData=create_struct(widgets, additionalWData)

;--- add the papco panel widget for choosing the y-axis-ranges ----------------
widgets=papco_panel_AddYScale_Widgets(panel_right, plotinfo,/VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
widgets=papco_panel_AddZScale_Widgets(panel_right, plotinfo,/VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
widgets=papco_panel_AddActionButtons(panel_left, Action, plotinfo, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
thisWidgetsData={helpfile:'crres_mea_panel.help', $
                 fld_mu:fld_mu, fld_pa_bin:fld_pa_bin, $
                 fld_slice_av:fld_slice_av, fld_time_av:fld_time_av, $
                 plot_style_base:plot_style_base, $
                 colorbar_base:colorbar_base }

widgetData=papco_PANEL_CreateWidgetData(control, thisWidgetsData)
widgetData=papco_PANEL_CreateWidgetData(widgetData, additionalWData)

;--- realize the window -------------------------------------------------------
crres_mea_refresh, /all
papco_Panel_Realize, panel_base, widgetData
papco_xmanager, 'crres_mea_panel', panel_base

;now return data. Since all the widget's info in in a control structure,
;only load it's pointer into USR_PTR1 and return that.

;control is part of the widgetData structure. Extract it
struct_assign, widgetData, *plotinfo.USR_PTR1

;work out if there has been a change in the control structure
plotinfo.modified = plotinfo.modified + $
                    papco_compare_struct(control, *plotinfo.USR_PTR1)

ReturnData={USR_PTR1:plotinfo.USR_PTR1, $
            MODIFIED:plotinfo.modified}

additionalRData=papco_panel_ReturnData(widgetData)

RETURN, create_struct(ReturnData, additionalRData)
   
END

;******************************************************************************
;* PROCEDURE:     
;*     PRO crres_mea_refresh, SL_ENERGY=SL_ENERGY
;* 
;* DESCRIPTION:  
;*     update the display of the editor-widgets
;* 
;* INPUTS:       
;*     none
;* 
;* KEYWORDS:
;*     only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*     have a look at PRO crres_mea_panel_event
;*
;* MODIFICATION HISTORY:       
;*     August 1999, written by R.Friedel
;******************************************************************************
PRO crres_mea_refresh, ALL = ALL, COL_LINE = COL_LINE

COMMON crres_mea_panel_data, widgetData
COMMON crres_mea ;see m_crres_mea_commons.pro for contents of this block

; ---- Handle ALL   -----------------------------------------------------------
IF keyword_set(ALL) THEN BEGIN 
    COL_LINE = 1
ENDIF   

; ---- COLORBAR or PLT SYYLE widget -------------------------------------------
IF keyword_set(COL_LINE) THEN BEGIN    
    IF widgetData.spec EQ 0 THEN BEGIN 
        widget_control, widgetData.plot_style_base, MAP=1
        widget_control, widgetData.colorbar_base, MAP=0
    endif else BEGIN 
        widget_control, widgetData.plot_style_base, MAP=0
        widget_control, widgetData.colorbar_base, MAP=1
    ENDELSE
ENDIF 


END

;******************************************************************************
;* PROCEDURE:     
;*      PRO crres_mea_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO crres_mea_panel_event, event
  
COMMON crres_mea_panel_data, widgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id

;test uvalue for structure - select variable sliders return structure uvals.
r = size(uval, /st)
IF r.type_name EQ 'STRUCT' THEN BEGIN
    uval_str = uval & uval = uval.uvalue
ENDIF

CASE uval OF
    'spb_pl_type'  : BEGIN
        widgetData.pl_type = event.value
    END 

    'spb_reload'   : widgetData.reload = event.value

    'spb_spec'   : BEGIN
        widgetData.spec = event.value
        crres_mea_refresh, /COL_LINE
    END 
    
    ELSE: BEGIN 
        WIDGET_CONTROL, widgetData.fld_mu, GET_VALUE = X
        widgetData.mu = x
        WIDGET_CONTROL, widgetData.fld_pa_bin, GET_VALUE = X
        widgetData.pa_bin = x
        WIDGET_CONTROL, widgetData.fld_slice_av, GET_VALUE=x
        widgetData.slice_av = x   
        WIDGET_CONTROL, widgetData.fld_time_av, GET_VALUE=x
        widgetData.time_av = x > 30  
        widgetData=papco_panel_Event(event, widgetData)
    END 
ENDCASE 
    
END
