;******************************************************************************
;* FILE: crres_mea_panel.pro
;*
;* DESCRIPTION: 
;*    This is a demonstration on how to implement a papco-panel-editor.
;*    It makes use of all the features provided by $papco_XX/papco_panel.pro
;*    (a collection of standard papco widgets). It constructs an editor for a
;*    very simple papco-plot, that has all the default-functionality.
;*
;* COMMON-BLOCKS:
;*    crres_mea_panel_data   contains the window-information
;*
;* STRUCTURES:
;*    This widget inherits a PlotInfo structure as an argument. This
;*    structure is then modified by the user according to his/her needs:
;*    Either by including standard PAPCO widgets from papco_panel.pro,
;*    which extend this structure, or by modifying its tags. The modified
;*    structure is then handed back to PAPCO.
;* 
;*    The PlotInfo structure is created by a call to papco_getPlotInfoStruct
;*    and contains a series of tags which are used to describe a given panel.
;*    These tags come in three categories:
;*       1. Tags used by PAPCO internally - don't touch!          
;*       2. Tags added by including standard PAPCO widgets. You can edit 
;*          or preset these tags if you wish.
;*       3. User Tags for control of user-specified features. There
;*          are enough user tags specified to handle most needs. 
;*    The main limitation of the User Tags is that you're stuck with
;*    what has been defined in PAPCO. If you REALLY need additional tags,
;*    this entails changing the PAPCO core, and should be done by
;*    whoever maintains the core. You can do this yourself too, by editing
;*    pro papco_getPlotInfoStruct contained in $papco_XX/papco/papco.pro
;*    but your changes will then not propagate to the next PAPCO version.
;*
;*    The PAPCO 09 set of user tags is:
;*
;*       typeVector:intarr(4), $ ; user keyword (old, for compatibility)
;*       swittch:lonarr(3),     $ ; user keyword (old, for compatibility)
;*       options:fltarr(20),   $ ; user keyword (float options)
;*       ioptions:lonarr(16),  $ ; user keyword (long options)
;*       soptions:strarr(16)   $ ; user keyword (string options)
;*
;*    NB! If you need more variables to commincate between your panel 
;*    editor and your plot routine then you can use your own common block
;*
;*    For a more complete description of the other PlotInfo tags and the
;*    tags used by PAPCO supplied widget refer to
;*    $papco_doc/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  crres_mea_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;*    To test you might try this to get a stand-alone widget:
;*       a=papco_getPlotInfoStruct()   which creates the structure
;*                                     containing the plot information 
;*	 a=crres_mea_panel(a)          which calls up the panel window.
;*    Interrogating a with help,a,/str shows the contents. 
;*
;* MODIFICATION HISTORY:       
;*     august 1995, written by A.Keese
;*     december 1995, modified by R.Friedel at Goddard
;*     april 1996, modified by R.Friedel
;*     march 1997, modified by R.Friedel
;*     August 1998, modified by R.Friedel
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  crres_mea_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor widget for a panel, using the methods defined in the 
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from aplotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  crres_mea_panel, aPlotInfo, GROUP=group, ACTION=ACTION

  COMMON crres_mea_panel_data, widgetData

;--- if GROUP is not specified, the group-leader will be the desktop
  IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
  IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
  additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
;    since this can be a large widget, make a top, left and right, bottom side.
  
  panel_base=papco_WIDGET_BASE(title=aPlotInfo.panelKind, column=1, $
                               GROUP_LEADER=group, /MODAL)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
  widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
  
  panel_middle=widget_base(panel_base,row=1)
  panel_left=widget_base(panel_middle,column=1)
  panel_right=widget_base(panel_middle,column=1) 
  
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
  
;--- data swittch base ---------------------------------------------------------
  swittch_base=WIDGET_BASE(panel_left,ROW=1, /ALIGN_CENTER) 
  
; --- Integral / differential -------------------------------------------------
  int_diff_base=WIDGET_BASE(swittch_base, /FRAME, COLUMN=1, /EXCLUSIVE)
  spb_integral=WIDGET_BUTTON(int_diff_base, $
                             VALUE=strcenter('Integral',12), $
                             UVALUE='spb_integral')
  spb_differential=WIDGET_BUTTON(int_diff_base, $
                                 VALUE=strcenter('Differential',12), $
                                 UVALUE='spb_differential')

  IF aPlotInfo.typeVector(1) EQ 0 THEN BEGIN
    WIDGET_CONTROL, spb_integral, SET_BUTTON=1
    WIDGET_CONTROL, spb_differential, SET_BUTTON=0
  ENDIF ELSE BEGIN
    WIDGET_CONTROL, spb_integral, SET_BUTTON=0
    WIDGET_CONTROL, spb_differential, SET_BUTTON=1
  ENDELSE
    
  reload_base=WIDGET_BASE(swittch_base, /FRAME, COLUMN=1, /EXCLUSIVE)
  reloadNames=['Relaod', 'Adopt' ]  
  spb_reload=lonarr(n_elements(reloadNames))
  for i=0,n_elements(reloadNames)-1 do $
    spb_reload(i)=WIDGET_BUTTON(reload_base, $
		VALUE=reloadNames(i), $
		UVALUE='spb_reload'+strtrim(string(i), 2))
  WIDGET_CONTROL, spb_reload(aPlotInfo.ioptions(1)), SET_BUTTON=1     
  
; --- Energy / pitch ----------------------------------------------------------
  ener_pitch_outerBase=WIDGET_BASE(panel_left,/FRAME,COLUMN=1,/ALIGN_CENTER) 
   
  ener_pitch_buttonBase=WIDGET_BASE(ener_pitch_OuterBase, ROW=1, /EXCLUSIVE) 
  spb_energy=WIDGET_button(ener_pitch_buttonbase, $
              VALUE='energy spectrum @ pitch angle:', $
              UVALUE='spb_energy')
  spb_pitch=widget_button(ener_pitch_buttonbase, $
	      VALUE='pitch angle distribution @ energy:', $
              UVALUE='spb_pitch')
   
   WIDGET_CONTROL, spb_energy, SET_BUTTON=1
   WIDGET_CONTROL, spb_pitch, SET_BUTTON=0
   sl_energy_Pitch=WIDGET_SLIDER(ener_pitch_outerBase, /DRAG, $
		     VALUE=aPlotInfo.typeVector(3) < 14 > 1, $
                     UVALUE='sl_energy_pitch',$
                     MINIMUM=0, MAXIMUM=14,XSIZE=400,/ALIGN_CENTER)
   lbl_energy_Pitch=WIDGET_LABEL(ener_pitch_outerBase, $
                     VALUE=strcenter('pitch angle energy range',50), $
                     /ALIGN_CENTER)			      
  
;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
;--- for examples of standard buttons or sliders look at other panel editors.
;    some standard buttons which have been recycled before can be found in
;    papco_XX/papco_lib/more_panel_widgets.

;--- As the same panel is called when you edit a plot, interrogate the tags in 
;    the plotInfoStruct (aplotInfo) passed in and set your widgets accordingly.
;    In this way the widget is always reinitialized to its last state.
   
;--- The next five papco panel widgets will be added only if you want to
;    support that functionality, and if they make sense for your
;    crres_mea. If you do not want to use them, just comment them out.

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
;  widgets=papco_panel_AddFile_Widgets(panel_left, aPlotInfo)
;  additionalWData=create_struct(widgets, additionalWData)   
;--- add the papco panel widget for choosing the color bar for color plots ----
  widgets=papco_panel_Addcolorbar_Widgets(panel_left, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
  widgets=papco_panel_add_PlotStyleButtons(panel_left, aPlotInfo, 0)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
  widgets=papco_panel_AddYScale_Widgets(panel_right, aPlotInfo,/VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
  widgets=papco_panel_AddZScale_Widgets(panel_right, aPlotInfo,/VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
  widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, $
                                       /CONFIG)
  additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (a template is in the crres_mea example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.
  thisWidgetsData={helpfile:'crres_mea_panel.help', $
                   slider_displaysEnergy:0, $
                   lbl_energy_Pitch:lbl_energy_Pitch, $
                   sl_energy_pitch:sl_energy_Pitch, $
                   spb_integral:spb_integral, $
                   spb_differential:spb_differential, $
                   ioptions:aPlotInfo.ioptions, $
                   spb_pitch:spb_pitch, $
                   spb_energy:spb_energy, $
                   typevector:aPlotInfo.typevector}
  widgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window -------------------------------------------------------
  papco_Panel_Realize, panel_base, widgetData
  crres_mea_panel_refresh
  papco_xmanager, 'crres_mea_panel', panel_base

;--- the returned data consists of the standard-data inherited from
;    papco_panel.pro and your new data. Stick to the tags provided in
;    aPlotInfo, there are ample variables for the user
  returnData={typevector:widgetData.typevector, $
              ioptions:widgetData.ioptions}
  additionalRData=papco_panel_ReturnData(widgetData)

  return, create_struct(ReturnData, additionalRData)
   
END

;******************************************************************************
;* PROCEDURE:     
;*     PRO crres_mea_panel_refresh, SL_ENERGY=SL_ENERGY
;* 
;* DESCRIPTION:  
;*     update the display of the editor-widgets
;* 
;* INPUTS:       
;*     none
;* 
;* KEYWORDS:
;*     only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*     have a look at PRO crres_mea_panel_event
;*
;* MODIFICATION HISTORY:       
;*     August 1999, written by R.Friedel
;******************************************************************************
PRO crres_mea_panel_refresh, SL_ENERGY_PITCH=SL_ENERGY_PITCH

  COMMON crres_mea_panel_data, widgetData
  COMMON crres_mea ;see m_crres_mea_commons.pro for contents of this block
  
  IF widgetData.typeVector(1) THEN BEGIN     ; differential
    WIDGET_CONTROL, widgetData.spb_energy, SENSITIVE=1
    WIDGET_CONTROL, widgetData.sl_energy_pitch, SENSITIVE=1
  ENDIF ELSE BEGIN			     ; integral
    WIDGET_CONTROL, widgetData.spb_energy, SENSITIVE=0
    WIDGET_CONTROL, widgetData.sl_energy_pitch, SENSITIVE=0
  ENDELSE

 IF widgetData.typeVector(2) THEN BEGIN     ; pitch angle
    WIDGET_CONTROL, widgetData.spb_energy, SET_BUTTON=0
    WIDGET_CONTROL, widgetData.spb_pitch, SET_BUTTON=1
    WIDGET_CONTROL, widgetData.spb_integral, SENSITIVE=1
    WIDGET_CONTROL, widgetData.spb_differential, SET_BUTTON=1
    widgetData.slider_displaysEnergy=1
    widgetData.YLINLOG=0
    papco_panel_Refresh,widgetData,/YSCALE
  ENDIF ELSE BEGIN			     ; energy range
    WIDGET_CONTROL, widgetData.spb_integral, SENSITIVE=0
    WIDGET_CONTROL, widgetData.spb_energy, SET_BUTTON=1
   WIDGET_CONTROL, widgetData.spb_pitch, SET_BUTTON=0    
    widgetData.slider_displaysEnergy=0
    widgetData.YLINLOG=1
    papco_panel_Refresh,widgetData,/YSCALE
  ENDELSE

; ---- Energy / Pitch - Slider ------------------------------------------------
  IF widgetData.slider_displaysEnergy THEN BEGIN ;slider shows energy range
    yarr=ech
    n_en=n_elements(yarr(*,0))  
    WIDGET_CONTROL, widgetData.sl_energy_pitch, $
      	            SET_SLIDER_MIN=1, SET_SLIDER_MAX=n_en, $
                    SET_VALUE=(widgetData.typeVector(3)<n_en)>1
    range=((widgetData.typeVector(3)<n_en)>1)-1
    rangeLabel='electrons ' + $
      strtrim(string(yarr(range,0), format='(F8.1)'),2) + ' - ' + $
      strtrim(string(yarr(range,1), format='(F8.1)'),2) + ' keV'
    if range eq 16 then rangeLabel= 'background protons'
    WIDGET_CONTROL, widgetData.lbl_energy_Pitch, $
        			 SET_VALUE=strcenter('energy range '+$
                                                     rangeLabel, 50)
  ENDIF ELSE BEGIN              ; slider shows pitch angle range
    rLow=varprt(pa_bins(widgetData.typeVector(3),0))
    rHigh=varprt(pa_bins(widgetData.typeVector(3),1))
    rangeLabel=strleft(rLow,5)+' - '+ strleft(rHigh,5)
    
    rangeLabel='Electron pitch angle range '+rangelabel+' degrees'
    WIDGET_CONTROL, widgetData.lbl_energy_Pitch, $
                    SET_VALUE=strcenter(rangeLabel, 50)
    WIDGET_CONTROL, widgetData.sl_energy_pitch, $
        	    SET_SLIDER_MIN=0, SET_SLIDER_MAX=18
  ENDELSE ; END energy/pitch slider  

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO crres_mea_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO crres_mea_panel_event, event
  
  COMMON crres_mea_panel_data, widgetData

  WIDGET_CONTROL,GET_UVALUE=uval, event.id
   
  IF strleft(uval, 8) EQ 'spb_relo' THEN BEGIN
    widgetData.ioptions(1)=fix(strfrom(uval, 10))
    return
  ENDIF 
   
  CASE uval OF
    'spb_integral' : BEGIN
      widgetData.typeVector(1)=0
      crres_mea_panel_refresh
    END
    'spb_differential' : BEGIN
      widgetData.typeVector(1)=1
      crres_mea_panel_refresh
    END
    'spb_energy' : BEGIN
      widgetData.typeVector(2)=0
      crres_mea_panel_refresh
    END
    'spb_pitch' : BEGIN
      widgetData.typeVector(2)=1
      crres_mea_panel_refresh
    END
    'sl_energy_pitch': BEGIN
      widgetData.typeVector(3)=event.value
      crres_mea_panel_refresh, /SL_ENERGY_PITCH
    END
    
    ELSE: widgetData=papco_panel_Event(event, widgetData)

   ENDCASE
    
END
