;******************************************************************************
;* file crres_mea_slice.pro
;*
;* this file contains the routines needed to enable the extra mouse features
;* provided for in papco. 
;*
;* Currently possible:
;*
;*    mouse zooms                        no routine here, in papco main
;*    mouse vertical slice               routine here
;*    mouse views slice (polar)          specialty slice - no routine here
;*    mouse views ephemeris              under development (POLAR only)
;*    mouse accesses level zero          routine here
;*    mouse writes paneldata             routine here
;*    mouse horizontal slice             routine here
;* 
;* Here the procedure names must again adhere to a strict naming convention so 
;* that they can be automatically called when needed.
;*
;* The slice types supported by these routines must be ENABLED by the user, so 
;* that papco can call them. This is done by setting the value of 
;* slice_type in draw_crres_mea.pro (contained in papcoadd_crres_mea.pro).
;* slice_type is defined in the following way:
;*
;*   slice_type:   Selects which slices are supported and for which slice
;*                 routines have been written.
;*                 This is an integer logical with 16 swittches. Currently
;*                 there are 5 papco slices. You may extend this by
;*                 further USER slices up to a total of 16 slices.
;*                     0 = disable all slices
;*                     1 = papco Zoom                 (2^0)
;*                     2 = papco View Slice           (2^1)
;*                     4 = papco Access level Zero    (2^2)
;*                     8 = papco Write Paneldata      (2^3)
;*                    16 = papco Hair Cursor          (2^4)
;*                    32 = papco horizontal Slice     (2^5)    
;*                    64 = USER slice A or not used   (2^6)
;*                   128 = USER slice B or not used   (2^7)
;*                    "     "     "   "  "  "    "      "    
;*                 32768 = USER slice Z or not used  (2^15)
;*                        
;*   You can support more than one type of slice by arithmetically adding the
;*   options, eg slice_type=1+4  support Zoom and Level Zero
;*
;* The "papco Zoom" function uses the mouse to define an area of a panel. 
;* Dragging the mouse with the left button pressed marks out a shaded rectangle
;* in the panel. The time range and y-range of the rectangle are read out in a
;* window and the user has the option of adopting these new ranges. The papco 
;* main common block for time (common mjdt, mjdt_start, mjdt_end) and y-range
;* (common yscale, yscl) are updated accordingly. Provided within PAPCO.
;*
;* The "papco Views Slice" function uses the time returned by papco and the
;* get_data_call specified for the crres_mea to read the data and to draw a 
;* slice at the clicked position. This is commonly used in spectra color plots 
;* to draw a spectrum at that time, or a pitch angle distribution etc, or to
;* call up any other plot at that time as specified by the user. 
;* -> user defined interface routine needed, template included here
;*
;* The "papco Accesses level Zero" uses the time passed and calls up another
;* widget application to read and work with level zero data. This application
;* must be entirely independent of papco and is only called from papco via
;* a mouse click in the panel required for the time required.
;* -> user defined interface routine needed, template included here
;*
;* The "papco Write Paneldata" function allows the user to read off data from 
;* panels using the mouse and to write that data to a file. This is a papco
;* supplied utility slice which by default only writes the time and y-value of
;* the mouse-click to file. The user can however interface his own routine to
;* add auxiliary data to the string written to file.
;* The output file is written in ASCII and has the same format as the .event
;* files used by the event times utility module. The files can thus be used
;* in the events time module to mark events on panels.
;* -> user defined interface routine possible, template included here
;*
;* The "papco Hair Cursor" function draws a large hair cursor and outputs
;* x,y,and z corrdinates onto the plot.  Provided within PAPCO.
;*
;* If your new data contains several sub-data types, one slice and/or levelzero
;* routine has to be provided for each type, since in general they would do
;* different things. See 
;* $papco_modules\crres_meb\papco_interface\crres_slices.pro  
;* for such an example.
;*
;*      PRO DrawSlice_crres_mea, panelNr, seconds, yValue
;*        -> Draw a slice for crres_mea
;*
;*      PRO levelzero_crres_mea, seconds, yValue
;*        -> This procedure calls the widget for levelzero data access 
;*           for this plottype.
;*
;*      PRO WriteSlice_crres_mea, panelNr, seconds, yValue
;*        -> This procedure calls the widget for writing panelinfo data to file
;*    
;* MODIFICATION HISTORY:       
;*      December 1995, written by R.Friedel at Goddard
;*      modified may 1997, Reiner Friedel
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO DrawSlice_crres_mea, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a slice for crres_mea
;*      calling a separate window for the plot.
;*      slice_type in draw_crres_mea must be set to +2 to enable this feature.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      December 1995, written by R.Friedel at Goddard
;******************************************************************************
PRO DrawSlice_crres_mea, panelNr, seconds, yValue, OUTPUT=OUTPUT

  COMMON PAPCO_DRAWSLICE, sliceWidgetData
  COMMON PLOT_COMPOSER, widgetData

; add in here any other common blocks you would need to make your plot data
; available. The common block included here is a dummy for illustration.
; papco uses the the getdatacall you set in papco_add_crres_mea to read the
; data you need for the slice. As long as you stay in one panel with the 
; mouse, the data is not read again. When you swittch panels, the new data is 
; read. 
  common crres_mea_data, input_header, input_data
  common crres_mea ;see m_crres_mea_commons.pro for contents of this block
  common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
  
; keep all data needed for slice in common block for quick access when
; dragging the mouse in panel
  common drawslice_crres_mea, yarr,zmat, extra_plotPar, utitle
  
  time=input_data.time           ; dummy, remove once your own is here

  WIDGET_CONTROL,/HOURGLASS
  thisPlot=widgetData.plotsDrawn(panelNr)

; use the same typevector, swittch etc as is used in the call to your plot
; routine. thisPlot contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 
  thisPlot=widgetData.plotinfos(panelNr)
  
  type=thisPlot.typeVector
  ioptions=thisPlot.ioptions
  
  thisPlot=widgetData.plotsdrawn(panelNr)

; set the panel window for the slice to full window
  !p.position=[0.15,0.1,0.8,0.9]

; select the index for the time chosen out of the time array
  DiffTimes=ABS(time - seconds)        ; this sets timeIndex to the index with
  result=MIN(DiffTimes, timeIndex)     ; t90(index) is closest to 'seconds'

; these two info strings are user-definable output strings to label the slice 
; window. To calculate the time use the start time in mjdt and the seconds 
; passed in. If your plot uses the papco philosophy then the time axis is in 
; running seconds since the start of the start day.
;
; However, you of course can use your own time format, then the information
; in seconds is whatever your time format is. papco provides a function
; to return a verbose time for ANY time format.    
; papco always knows the start end time of a plot from the time common
; block, and the start/end value of the x_axis shouldalways be stored in 
; papco by the user's plot routine:;
;
;   common coordinateSystems, plotted_x, plotted_y  
;
;   plotted_x = !x
;   plotted_y = !y     -   after the user has set up his axis.
;
; If this is done properly then the call
;   
;   time_str=papco_cursor_time(seconds,panelNr)
;   
; should return the correct time.
   
  time_str=papco_cursor_time(seconds,panelNr)
  time_mjdt=papco_cursor_time(seconds,panelNr,/mjdt)
   
  info1='You got to the default slice window'
  info2=string(time_str, yValue, $
               format="('time: ',A20,'   Y-value: ',f9.2)")

; select data for the slice - only once, first time panel is clicked in
  if sliceWidgetData.lastPanelDrawn NE panelNr then begin  
  
  utitle='CRRES MEA!COrbit '+string(input_header.orbit,format="(i4.4,'!C')")
  counts=transpose(input_data.counts)
  
  IF type(2) THEN BEGIN         ; pitch angle
    zmat=fltarr(input_header.ndat,19)
    for i=0, input_header.ndat-1 do zmat(i,*)=input_data(i).data(type(3),*)
    index=where(counts ne 0,c)
    zmat(index)=zmat(index)/counts(index)
    index=where(counts eq 0,c)
    zmat(index)=1e-20
    yarr=input_header.pa_bins
    n_channels=19
    ;utitle=utitle+'Elec. '+strleft(varprt(ech(type(3),0)),6) +'-'+$
    ;  strleft(varprt(ech(type(3),1)),6)+' keV'
    utitle=utitle+'Elec. '+string(fix(ech(type(3)-1,*)), $
                                      format="(i4,'-',i4,' keV')")
    uytitle='Pitch Angle (deg)'
    resy=5
  ENDIF ELSE BEGIN			     ; energy range
    if ioptions(2) eq 1 then begin ;spin average, average all pitch angles
      zmat=fltarr(input_header.ndat,17) & spc=fltarr(17) 
      for j=0,input_header.ndat-1 do begin
        for i=0,18 do begin
          count=input_data(j).counts(i) > 1e-20
          spc(*)=spc(*)+[input_data(j).data(0:15,i),input_data(j).data(17,i)] $
                      / count   
        endfor  
        index=where(spc ne 0,c)
        if c ne 0 then spc(index)=spc(index) / 19.0
        zmat(j,*)=spc
        spc(*)=0  
      endfor  
      index=where(zmat eq 0,c)
      if c ne 0 then zmat(index)=1e-20
      pa_range='Spin Average'
    endif else begin            ;select pitch angle
      zmat=transpose(input_data.data(*,type(3)))
      counts=counts(*,type(3))
      index=where(counts ne 0,c)
      for i=0,17 do zmat(index,i)=zmat(index,i)/counts(index)
      index=where(counts eq 0,c)
      for i=0,17 do zmat(index,i)=1e-20
      dummy=fltarr(input_header.ndat,17)
      for i=0, input_header.ndat-1 do begin
        dummy2=transpose(zmat(i,0:15))
        dummy(i,*)=[dummy2,zmat(i,17)]
      endfor
      zmat=temporary(dummy)
      par=input_header.pa_bins(type(3),*)
      pa_range=strleft(varprt(par(0)),5)+'-'+ strleft(varprt(par(1)),5)
    endelse  
    
    yarr=[ech(0:15,*),ech(17,*)]
    n_channels=17  
    utitle=utitle+'Elec. '+pa_range
    uytitle='Energy (keV)'
    resy=0.005
  ENDELSE
    
  uztit='flux'
  
  yst=thisPlot.ZSCL_MIN & yen=thisPlot.ZSCL_MAX
  xst=thisPlot.YSCL_MIN & xen=thisPlot.YSCL_MAX
  zst=1  & zen=2
  xlog=thisPlot.YLINLOG  &  ylog=thisPlot.ZLINLOG  & zlog=0
  uxtitle=uytitle  &  uytitle=uztit &  uztitle='' 
  ytickno=0  
  
  extra_plotPar={yrange:[yst, yen], xrange:[xst, xen], zrange:[zst,zen], $
                 xlog:xlog, ylog:ylog, zlog:zlog, $
                 xminor:0, xticks:0, xstyle:1, $
                 xtitle:uxtitle, ytitle:uytitle, ztitle:uztitle, $
                 yticks:ytickno, color:1, psym:-4}  
  
  endif ; make data arrays etc.
  
  result=str_sep(time_str,' ') & date=result(0)
  data_time=time(timeIndex) mod 86400
  data_time_str= date+convert_secondsToTime(data_time)
  info1=data_time_str
  title='Mouse time '+time_str+'!CData time '+data_time_str
  x=(yarr(*,0)+yarr(*,1))/2.0
  y=fltarr(n_elements(x))
  
  ;average more than one distribution if needed
  if ioptions(3) eq 0 then begin
    y(*)=zmat(timeIndex,*) 
    av_no=1
  endif else begin  
    for i=ioptions(3)*(-1),ioptions(3) do begin
      idx=((timeIndex+i) > (-1)) < (input_header.ndat-1)
      ;print, i, idx
      y(*)=y(*)+zmat(idx,*)
    endfor  
    av_no=ioptions(3)*2+1
    y=y / av_no
  endelse  
  av_str='!Ct (sample)='+varprt(av_no*30)+'s'
  
  if keyword_set(OUTPUT) then begin
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format if its time
;      CHANNELS      an (ch,2) array giving the channel start/end
;      X_LABEL       a string with the x-axis label
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.    
    message,'Writing slice data out to file',/cont
    description='CRRES MEA slice at '+data_time_str
    x_label=extra_plotPar.xtitle
    y_label=extra_plotPar.ytitle
    rs_label=utitle
    nodata=0
    format='TXT'
    fln_t_str=strdel(data_time_str,'/') & fln_t_str=strsubst(fln_t_str,' ','_')
    file='crres_mea_slice_'+varprt(input_header.ORBIT)+'_'+fln_t_str
    papco_write_data, x,y, $
      DESCRIPTION=description, TIME_LABEL=time_label, $
      X_LABEL=x_label, Y_LABEL=y_label, $
      RS_LABEL=rs_label, NODATA=nodata, $
      FORMAT=format, FILE=file
    message,'Write O.K.',/cont
  endif else begin
    erase
    plot,x,y,title=title,_extra=extra_plotPar
    right_side_label,panel,utitle+av_str
  endelse
  
; these calls put your info strings in the provided labels of the slice widget
  WIDGET_CONTROL, sliceWidgetData.lbl_slice1, SET_VALUE=info1
  WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO levelzero_crres_mea, seconds, yvalue
;*                                     
;* DESCRIPTION:  
;*      This procedure calls the widget for level zero data access 
;*      for this plottype. 
;*      slice_type in draw_crres_mea must be set to +4 to enable this feature.
;*
;* INPUTS:       
;*      seconds           T90 time at which mouse is clicked
;*      yvalue            y-value at which mouse is clicked
;*
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      levelzero_crres_mea, seconds, yvalue
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel at Goddard
;******************************************************************************
pro levelzero_crres_mea, seconds, yvalue

; The actual procedure is very dependent on what the level zero widget 
; application needs. An example is in $papco_MODULES/addpapco_swe_plots.pro.
; Provided as parameters to be passed to the new application, are the values
; of the mouse pointer when clicked. 
;
; The main widget remains active and can be swittched to (e.g. to get a new
; time from the graph) by using a "Parent" button in the new application
; which then executes the line
;            current_appl=xregistered('papco_main') 
; to swittch back to the papco main calling widget. 

; call the levelzero widget.  
;   crres_mea_levelzero, seconds, yvalue

; the following line is printed on your xterm as a check to see if papco
; correctly called this routine. If there is a problem make sure your naming 
; convention is O.K. : This routine MUST be called levelzero_crres_mea, 
; where crres_mea is the name of the panel you have set in papcoadd_crres_mea.

    print, 'got to the levelzero interface  ',seconds, yvalue

END


;******************************************************************************
;* PROCEDURE:     
;*      PRO WriteSlice_crres_mea, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a Write Paneldata slice for crres_mea
;*      calling a seperate window for the widget
;*      slice_type in draw_ceppad must be set to +32 to enable this feature.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      READDATA    if set the data needs to be re-read from file
;* 
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      may 1997, written by R.Friedel
;******************************************************************************
PRO WriteSlice_crres_mea, panelNr, seconds, yValue,READDATA=READDATA

   COMMON PAPCO_WRITE_PANELDATA, sliceWidgetData
   COMMON PLOT_COMPOSER, widgetData

; THIS TEMPLATE ACTUALLY DOES NOTHING, ONLY THE DEFAULT WRITE SLICE 
; FUNCTIONALITY WILL WORK (WRITE TIME AND Y-VALUE OF MOUSECLICK TO FILE). THE 
; EXAMPLE HERE IS TO SHOW HOW ONE COULD ADD ADDITIONAL INFORMATION.

; add in here any other common blocks you would need to make your additional
; data available (ie ephemeris).

; papco uses the the getdatacall you set in papco_add_crres_mea to read the
; plotdata For your panel. As long as you stay in one panel with the 
; mouse, the data does not need to be read again (READDATA=0). When you swittch
; panels, new data needs to be read (READDATA=0). It's up to the user to read 
; in the data he needs here. 

;   common crres_mea_data, input_header, input_data
;   common crres_mea_ephem, crres_mea_ephem
;   common mjdt, mjdt_start, mjdt_end    ;common time limits in mjdt

   WIDGET_CONTROL,/HOURGLASS
   thisPlot=widgetData.plotsDrawn(panelNr)

; use the same typevector, swittch etc as is used in the call to your plot
; routine. thisPlot contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 

   thisPlot=widgetData.plotsDrawn(panelNr)
   type=thisPlot.typeVector
   getdatacall=thisPlot.get_data_call

;read the data if needed

;   if READDATA then begin 
;      callgood=execute(getdatacall)
;      r_crres_mea_ephem
;   endif

; select the index for the time chosen out of the time array

;   DiffTimes=ABScrres_mea(.time - seconds)
                                        ; this sets timeIndex to the index 
;   result=MIN(DiffTimes, tidx)         ; closest to 'seconds'

; these strings are user-definable output strings to label the slice window.
; The "write Panelinfo" window papco supplies has 6 lines of user-definable
; text. 

   info1='User label 1'
   info2='User data  1'
   info3='User label 2'
   info4='User data 2'
   info5='User label 3'
   info6='User data 3'

; Time and y-value of mouse click are automatically put into labels.
; The calls below put your info strings in the provided user labels (max 6).
;
; NB NB NB:
; However, only info2, info4, info6 are written to file in the form
; info2+info4+info6.
;
; The idea is that info1 will have the labels for the data in info2, info3 the
; labels for the data in info4, and info5 the labels for the data in info6.
; Time and y-value of mouse click are written first, then info2+info4+info6
; followed by a user defined comment from the Write Paneldata window.

   WIDGET_CONTROL, sliceWidgetData.lbl_slice1, SET_VALUE=info1
   WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2
   WIDGET_CONTROL, sliceWidgetData.lbl_slice3, SET_VALUE=info3
   WIDGET_CONTROL, sliceWidgetData.lbl_slice4, SET_VALUE=info4
   WIDGET_CONTROL, sliceWidgetData.lbl_slice5, SET_VALUE=info5
   WIDGET_CONTROL, sliceWidgetData.lbl_slice6, SET_VALUE=info6

END

function dewc_day2time, a
  
  b=a*24.0
  hr=fix(b)
  
  c=(b-hr)*60.0
  min=fix(c)
  
  d=(c-min)*60.0
  sec=fix(d)
  
  return,string(hr,min,sec,format="(i2.2,':',i2.2,':',i2.2)")
  
end  
  
  
  
