;******************************************************************************
;* file p_crres_mea_spec.pro
;*
;* This file is a template for a plot routine which can be used by PAPCO.
;* It includes all the periphery needed. This routine makes use of a
;* color spectrogram routine that is supplied with PAPCO:
;* papco_XX/papco_lib/papco_plot_colorspec.pro. This routine should
;* meet all the requirements for the display of 3-D data (tyhe third
;* dimension being color). This routine also adheres to the PAPCO color scheme.
;* You of course can use your own color plot rotuine if desired!
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_crres_mea_spec, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (and/or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure PAPCD_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to make use of
;*              an extra common block.
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_crres_mea, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_crres_mea, panel, plotinfo, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

; The following common blocks are needed for PAPCO:
COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
COMMON yscale, yscl			;man/auto yscaling
COMMON zscale, zscl			;man/auto zscaling
COMMON shift_label, down             ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_crres_mea. 
COMMON crres_mea_data, input_header, input_data
COMMON crres_mea ;see m_crres_mea_commons.pro for contents of this block
COMMON crres_mea_slice, time, yarr, zmat, data, $
                        extra_plotPar, utitle, uytitle, uztitle
COMMON crres_mea_plot, e1, e2, p1, p2, n_ch, n_sec, n_det, ndat, nodata

COMMON papco_color_names
cols=[black, red, green,  blue, magenta, cyan, burgundy, $
      olive, dark_green, teal, royal_blue, violet]

control = *plotinfo.USR_PTR1
control.time_av = control.time_av > 30
e1 = fix(control.ENERGY_AVERAGE_FROM_VAL)
e2 = fix(control.ENERGY_AVERAGE_TO_VAL)
p1 = fix(control.PITCH_AVERAGE_FROM_VAL)
p2 = fix(control.PITCH_AVERAGE_TO_VAL)
n_ch = 18
n_det = 1

IF keyword_set(OUTPUT) THEN output=OUTPUT ELSE output=0

; restrict data to actual time range requested - this makes for faster zooming
; input data is in TAI. For plotting, always start at zero to get maximum
; resolution (plot works in float only, large values get rounded!) 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),ndat)
IF ndat NE 0 THEN data=input_data(index) ELSE BEGIN
    msg = 'No data to plot' & GOTO, noplot
ENDELSE

xut1=0  &  xut2=tai_2-tai_1 & time = data.tai-tai_1
nodata = -99.9

;make title strings for pa and ek ranges
e1 = e1 >  0 <  (n_ch-1) & e2 = e2 >  0 <  (n_ch-1) 
tit_ek = 'EK '+ strcompress(string(ech(e1,0),ech(e2,1),$
                 format = "(f6.1,'-',f6.1)"), /REMOVE_ALL)
tit_pa = 'PA '+ varprt(p1)+'-'+varprt(p2)

utitle='CRRES MEA'
uztit= 'diff. flux'

CASE 1 OF 
    (control.pl_type EQ 0): BEGIN ;E @ Pa: Sum EK, choose PArange
        idx = [findgen(16), 17] & n_ch = n_elements(idx)
        yarr = ech(idx, *)
        m_crres_mea_zmat, control
        zmat = zmat(*, idx)
        utitle=utitle+'!CElec. '+tit_pa
        uytitle = 'Energy (keV)'
    END 
    (control.pl_type EQ 1): BEGIN ;Pa @ E: Average Pa, E range chosen
        m_crres_mea_zmat, control        

        utitle=utitle+'!CElec. '+tit_ek
        uytitle = 'Pitch Angle (deg)'
    END 
ENDCASE

utitle = utitle+'!CTime bin: '+varprt(control.time_av)+' s'

GOTO, m_plot
  

    uytitle='Pitch Angle (deg)'
    if type(3) ne 17 then ch_str=string(fix(ech(type(3)-1,*)), $
                                        format="(i4,'-',i4,' keV')") $
      else ch_str='Background'
    utitle=utitle+'Elec. '+ch_str
    resy=5  


m_plot:

yst=yarr(0,0) & yen=yarr(n_ch-1,1)

panelset,panel	               ;sets the panel position viewport
down=0

IF control.spec EQ 1 THEN BEGIN  ; do spec

    PAPCO_Set_SubTable, SUBTABLE ;sets the papco color table index

    IF (yscl(panel(0),0) EQ 1) THEN BEGIN 
        yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
        message,'Using manual y-scale:'+varprt(yst)+' '+varprt(yen), /info
    ENDIF ELSE BEGIN 
        yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
    ENDELSE 

    exclude = plotinfo.MANUALZSCALING

    IF (zscl(panel(0),0) EQ 1) THEN BEGIN 
        zst=zscl(panel(0),1)  &  zen=zscl(panel(0),2)
    ENDIF ELSE BEGIN 
        papco_autorange,zmat,zst,zen, $
            log=zscl(panel(0),3), exclude=exclude, nodata=nodata
        zscl(panel(0),1)=zst  &  zscl(panel(0),2)=zen
    ENDELSE

    papco_y_label, yst, yen, log=yscl(panel(0),3), VALID = valid, /verb
    yst = yst*0.99 & yen = yen*1.01
   
    extra_plotPar={yrange:[yst, yen],xrange:[xut1, xut2], zrange:[zst,zen], $
               ylog:yscl(panel(0),3), zlog:zscl(panel(0),3),  $
               xtickformat:'noticks', $;ztitle:uztit, $
               ztickformat:'papco_color_bar_log_ticks'}

    ; add keyword structure set here with the one passed in
    extra_plotPar=create_struct(extra_plotPar, extra_par)

    papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

    resx=control.time_av/3.0
    IF yscl(panel(0),3) THEN resy=(alog10(yen)-alog10(yst))/100 $
                        ELSE resy=(yen-yst)/100

    papco_plot_colorspec,zmat,time,yarr, nodata = nodata, $
      resx=resx, resy=resy, $
      special_colors=special_colors, xstyle=5, ystyle=5, $
      _extra=extra_plotPar
    papco_color_bar,_EXTRA=extra_plotPar

    ;plot color bar label next to color bar.Use scalable routine!
    papco_colorbar_label, panel, uztit, /ROT90  
 
    ; plot y-axis label at left of plot. Use scalable routine!  
    left_side_label, panel, uytitle, /ROT90  

    right_side_label, panel, utitle, /ROT90, _extra=extra_plotpar

    ; store the coordinate information into common block
    plotted_x = !x  &  plotted_y = !y

ENDIF 

; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
IF KEYWORD_SET(OUTPUT) THEN BEGIN
    message,'Writing plot data out to file',/cont
    description='crres_mea template sample data - Energy spectra, 10 channels'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, time, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
ENDIF 
   
return 

;jump to here if no data to plot 
noplot: 
get_err_no = 10 & get_err_msg = msg
message, get_err_msg, /cont
papco_Draw_ErrorPanel, panel, plotinfo.panelkind+': ' + get_err_msg
plotted_x = !x  &  plotted_y = !y

END


;******************************************************************************
;* PROCEDURE: m_crres_mea_zmat
;*
;* DESCRIPTION:  Make plot arrays for pitch mea data
;******************************************************************************
PRO m_crres_mea_zmat, control

COMMON crres_mea_slice    ;see p_crres_mea for definition
COMMON crres_mea_plot     ;see p_crres_mea for definition

;first, make a new time array for the time average range chosen. 
t_s = floor(time(0) / control.time_av) * control.time_av
t_e = ceil(time(ndat-1) / control.time_av) * control.time_av
n = (t_e-t_s)/control.time_av
new_t = fltarr(n, 2)
FOR i = 0l, n-1 DO BEGIN
    new_t(i, 0) = t_s + i*control.time_av
    new_t(i, 1) = t_s + (i+1)*control.time_av
ENDFOR

CASE control.pl_type OF

    0: BEGIN                    ;E @ Pa: Sum energy, Pa range chosen.
        zmat = fltarr(n, n_ch) & zmat(*) = nodata & numb = fltarr(n, n_ch)
        message, 'Binning data in time and pitch angle range', /info
        FOR i = 0l, n-1 DO BEGIN ;loop new time steps
            idx = where((time GE new_t(i, 0)) AND (time LT new_t(i, 1)) AND $
                        (data.pa GE p1) AND (data.pa LT p2), c)
            IF c NE 0 THEN BEGIN ;average spectra for matches found
                new_spec = fltarr(n_ch)
                new_numb = fltarr(n_ch)
                FOR j = 0, c-1 DO BEGIN
                    spec = data(idx(j)).data
                    idx2 = where(spec NE 0, c2)
                    IF c2 NE 0 THEN BEGIN
                        new_spec(idx2) = new_spec(idx2)+spec(idx2)
                        new_numb(idx2) = new_numb(idx2) + 1
                    ENDIF 
                ENDFOR 
                zmat(i, *) = new_spec
                numb(i, *) = new_numb
            ENDIF     

        ENDFOR
        
        x1 = where(numb EQ 0, c1, COMPLEMENT = x2, NCOMPLEMENT=c2)
        IF c2 NE 0 THEN zmat(x2) = zmat(x2) / numb(x2)
        IF c1 NE 0 THEN zmat(x1) = nodata

    END

    1: BEGIN                    ;Pa @ E: Average Pa, E range chosen.
        yarr = papco_pitch_bin(abs(control.pa_bin),  PA_IDX = pa_idx)
        n_ch = n_elements(yarr(*, 0))
        zmat = fltarr(n, n_ch) & numb = fltarr(n, n_ch)
  
        ;first sum the energy range chosen. Exclude ch=16 (bckgrng channel)
        message, 'Summing energy range', /info
        f = fltarr(ndat)
        FOR i = 0l, ndat-1 DO BEGIN
            FOR e = e1, e2 DO BEGIN
                IF e EQ 16 THEN CONTINUE
                f(i) = f(i)+data(i).data(e)
            ENDFOR 
        ENDFOR

        idx = where(data.pa EQ 180, c) & IF c NE 0 THEN data(idx).pa = 179.9
      
        message, 'Binning data in time and pitch angle range', /info
        FOR i = 0l, n-1 DO BEGIN ;loop new time steps
            idx = where((time GE new_t(i, 0)) AND (time LT new_t(i, 1)), c)
            IF c NE 0 THEN BEGIN      
                FOR j = 0, c-1 DO BEGIN
                    pa = pa_idx(data(idx(j)).pa)
                    zmat(i, pa) = zmat(i, pa) + f(idx(j))
                    numb(i, pa) = numb(i, pa) + 1
                ENDFOR 
            ENDIF     

        ENDFOR
        
        x1 = where(numb EQ 0, c1, COMPLEMENT = x2, NCOMPLEMENT=c2)
        IF c2 NE 0 THEN zmat(x2) = zmat(x2) / numb(x2)
        IF c1 NE 0 THEN zmat(x1) = nodata                
                
    END

ENDCASE

time =  new_t

END 
