;******************************************************************************
;*  NAME:        r_crres_mea
;*
;*  DESCRIPTION: Routine to read crres_mea files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
pro r_crres_mea, in_orbit, VERBOSE=VERBOSE

COMMON crres_mea_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg

get_err_no=0 & get_err_msg='crres_mea read succesfully'

; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
input_header=0  &  input_data=0   

; loop through from start to end time concatonating data files into one
; array. Here data are stored by orbit, so we need to find the orbit
; range corresponding to the start and end times.
time=0  &  no_files_read=0

; check if orbit is non-zero, then read that orbit 
; - otherwise get list of orbits to read
IF in_orbit EQ 0 THEN orbit_list=papco_getOrbitList('CRRES') $
    ELSE orbit_list=[in_orbit]

FOR i=0,n_elements(orbit_list)-1 DO BEGIN 
    orbit=orbit_list(i)

    ; read one orbit of data
    r_crres_mea_oneorbit, orbit, VERBOSE = verbose
  
    ; check for an error in get_error - and then get the file remotely!
    ; comment this section out if you do not want this feature.
    if get_err_no ne 0 then begin      
        ;c_url=papco_getenv('CRRES_MEA_FTP_ARCHIVE')
        ;c_file=fln+'.idl'
        ;c_download_dir=path
        ;papco_get_remote_data, c_url, c_file, c_download_dir
        ;check if file "arrived" and then read it
        ;if get_err_no eq 0 then $
        ;    r_crres_mea_onefile, path+fln, /VERBOSE $
        ;else begin
        ;    get_err_msg='No data locally, remote get failed!C  ' + get_err_msg
        ;return
        ;endelse  
    ENDIF 
     
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
        if no_files_read eq 0 then begin   ;start the array buffer
            input_data_buf=temporary(input_data)
        endif else begin                   ;otherwise concatonate arrays
            input_data_buf=[temporary(input_data_buf),temporary(input_data)]
        endelse        
        input_data=0                     ;clear array for reuse
        no_files_read=no_files_read+1
    ENDIF 

ENDFOR                    ;next orbit

   
; check if any files were read   
if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
endif else begin
    get_err_no=1
    get_err_msg='no files found for timerange'
    return
ENDELSE  

END 


;******************************************************************************
PRO r_crres_mea_oneorbit, orbit, VERBOSE = verbose
  
COMMON crres_mea_data, input_header, input_data
COMMON crres_eph_data, crres_eph_header, crres_eph_data
COMMON get_error, get_err_no, get_err_msg
  
get_err_no=0 & get_err_msg='crres_mea read succesfully'
   
IF NOT (papco_check_data_env('CRRES_MEA_DATA', path=PATH)) THEN return

; check if file already exists
fln1 = 'cr_mea_' + string(orbit,format="(i4.4)") + '_p.idl'
result=findfile(path+fln1,count=c)
IF c EQ 1 THEN BEGIN 
    restore, path+fln1, verbose = verbose
    message, fln1, /info
    return
ENDIF 

; check if file needs to be extracted from archive  
fln2 = 'cr' + string(orbit,format="(i4.4)") + 'f.csv'
result=findfile(path+fln2, count=c)

IF c EQ 0 THEN BEGIN
    CASE 1 OF
        (orbit le 0100) AND (orbit ge 0006): zip_fln='crres_0006_0100.zip'
        (orbit le 0198) AND (orbit ge 0101): zip_fln='crres_0101_0198.zip'
        (orbit le 0290) AND (orbit ge 0198): zip_fln='crres_0198_0290.zip'
        (orbit le 0442) AND (orbit ge 0416): zip_fln='crres_0416_0442.zip'
        (orbit le 0539) AND (orbit ge 0443): zip_fln='crres_0443_0539.zip'
        (orbit le 0634) AND (orbit ge 0540): zip_fln='crres_0540_0634.zip'
        (orbit le 0740) AND (orbit ge 0635): zip_fln='crres_0635_0740.zip'
        (orbit le 0834) AND (orbit ge 0741): zip_fln='crres_0741_0834.zip'
        (orbit le 0925) AND (orbit ge 0835): zip_fln='crres_0835_0925.zip'
        (orbit le 1013) AND (orbit ge 0926): zip_fln='crres_0926_1013.zip'
        (orbit le 1067) AND (orbit ge 1014): zip_fln='crres_1014_1067.zip'
        else: begin
            get_err_no=1
            get_err_msg='No data for orbit '+ string(orbit,format="(i4.4)")
            message, get_err_msg, /info
            return
        END      
    ENDCASE 
    message,'extracting orbit data from archive file '+zip_fln, /cont
    print,'unzip '+path+'archive/'+zip_fln+' '+fln2+' -d '+ path
    spawn,'unzip '+path+'archive/'+zip_fln+' '+fln2+' -d '+ path
    unzip_flag = 1
ENDIF ELSE unzip_flag = 0

;read data and make idl saveset.

in_str=''  &  in_arr=fltarr(26)
openr, unit, path+fln2, /get_lun
;read in first line - header for columns
readf,unit,in_str
header=str_sep(in_str,',')
for i=0,n_elements(header)-1 do header(i)=strdel(header(i),'.')

dummy = {CRRES_MEA, tai:0.0d, time:0.0, data:fltarr(18), pa:0.0}
input_data = replicate(dummy, 80000)
ndat = 0l
message,'Orbit '+varprt(orbit)+', reading ascii data...', /info
WHILE NOT eof(unit) DO BEGIN 
    readf,unit,in_arr
    input_data(ndat).time = in_arr(0)
    input_data(ndat).data = in_arr(1:18)
    input_data(ndat).pa = in_arr(19)
    ndat = ndat+1
ENDWHILE 
ndat = ndat-1
close,unit  &  free_lun, unit

input_data = input_data(0:ndat)

;get time range for orbit
info = papco_getOrbitInfoFor('CRRES', orbit)
;get TAI of start time
year = 1900+info.year & doy = info.day & sec = info.xut1
utc_start = doy2utc(doy, year) & utc_start.time = sec*1000.0d
tai_start = utc2tai(utc_start)
message, 'Orbit start time: '+tai2utc(tai_start, /ecs), /info
FOR i = 0l, ndat-1 DO input_data(i).tai = tai_start+input_data(i).time-sec

IF unzip_flag THEN BEGIN         ;delete unziped file
    file_delete, path+fln2, /verbose
ENDIF 

; make input_header info structure
input_header={CRRES_MEA_HEADER, orbit:orbit, ndat:ndat }
                
save,file=path+fln1,input_header,input_data, /verb, /compress
  
END   
