;******************************************************************************
;* FILE:
;*    crres_meb_overview_panel.pro
;* 
;*    This file belongs to the papco-project. See $papco_XX/papco/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains a demonstration on how to implement a papco-
;*    panel-editor. It makes use of all the features provided by 
;*    $papco_XX/papco_panel.pro (a collection of standard papco widgets)
;*    
;*    It constructs an editor for a very simple papco-plot, that has only
;*    default-functionality.
;*
;* FILES YOU MAY WANT TO READ (under $papco_XX/papco_doc)
;*    papco_conventions.text - which conventions were used when developing ?
;*    papco_variables.text   - what structures and global variables
;                              are used by which file ?
;*
;* COMMON-BLOCKS:
;*    papco_PANEL_TEMPLATE_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    papco_PANEL_EDITOR-structure (see $papco_XX/papco_panel.pro for a
;*    description). 
;*       {papco_PANEL_TEMPLATE_DATA, 
;*	     ; imagine the papco_PANEL_EDITOR-structure was inserted...
;*	      
;*    For a more complete description see $papco_DOC/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  papco_PANEL_TEMPLATE, aPlotInfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*       a=papco_getPlotInfoStruct()   which creates the structure
;*                                     containing the plot information 
;*	 a=crres_meb_overview_panel(a)
;*       which calls up the window.
;*       Interrogating a with help,a,/str shows the contents. This is a good
;*       way of testing your own panel editor. 
;*
;* MODIFICATION HISTORY:       
;*     august 1995, written by A.Keese
;*     december 1995, modified by R.Friedel at Goddard
;*     april 1996, modified by R.Friedel
;*     march 1997, modified by R.Friedel
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  crres_meb_overview_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_PLOTINFO-type
;*                      this structure is edited in the panel  editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at last some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I 
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      swittch:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, swittch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          swittch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commincate between your
;*     panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which papco supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      papco internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $                       
;*                      xut2:0.0}                           
;*
;*                  
;*      For an enhanced description read 
;*                          $papco_XX/papco_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  crres_meb_overview_panel, aPlotInfo, GROUP=group, ACTION=ACTION

   COMMON crres_meb_overview_panel_DATA, widgetData

; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
   panel_base=papco_widget_base(title=aPlotInfo.panelKind,$
                                column=1, $
                                GROUP_LEADER=group)

; All papco_panel_Add* routines are papco panel widgets that can be included
; in the panel editor or not. They provide functionality that papco supports.
; They dynamically extend the panel editors widget structure, and all the 
; events they generate are supported by papco_panel_Event, transparent to the
; user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
   widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
; --- Electron / Ion ----------------------------------------------------------
   elec_ion_bt_base=WIDGET_BASE(panel_base, /FRAME, ROW=1, /EXCLUSIVE) 
   spb_electrons=WIDGET_BUTTON(elec_ion_bt_base, $ 
			VALUE='electrons', $
			UVALUE='spb_electrons')
   spb_ions=WIDGET_BUTTON(elec_ion_bt_base, $
			VALUE='ions', $
			UVALUE='spb_ions')

   IF aPlotInfo.typeVector(0) EQ 0 THEN BEGIN
      WIDGET_CONTROL, spb_electrons, SET_BUTTON=1
      WIDGET_CONTROL, spb_ions, SET_BUTTON=0
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_electrons, SET_BUTTON=0
      WIDGET_CONTROL, spb_ions, SET_BUTTON=1
   ENDELSE

; --- Integral / differential -------------------------------------------------
   int_diff_base=WIDGET_BASE(panel_base, /FRAME, ROW=1, /EXCLUSIVE)
   spb_integral=WIDGET_BUTTON(int_diff_base, $
			VALUE='Integral', $
			UVALUE='spb_integral')
   spb_differential=WIDGET_BUTTON(int_diff_base, $
			VALUE='Differential', $
			UVALUE='spb_differential')

   IF aPlotInfo.typeVector(1) EQ 0 THEN BEGIN
      WIDGET_CONTROL, spb_integral, SET_BUTTON=1
      WIDGET_CONTROL, spb_differential, SET_BUTTON=0
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_integral, SET_BUTTON=0
      WIDGET_CONTROL, spb_differential, SET_BUTTON=1
   ENDELSE

; --- Plot swittches -----------------------------------------------------------
   swittch_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=1)

   smooth_base=WIDGET_BASE(swittch_base, /FRAME, ROW=1, /EXCLUSIVE)

   smoothNames=['no smooth', $
              'smooth vertically', $
              'smooth horizontally', $
              'smooth both']
   spb_type=lonarr(N_ELEMENTS(smoothNames))

   FOR i=0, N_ELEMENTS(smoothNames)-1 DO $
      spb_type(i)=WIDGET_BUTTON(smooth_base, $
			VALUE=smoothNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))

   WIDGET_CONTROL, spb_type(aPlotInfo.ioptions(17)), SET_BUTTON=1

   fill_base=WIDGET_BASE(swittch_base, /FRAME, ROW=1, /EXCLUSIVE)

   fillNames=['no fill', $
              'fill missing data', $
              'fill missing orbit', $
              'fill both']
   spb_fill=lonarr(N_ELEMENTS(fillNames))

   FOR i=0, N_ELEMENTS(fillNames)-1 DO $
      spb_fill(i)=WIDGET_BUTTON(fill_base, $
			VALUE=fillNames(i), $
			UVALUE='spb_fill'+strtrim(string(i), 2))

   WIDGET_CONTROL, spb_fill(aPlotInfo.ioptions(18)), SET_BUTTON=1

   orbitlabel_base=WIDGET_BASE(swittch_base, /FRAME, ROW=1, /EXCLUSIVE)
   spb_noorbit=WIDGET_BUTTON(orbitlabel_base, $
			VALUE='no orbit labels ', $
			UVALUE='spb_noorbit')
   spb_orbit=WIDGET_BUTTON(orbitlabel_base, $
			VALUE='label orbits', $
			UVALUE='spb_orbit')

   IF aPlotInfo.ioptions(19) EQ 0 THEN BEGIN
      WIDGET_CONTROL, spb_noorbit, SET_BUTTON=1
      WIDGET_CONTROL, spb_orbit, SET_BUTTON=0
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_noorbit, SET_BUTTON=0
      WIDGET_CONTROL, spb_orbit, SET_BUTTON=1
   ENDELSE

   norm_base=WIDGET_BASE(swittch_base, /FRAME, ROW=1, /EXCLUSIVE)
   spb_nonorm=WIDGET_BUTTON(norm_base, $
			VALUE='no normalization', $
			UVALUE='spb_nonorm')
   spb_norm=WIDGET_BUTTON(norm_base, $
			VALUE='normalization', $
			UVALUE='spb_norm')

   IF (aPlotInfo.ioptions(19) AND 2) EQ 0 THEN BEGIN
      WIDGET_CONTROL, spb_nonorm, SET_BUTTON=1
      WIDGET_CONTROL, spb_norm, SET_BUTTON=0
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_nonorm, SET_BUTTON=0
      WIDGET_CONTROL, spb_norm, SET_BUTTON=1
   ENDELSE

; --- SLIDERS: ----------------------------------------------------------------

; --- Energy ------------------------------------------------------------------
   energyBase=WIDGET_BASE(panel_base, COLUMN=2, /FRAME)
   energyLeftBase=WIDGET_BASE(energyBase, /EXCLUSIVE, ROW=2)
   energyRightBase=WIDGET_BASE(energyBase, COLUMN=1)
   spb_allenergies=WIDGET_BUTTON(energyLeftBase, $
				VALUE='All energies        ',$
				UVALUE='spb_allenergies')
   spb_Selectenergies=WIDGET_BUTTON(energyLeftBase, $
				VALUE='Select energy   ',$
				UVALUE='spb_selectenergies')

   IF aPlotInfo.typeVector(2) ne 0 then BEGIN
       WIDGET_CONTROL, spb_allenergies, SET_BUTTON=0 
       WIDGET_CONTROL, spb_Selectenergies, SET_BUTTON=1
   ENDIF ELSE BEGIN
       WIDGET_CONTROL, spb_allenergies, SET_BUTTON=1
       WIDGET_CONTROL, spb_Selectenergies, SET_BUTTON=0
   ENDELSE

   sl_energy=WIDGET_SLIDER(energyrightBase, $
			/DRAG, $
		        VALUE=aPlotInfo.typeVector(2) < 14 > 1, $
                        UVALUE='sl_energy',$
                        MINIMUM=1, MAXIMUM=14)
   lbl_energy=WIDGET_LABEL(energyrightBase, $
			VALUE=strcenter('energy range',50), $
			/ALIGN_CENTER)			      
       
; --- Pitch angle for uncorrected pitch angle plots-----------------------------
   pitchBase=WIDGET_BASE(panel_base, COLUMN=2, /FRAME)
   pitchLeftBase=WIDGET_BASE(pitchBase, /EXCLUSIVE, ROW=2)
   pitchRightBase=WIDGET_BASE(pitchBase, COLUMN=1)
   spb_allpitch=WIDGET_BUTTON(pitchLeftBase, $
				VALUE='All pitch angles    ',$
				UVALUE='spb_allpitch')
   spb_Selectpitch=WIDGET_BUTTON(pitchLeftBase, $
				VALUE='Select pitch angles  ',$
				UVALUE='spb_selectpitch')

   sl_pitch=WIDGET_SLIDER(pitchrightBase, $
			/DRAG, $
		        VALUE=aPlotInfo.typeVector(3) < 11 > 1, $
                        UVALUE='sl_pitch',$
                        MINIMUM=1, MAXIMUM=10)
   lbl_pitch=WIDGET_LABEL(pitchrightBase, $
			VALUE=strcenter('pitch angle range',50), $
			/ALIGN_CENTER)			      
       
; --- Pitch angle for corrected pitch angle at equator--------------------------
   eqpitchBase=WIDGET_BASE(panel_base, COLUMN=1, /FRAME)
   eqpitchLeftBase=WIDGET_BASE(eqpitchBase, /EXCLUSIVE, ROW=1)
   eqpitchRightBase=WIDGET_BASE(eqpitchBase, COLUMN=1)

   spb_eqpitch=WIDGET_BUTTON(eqpitchLeftBase, $
				VALUE='Corrected pitch angles at equator',$
				UVALUE='spb_eqpitch')

   IF aPlotInfo.typeVector(3) ge 0 then BEGIN
       IF aPlotInfo.typeVector(3) ne 0 then BEGIN
           WIDGET_CONTROL, spb_allpitch, SET_BUTTON=0 
           WIDGET_CONTROL, spb_Selectpitch, SET_BUTTON=1
           WIDGET_CONTROL, spb_eqpitch, SET_BUTTON=0
;           print,'here, spb_eqpitch   ',spb_eqpitch
       ENDIF ELSE BEGIN
           WIDGET_CONTROL, spb_allpitch, SET_BUTTON=1
           WIDGET_CONTROL, spb_Selectpitch, SET_BUTTON=0
           WIDGET_CONTROL, spb_eqpitch, SET_BUTTON=0
           print,'here2'
       ENDELSE
   ENDIF ELSE BEGIN
       WIDGET_CONTROL, spb_allpitch, SET_BUTTON=0
       WIDGET_CONTROL, spb_Selectpitch, SET_BUTTON=0
       WIDGET_CONTROL, spb_eqpitch, SET_BUTTON=1
   ENDELSE

   sl_eqpitch=WIDGET_SLIDER(eqpitchrightBase, $
			/DRAG, $
		        VALUE=aPlotInfo.typeVector(3) < 3 > 1, $
                        UVALUE='sl_eqpitch',$
                        MINIMUM=1, MAXIMUM=3)
   lbl_eqpitch=WIDGET_LABEL(eqpitchrightBase, $
			VALUE=strcenter('Corrected pitch angle range',50), $
			/ALIGN_CENTER)			      

; --- L-value ----------------------------------------------------------------
   LvalueBase=WIDGET_BASE(panel_base, COLUMN=1, /FRAME)
   LvalueLeftBase=WIDGET_BASE(LvalueBase, /EXCLUSIVE, ROW=1)
   LvalueRightBase=WIDGET_BASE(LvalueBase, COLUMN=1)
   spb_allLvalues=WIDGET_BUTTON(LvalueLeftBase, $
				VALUE='All L-values        ',$
				UVALUE='spb_allLvalues')
   spb_SelectLvalues=WIDGET_BUTTON(LvalueLeftBase, $
				VALUE='Select L-value   ',$
				UVALUE='spb_selectLvalues')

   tmp=WIDGET_INFO(panel_base, /GEOMETRY)
   sl_Lvalue=widget_slider(LvalueRightBase, $
       MINIMUM=1, MAXIMUM=35, /DRAG, $
 	 VALUE=(aPlotinfo.options(0)>1), $
	 UVALUE='sl_Lvalue')
   lbl_Lvalue =WIDGET_LABEL(LvalueRightBase, $
			VALUE=strcenter('L-value', 50), $
			/ALIGN_CENTER)
   IF aPlotinfo.options(0) ne 0 THEN BEGIN
      WIDGET_CONTROL, spb_allLvalues, SET_BUTTON=0
      WIDGET_CONTROL, spb_selectLvalues, SET_BUTTON=1
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_allLvalues, SET_BUTTON=1
      WIDGET_CONTROL, spb_selectLvalues, SET_BUTTON=0
      WIDGET_CONTROL, sl_Lvalue, SENSITIVE=0      
   ENDELSE
   
; The next four papco panel widgets will be added only if you want to
; support that functionality, and if they make sense for your crres_meb_overview.

;--- add the papco panel widget for choosing the color bar for color plots ----
   widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
   widgets=papco_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
   widgets=papco_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.
;    if you want to make use of the configure feature, then add the
;    keyword CONFIG
   widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create 
;    that helpfile in the papco_interface directory of your module (a template
;    for this help file exist in the crres_meb_overview example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
   thisWidgetsData={helpFile:'crres_meb_overview_panel.help', $
                    typeVector:aPlotInfo.typeVector, $
                    options:aPlotInfo.options, $
                     $
                    spb_integral:spb_integral, $
                    spb_differential:spb_differential, $
                    spb_allenergies:spb_allenergies, $
                    spb_Selectenergies:spb_Selectenergies, $
                    spb_noorbit:spb_noorbit, $
                    spb_orbit:spb_orbit, $
                    spb_nonorm:spb_nonorm, $
                    spb_norm:spb_norm, $
                    sl_energy:sl_energy, $
                    lbl_energy:lbl_energy, $
                    spb_allpitch:spb_allpitch, $
                    spb_Selectpitch:spb_Selectpitch, $
                    spb_eqpitch:spb_eqpitch, $
                    sl_pitch:sl_pitch, $
                    lbl_pitch:lbl_pitch, $
                    sl_eqpitch:sl_eqpitch, $
                    lbl_eqpitch:lbl_eqpitch, $
                    spb_allLvalues:spb_allLvalues, $
                    spb_selectLvalues:spb_selectLvalues, $
                    sl_Lvalue:sl_Lvalue,$
                    lbl_Lvalue:lbl_Lvalue,$
                    spb_type:spb_type,$
                    spb_fill:spb_fill}

   widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   papco_Panel_Realize, panel_base, widgetData
   crres_meb_overview_panel_refresh,/SL_PITCH,/SL_LVALUE,/SL_ENERGY, $
     /SL_EQPITCH, /PB_PITCH,/PB_ENERGY,/PB_LVALUE
   crres_meb_overview_panel_refresh,/SL_PITCH,/SL_LVALUE,/SL_ENERGY, $
     /SL_EQPITCH
; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reported but the 
; data contained in aPlotInfo would be returned.
   papco_xmanager, 'crres_meb_overview_panel', panel_base, /MODAL

; the returned data consists of the standard-data inherited from
; papco_PANEL.PRO and your new data - the typeVector-tag and swittch-tag must
; always be included, but you can add more. However, if possible stick to the 
; tags provided in aPlotInfo, it makes life easier. 
   returnData={typevector:widgetData.typevector, $
               options:widgetData.options, $
               ioptions:widgetData.ioptions}
   additionalRData=papco_panel_ReturnData(widgetData)

   return, create_struct(ReturnData, additionalRData)
END

;***************************************************************************
;* 
;* PROCEDURE:     
;*      PRO crres_meb_overview_panel_refresh, $
;*                SL_PITCH=SL_PITCH, SL_ENERGY=SL_ENERGY, SL_LVALUE=SL_LVALUE $
;*                PB_PITCH=PB_PITCH, PB_ENERGY=PB_ENERGY, PB_LVALUE=PB_LVALUE
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*      have a look at PRO crres_meb_overview_panel_event
;*
;* MODIFICATION HISTORY:       
;*      october 1995, written by R.Friedel 
;*
;***************************************************************************
PRO crres_meb_overview_panel_refresh,  $
                 SL_PITCH=SL_PITCH, SL_ENERGY=SL_ENERGY, SL_LVALUE=SL_LVALUE, $
                 PB_PITCH=PB_PITCH, PB_ENERGY=PB_ENERGY, PB_LVALUE=PB_LVALUE, $
                 PB_EQPITCH=PB_EQPITCH, SL_EQPITCH=SL_EQPITCH

   COMMON crres_meb_overview_panel_DATA, widgetData

   WIDGET_CONTROL, widgetData.spb_integral, SENSITIVE=0

   IF KEYWORD_SET(PB_LVALUE) then BEGIN
     IF widgetData.options(0) eq 0 then BEGIN  ; all L-value
        WIDGET_CONTROL, widgetData.spb_allLvalues, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.spb_selectLvalues, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_allenergies, SENSITIVE=0
        WIDGET_CONTROL, widgetData.spb_allpitch, SENSITIVE=0
        WIDGET_CONTROL, widgetData.spb_selectenergies, SENSITIVE=1
        WIDGET_CONTROL, widgetData.spb_selectpitch, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.spb_selectenergies, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.spb_selectpitch, SENSITIVE=1
        WIDGET_CONTROL, widgetData.sl_Lvalue, SENSITIVE=0
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=1
        IF widgetData.typeVector(3) lt 0 then BEGIN
           WIDGET_CONTROL, widgetData.spb_eqpitch, SET_BUTTON=0
           WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=1
           WIDGET_CONTROL, widgetData.sl_eqpitch, SENSITIVE=0
           WIDGET_CONTROL, widgetData.spb_eqpitch, SENSITIVE=1
        ENDIF ELSE BEGIN
           WIDGET_CONTROL, widgetData.spb_selectpitch, SET_BUTTON=0
           WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=0
           WIDGET_CONTROL, widgetData.spb_eqpitch, SENSITIVE=1
           WIDGET_CONTROL, widgetData.sl_eqpitch, SENSITIVE=1
        ENDELSE
;        print,'here return all L'
        return 
     ENDIF ELSE BEGIN                            ; select L-value
        IF widgetData.typeVector(3) gt 0 then BEGIN
           WIDGET_CONTROL, widgetData.spb_eqpitch, SET_BUTTON=1
           WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=0
           WIDGET_CONTROL, widgetData.sl_eqpitch, SENSITIVE=1
           WIDGET_CONTROL, widgetData.spb_selectpitch, SET_BUTTON=0
        ENDIF ELSE BEGIN
           WIDGET_CONTROL, widgetData.spb_selectpitch, SET_BUTTON=1
           WIDGET_CONTROL, widgetData.spb_eqpitch, SET_BUTTON=0
           WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=1
           WIDGET_CONTROL, widgetData.sl_eqpitch, SENSITIVE=0
        ENDELSE
        WIDGET_CONTROL, widgetData.spb_allLvalues, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_selectLvalues, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.sl_Lvalue, SENSITIVE=1
        WIDGET_CONTROL, widgetData.spb_allenergies, SENSITIVE=1
        WIDGET_CONTROL, widgetData.spb_allpitch, SENSITIVE=1
        WIDGET_CONTROL, widgetData.spb_selectenergies, SENSITIVE=1
        WIDGET_CONTROL, widgetData.spb_selectpitch, SENSITIVE=1
        IF widgetData.typeVector(2) eq 0 then BEGIN
           WIDGET_CONTROL, widgetData.spb_eqpitch, SENSITIVE=1
        ENDIF ELSE BEGIN
           WIDGET_CONTROL, widgetData.spb_eqpitch, SENSITIVE=0
        ENDELSE
        WIDGET_CONTROL, widgetData.sl_eqpitch, SENSITIVE=0
;        print,'here select L'
     ENDELSE
   ENDIF
  
   IF KEYWORD_SET(PB_ENERGY)  then BEGIN
     IF widgetData.typeVector(2) eq 0 then BEGIN  ; all energy
        WIDGET_CONTROL, widgetData.spb_allenergies, SET_BUTTON=1 
        WIDGET_CONTROL, widgetData.spb_Selectenergies, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_allpitch, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_Selectpitch, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=0                
        IF widgetData.typeVector(3) gt 0 then BEGIN 	;corrected pitch
           WIDGET_CONTROL, widgetData.spb_eqpitch, SET_BUTTON=1
           WIDGET_CONTROL, widgetData.spb_eqpitch, SENSITIVE=1
           WIDGET_CONTROL, widgetData.sl_eqpitch, SENSITIVE=1
           WIDGET_CONTROL, widgetData.spb_allpitch, SET_BUTTON=0
           WIDGET_CONTROL, widgetData.spb_selectpitch, SET_BUTTON=0
           WIDGET_CONTROL, widgetData.spb_allpitch, SENSITIVE=0
           WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=0 
           WIDGET_CONTROL, widgetData.sl_eqpitch, $
                           SET_VALUE=2
           if widgetData.typeVector(3) eq 0 then widgetData.typeVector(3)=2
;           print,'here all energy corrected pitch'
        ENDIF ELSE BEGIN				; at sat pitch
           WIDGET_CONTROL, widgetData.spb_eqpitch, SET_BUTTON=0
           WIDGET_CONTROL, widgetData.spb_eqpitch, SENSITIVE=1
           WIDGET_CONTROL, widgetData.sl_eqpitch, SENSITIVE=0
           WIDGET_CONTROL, widgetData.spb_allpitch, SET_BUTTON=0
           WIDGET_CONTROL, widgetData.spb_allpitch, SENSITIVE=0
           WIDGET_CONTROL, widgetData.spb_selectpitch, SET_BUTTON=1
           WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=1 
           WIDGET_CONTROL, widgetData.sl_pitch, $
                           SET_VALUE=10
           if widgetData.typeVector(3) eq 0 then widgetData.typeVector(3)=-10
;           print,'here all energy at sat pitch'        
        ENDELSE
     ENDIF ELSE BEGIN                             ; SELECT ENERGY
        WIDGET_CONTROL, widgetData.spb_allenergies, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_Selectenergies, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.spb_allpitch, SET_BUTTON=1 
        WIDGET_CONTROL, widgetData.spb_Selectpitch, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=1
        WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=0
        widgetData.typeVector(3)=0
     ENDELSE
   ENDIF

   IF KEYWORD_SET(PB_PITCH) then BEGIN
      IF widgetData.typeVector(3) eq 0 then BEGIN  ; all pitch angle
        WIDGET_CONTROL, widgetData.spb_allpitch, SET_BUTTON=1 
        WIDGET_CONTROL, widgetData.spb_Selectpitch, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_allenergies, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_Selectenergies, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=0
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=1
        WIDGET_CONTROL, widgetData.spb_eqpitch, SENSITIVE=0
        WIDGET_CONTROL, widgetData.sl_eqpitch, SENSITIVE=0
;        print,'here all pitch angle'
      ENDIF 
      IF widgetData.typeVector(3) lt 0 then BEGIN  ; select pitch angle
        WIDGET_CONTROL, widgetData.spb_allpitch, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.spb_Selectpitch, SET_BUTTON=1
        WIDGET_CONTROL, widgetData.spb_allenergies, SET_BUTTON=1 
        WIDGET_CONTROL, widgetData.spb_Selectenergies, SET_BUTTON=0
        WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=1
        WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=0
        WIDGET_CONTROL, widgetData.sl_eqpitch, SENSITIVE=0
;        print,'here select pitch angle'
        widgetData.typeVector(2)=0
      ENDIF
   ENDIF

   IF KEYWORD_SET(PB_EQPITCH) then BEGIN
      WIDGET_CONTROL, widgetData.spb_allpitch, SET_BUTTON=0
      WIDGET_CONTROL, widgetData.spb_Selectpitch, SET_BUTTON=0
      WIDGET_CONTROL, widgetData.spb_allenergies, SET_BUTTON=1 
      WIDGET_CONTROL, widgetData.spb_Selectenergies, SET_BUTTON=0
      WIDGET_CONTROL, widgetData.spb_eqpitch, SET_BUTTON=1
      WIDGET_CONTROL, widgetData.sl_pitch, SENSITIVE=0
      WIDGET_CONTROL, widgetData.sl_energy, SENSITIVE=0
      WIDGET_CONTROL, widgetData.spb_eqpitch, SENSITIVE=1
      WIDGET_CONTROL, widgetData.sl_eqpitch, SENSITIVE=1 
;      print,'here5'           
   ENDIF

   IF KEYWORD_SET(SL_ENERGY) then BEGIN
      IF widgetData.typeVector(0) THEN BEGIN	;----------------- ions
         WIDGET_CONTROL, widgetData.sl_energy, $
        			 SET_SLIDER_MIN=1, $
        			 SET_SLIDER_MAX=12
          IF widgetData.typeVector(2) GT 12 THEN BEGIN 
        	    widgetData.typeVector(2)=12
          	    WIDGET_CONTROL, widgetData.sl_energy, $
        			    SET_VALUE=widgetData.typeVector(2)>1
          ENDIF	 
          widgetData.typeVector(2)= widgetData.typeVector(2) < 12
          range=(widgetData.typeVector(2)-1) > 0
          yminarr=[37.,54.,69.,85.,113.,147.,193.,254.,335.,447.,602.,805.] 
          ymaxarr=[54.,69.,85.,113.,147.,193.,254.,335.,447.,602.,805.,3200.] 
          rangeLabel='(ion) '+$
	       strtrim(string(yminarr(range),format='(F8.1)'),2) + ' - ' + $
	       strtrim(string(ymaxarr(range),format='(F8.1)'),2) + ' keV'
      ENDIF ELSE BEGIN  ;------------------------------------- electrons
          WIDGET_CONTROL, widgetData.sl_energy, $
        			 SET_SLIDER_MIN=1, $
        			 SET_SLIDER_MAX=14
          IF widgetData.typeVector(2) GT 14 THEN BEGIN 
        	    widgetData.typeVector(2)=14
          	    WIDGET_CONTROL, widgetData.sl_energy, $
        			    SET_VALUE=widgetData.typeVector(2)>1
          ENDIF	 
          range=(widgetData.typeVector(2)-1) > 0
          yminarr=[21.,31.5,40.,49.5,59.,69.,81.,94.5,$
        	 	  112.,129.5,151.,177.5,208.,242.5]
          ymaxarr=[31.5,40.,49.5,59.,69.,81.,94.5,112.,$
        	 	  129.5,151.,177.5,208.,242.5,285.]
          rangeLabel='(electron) '+$
	       strtrim(string(yminarr(range),format='(F8.1)'),2) + ' - ' + $
	       strtrim(string(ymaxarr(range),format='(F8.1)'),2) + ' keV'
     ENDELSE
     WIDGET_CONTROL, widgetData.lbl_energy, $
        	SET_VALUE=strcenter('energy range '+ rangeLabel, 50)
   ENDIF

   IF KEYWORD_SET(SL_PITCH) then BEGIN
      WIDGET_CONTROL, widgetData.sl_pitch, $
			       SET_SLIDER_MIN=1, $
			       SET_SLIDER_MAX=10
      WIDGET_CONTROL, widgetData.sl_pitch, $
			       SET_VALUE=abs(widgetData.typeVector(3)) < 11 > 1
               pitch=(abs(widgetData.typeVector(3))-1) < 10 > 0
      IF pitch EQ 0 THEN pitchlabel='0 - 5'
         IF pitch EQ 18 THEN pitchlabel='175 - 180'
         IF 0 LT pitch AND pitch LT 18 THEN BEGIN
        	 rLower=5+10*(pitch-1)
        	 pitchlabel=strtrim(string(rLower),2)+' - '+ $
        		     strtrim(string(rLower+10),2)
      ENDIF
       
      pitchLabel=pitchLabel + ' degrees'
      WIDGET_CONTROL, widgetData.lbl_pitch, $
			 SET_VALUE=strcenter('pitch angle '+$
			         pitchLabel,50)   
   ENDIF

   IF KEYWORD_SET(SL_EQPITCH) then BEGIN
      WIDGET_CONTROL, widgetData.sl_eqpitch, $
			       SET_SLIDER_MIN=1, $
			       SET_SLIDER_MAX=3
      WIDGET_CONTROL, widgetData.sl_eqpitch, $
			       SET_VALUE=widgetData.typeVector(3) < 3 > 1
               pitch=(widgetData.typeVector(3)-1) < 2 > 0
      Parr=[30,45,60] 
      pitchlabel=string(parr(pitch),format="(i3.3)")
      
      pitchLabel=pitchLabel + ' degree'
      WIDGET_CONTROL, widgetData.lbl_eqpitch, $
			 SET_VALUE=strcenter('corr. pitch angle at equ. '+$
			         pitchLabel,50)   
   ENDIF

   IF KEYWORD_SET(SL_LVALUE) then BEGIN
      WIDGET_CONTROL, widgetData.sl_Lvalue, $
			       SET_SLIDER_MIN=1, $
			       SET_SLIDER_MAX=35
      WIDGET_CONTROL, widgetData.sl_Lvalue, $
			       SET_VALUE=(widgetData.options(0)>1)
      Lvalue=float((widgetData.options(0)>1)-1)*0.2 +1.0
      LvalueLabel=string(Lvalue,Lvalue+0.2,format="(f4.1,' - ',f4.1)")

      WIDGET_CONTROL, widgetData.lbl_Lvalue, $
			 SET_VALUE=strcenter('L-value '+$
			         LvalueLabel,50)   
   ENDIF

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO crres_meb_overview_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO crres_meb_overview_panel_event, event

  COMMON crres_meb_overview_panel_DATA, widgetData
  
  WIDGET_CONTROL,GET_UVALUE=uval, event.id
  
; in this editor, we only need to process the options(0) slider events. events
; other than those from your own widgets need to be passed on to 
; 'papco_panel_Event', which handles the events generated
; by the papco-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'papco_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.
  
  
  if strleft(uval, 8) EQ 'spb_type' THEN BEGIN
    smooth_choice=fix(strfrom(uval, 8)) 
    uval=strleft(uval, 8)
  endif
  
  if strleft(uval, 8) EQ 'spb_fill' THEN BEGIN
    fill_choice=fix(strfrom(uval, 8)) 
    uval=strleft(uval, 8)
  endif
  
  
  CASE uval OF
    'spb_electrons' : BEGIN
      widgetData.typeVector(0)=0
      crres_meb_overview_panel_refresh,/SL_ENERGY
    END
    'spb_ions' : BEGIN
      widgetData.typeVector(0)=1
      crres_meb_overview_panel_refresh,/SL_ENERGY
    END
    'spb_differential' : BEGIN
      widgetData.typeVector(1)=1
    END
    'spb_integral'	 : BEGIN
      widgetData.typeVector(1)=0
    END
    
    'spb_type'       : BEGIN
      widgetData.ioptions(17)=smooth_choice
    END
    
    'spb_fill'       : BEGIN
      widgetData.ioptions(18)=fill_choice
    END
    
    'spb_noorbit'    : widgetData.ioptions(19)=0
    'spb_orbit'      : widgetData.ioptions(19)=1
    'spb_nonorm'     : widgetData.ioptions(19)=widgetData.ioptions(19) and 253
    'spb_norm'       : widgetData.ioptions(19)=widgetData.ioptions(19) OR 2
    
    'spb_allenergies' : BEGIN
      widgetData.typeVector(2)=0
      crres_meb_overview_panel_refresh,/PB_LVALUE,/PB_ENERGY
    END
    'spb_selectenergies' :BEGIN
      widgetData.typeVector(2)=3
      crres_meb_overview_panel_refresh,/PB_LVALUE,/PB_ENERGY
    END
    'sl_energy': BEGIN
      widgetData.typeVector(2)=event.value
      crres_meb_overview_panel_refresh, /SL_ENERGY
    END
    'spb_allpitch' : BEGIN
      widgetData.typeVector(3)=0
      crres_meb_overview_panel_refresh,/PB_LVALUE,/PB_PITCH
    END
    'spb_selectpitch' : BEGIN
      widgetData.typeVector(3)=-10
      crres_meb_overview_panel_refresh,/PB_LVALUE,/PB_PITCH
    END
    'spb_eqpitch': BEGIN
      widgetData.typeVector(3)=2
      crres_meb_overview_panel_refresh,/PB_LVALUE,/PB_PITCH,/PB_EQPITCH
      crres_meb_overview_panel_refresh,/SL_EQPITCH
;               print,'spb_eqpitch','  ', widgetData.typeVector(3)
    END        
    'sl_pitch': BEGIN
      widgetData.typeVector(3)=event.value*(-1)
;               print,widgetData.typeVector
      crres_meb_overview_panel_refresh,/PB_LVALUE,/SL_PITCH
      crres_meb_overview_panel_refresh,/SL_PITCH
    END
    'sl_eqpitch': BEGIN
      widgetData.typeVector(3)=event.value
      crres_meb_overview_panel_refresh,/SL_EQPITCH
;               print,'sl_eqpitch','  ', widgetData.typeVector(3)
    END
    'spb_allLvalues': BEGIN
      widgetData.options(0)=0
      crres_meb_overview_panel_refresh,/PB_LVALUE
    END
    'spb_selectLvalues': BEGIN
      widgetData.options(0)=25
      crres_meb_overview_panel_refresh,/PB_LVALUE,/PB_ENERGY,/PB_PITCH
    END
    'sl_Lvalue' : BEGIN
      widgetData.options(0)=event.value
      crres_meb_overview_panel_refresh, /SL_LVALUE
    END
    
    ELSE: $
      widgetData=papco_panel_Event(event, widgetData)
    
  ENDCASE   
  
  
END
