;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  crres_meb_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_olotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I 
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      swittch:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, swittch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          swittch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commincate between your
;*      panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which papco supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      papco internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $                       
;*                      xut2:0.0}                           
;*
;*                  
;*      For an enhanced description read 
;*                          $papco_XX/papco_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION crres_meb_panel, aPlotInfo, GROUP=group, ACTION=ACTION

COMMON crres_meb_panel_data, widgetData
COMMON crres_meb

;--- if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0
;--- set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
title = aplotinfo.panelKind + ' - ' + papco_module_version('crres_meb')
panel_base=papco_WIDGET_BASE(TITLE=title, ROW=1, GROUP_LEADER=group)
panel_left=widget_base(panel_base,column=1, /frame)
panel_right=widget_base(panel_base,column=1,/frame)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
widgets=papco_panel_AddHeightWidgets(panel_left, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
 
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
  
; --- Species /  Integral Differental choices ---------------------------------
choice_base=WIDGET_BASE(panel_left, ROW=2, /ALIGN_CENTER)   
; --- Electron / Ion ---
species_names=['Electrons','Ions']
species_base=CW_BGROUP(choice_base,species_names,/FRAME,ROW=1,/EXCLUSIVE, $
                         UVALUE='spb_species', $
                         SET_VALUE=aPlotInfo.ioptions(0) )
  ; --- Integral / differential ---
  erange_names=['Integral','Differential']
  erange_base=CW_BGROUP(choice_base,erange_names,/FRAME,ROW=1,/EXCLUSIVE, $
                        UVALUE='spb_erange', IDS=erange_IDS, $
                        SET_VALUE=aPlotInfo.ioptions(1) )
  spb_integral=erange_IDS(0)  &  spb_differential=erange_IDS(1)
   
  ; --- Smooth choice ---
  smooth_base=CW_FIELD(choice_base, /FRAME, ROW=1, /INTEGER, XSIZE=4, $
                       TITLE='Smooth points:', $
                       UVALUE='ef_smooth', $
                       VALUE=aPlotInfo.ioptions(4)) 
   
  ; --- Reload choice ---
  reload_names=['Relaod', 'Adopt']
  reload_base=CW_BGROUP(choice_base, reload_names, /FRAME, ROW=1, /EXCLUSIVE, $
                        UVALUE='spb_reload', $
                        SET_VALUE=aPlotInfo.ioptions(5))   
   
; --- Energy / pitch ----------------------------------------------------------
   ener_pitch_outerBase=WIDGET_BASE(panel_left, /FRAME, ROW=1, /ALIGN_CENTER)
   ener_pitch_leftBase=WIDGET_BASE(ener_pitch_OuterBase, COLUMN=1) 
   ener_pitch_rightBase=WIDGET_BASE(ener_pitch_OuterBase, COLUMN=1) 
   
   ener_pitch_Base=WIDGET_BASE(ener_pitch_leftBase, ROW=1, /EXCLUSIVE)
   spb_energy=WIDGET_button(ener_pitch_base, $
                            VALUE=strcenter('all energies',16), $
                            UVALUE='spb_energy')
   spb_pitch=widget_button(ener_pitch_base, $
                           VALUE=strcenter('all pitch angles',16), $
                           UVALUE='spb_pitch')
   
   IF aPlotInfo.ioptions(2) EQ 0 THEN BEGIN
     WIDGET_CONTROL, spb_energy, SET_BUTTON=1
     WIDGET_CONTROL, spb_pitch, SET_BUTTON=0
     sl_energy_Pitch=WIDGET_SLIDER(ener_pitch_leftBase, xsize=440, /DRAG, $
                                   VALUE=aPlotInfo.ioptions(3) < 14 > 1, $
                                   UVALUE='sl_energy_pitch',$
                                   MINIMUM=1, MAXIMUM=14)
     lbl_energy_Pitch=WIDGET_LABEL(ener_pitch_leftBase, $
                                   VALUE=strcenter('pitch angle range',50), $
                                   /ALIGN_CENTER)			      
     WIDGET_CONTROL, spb_differential, SET_BUTTON=1     
   ENDIF ELSE BEGIN
     WIDGET_CONTROL, spb_energy, SET_BUTTON=0
     WIDGET_CONTROL, spb_pitch, SET_BUTTON=1
     sl_energy_Pitch=WIDGET_SLIDER(ener_pitch_leftBase, xsize=440, /DRAG, $
                                   VALUE=aPlotInfo.ioptions(3) < 19 > 1, $
                                   UVALUE='sl_energy_pitch',$
                                   MINIMUM=1, MAXIMUM=19)
     lbl_energy_Pitch=WIDGET_LABEL(ener_pitch_leftBase, $
                                   VALUE=strcenter('energy range',50), $
                                   /ALIGN_CENTER)			      
   ENDELSE
   
;--- buttons for choosing channels. Any combination of buttons may be pressed.
;    Default is all chosen.  Not all choices are mapped together!
;    Need one base for elec , ion energies and pitch angles.   
  ch_outer_base=WIDGET_BASE(panel_left, COLUMN=1, /align_center)
  ch_base=WIDGET_BASE(ch_outer_base, /align_center)     
  ch_sel=aPlotInfo.typevector(0)
  ;--- electron energy channels ---
  n_ch=n_elements(ele_lbl)  &  iset=bytarr(n_ch)
  for i=0,n_ch-1 do if (ch_sel and 2l^i) eq 2l^i then iset(i)=1 else iset(i)=0
  mpb_ele=CW_BGROUP(ch_base, ele_lbl, /FRAME, COLUMN=4, /NONEXCLUSIVE, $
                    LABEL_TOP='Select electron energy channels (keV)', $
                    UVALUE='mpb_ele', SET_VALUE=iset)
  ;--- ion energy channels ---
  n_ch=n_elements(ion_lbl)  &  iset=bytarr(n_ch)
  for i=0,n_ch-1 do if (ch_sel and 2l^i) eq 2l^i then iset(i)=1 else iset(i)=0
  mpb_ion=CW_BGROUP(ch_base, ion_lbl, /FRAME, COLUMN=4, /NONEXCLUSIVE, $
                    LABEL_TOP='Select ion energy channels (keV)', $
                    UVALUE='mpb_ion', SET_VALUE=iset)
  ;--- pitch angle channels ---
  n_ch=n_elements(pa_lbl)  &  iset=bytarr(n_ch)
  for i=0,n_ch-1 do if (ch_sel and 2l^i) eq 2l^i then iset(i)=1 else iset(i)=0
  mpb_pa=CW_BGROUP(ch_base, pa_lbl, /FRAME, COLUMN=5, /NONEXCLUSIVE, $
                   LABEL_TOP='Select pitch angle channels (deg)', $
                   UVALUE='mpb_pa', SET_VALUE=iset)  
  
;--- Add limit choices to select L-range --------------------------------------
  l_base= WIDGET_BASE(panel_left, /FRAME, COLUMN=1, /align_center)
  l_label_base=WIDGET_BASE(l_base, ROW=1, /align_center)
  l_low_label= WIDGET_LABEL(l_label_base, /align_center, $
  VALUE=strformat(string(aPlotInfo.options(2),format="(f4.1)"),35,/center), $
                             UVALUE='l_low_label')
  l_high_label=WIDGET_LABEL(l_label_base, /align_center, $
  VALUE=strformat(string(aPlotInfo.options(3),format="(f4.1)"),35,/center), $
                             UVALUE='l_high_label')

  l_slider_base=WIDGET_BASE(l_base, row=1, /align_center)
  ; --- Low L-value ---
  l_low=WIDGET_SLIDER(l_slider_Base, /DRAG, /SUPPRESS_VALUE, xsize=220, $
                  title=strformat('Low L-value',35,/center), $
         	  VALUE=(((aPlotInfo.options(2) < 12)-1) * 10) < 111 > 0, $
                  UVALUE='l_low', MINIMUM=0, MAXIMUM=110)	      
  ; --- High L-value ---
  l_high=WIDGET_SLIDER(l_slider_Base, /DRAG, /SUPPRESS_VALUE, xsize=220, $
                  title=strformat('High  L-value',35,/center), $
		  VALUE=(((aPlotInfo.options(3) < 12)-1) * 10) < 111 > 0, $
                  UVALUE='l_high', MINIMUM=0, MAXIMUM=110)   

;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
;--- for examples of standard buttons or sliders look at other panel editors.
;    some standard buttons which have been recycled before can be found in
;    papco_XX/papco_lib/more_panel_widgets.

;--- As the same panel is called when you edit a plot, interrogate the tags in 
;    the plotInfoStruct (aplotInfo) passed in and set your widgets accordingly.
;    In this way the widget is always reinitialized to its last state.
   
;--- The next five papco panel widgets will be added only if you want to
;    support that functionality, and if they make sense for your
;    crres_meb. If you do not want to use them, just comment them out.

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
;   widgets=papco_panel_AddFile_Widgets(panel_left, aPlotInfo)
;   additionalWData=create_struct(widgets, additionalWData)   
   
widgets=papco_PANEL_AddDescr_Widgets(panel_right, aPlotInfo, /VERT)
additionalWData=create_struct(widgets, additionalWData)   
   
if aPlotInfo.panelKind EQ 'crres_meb_line' then begin
;--- add the  papco panel widget for choosing the plot-style for line plots ---
     widgets=papco_panel_add_PlotStyleButtons(panel_left, aPlotInfo, 0)
     additionalWData=create_struct(widgets, additionalWData)
endif
         
if aPlotInfo.panelKind EQ 'crres_meb_spec' then begin
;--- add the papco panel widget for choosing the color bar for color plots ----
     widgets=papco_panel_Addcolorbar_Widgets(panel_left, aPlotInfo)
     additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
     widgets=papco_PANEL_AddZScale_Widgets(panel_left, aPlotInfo)
     additionalWData=create_struct(widgets, additionalWData)
endif
   
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
widgets=papco_PANEL_AddYScale_Widgets(panel_right, aPlotInfo, /VERT)
additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
widgets=papco_panel_AddActionButtons(panel_left, Action, aPlotInfo, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create 
;    the helpfile in the papco_interface directory of your module (a template
;    for this help file is in the crres_meb example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
thisWidgetsData={helpFile:'crres_meb_panel.help', $
                    typevector:aPlotInfo.typevector, $
                    options:aPlotInfo.options, $
                    ioptions:aPlotInfo.ioptions, $
                    l_low:l_low, $
                    l_high:l_high, $
                    l_low_label:l_low_label, $
                    l_high_label:l_high_label, $
                    slider_displaysEnergy:0, $
                    lbl_energy_Pitch:lbl_energy_Pitch, $
                    sl_energy_pitch:sl_energy_Pitch, $
                    spb_integral:spb_integral, $
                    spb_differential:spb_differential, $
                    spb_energy:spb_energy, $
                    mpb_pa:mpb_pa, $
                    mpb_ele:mpb_ele, $
                    mpb_ion:mpb_ion, $
                    smooth_base:smooth_base}
   
widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)
   
;--- realize the window -------------------------------------------------------
papco_Panel_Realize, panel_base, widgetData
crres_meb_panel_refresh, /ALL
   
;--- As this is a function that will return all changes made by the
;    user, the xmanager has to be called with the keyword MODAL. If we
;    wouldn't specify /MODAL, no changes would be reported but the 
;    data contained in aPlotInfo would be returned.
papco_xmanager, 'crres_meb_panel', panel_base

; --- the returned data consists of the standard-data inherited from
;     papco_PANEL.PRO and your new data - the ioptions-tag and swittch-tag 
;     must always be included, but you can add more. However, if possible stick
;     to the tags provided in aPlotInfo, it makes life easier. 
returnData={ioptions:widgetData.ioptions, $
               options:widgetData.options,$
               typevector:widgetData.typevector}
additionalRData=papco_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO crres_meb_panel_refresh, $
;*                      CHANNEL=CHANNEL, SL_ENERGY_PITCH=SL_ENERGY_PITCH
;* 
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*      have a look at PRO crres_panel_event
;*
;* MODIFICATION HISTORY:       
;*      November 1998, R Friedel 
;******************************************************************************
PRO crres_meb_panel_refresh, ALL=ALL, CHANNEL=CHANNEL, $
                             SL_ENERGY_PITCH=SL_ENERGY_PITCH, $
                             LOW_L=LOW_L, HIGH_L=HIGH_L
  
  common crres_meb_panel_data, widgetData
  common crres_meb
  
  if keyword_set(ALL) then begin
    CHANNEL=1  &  LOW_L=1  &  HIGH_L=1
  endif
  
  if keyword_set(CHANNEL) then begin
    if widgetData.ioptions(2) eq 0 then begin ; all energies
      widget_control, widgetData.mpb_pa, MAP=0
      if widgetData.ioptions(0) eq 0 then begin ; electron energies
        widget_control, widgetData.mpb_ele, MAP=1
        widget_control, widgetData.mpb_ion, MAP=0
      endif else begin            ;ion energies
        widget_control, widgetData.mpb_ele, MAP=0
        widget_control, widgetData.mpb_ion, MAP=1
      endelse
    endif else begin            ; all pitch angles
      widget_control, widgetData.mpb_ele, MAP=0
      widget_control, widgetData.mpb_ion, MAP=0  
      widget_control, widgetData.mpb_pa, MAP=1
    endelse
  endif
  
  IF widgetData.ioptions(1) THEN BEGIN ; differential
    WIDGET_CONTROL, widgetData.spb_energy, SENSITIVE=1
  ENDIF ELSE BEGIN              ; integral
    WIDGET_CONTROL, widgetData.spb_energy, SENSITIVE=0
  ENDELSE
  
  IF widgetData.ioptions(2) THEN BEGIN ; pitch angle
    WIDGET_CONTROL, widgetData.spb_integral, SENSITIVE=1
    widgetData.slider_displaysEnergy=1
    papco_panel_Refresh,widgetData,/YSCALE
  ENDIF ELSE BEGIN              ; energy range
    WIDGET_CONTROL, widgetData.spb_integral, SENSITIVE=0
    widgetData.slider_displaysEnergy=0
    papco_panel_Refresh,widgetData,/YSCALE
  ENDELSE
  
; ---- Energy / Pitch - Slider ------------------------------------------
  IF widgetData.slider_displaysEnergy THEN BEGIN  ; energies
      IF widgetData.ioptions(0) THEN BEGIN ;----------------- ions
        WIDGET_CONTROL, widgetData.sl_energy_pitch, $
          SET_SLIDER_MIN=1, SET_SLIDER_MAX=12
        IF widgetData.ioptions(3) GT 12 THEN BEGIN 
          widgetData.ioptions(3)=12
          WIDGET_CONTROL, widgetData.sl_energy_pitch, $
            SET_VALUE=widgetData.ioptions(3)
        ENDIF	 
        widgetData.ioptions(3)= widgetData.ioptions(3) < 12
        range=widgetData.ioptions(3)-1
        yminarr=ion_ymin  &  ymaxarr=ion_ymax
        rangeLabel='(ion) '+$
          strtrim(string(yminarr(range), format='(F8.1)'),2) + ' - ' + $
          strtrim(string(ymaxarr(range), format='(F8.1)'),2) + ' keV'
      ENDIF ELSE BEGIN         ;------------------------------------- electrons
        WIDGET_CONTROL, widgetData.sl_energy_pitch, $
          SET_SLIDER_MIN=1, SET_SLIDER_MAX=14
        IF widgetData.ioptions(3) GT 14 THEN BEGIN 
          widgetData.ioptions(3)=14
          WIDGET_CONTROL, widgetData.sl_energy_pitch, $
            SET_VALUE=widgetData.ioptions(3)
        ENDIF	 
        range=widgetData.ioptions(3)-1
        yminarr=ele_ymin  &  ymaxarr=ele_ymax
        rangeLabel='(electron) '+ strtrim(string(yminarr(range), $
                         format='(F8.1)'),2) +$
          ' - ' + strtrim(string(ymaxarr(range), format='(F8.1)'),2) +' keV'
      ENDELSE
      WIDGET_CONTROL, widgetData.lbl_energy_Pitch, $
        SET_VALUE=strcenter('energy range '+ rangeLabel, 50)
  ENDIF ELSE BEGIN   ; pitch angles, NOT widgetData.slider_displaysEnergy
    range=(widgetData.ioptions(3)-1)>0
    IF range EQ 0  THEN rangeLabel='0 - 5'
    IF range EQ 18 THEN rangeLabel='175 - 180'
    IF 0 LT range AND range LT 18 THEN BEGIN
      rLow=5+10*(range-1)
      rangeLabel=strtrim(string(rLow),2)+' - '+ strtrim(string(rLow+10),2)
    ENDIF
    rangeLabel=rangeLabel + ' degree'
    WIDGET_CONTROL, widgetData.lbl_energy_Pitch, $
      SET_VALUE=strcenter('pitch range ' + rangeLabel, 50)
    WIDGET_CONTROL, widgetData.sl_energy_pitch, $
      SET_SLIDER_MIN=1, SET_SLIDER_MAX=19
  ENDELSE                     ; END slider_displaysEnergy ?
      
  if keyword_set(LOW_L) then begin
    high_l=widgetData.options(3) > (widgetData.options(2)+0.1)
    widgetData.options(3)=high_l
    WIDGET_CONTROL, widgetData.l_high, $
      SET_VALUE=(widgetData.options(3) -1) * 10
    label=string(widgetData.options(3),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_high_label, $
      SET_VALUE=strformat(label,35,/center)
    label=string(widgetData.options(2),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_low_label, $
      SET_VALUE=strformat(label,35,/center)
  end
  
  if keyword_set(HIGH_L) then begin
    low_l=widgetData.options(2) < (widgetData.options(3)-0.1)
    widgetData.options(2)=low_l
    WIDGET_CONTROL, widgetData.l_low, $
      SET_VALUE=(widgetData.options(2) -1) * 10
    label=string(widgetData.options(3),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_high_label, $
      SET_VALUE=strformat(label,35,/center)
    label=string(widgetData.options(2),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_low_label, $
      SET_VALUE=strformat(label,35,/center)    
  end  

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO crres_meb_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO crres_meb_panel_event, event

  COMMON crres_meb_panel_data, widgetData
  
  WIDGET_CONTROL,GET_UVALUE=uval, event.id
    
  CASE uval OF
    'spb_species': BEGIN
      widgetData.ioptions(0)=event.value
      crres_meb_panel_refresh, /CHANNEL
    END
    
    'spb_erange' : BEGIN
      widgetData.ioptions(1)=event.value
      crres_meb_panel_refresh
    END 
    
    'spb_reload': BEGIN
      widgetData.ioptions(5)=event.value
      crres_meb_panel_refresh
    END 
    
    'spb_energy' : BEGIN
      widgetData.ioptions(2)=0
      crres_meb_panel_refresh, /CHANNEL
    END
    
    'spb_pitch' : BEGIN
      widgetData.ioptions(2)=1
      crres_meb_panel_refresh, /CHANNEL
    END
    
    'sl_energy_pitch': BEGIN
      widgetData.ioptions(3)=event.value
      crres_meb_panel_refresh, /SL_ENERGY_PITCH
    END
    
    'mpb_ele': BEGIN
      WIDGET_CONTROL,widgetData.mpb_ele,GET_VALUE=channels
      widgetData.typevector(0)=0
      for i=0,n_elements(channels)-1 do if channels(i) then $
        widgetData.typevector(0)=widgetData.typevector(0)+2l^i      
    END  
    'mpb_ion': BEGIN
      WIDGET_CONTROL,widgetData.mpb_ion,GET_VALUE=channels
      widgetData.typevector(0)=0
      for i=0,n_elements(channels)-1 do if channels(i) then $
        widgetData.typevector(0)=widgetData.typevector(0)+2l^i      
    END  
    'mpb_pa': BEGIN
      WIDGET_CONTROL,widgetData.mpb_pa,GET_VALUE=channels
      widgetData.typevector(0)=0
      for i=0,n_elements(channels)-1 do if channels(i) then $
        widgetData.typevector(0)=widgetData.typevector(0)+2l^i    
    END  
    
    'l_low'    : begin
      widgetData.options(2)=event.value / 10.0 + 1.0
      crres_meb_panel_refresh, /LOW_L
    end  
   
    'l_high'   : begin
      widgetData.options(3)=event.value / 10.0 + 1.0
      crres_meb_panel_refresh, /HIGH_L
    end  

    ELSE: BEGIN
      widget_control, widgetData.smooth_base, get_value=x
      widgetData.ioptions(4)=x
      widgetData=papco_panel_Event(event, widgetData)
    end
    
  ENDCASE
  
END

FUNCTION crres_meb_line_panel, aPlotInfo, GROUP=group, ACTION=ACTION

return, crres_meb_panel(aPlotInfo, GROUP=group, ACTION=ACTION)

END

FUNCTION crres_meb_spec_panel, aPlotInfo, GROUP=group, ACTION=ACTION

return, crres_meb_panel(aPlotInfo, GROUP=group, ACTION=ACTION)

END
