;******************************************************************************
;******************************************************************************
;* FILE:
;*    $papco_PATH/crres_meb_plaspar_panel.pro
;* 
;*    This file belongs to the papco-project. See $papco_PATH/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains the panel-editor for CRRES-plaspar-plots.
;*
;*    For 'crres_meb_plaspar', 'typeVector' has the following meaning:
;*
;*    typeVector chooses plot type (4 element vector)
;*               typeVector(0)    -    0 electrons, 1 ions
;*               typeVector(1)    -    0 energy density
;*               typeVector(1)    -    1 number density
;*               typeVector(1)    -    2 perpendicular plasma pressure
;*               typeVector(1)    -    3 parallel plasma pressure
;*               typeVector(1)    -    4 total plasma pressure
;*               typeVector(1)    -    5 beta perpendicular
;*               typeVector(1)    -    6 beta parallel
;*               typeVector(1)    -    7 beta total
;*               typeVector(1)    -    8 anisotropy
;*               typeVector(1)    -    9 % data missing in pw/e array
;*
;* COMMON-BLOCKS:
;*    crres_mag_panel_data	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    papco_PANEL_EDITOR-structure (see $papco_PATH/papco_panel.pro for a
;*    description). 
;*      
;*    the additional fields are :
;*         {helpFile:'crres_meb_plaspar_panel.help', $
;*                    typeVector:aPlotInfo.typeVector, $
;*                    spb_type:spb_type}
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  crres_meb_plaspar_panel, aPlotInfo, $
;*                                          GROUP=group, ACTION=ACTION
;*    You might try this:
;*	 print, crres_meb_plaspar_panel(papco_getPlotInfoStruct())
;*
;* MODIFICATION HISTORY:       
;*     February 1999, re-written by R. Friedel
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  crres_meb_plaspar_panel GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for an crres_meb_plaspar
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the papco_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=papco_getPlotInfoStruct()
;*      print, crres_meb_plaspar_panel(aPlotInfoStruct)
;******************************************************************************
FUNCTION  crres_meb_plaspar_panel, aPlotInfo, GROUP=group, ACTION=ACTION
  
  common crres_meb_plaspar_panel_data, widgetData
 
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

   additionalWData={canceled:1}

   panel_base=papco_widget_base(TITLE=aPlotInfo.panelKind, COLUMN=1, $
			        GROUP_LEADER=GROUP) 
  
; --- height-Widgets ----------------------------------------------------------
   widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

; --- Choose components / plot type--------------------------------------------
   type_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=3, /EXCLUSIVE) 
   typeNames=['number density' ,$ 
              'energy density',$
              'perp. plasma pressure', $
              'paral. plasma pressure' ,$
              'total plasma pressure',$
              'beta perp.' , $
              'beta paral.', $
              'beta total' ,$
              'anisotropy' ,$
              '% data missing ']

   spb_type=lonarr(N_ELEMENTS(typeNames))

   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
      spb_type(i)=WIDGET_BUTTON(type_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))
 
; --- validate the info in 'aPlotInfo -----------------------------------------
   aPlotInfo.typeVector(1)=aPlotInfo.typeVector(1) < (N_ELEMENTS(typeNames)-1)
   aPlotInfo.typeVector(1)=aPlotInfo.typeVector(1) > 0
   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(1)), SET_BUTTON=1

;--- add widget to choose low/high res data
   elio_base=WIDGET_BASE(panel_base, /FRAME, ROW=1, /EXCLUSIVE)
   spb_electrons=WIDGET_BUTTON(elio_base, $
			VALUE='electrons', $
			UVALUE='spb_electrons')
   spb_ions=WIDGET_BUTTON(elio_base, $
			VALUE='ions', $
			UVALUE='spb_ions')

   IF aPlotInfo.typeVector(0) EQ 0 THEN BEGIN
      WIDGET_CONTROL, spb_electrons, SET_BUTTON=1
      WIDGET_CONTROL, spb_ions, SET_BUTTON=0
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, spb_electrons, SET_BUTTON=0
      WIDGET_CONTROL, spb_ions, SET_BUTTON=1
   ENDELSE


;--- add the widgets for choosing the plot-style ------------------------------
   widgets=papco_panel_add_PlotStyleButtons(panel_base, aPlotInfo, $
                                           aPlotInfo.typeVector(2) GE 4)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- now, add the widgets for choosing the y-axis-ranges ----------------------
   widgets=papco_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
   widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

   thisWidgetsData={helpFile:'crres_meb_plaspar_panel.help', $
                    typeVector:aPlotInfo.typeVector, $
                    spb_type:spb_type, $
                    spb_electrons:spb_electrons, $
                    spb_ions:spb_ions}

   widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)
   papco_Panel_Realize, panel_base, widgetData

   papco_xmanager, 'crres_meb_plaspar_panel', panel_base, /MODAL

;--- return the data  ---------------------------------------------------------
   additionalRData=papco_panel_ReturnData(widgetData)

   thisReturnData={channel:0, $
                   typeVector:widgetData.typeVector, $
                   ioptions:aPlotInfo.ioptions}	 

   return, create_struct(thisReturnData, additionalRData)
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO crres_meb_plaspar_panel_refresh, event
;* 
;* DESCRIPTION:  
;*	This processes refreshes for this panel
;*
;* INPUTS:       
;*	none
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER     
;******************************************************************************
PRO crres_meb_plaspar_panel_refresh, YLOG=YLOG
  
  common crres_meb_plaspar_panel_data, widgetData
      
  if keyword_set(YLOG) then begin
    lin=[8,9]
    index=where(lin eq widgetData.typeVector(1),count)
    if count eq 1 then begin
      widgetData.YLINLOG=0
    endif else begin
      widgetData.YLINLOG=1     
    endelse
    papco_panel_Refresh, widgetData, /YSCALE
  endif  
    
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO crres_meb_plaspar_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;******************************************************************************
PRO crres_meb_plaspar_panel_event, EVENT
  
  common crres_meb_plaspar_panel_data, widgetData
  WIDGET_CONTROL,GET_UVALUE=uval, event.id
      
  IF strleft(uval, 8) EQ 'spb_type' THEN BEGIN
    widgetData.typeVector(1)=fix(strfrom(uval, 8))
    crres_meb_plaspar_panel_refresh, /YLOG
    return
  ENDIF 

  CASE uval OF
    'spb_electrons' : BEGIN
        widgetData.typeVector(0) = 0
    END
    'spb_ions' : BEGIN
        widgetData.typeVector(0) = 1
    END

    ELSE: widgetData=papco_panel_Event(event, widgetData)
  ENDCASE
    
END
