;******************************************************************************
;* file p_crres_meb_overview.pro
;*
;* This file is a template for a plot routine which can be used by PAPCO.
;* It includes all the periphery needed. This routine makes use of a
;* color spectrogram routine that is supplied with PAPCO:
;* papco_XX/papco_lib/papco_plot_colorspec.pro. This routine should
;* meet all the requirements for the display of 3-D data (tyhe third
;* dimension being color). This routine also adheres to the PAPCO color scheme.
;* You of course can use your own color plot rotuine if desired!
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_crres_meb_overview, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (and/or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure PAPCD_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to make use of
;*              an extra common block.
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_plot_type, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;******************************************************************************
pro p_crres_meb_overview, panel, dtype, ylim, swittch, $
                      OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Line styles are also not useful. So the keywords 
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,_EXTRA=extra_par
; are not used at all here, but are in the call for consistency...

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_plot_type_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags which are meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type and/or swittch. Replace type in the call with whatever variables
; you need, or add more. 

; The following common blocks are needed for PAPCO:
  common mjdt, mjdt_start, mjdt_end ;common time limit in mjdt
  common yscale, yscl           ;man/auto yscaling
  common zscale, zscl           ;man/auto zscaling
  common shift_label, down      ;common for x-axis label shifting
  common coordinateSystems, plotted_x, plotted_y  
                                ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_plot_type. 
  common crres_meb_overview, info, lsum, valdat
  common crres_meb_overview_extra, intime, sp, lbin, lrange
  
  papco_Set_SubTable, SUBTABLE         ;sets the papco color table index
  panelset,panel			;sets the panel position viewport

; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.

   ytickno=50*(!p.position(3)-!p.position(1))
   if ytickno gt 10 then ytickno=10

;   xtickno=6

;  r_apogees  ; put apogee info into commons for axis labeling
   
; Energy channel to be plotted
   ch=dtype(2)-1   
   
; Setup labels   
   if (dtype(1) eq 1) then begin
     intdiff='Diff. '
     if (dtype(0) eq 1) then begin
       cspecies='ions'
       ekarr=[45.5, 61.5, 77.0, 99.0, 130.0, 170.0, 223.5, 294.5, 391.0, $
              524.5, 703.0, 2002.5, 2002.5, 2002.5]
     endif else begin 
       cspecies='electrons'
       ekarr=[26.0, 35.0, 45.0, 54.0, 64.0, 75.0, 88.0, 103.0, 121.0,  $
              140.0,164.0, 193.0, 225.0, 264.0]
     endelse
   endif else begin 
     intdiff='Int. '
     if (dtype(0) eq 1) then begin
       cspecies='ions'
     endif else begin 
       cspecies='electrons'
     endelse
   endelse
   
   cspecies=cspecies+' (MEB)'+'!C'+$
     strtrim(string(ekarr(ch),format="(f6.1,' keV')"),2)
   coltit='[cm!e-2!ns!e-1!nsr!e-1!nkeV!e-1!n]'
   cytit='L-value'
   linlogbit=1 & relbit=0
   
   if (panel(1) eq 1) then begin
     utitle=' '
   endif else begin
     utitle= cspecies
   endelse
   
;Setup variables
   nobins=(lrange-1)/float(lbin)+1
   zarr=fltarr(sp,nobins)
   yval=fltarr(sp,nobins)
   
   
;y-axis values
   yarr=fltarr(nobins,2)
   for i=0,nobins-1 do begin
     yarr(i,0)=1+i*lbin
     yarr(i,1)=1+(i+1)*lbin
   endfor
   
; time arrays
   time=(intime(*,0)+intime(*,1))/2.
   xarr=intime
   
   xut1=mjdt_start.mjd+mjdt_start.t / 86400.0                     
   xut2=mjdt_end.mjd  +mjdt_end.t   / 86400.0  
   
; flux array
   for i=0,sp-1 do begin
     zarr(i,*)=  lsum(i,ch,*)/19.
     yval(i,*)=valdat(i,ch,*)
   endfor

   index=where(yval ne 0,count)
   if count gt 0 then $
     zarr(index)=zarr(index)/yval(index)
   index=where(yval eq 0, count)
   if count gt 0 then $
     zarr(index)=1e-20

; restrict data to actual time range requested - this makes for faster zooming
   index=where((time gt xut1) and (time lt xut2),count)
   if count gt 1 then begin
     time=time(index)
     xarr=xarr(index,*)
     zarr=zarr(index,*)
   endif

   index=where(xarr(1:*,0)-xarr(0:*,0) gt 0.,count) + 1
   if count gt 1 then begin
     time=time(index)
     xarr=xarr(index,*)
     zarr=zarr(index,*)
   endif

;fill in missing data from previous or next half orbit's data if they exist
   if (ioptions(18) eq 1 or ioptions(18) eq 3) then fill_lav, zarr
   if (ioptions(18) eq 2 or ioptions(18) eq 3) then $ 
      fill_lav_orb, orbit, leg, xarr, zarr

;set flag for vertical smoothing if needed
   if (ioptions(17) eq 1) or (ioptions(17) eq 3) then smooth_vert, zarr, 3

;do horizontal smoothing if needed
   if (ioptions(17) eq 2) or (ioptions(17) eq 3) then smooth_horiz, zarr, 5
   
; the plot-calls normally use the arrays time for the time axis and yarr (zmat 
; for color plots) for the y or z axis.
; depending on your plot options different data may be selected and different
; axis-labeling may be needed. This is normally done using a case statement.

; in this example we have a single test array, so no case statement is needed.

; set the begin end end times. PAPCO convention is to use running secconds
; starting with the no of seconds since the beginning of the start
; day.
   
   uztit=coltit
   uytitle='L-value'
   utitle=cspecies

   yst=yarr(0,0)
   yen=yarr(nobins-1,1)
   
   index=where(zarr lt 0.1, count)
   if count gt 0 then begin
     zarr(index)=1e-20
   endif

; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data

; the procedure papco_autorange (in papco_XX/papco_lib/autoscaling is
; optional and sets the zlimits automatically for the data, attempting
; to use the full dynamic range of the color table. you may need to
; supply your own procedure for your own data.

; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling

   if (zscl(panel(0),0) eq 1) then begin
     zmin=zscl(panel(0),1)
     zmax=zscl(panel(0),2)
   endif else begin
     papco_autorange,zarr,zmin,zmax, $
       log=zscl(panel(0),3), exclude=0, nodata=1e-20
     zscl(panel(0),1)=zmin
     zscl(panel(0),2)=zmax
   endelse

   if (yscl(panel(0),0) eq 1) then begin
     yst=yscl(panel(0),1)
     yen=yscl(panel(0),2)
   endif else begin
     yscl(panel(0),1)=yst
     yscl(panel(0),2)=yen
   endelse
   
; use auto y-labeling routine to get "good" y-lables
  papco_y_label, yst, yen       
   
; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO:
;
;    !P.CHARSIZE  = widgetData.default_charsize    
;    !P.CHARTHICK = widgetData.default_charthick
;    !P.FONT      = -1
;    !P.NOERASE   = 1
;    !P.TICKLEN   = widgetData.default_TICKLEN
;    !X.TICKS     = widgetData.default_XTICKS
;    !X.STYLE     = widgetData.default_XSTYLE
;    !Y.STYLE     = 1
;    !X.MINOR     = widgetData.default_XMINOR
;
; Overriding these values by setting keywords is of course possible
; and allowed but discouraged, since 
; then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure.

   extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zmin,zmax], $
                  ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
                  ytitle:uytitle, ztitle:uztit, $
                  xtickformat:'noticks', $
                  ztickformat:'papco_color_bar_log_ticks'}

; check if the passed-in structure extra_par was set. If not, create
; it and set color to default black; which for the color table used by
; papco is 1. If extra_par was set, see if the tag color exists. If it
; does, leave it unchanged, if not, add it and set color to default
; black. 
  if n_elements(extra_par) EQ 0 then extra_par={color:1} else begin
    names=tag_names(extra_par)
    res=where(names eq 'COLOR',count)
    if count eq 0 then extra_par=create_struct({color:1}, extra_par)
  endelse  

; add keyword structure set locally and the one passed in
  extra_plotPar=create_struct(extra_plotPar, extra_par)
  
; use papco rotuine to draw time axis. This checks for bottom plot
; and uses the user's  xtickformat if it is something other than 'noticks'.
  down=0
  papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar   

;do color plot and color bar plot. nodata specifies the "nodata" flag
;in your data. Here we do not plot axis and data seperate, as
;papco_plot_colorspec cannot be used as an overplot!
  papco_plot_colorspec,zarr,xarr,yarr, nodata=1e-20,xstyle=5,$
    _extra=extra_plotPar
  papco_color_bar,_extra=extra_plotPar
  
; store the coordinate information into common block
  plotted_x = !x  &  plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label - 0r use any other function you define yourself.
; The routine x_side_label puts x-axis label to the right of the axis.
  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
    extra_plotPar.xtickformat='mjd_dec_1'
    extra_plotPar.color=1
    down=down+1
    plot,time,time, _extra=extra_plotPar, /NODATA
    extra_plotPar.xtickformat='crres_meb_orb_ticks'
    extra_plotPar.color=1
    down=down+1
    plot,time,time, _extra=extra_plotPar, /NODATA
    x_side_label,panel,'Time(UT)!CDate!COrbit'
  endif
   
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.
  right_side_label,panel,utitle,/rot90
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  
  if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file', /cont
    description='plot_type template sample data - Energy spectra, 10 channels'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, xarr, zarr, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
  endif  
   
end 

;------------------------------------------------------------------------------
; This function labels orbits given that a mag-lav file was read

function crres_meb_orb_ticks,axis,index,t
  
  common crres_meb_overview, info, lsum, valdat
  common crres_meb_overview_extra, intime, sp, lbin, lrange
  
  time=(intime(*,0)+intime(*,1))/2.
  res=min(abs(time-t),mindex)
  orbit=info(mindex,3)
  if res lt 1.0 then $
    return,string(orbit,format="('!C!C',i4.4)") $
  else $
    return,''
  
end 
;------------------------------------------------------------------------------
pro fill_lav, zarr, swittch

; procedure to smooth and fill in lav plots for "good" movies

   print,'% fill_lav: in progress'

     szarr=size(zarr)
     count=0
     for j=0,szarr(1)-1 do begin	;loop along time axis
       spectrum=zarr(j,*)
       idx=where(spectrum eq 0.0, nodat)
       if nodat ne 0 then begin
         for i=0,nodat-1 do begin	;fill with average value of neighbours
           k=0 & left_val=zarr((j-k),idx(i))
           while ((j-k) ge 0) do $
             if (zarr((j-k),idx(i))) eq 0.0 then k=k+1 else begin
               left_val=zarr((j-k),idx(i))
               goto,f_left
             endelse
f_left:    l=0 & right_val=zarr((j+l),idx(i))
           while ((j+l) le szarr(1)-1) do $
             if (zarr((j+l),idx(i))) eq 0.0 then l=l+1 else begin
               right_val=zarr((j+l),idx(i))
               goto,f_right
             endelse
f_right :  if (right_val eq 0) then right_val=left_val
           if (left_val eq 0) then left_val=right_val
           spectrum(idx(i))=(left_val+right_val)/2
;           print,j,idx(i),left_val,k,right_val,l,(left_val+right_val)/2
           count=count+1
         endfor
       endif  
       zarr(j,*)=spectrum		;reassign new spectrum
     endfor

     print,'% fill_lav: Missing Data Points filled: ',count

end

;------------------------------------------------------------------------------

pro fill_lav_orb, orbit, leg, xarr, zarr

print,'% fill_lav_orb: in progress'

   nominal_diff=4.933		;nominal time between half orbits
   half_orb_diff=fltarr(n_elements(leg)-1)

;find out between which indicies of the time array there is a gap: if more
;than one half orbit - then there is something missing.  

   orbfrac=orbit+leg/2

   for i=0, n_elements(leg)-2 do begin
      half_orb_diff(i)=orbfrac(i+1) - orbfrac(i)
   endfor

   mis=where(half_orb_diff ne 0.5 and half_orb_diff ne 0.0)
   no_half_orb_mis=half_orb_diff(mis) / 0.5 -1

   k=n_elements(mis)
;   for i=0,k-1 do print,orbfrac(mis(i)),orbfrac(mis(i)+1),no_half_orb_mis(i)

;make a list of array index positions for the missing data. E.g., if data
;is missing between indices 4 and 5, and it's two half orbits, make extra
;indices 4.33,4,66
   
   extra_idx=fltarr(total(no_half_orb_mis))
   l=0
   for i=0,k-1 do begin
      for j=1,no_half_orb_mis(i) do begin
         extra_idx(l)=mis(i)+j/(no_half_orb_mis(i)+1)
         l=l+1
      endfor
   endfor

;add exisiting indices to the new ones and sort

   new_idx=[findgen(n_elements(orbit)),extra_idx]
   new_idx=new_idx(sort(new_idx))

;array new_idx now contains the indices the exisiting arrays have to 
;be interpolated into.

;get a new array of orbfrac, orbit, leg, xarr, zarr

   orbfrac=interpolate(orbfrac,new_idx)
   orbit=round(round(orbfrac/0.5) * 0.5 - 0.01)
   leg=round(orbfrac/0.5) mod 2
   xarr=interpolate(xarr,new_idx)
   zarr=interpolate(zarr,new_idx,findgen(32),/grid,/cubic)

   print,'% fill_lav_orb: done'

end

;------------------------------------------------------------------------------

pro smooth_horiz, zarr, smrange

   if smrange ge 3 then begin 
      print,'% smooth_horiz: in progress'
      szarr=size(zarr)
      for i=0,szarr(2)-1 do begin
         l_slice=zarr(*,i)
         l_slice=smooth(l_slice,smrange)
         zarr(*,i)=l_slice
      endfor
      print,'% smooth_horiz: done'
   endif else return

end

;------------------------------------------------------------------------------

pro smooth_vert, zarr, smrange

   if smrange ge 2 then begin 
      print,'% smooth_vert: in progress'
      szarr=size(zarr)
      for i=0,szarr(1)-1 do begin
         spec=zarr(i,*)
         spec=smooth(spec,smrange)
         zarr(i,*)=spec
      endfor
      print,'% smooth_vert: done'
   endif else return

end
