;******************************************************************************
;* file p_crres_meb_plaspar_overview.pro
;*
;******************************************************************************
;* PROCEDURE:     
;*      p_crres_meb_plaspar_overview, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (and/or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure PAPCD_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to make use of
;*              an extra common block.
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_crres_meb_plaspar_overview, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;******************************************************************************
pro p_crres_meb_plaspar_overview, panel, dtype, swittch, $
                                  OUTPUT=OUTPUT, $
                                  OVERPLOT=OVERPLOT, $
                                  PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                                  SUBTABLE=SUBTABLE, $
                                  _EXTRA=extra_par

; The following common blocks are needed for PAPCO:

   common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
   common yscale, yscl			;man/auto yscaling
   common zscale, zscl			;man/auto zscaling
   common shift_label, down             ;common for x-axis label shifting
   common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_crres_meb_plaspar_overview. 
   
   common crres_meb_plaspar_overview, info, lsum, valdat
   common crres_meb_plaspar_overview_extra, intime, sp, lbin, lrange
   
   papco_Set_SubTable, SUBTABLE         ;sets the papco color table index
   panelset,panel			;sets the panel position viewport

; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.

   ytickno=50*(!p.position(3)-!p.position(1))
   if ytickno gt 10 then ytickno=10

;   xtickno=6

   ch=dtype(1)

; Setup labels 
   if (dtype(0) eq 1) then begin
     cspecies='ions'
     totno=228
   endif else begin 
     cspecies='electrons'
     totno=266
   endelse
   
   uytitle='L-value'
   
;Setup variables
   
   nobins=(lrange-1)/float(lbin)+1
   zarr=fltarr(sp,nobins)
   yval=fltarr(sp,nobins)
   lsumn=lsum
   
;y-axis values
   
   yarr=fltarr(nobins,2)
   for i=0,nobins-1 do begin
     yarr(i,0)=1+i*lbin
     yarr(i,1)=1+(i+1)*lbin
   endfor
   
; time arrays
   
   time=(intime(*,0)+intime(*,1))/2.
   xarr=intime
   
   xut1=mjdt_start.mjd+mjdt_start.t / 86400.0                     
   xut2=mjdt_end.mjd  +mjdt_end.t   / 86400.0  
   
; plaspar array
   
;divide sum in bins by number of bins where not zero
   index=where(valdat ne 0,count)
   if count gt 1 then $
     lsumn(index)=lsum(index)/valdat(index)
   
   case ch of
     0:begin
       zarr(0:sp-1,*)=lsumn(0:sp-1,1,*)
       utitle='!Cnumber density' & coltit='cm!E-3'
     end
     1:begin
       zarr(0:sp-1,*)=lsumn(0:sp-1,0,*)
;         utitle='!Cenergy density' & coltit='keVcm!E-3' 
       utitle='!C54-147keV' & coltit='[keVcm!E-3!n]'   
     end
     2:begin
       zarr(0:sp-1,*)=lsumn(0:sp-1,2,*)
       utitle='!C!9x!3 plasma pres.' & coltit='Nm!E-2' 
     end
     3:begin
       zarr(0:sp-1,*)=lsumn(0:sp-1,3,*)
       utitle='!C!9#!3 plasma pres.' & coltit='Nm!E-2' 
     end
     4:begin
       zarr(0:sp-1,*)=2./3. * lsumn(0:sp-1,0,*)
       utitle='!Ctot. plasma pres.' & coltit='Nm!E-2' 
     end
     5:begin
       zarr(0:sp-1,*)=lsumn(0:sp-1,5,*)
       endenb=2.4834E-3 * zarr^2
       utitle='!Cbeta perp.' & coltit='!7b!D!9x!3'
     end
     6:begin
       endenb=2.4834E-3*plaspararr(*,5)^2
       zarr=plaspararr(*,3) / endenb
       utitle='!Cbeta paral.' & coltit='!7b!D!9#!3'
     end
     7:begin
       endenb=2.4834E-3*plaspararr(*,5)^2
       zarr=2./3. * plaspararr(*,0) / endenb
       utitle='!Cbeta total' & coltit='!7b!3'
     end
     8:begin
       zarr(0:sp-1,*)=lsumn(0:sp-1,3,*)
       zarr2=zarr
       zarr(0:sp-1,*)=lsumn(0:sp-1,2,*)
       index=where(zarr ne 0,count)
       if count gt 2 then zarr(index)=zarr2(index)/zarr(index)
       zmin=0 & zmax=1.2
       linlogbit=0 & relbit=1
       utitle='!CPress. aniso.' & coltit='P!Dpar!N/P!Dperp'
     end
     9:begin
       zarr(0:sp-1,*)=lsumn(0:sp-1,4,*)
       zarr= zarr / totno * 100
       zmin=0 & zmax=30
       linlogbit=0 & relbit=1
       utitle='!Cdata missing' & coltit='%'
     end
   endcase
   
   uztit=coltit
   utitle=cspecies+utitle
   
   yst=yarr(0,0)
   yen=yarr(nobins-1,1)

; restrict data to actual time range requested - this makes for faster zooming

   index=where((time gt xut1) and (time lt xut2),count)
   if count gt 1 then begin
     zarr=zarr(index,*)
     xarr=xarr(index,*)
     time=time(index)
   endif
   
   index=where(xarr(1:*,0)-xarr(0:*,0) gt 0.,count) + 1
   if count gt 1 then begin
     time=time(index)
     xarr=xarr(index,*)
     zarr=zarr(index,*)
   endif
   
; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data

; the procedure papco_autorange (in papco_XX/papco_lib/autoscaling is
; optional and sets the zlimits automatically for the data, attempting
; to use the full dynamic range of the color table. you may need to
; supply your own procedure for your own data.

; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling
   
   if (zscl(panel(0),0) eq 1) then begin
     zmin=zscl(panel(0),1)
     zmax=zscl(panel(0),2)
   endif else begin
     papco_autorange,zarr,zmin,zmax, $
       log=zscl(panel(0),3), exclude=0, nodata=1e-20
     zscl(panel(0),1)=zmin
     zscl(panel(0),2)=zmax
   endelse

   if (yscl(panel(0),0) eq 1) then begin
     yst=yscl(panel(0),1)
     yen=yscl(panel(0),2)
   endif else begin
     yscl(panel(0),1)=yst
     yscl(panel(0),2)=yen
   endelse
   
; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO:
;
;    !P.CHARSIZE  = widgetData.default_charsize    
;    !P.CHARTHICK = widgetData.default_charthick
;    !P.FONT      = -1
;    !P.NOERASE   = 1
;    !P.TICKLEN   = widgetData.default_TICKLEN
;    !X.TICKS     = widgetData.default_XTICKS
;    !X.STYLE     = widgetData.default_XSTYLE
;    !Y.STYLE     = 1
;    !X.MINOR     = widgetData.default_XMINOR
;
; Overriding these values by setting keywords is of course possible
; and allowed but discouraged, since 
; then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure.

   extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zmin,zmax], $
                  ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
                  ytitle:uytitle, xtickformat:'noticks'}

; check if the passed-in structure extra_par was set. If not, create
; it and set color to default black; which for the color table used by
; papco is 1. If extra_par was set, see if the tag color exists. If it
; does, leave it unchanged, if not, add it and set color to default
; black. 

  if n_elements(extra_par) EQ 0 then extra_par={color:1} else begin
    names=tag_names(extra_par)
    res=where(names eq 'COLOR',count)
    if count eq 0 then extra_par=create_struct({color:1}, extra_par)
  endelse  

; add keyword structure set locally and the one passed in

  extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; change the xtickformat if the panel is the bottom one and needs to
; plot a time axis   
   
  if panel(0) eq 0 then extra_plotPar.xtickformat='mjd_dec_2'
   
; You might have to add your own xtickformat for time axis
; labeling. If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; You can at this point also define a custom ytickformat, or leave it out for
; IDL default plot behaviour.
   
; "down" sets the axis labeling level. 
  
  down=0

;do color plot and color bar plot. nodata specifies the "nodata" flag
;in your data. Here we do not plot axis and data seperate, as
;papco_plot_colorspec cannot be used as an overplot!

  papco_plot_colorspec,zarr,xarr,yarr, $
    nodata=1e-20, _extra=extra_plotPar
  papco_colorbar,zmin,zmax,0,0,uztit
  
; store the coordinate information into common block

  plotted_x = !x
  plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label - 0r use any other function you define yourself.
; The routine x_side_label puts x-axis label to the right of the axis.
  
  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
    extra_plotPar.xtickformat='mjd_dec_1'
    extra_plotPar.color=1
    down=1
    plot,time,time, _extra=extra_plotPar, /NODATA
    extra_plotPar.xtickformat='crres_meb_plaspar_orb_ticks'
    extra_plotPar.color=1
    down=2
    plot,time,time, _extra=extra_plotPar, /NODATA
    x_side_label,panel,'Time(UT)!CDate!COrbit'
  endif
   
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.

  right_side_label,panel,utitle,/rot90
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  
  if KEYWORD_SET(OUTPUT) then begin
    print,'% p_crres_meb_plaspar_overview: Writing plot data out to file'
    description='crres_meb_plaspar_overview template sample data - Energy spectra, 10 channels'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, time, zarr, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
  endif  
   
end 

;------------------------------------------------------------------------------
; This function labels orbits given that a mag-lav file was read

function crres_meb_plaspar_orb_ticks,axis,index,t
  
  common crres_meb_plaspar_overview, info, lsum, valdat
  common crres_meb_plaspar_overview_extra, intime, sp, lbin, lrange
  
  time=(intime(*,0)+intime(*,1))/2.
  res=min(abs(time-t),mindex)
  orbit=info(mindex,3)
  if res lt 1.0 then $
    return,string(orbit,format="('!C!C',i4.4)") $
  else $
    return,''
  
end 






