;******************************************************************************
;*  NAME:        r_crres_meb
;*
;*  DESCRIPTION: Routine to read crres_meb files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*                                  Reads data by orbit only!
;*
;*  INPUT:       orbit - orbit number of crres orbit
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
PRO r_crres_meb, in_orbit, ty, VERBOSE=VERBOSE

COMMON crres_meb_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg

get_err_no=0  &  get_err_msg='crres_meb read succesfully'
   
; check env. variable pointing to data directory
if not (papco_check_data_env('CRRES_MEB_DATA', path=PATH)) then return
   
; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
input_header=0  &  input_data=0   
input_header={crres_meb, orbit:0, dim1:0l, dim2:0l, ndat:0l}
input_header.orbit=in_orbit
   
; loop through from start to end time concatonating data files into one
; array. Here data are stored by orbit, so we need to find the orbit
; range corresponding to the start and end times.
time=0  &  no_files_read=0
   
; check if orbit is non-zero, then read that orbit 
; - otherwise get list of orbits to read
if in_orbit eq 0 then orbit_list=papco_getOrbitList('CRRES') $
    else orbit_list=[in_orbit]
      
for i=0,n_elements(orbit_list)-1 do begin
    orbit=orbit_list(i)

    ; read one orbit of data
    r_crres_meb_oneorbit_idlsunbin, orbit, path, ty, /VERBOSE
     
    ; check for an error in get_error - and then get the file remotely!
    ; comment this section out if you do not want this feature.
    if get_err_no ne 0 then begin      
      c_url=papco_getenv('CRRES_MEB_FTP_ARCHIVE')
    ; assemble the data file from the orbit requested
      orbitstr=string(format='(i4.4)',orbit)
      fln='bin'+orbitstr
      c_file=fln+'.idl'
      c_download_dir=path
      papco_get_remote_data, c_url, c_file, c_download_dir
      ;check if file "arrived" and then read it
      if get_err_no eq 0 then $
        r_crres_meb_oneorbit_idlsunbin, path+fln, /VERBOSE $
      else begin
        get_err_msg='% r_crres_meb: No data locally, remote get failed!C  ' $
          + get_err_msg
        return
      endelse  
    endif   
     
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
      if no_files_read eq 0 then begin   ;start the array buffer
        input_data_buf=temporary(input_data)
        ndat=input_header.ndat
      endif else begin                   ;otherwise concatonate arrays
        input_data_buf=[temporary(input_data_buf),temporary(input_data)]
        ndat=ndat+input_header.ndat
      endelse        
      input_data=0                     ;clear array for reuse
      no_files_read=no_files_read+1
    endif
  endfor         
   
; check if any files were read   
  if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
    input_header.ndat=ndat
  endif else begin
    get_err_no=1
    get_err_msg='no files found for timerange'
    return
  endelse      
  message, get_err_msg, /cont
   
end

;******************************************************************************
;*  NAME:        r_crres_meb_oneorbit_idlsunbin, orbit, path, ty
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       orbit  - orbit number
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_crres_meb_oneorbit_idlsunbin, orbit, path, ty, VERBOSE=VERBOSE
  
common crres_meb_data, input_header, input_data
common get_error, get_err_no, get_err_msg
  
get_err_no=0 & get_err_msg='crres_meb oneorbit read succesfully'
  
; clear data arrays for reuse
input_data=0  
    
; assemble the data file from the orbit requested
orbitstr=string(format='(i4.4)',orbit)
fln='bin'+orbitstr 
    
; see if file is gzipped, and unzip  
result=findfile(path+fln+'.idl.gz',count=c)
if c eq 1 then begin
    if verbose then message,'unzipping file '+fln+'.idl.gz',/cont
    spawn, 'gunzip '+ result
    gzipflag=1
endif else gzipflag=0
  
; NOTE: thse IDL binaries where written on SunOS and are NOT portable
; as is. This routine tests for linux and coverts variables if needed.  
  
; try to open the file, and if not found, read original  
openr,unit,path+fln+'.idl', /GET_LUN, ERROR=err
if err ne 0 then goto,read_original
if verbose then message,'reading file '+fln+'.idl',/cont
  
;------------------------------------------------------------------------------
; read one orbit of data - idl binaries
  nn=700  ; max no of data points: 10 hours at 1 min resolution 
          ; enough for one orbit (9h50)

;initialise variables
  iorb=long(0)  &  iyear=long(0)  &  iday=long(0)
  iut1=long(0)  &  iut2=long(0)
  iop1=long(0)  &  isec=long(0)   & idetek=long(0)  & intv=long(0)

  ut=float(0)  &  wmlt=float(0)  &  wilat=float(0)
  wlshel=float(0)  &  wrad=float(0)  &  dmin=float(0)

;Arrays for ephemeris information
  xutepf=fltarr(nn)  & xmlt=fltarr(nn)  & xilat=fltarr(nn)
  xlshel=fltarr(nn)  & xrad=fltarr(nn)  & xdmin=fltarr(nn)

;Arrays for data returned
  if (ty(0) eq 0) and (ty(1) eq 1) and (ty(2) eq 0) then ymat=fltarr(nn,14)
  if (ty(0) eq 1) and (ty(1) eq 1) and (ty(2) eq 0) then ymat=fltarr(nn,12)
  if (ty(1) eq 1) and (ty(2) eq 1) then ymat=fltarr(nn,19)
  if (ty(1) eq 0) then ymat=fltarr(nn)
  if (ty(0) eq 0) and (ty(2) eq 2) then ymat=fltarr(nn,19,14)
  if (ty(0) eq 1) and (ty(2) eq 2) then ymat=fltarr(nn,19,12)
  if (ty(1) eq 0) and (ty(2) eq 1) then ymat=fltarr(nn,19)

;  Arrays fuer Pitchwinkel
  PWIP =fltarr(19)     &  PWIE=fltarr(19)
  PWDP =fltarr(19,12)  &  PWDE=fltarr(19,14)
  IPWIP=lonarr(19)     &  IPWIE=lonarr(19)
  IPWDP=lonarr(19,12)  &  IPWDE=lonarr(19,14)  
  NPWIP=intarr(19)     &  NPWIE=intarr(19)
  NPWDP=intarr(19,12)  &  NPWDE=intarr(19,14)

  ndat=-1
  on_ioerror, errorLoop ; error handler for file IO
  readu,unit,IORB,IYEAR,IDAY,IUT1,IUT2,IOP1,ISEC,IDETEK,INTV
  
  if strupcase(!version.os) eq 'LINUX' then begin
    byteorder,IORB,IYEAR,IDAY, IUT1,IUT2,IOP1,/LSWAP
    byteorder,ISEC,IDETEK,INTV,/LSWAP
  endif
  
  ;loop through data file reading records by the minute
  while(not eof(unit)) do begin
    readu,unit,UT,WMLT,WILAT,WLSHEL,WRAD,DMIN 
    readu,unit,PWIP,IPWIP,NPWIP,PWIE,IPWIE,NPWIE,PWDP,IPWDP,NPWDP,PWDE,NPWDE
    
    if strupcase(!version.os) eq 'LINUX' then begin
      byteorder, UT, WMLT,WILAT,WLSHEL,WRAD,DMIN, /FTOXDR
      byteorder, PWIP,  PWIE,  PWDP,  PWDE,  /FTOXDR
      byteorder, IPWIP, IPWIE, IPWDP, IPWDE, /LSWAP
      byteorder, NPWIP, NPWIE, NPWDP, NPWDE, /SSWAP
    endif
    
    ndat=ndat+1
    ;print,ndat,ut, wmlt,wilat,wlshel,wrad
    XUTEPF(NDAT) = UT      &  XMLT  (NDAT) = WMLT  &  XILAT (NDAT) = WILAT
    XLSHEL(NDAT) = WLSHEL  &  XRAD  (NDAT) = WRAD  &  XDMIN (NDAT) = DMIN
    
    ;select data required
    if (ty(0) eq 0) then begin ;electrons
         if (ty(1) eq 0) then begin ;integral
           index=where(npwie ne 0,count)
           if count ne 0 then ymat(ndat,index)=pwie(index)/npwie(index)
         endif else begin       ;differential
           index=where(npwde ne 0,count)
           if count ne 0 then pwde(index)=pwde(index)/npwde(index)
           case ty(2) of
              0 : ymat(ndat,*)=pwde(ty(3)-1,*) ;select a pitch angle
              1 : ymat(ndat,*)=pwde(*,ty(3)-1) ;select an energy
              2 : ymat(ndat,*,*)=pwde ;select full pa/E array
           endcase
         endelse
    endif else begin          ;ions
         if (ty(1) eq 0) then begin ;integral
           index=where(npwip ne 0,count)
           if count ne 0 then ymat(ndat,index)=pwip(index)/npwip(index)
         endif else begin       ;differential
           index=where(npwdp ne 0,count)
           if count ne 0 then pwdp(index)=pwdp(index)/npwdp(index)
           case ty(2) of
              0 : ymat(ndat,*)=pwdp(ty(3)-1,*) ;select a pitch angle
              1 : ymat(ndat,*)=pwdp(*,ty(3)-1) ;select an energy
              2 : ymat(ndat,*,*)=pwdp ;select full pa/E array
            endcase
         endelse
    endelse

  endwhile

  close,unit  &  free_lun, unit

;--select time interval

;  if xut1=xut2 then the whole orbit is returned   
  ixut1=0
  ixut2=ndat-1
  xut1=0 & xut2=0
  if xut1 eq xut2 then begin
    xut1=xutepf(ixut1)
    xut2=xutepf(ixut2)
  endif else begin
    for i=0,ndat-2 do begin
      if (xutepf(i) le xut1) and (xutepf(i+1) gt xut1) then ixut1=i
      if (xutepf(i) lt xut2) and (xutepf(i+1) ge xut2) then ixut2=i+1
    endfor
  endelse
  ndat=ixut2-ixut1
  xutepf=xutepf(ixut1:ixut2)
  xmlt=xmlt(ixut1:ixut2)
  xilat=xilat(ixut1:ixut2)
  xlshel=xlshel(ixut1:ixut2)
  xrad=xrad(ixut1:ixut2)
   
  if (ty(1) eq 1) and (ty(2) ne 2) then ymat=ymat(ixut1:ixut2,*)
  if (ty(1) eq 0) and (ty(2) ne 2) then ymat=ymat(ixut1:ixut2,*)
  if (ty(1) eq 1) and (ty(2) eq 2) then ymat=ymat(ixut1:ixut2,*,*)

;check if there are valid data points: return error if all zero 
  pts=n_elements(ymat)
  index=where(ymat le 0.0,nodat)
  if nodat eq pts then begin
    get_err_msg='no valid data points in this timerange'
    message,get_err_msg,/cont
    get_err_no=3
  endif
   
;fold data into input_data, input_header arrays.   
  result=size(ymat) & ndat=result(1) & dim1=0 & dim2=0
  case result(0) of
    1: begin
      data=0.0
    end
    2: begin
      dim1=result(2)
      data=fltarr(dim1)
    end
    3: begin
      dim1=result(2)
      dim2=result(3)
      data=fltarr(dim1,dim2)      
    end
  endcase
     
  input_header.dim1=dim1
  input_header.dim2=dim2
  input_header.ndat=ndat
  
  input_data_structname=string(ty,format="(4(i2.2))")
  dummy=create_struct(name=input_data_structname, $
                      'time',0.0d, $
                      'mlt',0.0, 'ilat',0.0, 'lshell',0.0, 'rad',0.0, $
                      'data',data)
  input_data=replicate(dummy,ndat)
   
  input_data.time   = xutepf
  input_data.mlt    = xmlt
  input_data.ilat   = xilat
  input_data.lshell = xlshel
  input_data.rad    = xrad
   
  case result(0) of
    1: for i=0,ndat-1 do input_data(i).data  = ymat(i)  
    2: for i=0,ndat-1 do input_data(i).data  = ymat(i,*)  
    3: for i=0,ndat-1 do input_data(i).data  = ymat(i,*,*)  
  endcase
  
;convert time to TAI for easy concatonation of files.  
  message,'converting to TAI...', /cont
  utc=doy2utc(IDAY,1900+IYEAR)
  for i=0,ndat-1 do begin
    utc.time=input_data(i).time*1000
    input_data(i).time=utc2tai(utc)
  endfor  
  return
  
;------------------------------------------------------------------------------

; this label is jumped at, when an IO-error occured.
errorLoop:
  close, unit
  free_lun, unit

  get_err_no=3
  get_err_msg=!ERR_STRING
  return   
;------------------------------------------------------------------------------
  
  read_original:
  ; if the .idl save file could not be found, then try to read the original
  message, 'File '+path+fln+'.idl' +' not found', /info
  r_crres_meb_oneorbit_original, fln, VERBOSE=VERBOSE
      
end   
  

PRO r_crres_meb_oneorbit_original, fln, VERBOSE=VERBOSE
common get_error, get_err_no, get_err_msg

message, 'Not implemented', /info
get_err_no = 5

END 
