;******************************************************************************
;* FUNCTION:     
;*    FUNCTION  crres_mics_plaspar_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_PLOTINFO-type
;*                      this structure is edited in the panel  editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at last some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I 
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, switch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commincate between your
;*     panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which papco supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      papco internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $                       
;*                      xut2:0.0}                           
;*
;*                  
;*      For an enhanced description read 
;*                          $papco_XX/papco_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  crres_mics_plaspar_panel, aPlotInfo, GROUP=group, ACTION=ACTION

  COMMON crres_mics_plaspar_panel_DATA, widgetData

; if GROUP is not specified, the group-leader will be the desktop
  IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
  IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
  additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
  panel_base=papco_widget_base(title=aPlotInfo.panelKind, /ALIGN_CENTER, $
                               column=1, GROUP_LEADER=group, /MODAL)

; All papco_panel_Add* routines are papco panel widgets that can be included
; in the panel editor or not. They provide functionality that papco supports.
; They dynamically extend the panel editors widget structure, and all the 
; events they generate are supported by papco_panel_Event, transparent to the
; user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
  widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
   
;
; --- Choose components / plot type--------------------------------------------

  typeNames=[	' H+', $
	   	' He+', $
		' He++', $
		' O+', $
                ' O++', $
                ' Proton', $
                ' DCR,', $
                ' O+ / Proton' , $
                ' O+ / H+' , $
                ' He++ / Proton' , $
                ' He++ / H+' , $
                ' He+ / Proton' , $
                ' He+ / H+' ]
  
  type_base=CW_BGROUP(panel_base, typeNames, /FRAME, COLUMN=5, /EXCLUSIVE, $
                      UVALUE='spb_type', SET_VALUE=aPlotInfo.typeVector(0) )
  
; --- Type: Energy Density ---------
  data_type_bt_base=WIDGET_BASE(panel_base, /FRAME, ROW=1, /EXCLUSIVE)
  spb_en_den=WIDGET_BUTTON(data_type_bt_base, $
                           VALUE='Energy Density', UVALUE='spb_en_den')
  IF aPlotInfo.typeVector(1) EQ 0  THEN WIDGET_CONTROL,spb_en_den, SET_BUTTON=1
  
; --- Reload choice ---
  reload_names=['Relaod', 'Adopt']
  reload_base=CW_BGROUP(panel_base, reload_names, /FRAME, ROW=1, /EXCLUSIVE, $
                        UVALUE='spb_reload', $
                        SET_VALUE=aPlotInfo.ioptions(5))     
  
; --- Time Average ------------------
  average_base =WIDGET_BASE(panel_base, COLUMN=1, /FRAME)
  sl_average=WIDGET_SLIDER(average_base, XSIZE=300, /DRAG, $
                          VALUE=aPlotInfo.typeVector(2) < 30 > 1, $
                          UVALUE='sl_average', MINIMUM=1, MAXIMUM=30)
  lbl_average=WIDGET_LABEL(average_base, /ALIGN_CENTER, $
                          VALUE=strcenter('Time average (min)',50) )	
  
;--- Add limit choices to select L-range --------------------------------------
  l_base= WIDGET_BASE(panel_base, /FRAME, COLUMN=1, /align_center)
  l_label_base=WIDGET_BASE(l_base, ROW=1, /align_center)
  l_low_label= WIDGET_LABEL(l_label_base, /align_center, $
  VALUE=strformat(string(aPlotInfo.options(2),format="(f4.1)"),35,/center), $
                             UVALUE='l_low_label')
  l_high_label=WIDGET_LABEL(l_label_base, /align_center, $
  VALUE=strformat(string(aPlotInfo.options(3),format="(f4.1)"),35,/center), $
                             UVALUE='l_high_label')

  l_slider_base=WIDGET_BASE(l_base, row=1, /align_center)
  ; --- Low L-value ---
  l_low=WIDGET_SLIDER(l_slider_Base, /DRAG, /SUPPRESS_VALUE, xsize=220, $
                  title=strformat('Low L-value',35,/center), $
         	  VALUE=(((aPlotInfo.options(2) < 12)-1) * 10) < 111 > 0, $
                  UVALUE='l_low', MINIMUM=0, MAXIMUM=110)	      
  ; --- High L-value ---
  l_high=WIDGET_SLIDER(l_slider_Base, /DRAG, /SUPPRESS_VALUE, xsize=220, $
                  title=strformat('High  L-value',35,/center), $
		  VALUE=(((aPlotInfo.options(3) < 12)-1) * 10) < 111 > 0, $
                  UVALUE='l_high', MINIMUM=0, MAXIMUM=110)        
  
; for examples of standard buttons or sliders look at other panel editors.
; some standard buttons which have been recycled before can be found in
; $papco_MODULES/panels&slice/more_panel_widgets

; As the same panel is called when you edit a plot, interrogate the tags in the
; plotInfoStruct (aplotInfo) passed in and set your widgets accordingly. 
; In this way the widget is always reinitialized to its last state.

; The next four papco panel widgets will be added only if you want to
; support that functionality, if they make sense for your crres_mics_plaspar.

;--- add the papco panel widget for specifying own plot description label -----
  widgets=papco_panel_AddDescr_Widgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
  widgets=papco_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
  widgets=papco_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.
;    if you want to make use of the configure feature, then add the
;    keyword CONFIG
  widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, $
                                       /CONFIG, ROW=2)
  additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create 
;    that helpfile in the papco_interface directory of your module (a template
;    for this help file exist in the crres_mics_plaspar example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
   thisWidgetsData={helpfile:'crres_mics_plaspar_panel.help', $
                    typevector:aPlotInfo.typevector, $
                    options:aPlotInfo.options, $
                    ioptions:aPlotInfo.ioptions, $
                    sl_average:sl_average, $
                    lbl_average:lbl_average, $
                    l_low:l_low, $
                    l_high:l_high, $
                    l_low_label:l_low_label, $
                    l_high_label:l_high_label}
   widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   papco_Panel_Realize, panel_base, widgetData
   papco_xmanager, 'crres_mics_plaspar_panel', panel_base

; the returned data consists of the standard-data inherited from
; papco_PANEL.PRO and your new data - the typeVector-tag and switch-tag must
; always be included, but you can add more. However, if possible stick to the 
; tags provided in aPlotInfo, it makes life easier. 
   returnData={typevector:widgetData.typevector, $
               ioptions:widgetData.ioptions, $
               options:widgetData.options}
   additionalRData=papco_panel_ReturnData(widgetData)
   return, create_struct(ReturnData, additionalRData)
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO crres_mics_plaspar_panel_refresh
;*
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only the fields corresponding to the set keyword are updated
;*
;* CALLING SEQUENCE:
;*      have a look at PRO crres_panel_event
;*
;* MODIFICATION HISTORY:       
;*      February 2001, R Friedel 
;******************************************************************************
pro crres_mics_plaspar_panel_refresh, ALL=ALL, LOW_L=LOW_L, HIGH_L=HIGH_L
  
  common crres_mics_plaspar_panel_data, widgetData
  
  if keyword_set(ALL) then begin
    LOW_L=1  &  HIGH_L=1
  endif
  
  if keyword_set(LOW_L) then begin
    high_l=widgetData.options(3) > (widgetData.options(2)+0.1)
    widgetData.options(3)=high_l
    WIDGET_CONTROL, widgetData.l_high, $
      SET_VALUE=(widgetData.options(3) -1) * 10
    label=string(widgetData.options(3),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_high_label, $
      SET_VALUE=strformat(label,35,/center)
    label=string(widgetData.options(2),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_low_label, $
      SET_VALUE=strformat(label,35,/center)
  end
  
  if keyword_set(HIGH_L) then begin
    low_l=widgetData.options(2) < (widgetData.options(3)-0.1)
    widgetData.options(2)=low_l
    WIDGET_CONTROL, widgetData.l_low, $
      SET_VALUE=(widgetData.options(2) -1) * 10
    label=string(widgetData.options(3),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_high_label, $
      SET_VALUE=strformat(label,35,/center)
    label=string(widgetData.options(2),format="(f4.1)")
    WIDGET_CONTROL, widgetData.l_low_label, $
      SET_VALUE=strformat(label,35,/center)    
  end  
  
END
  

;******************************************************************************
;* PROCEDURE:     
;*      PRO crres_mics_plaspar_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO crres_mics_plaspar_panel_event, event

  common crres_mics_plaspar_panel_data, widgetData

  WIDGET_CONTROL,GET_UVALUE=uval, event.id

  case uval of
    'spb_type' : begin
      widgetData.typeVector(0)=event.value
    end    
    'spb_en_den': begin
      widgetData.typeVector(1)=0
    end
    'spb_reload': begin
      widgetData.ioptions(5)=event.value
    end     
    'sl_average': begin
      widgetData.typeVector(2)=event.value
    end
    'l_low'    : begin
      widgetData.options(2)=event.value / 10.0 + 1.0
      crres_mics_plaspar_panel_refresh, /LOW_L
    end  
    'l_high'   : begin
      widgetData.options(3)=event.value / 10.0 + 1.0
      crres_mics_plaspar_panel_refresh, /HIGH_L
    end      
    else: widgetData=papco_panel_Event(event, widgetData)
  endcase
   
END
