;******************************************************************************
;* FILE:
;*    crres_mics_spectra_panel.pro
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  crres_mics_spectra_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_PLOTINFO-type
;*                      this structure is edited in the panel  editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at last some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I 
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      switch:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, switch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          switch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commincate between your
;*     panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which papco supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      papco internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $                       
;*                      xut2:0.0}                           
;*
;*                  
;*      For an enhanced description read 
;*                          $papco_XX/papco_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  crres_mics_spectra_panel, aPlotInfo, GROUP=group, ACTION=ACTION

   COMMON crres_mics_spectra_panel_DATA, widgetData

; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
   panel_base=papco_widget_base(title=aPlotInfo.panelKind,$
                                column=1, /modal, $
                                GROUP_LEADER=group)

; All papco_panel_Add* routines are papco panel widgets that can be included
; in the panel editor or not. They provide functionality that papco supports.
; They dynamically extend the panel editors widget structure, and all the 
; events they generate are supported by papco_panel_Event, transparent to the
; user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
   widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
; --- Choose components / plot type--------------------------------------------

   type_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=4, /EXCLUSIVE) 

   typeNames=[	'R0', $
		'R1 H+', $
	   	'R2 He+', $
		'R3 He++', $
		'R4 O+', $
		'R5 O++', $
		'R6', $
		'R7', $
		'Proton', $
		'HVY', $
		'FSR', $
		'DCR', $
		'TCR', $
		'MSS' ]

   spb_type=lonarr(N_ELEMENTS(typeNames))

   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
      spb_type(i)=WIDGET_BUTTON(type_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))
   
; --- Type Diff. Num. Flux, Diff. En. Flux )---------
   
   data_type_bt_base=WIDGET_BASE(panel_base, /FRAME, ROW=1, /EXCLUSIVE)
   spb_num_flux=WIDGET_BUTTON(data_type_bt_base, $
                              VALUE='Diff. Num. Flux', $
                              UVALUE='spb_num_flux')
   spb_en_flux=WIDGET_BUTTON(data_type_bt_base, $
                             VALUE='Diff. En. Flux', $
                             UVALUE='spb_en_flux')   
   
   IF aPlotInfo.typeVector(1) EQ 0  $
     THEN WIDGET_CONTROL, spb_num_flux, SET_BUTTON=1
   IF aPlotInfo.typeVector(1) EQ 1  $
     THEN WIDGET_CONTROL, spb_en_flux, SET_BUTTON=1
   
; The next four papco panel widgets will be added only if you want to
; support that functionality, and if they make sense for your crres_mics_spectra.

;--- add the papco panel widget for choosing the color bar for color plots ----
   widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
   widgets=papco_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
   widgets=papco_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.
;    if you want to make use of the configure feature, then add the
;    keyword CONFIG
   widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create 
;    that helpfile in the papco_interface directory of your module (a template
;    for this help file exist in the crres_mics_spectra example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
   thisWidgetsData={helpfile:'crres_mics_spectra_panel.help', $
                    typevector:aPlotInfo.typevector, $
                    spb_num_flux:spb_num_flux, $
                    spb_en_flux:spb_en_flux}
   widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   papco_Panel_Realize, panel_base, widgetData
   papco_xmanager, 'crres_mics_spectra_panel', panel_base

; the returned data consists of the standard-data inherited from
; papco_PANEL.PRO and your new data - the typeVector-tag and switch-tag must
; always be included, but you can add more. However, if possible stick to the 
; tags provided in aPlotInfo, it makes life easier. 
   returnData={typevector:widgetData.typevector}
   additionalRData=papco_panel_ReturnData(widgetData)

   return, create_struct(ReturnData, additionalRData)
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO crres_mics_spectra_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO crres_mics_spectra_panel_event, event

  COMMON crres_mics_spectra_panel_DATA, widgetData
  
  WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to 
; 'papco_panel_Event', which handles the events generated
; by the papco-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'papco_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.
   
  IF strleft(uval, 8) EQ 'spb_type' THEN BEGIN
    widgetData.typeVector(0)=fix(strfrom(uval, 8))
    return
  ENDIF 
  
  case uval of
    'spb_num_density': begin
      widgetData.typeVector(1)=2
    end
    'spb_en_density':begin
      widgetData.typeVector(1)=3
    end
    else: widgetData=papco_panel_Event(event, widgetData)
  endcase
  
END
