;------------------------------------------------------------------------------
; File: MICS_CALS.PRO	Calibration Routines for CRRES MICS 
; Revision: 02/11/98	J. L. Roeder
;------------------------------------------------------------------------------
; Routines:
;	mics_ch_num	Returns channel number from MICS Channel Id string
;	def_micseff	Define the mics efficiency structure
;	rd_mics_cals	Reads efficiencies into common block from file
;	effic_correct	Correction factor for microchannel plate efficiency
;	conv_fact	Extract the conversion factors for a particular channel
;------------------------------------------------------------------------------
function mics_ch_num,  chan_str

;       Returns channel number from MICS Channel Id string

;       Chan is the string name corresponding to the nch channel number
channels = ['FSR', 'DCR', 'TCR', 'MSS', 'P', 'ALPHA', 'R0', 'H+', 'He+',  $
	'He++', 'O+', 'O++', 'O+++', 'O2+']
ichan = where(chan_str eq channels)
if ichan(0) ne -1 then ichan = ichan(0) else stop, '%mics_ch_num: bad chan Id'

return, ichan
end
;------------------------------------------------------------------------------
function def_micseff,  dummy

;	Defines mics efficiency structure

micseff = {eff_mics,  		$
	fsr : fltarr(32),  	$
	dcr : fltarr(32), 	$
	tcr : fltarr(32), 	$
	mss : fltarr(32), 	$
	proton:fltarr(32), 	$
	alpha : fltarr(32), 	$
	r0 : fltarr(32),  	$
	h1 : fltarr(32), 	$
	he1 : fltarr(32), 	$
	he2 : fltarr(32), 	$
	o1 : fltarr(32), 	$
	o2 : fltarr(32), 	$
	o3 : fltarr(32), 	$
	omul : fltarr(32)	$
	}

return,  micseff
end
;------------------------------------------------------------------------------
pro rd_mics_cals

;	Read CRRES MICS calibrations from disk file

; Inputs: None
; Outputs: In common block mics_cals

common mics_cals,  eoq,  geom,  de_over_e,  effic,  $
	tcr_ssd_com,  tcr_ssd_ref,  ssd_hi_lo

if n_elements(eoq) ne 32 then begin

	geom = 7.26e-3		; Geometric factor in cm2 ster
	de_over_e = .0745		; Delta E over E factor
        
        path=papco_getenv('PAPCO_MODULES')
	fn = path+'/crres_mics/plot_read/mics_cals.dat'
	openr, unit, fn, /GET_LUN
	print, 'Reading ', fn

;	Read energy per charge for each esa step
	label = ' '
	for i = 0, 1 do readf, unit, label
	eoq = fltarr(32)
	readf, unit, eoq

	effic = def_micseff()
	effic = replicate(effic, 2)
	a = fltarr(32)

;	Read efficiencies for SSD for HVPS = off and HVPS = 3
	for i = 0, 2 do readf, unit, label
	readf, unit, a
	effic(0).mss = a
	readf, unit, label
	readf, unit, a
	effic(1).mss = a

;	Read ratios of TCR to SSD for Reference Case for HVPS = off and HVPS = 3
	for i = 0, 2 do readf, unit, label
	tcr_ssd_ref = fltarr(32, 2)
	readf, unit, a
	tcr_ssd_ref(0, 0) = a
	readf, unit, label
	readf, unit, a
	tcr_ssd_ref(0, 1) = a

;	Read ratios of TCR to SSD for Comparison Case for HVPS = off and HVPS = 3
	readf, unit, label
	tcr_ssd_com = fltarr(32, 2)
	readf, unit, a
	tcr_ssd_com(0, 0) = a
	readf, unit, label
	readf, unit, a
	tcr_ssd_com(0, 1) = a

;	Read ratio of efficiencies for SSD for hi and low discriminator modes
	for i = 0, 2 do readf, unit, label
	ssd_hi_lo = fltarr(32)
	readf, unit, ssd_hi_lo

;	Read efficiencies for DCR for HVPS = off and HVPS = 3
	for i = 0, 2 do readf, unit, label
	readf, unit, a
	effic(0).dcr = a
	readf, unit, label
	readf, unit, a
	effic(1).dcr = a

;	Read efficiencies for Protons for HVPS = off and HVPS = 3
	for i = 0, 2 do readf, unit, label
	readf, unit, a
	effic(0).proton = a
	readf, unit, label
	readf, unit, a
	effic(1).proton = a

;	Read efficiencies for R1 (H+) for HVPS = off and HVPS = 3
	for i = 0, 2 do readf, unit, label
	readf, unit, a
	effic(0).h1 = a
	readf, unit, label
	readf, unit, a
	effic(1).h1 = a

;	Read efficiencies for R2 (He+) for HVPS = off and HVPS = 3
	for i = 0, 2 do readf, unit, label
	readf, unit, a
	effic(0).he1 = a
	readf, unit, label
	readf, unit, a
	effic(1).he1 = a

;	Read efficiencies for R3 (He++) for HVPS = off and HVPS = 3
	for i = 0, 2 do readf, unit, label
	readf, unit, a
	effic(0).he2 = a
	readf, unit, label
	readf, unit, a
	effic(1).he2 = a

;	Read efficiencies for R4 (O+) for HVPS = off and HVPS = 3
	for i = 0, 2 do readf, unit, label
	readf, unit, a
	effic(0).o1 = a
	readf, unit, label
	readf, unit, a
	effic(1).o1 = a

;	Read efficiencies for R5 (O++) for HVPS = off and HVPS = 3
	for i = 0, 2 do readf, unit, label
	readf, unit, a
	effic(0).o2 = a
	readf, unit, label
	readf, unit, a
	effic(1).o2 = a

;	Read efficiencies for R6 (O+++) for HVPS = off and HVPS = 3
	for i = 0, 2 do readf, unit, label
	readf, unit, a
	effic(0).o3 = a
	readf, unit, label
	readf, unit, a
	effic(1).o3 = a

;	Read efficiencies for R7 (O2+) for HVPS = off and HVPS = 3
	for i = 0, 2 do readf, unit, label
	readf, unit, a
	effic(0).omul = a
	readf, unit, label
	readf, unit, a
	effic(1).omul = a

	free_lun, unit

endif

end
;------------------------------------------------------------------------------
function effic_correct,  orbits

;	MICS Efficiency correction based on inner zone data

; Inputs:
;	orbit	Orbit number, 1-1067

common eff_correct,  orbnumb,  factors

if n_elements(factors) eq 0 then begin

;	Corrections based on DCR decay
	norm  =  700.
	factor0  =  .75

	coeff  =  [[2.903090,   -0.001903090,   0.0000000],  	$
		 [4.488464,   -0.012853989,   1.0428488e-5], 	$
		 [3.0758368,  -0.0011864311,  2.0191274e-7]]

;	Corrections based on TCR decay (not used)
;	norm  =  110.
;	factor0  =  .25

;	coeff  =  [[2.21782240,   -0.0029884426,   0.0000000],  	$
;		 [3.18229390,   -0.0111806670,   5.6015380e-6], 	$
;		 [2.16885690,   -0.0013135268,   8.8240503e-8]]

	orb_rng  =  [[66, 141],  [142, 312],  [313, 339],  [348, 356],  $
		    [340, 347],  [357, 1065]]

	coef_numb  =  [0, 1, 1, 1, 2, 2]

	norbits  =  1066
	orbnumb  =  indgen(norbits)	
	factors  =  replicate(factor0, norbits)
	numb_coef  =  replicate(-1, norbits)

	numb_interv  =  n_elements(coef_numb)
	for i  =  0,  numb_interv - 1 do begin
		o1  =  orb_rng(0, i)
		o2  =  orb_rng(1, i)
		numb_coef(o1:o2)  =  coef_numb(i)
	endfor

	for i = fix(min(coef_numb)), fix(max(coef_numb)) do begin
		j  =  where(numb_coef eq i)
		orb  =  orbnumb(j)
		c  =  coeff(*, i)
		fact  =  poly(orb, c)
		factors(j)  =  factor0 * 10.^fact / norm 
	endfor

endif

factor  =  factors(orbits)

return,  factor
end
;------------------------------------------------------------------------------
pro conv_fact, orbit, chan, hvps, factor

;	Extract the conversion factors from count rate to flux 
;	for a particular channel

; Inputs:
;	orbit	Orbit number, 1-1067
;	chan	Channel identifier string, ie one of 'FSR', 'DCR', 'TCR', 'MSS', 'P', 'ALPHA', 
;          'H+', 'He+', 'He++', 'O+', 'O++', 'O+++'
;	hvps	Postacceleration High Voltage Status: 0=Off, 1=Step 3

;	Outputs:
;	factor	Multiplicative factor to convert counts/s to diff number flux

common mics_cals,  eoq,  geom,  de_over_e,  effic,  $
		tcr_ssd_com,  tcr_ssd_ref,  ssd_hi_lo

	;Read calibration file for first execution
if n_elements( eoq) eq 0L then rd_mics_cals

;	Integer corresponding to the requested channel ID
ichan = mics_ch_num(chan)

;	Extract efficiency from structure using channel number as a tag index
eta = effic.(ichan)
eta1 = eta(*, hvps)

;	Efficiency correction for channel plate degradation
corr = effic_correct(orbit)

;	Flux conversion factor
factor  =  1./(corr * geom * de_over_e*eoq * (eta1)>1.e-10)

;	Insert zero where efficiency is flagged as negative
j=where( eta1 le 1.e-10)
if j(0) ne -1 then factor( j) = 0.

end
