;******************************************************************************
;* file p_crres_mics_overview.pro
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_crres_mics_overview, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (and/or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure PAPCD_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.switch (three element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to make use of
;*              an extra common block.
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_crres_mics_overview, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;******************************************************************************
pro p_crres_mics_overview, panel, dtype, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

; The following common blocks are needed for PAPCO:

   common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
   common yscale, yscl			;man/auto yscaling
   common zscale, zscl			;man/auto zscaling
   common shift_label, down             ;common for x-axis label shifting
   common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_crres_mics_overview. 

   common crres_mics_overview, info, lsum, valdat
   common crres_mics_overview_extra, intime, sp, type,lbin,lrange
  
   papco_Set_SubTable, SUBTABLE         ;sets the papco color table index
   panelset,panel			;sets the panel position viewport

; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.

   ytickno=50*(!p.position(3)-!p.position(1))
   if ytickno gt 10 then ytickno=10

;   xtickno=6
   
; Coefficients for the calculation of energy density   
   
;   coeff    = [1.43059e-7, 2.86135e-7, 2.8612e-7, 5.7213e-7, $
;               5.7213e-7, 1.43059e-7, 3.71915e-07]
   coeff    = [1.43059e-7, 2.86135e-7, 2.8612e-7, 5.7213e-7, $
               5.7213e-7, 1.43059e-7, 1.43059e-7]
   charge   = [1, 1, 2, 1, 2, 1, 1]
   
; Energy channel to be plotted
   ch=dtype(2)-1   
   
; Setup labels
   
   en_array = [1.2, 1.5, 2.0, 2.6, 3.4, 4.4, 5.8, 7.7, 10.1, 13.4, 17.7, $
               23.4, 31.2, 41.2, 54.6, 72.4, 81.0, 90.5, 101.0, 113.0, $
               125.3, 140.3, 157., 175., 197., 220., 240., 275., 305., $
               341., 381., 426.]
   
   cspecies = ['H+','He+','He++','O+','O++','H+','ions (MICS)']   
   
   
   if dtype(1) ne 0 then begin ; differential 
     if dtype(0) eq 2 or dtype(0) eq 4 then $ ; He++ or O++
       utitle=cspecies(dtype(0))+ '!C'+ $
       string(2*en_array(ch),format="(f5.1,' keV')") $
     else $ ; H+ or He+ or O+ or Protons or DCR
       utitle=cspecies(dtype(0))+ '!C'+ $
       string(en_array(ch),format="(f5.1,' keV')") 
   endif else begin            ; integral
     if dtype(0) eq 6 then begin ; DCR
       utitle=cspecies(dtype(0))+'!C23-157keV'
       if dtype(3) eq 4 then utitle=cspecies(dtype(0))+'!C23-101keV'
       if dtype(3) eq 5 then utitle=cspecies(dtype(0))+'!C101-157keV'
       if dtype(3) eq 6 then utitle=cspecies(dtype(0))+'!C6-24keV'
     endif else begin
       if dtype(0) eq 5 then begin ; Protons
         utitle=cspecies(dtype(0))+'!C23-157keV'
         if dtype(3) eq 4 then utitle=cspecies(dtype(0))+'!C23-101keV'
         if dtype(3) eq 5 then utitle=cspecies(dtype(0))+'!C101-157keV'
         if dtype(3) eq 6 then utitle=cspecies(dtype(0))+'!C6-24keV'
       endif else begin
         if dtype(0) eq 2 or dtype(0) eq 4 then begin ; He++ or O++
           utitle=cspecies(dtype(0))+'!C62-314keV'
           if dtype(3) eq 4 then utitle=cspecies(dtype(0))+'!C62-109keV'
           if dtype(3) eq 5 then utitle=cspecies(dtype(0))+'!C109-314keV'
           if dtype(3) eq 6 then utitle=cspecies(dtype(0))+'!C6-27keV'
         endif else begin
           if dtype(0) eq 0 then begin ; H+
             utitle=cspecies(dtype(0))+'!C23-305keV'
             if dtype(3) eq 4 then utitle=cspecies(dtype(0))+'!C23-101keV'
             if dtype(3) eq 5 then utitle=cspecies(dtype(0))+'!C101-305keV'
             if dtype(3) eq 6 then utitle=cspecies(dtype(0))+'!C6-23keV'
           endif else begin  
             if dtype(0) eq 1 then begin ; He+
               utitle=cspecies(dtype(0))+'!C31-305keV'
               if dtype(3) eq 4 then utitle=cspecies(dtype(0))+'!C31-101keV'
               if dtype(3) eq 5 then utitle=cspecies(dtype(0))+'!C101-305keV'
               if dtype(3) eq 6 then utitle=cspecies(dtype(0))+'!C6-23keV'
             endif else begin  ; O+
               utitle=cspecies(dtype(0))+'!C54-305keV'
               if dtype(3) eq 4 then utitle=cspecies(dtype(0))+'!C54-101keV'
               if dtype(3) eq 5 then utitle=cspecies(dtype(0))+'!C101-305keV'
               if dtype(3) eq 6 then utitle=cspecies(dtype(0))+'!C6-23keV'
             endelse
           endelse
         endelse
       endelse
     endelse
   endelse
   
   
   if dtype(3) eq 0 then uztit='[cm!e-2!ns!e-1!nsr!e-1!nkeV!e-1!n]'
   if dtype(3) eq 1 then uztit='[cm!e-2!ns!e-1!nsr!e-1!n]'
   if dtype(3) eq 2 then uztit='[cm!E-3]'
   if dtype(3) eq 3 then uztit='[keVcm!E-3!n]'
   if dtype(3) eq 4 then uztit='[keVcm!E-3!n]'
   if dtype(3) eq 5 then uztit='[keVcm!E-3!n]'
   if dtype(3) eq 6 then uztit='[keVcm!E-3!n]'
   
   uytitle='L-value'
   
; Setup variables
   
   nobins=(lrange-1)/float(lbin)+1
   zarr=fltarr(sp,nobins)
   yval=fltarr(sp,nobins)
   
; y-axis valyes   
   
   yarr=fltarr(nobins,2)
   for i=0,nobins-1 do begin
     yarr(i,0)=1+i*lbin
     yarr(i,1)=1+(i+1)*lbin
   endfor
   
; time arrays
   
   time=(intime(*,0)+intime(*,1))/2.
   xarr=intime
   
   xut1=mjdt_start.mjd+mjdt_start.t / 86400.0
   xut2=mjdt_end.mjd  +mjdt_end.t   / 86400.0
   
; flux array   
   
   index=where(valdat gt 0,count)
   if count gt 0 then $
     lsum(index)=lsum(index)/valdat(index)
   
; Select energy (Diff. Num flux & diff. en. flux)
   if dtype(1) eq 1 then begin
     if dtype(3) eq 1 then en_fact=en_array(ch) else en_fact=1
     for i=0,sp-1 do begin
       zarr(i,*)=  lsum(i,ch,*)*en_fact
       yval(i,*)=valdat(i,ch,*)
     endfor
     index=where(yval eq 0, count)
     if count gt 0 then zarr(index)=1e-20
   endif
   
; Select energy ranges   
; for the calculation of total energy take into account the charge
   if dtype(0) eq 2 or dtype(0) eq 4 then begin ; for He++ & O++
     en_ch1=12                  ;  31.2 x 2 =  62.4 keV
     en_ch2=22                  ; 157.0 x 2 = 314.0 keV
     if dtype(3) eq 4 then begin
       en_ch1=12                ;  31.2 x 2 =  62.4 keV
       en_ch2=14                ;  54.6 x 2 = 109.2 keV
     endif 
     if dtype(3) eq 5 then begin
       en_ch1=15                ;  72.4 x 2 = 144.8 keV
       en_ch2=22                ; 157.0 x 2 = 314.0 keV
     endif
     if dtype(3) eq 6 then begin
       en_ch1=3                 ;  2.6 x 2 =  5.2 keV
       en_ch2=9                 ; 13.4 x 2 = 26.8 keV
     endif
   endif else begin
     if dtype(0) eq 0 or dtype(0) eq 5 then begin ; for H+ and protons
       en_ch1=11                ;  23.4 keV 
       en_ch2=28                ; 305.0 keV
       if dtype(0) eq 5 then en_ch2=22 ; 157.0 keV (due to bg)
       if dtype(3) eq 4 then begin
         en_ch1=11              ;  23.4 keV
         en_ch2=18              ; 101.0 keV
       endif
       if dtype(3) eq 5 then begin
         en_ch1=19              ; 113.0 keV
         en_ch2=28              ; 305.0 keV
         if dtype(0) eq 5 then en_ch2=22 ; 157.0 keV (due to bg)
       endif
       if dtype(3) eq 6 then begin
         en_ch1=6               ;  5.8 keV
         en_ch2=11              ; 23.4 keV
       endif
     endif else begin
       if dtype(0) eq 3 then begin ; for O+
         en_ch1=14              ;  54.6 keV 
         en_ch2=28              ; 305.0 keV
         if dtype(3) eq 4 then begin
           en_ch1=14            ;  54.6 keV
           en_ch2=18            ; 101.0 keV
         endif
         if dtype(3) eq 5 then begin
           en_ch1=19            ; 113.0 keV
           en_ch2=28            ; 305.0 keV
         endif
         if dtype(3) eq 6 then begin
           en_ch1=6             ;  5.8 keV
           en_ch2=11            ; 23.4 keV
         endif
       endif else begin
         if dtype(0) eq 1 then begin ; for He+
           en_ch1=12            ;  31.2 keV 
           en_ch2=28            ; 305.0 keV
           if dtype(3) eq 4 then begin
             en_ch1=12          ;  31.2 keV
             en_ch2=18          ; 101.0 keV
           endif
           if dtype(3) eq 5 then begin
             en_ch1=19          ; 113.0 keV
             en_ch2=28          ; 305.0 keV
           endif
           if dtype(3) eq 6 then begin
             en_ch1=6           ;  5.8 keV
             en_ch2=11          ; 23.4 keV
           endif
         endif else begin
           if dtype(0) eq 6 then begin ; for DCR
             en_ch1=11          ;  23.4 keV 
             en_ch2=22          ; 157.0 keV (due to bg)
             if dtype(3) eq 4 then begin
               en_ch1=11        ;  23.4 keV
               en_ch2=18        ; 101.0 keV
             endif
             if dtype(3) eq 5 then begin
               en_ch1=19        ; 113.0 keV
               en_ch2=22        ; 157.0 keV (due to bg)
             endif
             if dtype(3) eq 6 then begin
               en_ch1=6         ;  5.8 keV
               en_ch2=11        ; 23.4 keV
             endif
           endif
         endelse
       endelse
     endelse
   endelse
                    
; calculate total differential number flux
   
   if dtype(1) eq 0 and dtype(3) eq 0 then begin
     for i=0,sp-1 do begin                           ; all time steps
       ymat = 0.0       
       for chh=en_ch1,en_ch2 do begin
         enden = lsum(i,chh,*)                       ; number flux
         ymat = ymat + enden                        ; sum energies
       endfor
       zarr(i,*) = ymat                             ; total energy
     endfor
   endif
   
; calculate total differential energy flux
   
   if dtype(1) eq 0 and dtype(3) eq 1 then begin
     for i=0,sp-1 do begin                           ; all time steps
       ymat = 0.0       
       for chh=en_ch1,en_ch2 do begin
         energy  = en_array(chh)                        ; energy
         enden = lsum(i,chh,*)*energy                ; energy flux
         ymat = ymat + enden                       ; sum energies
       endfor
       zarr(i,*) = ymat                             ; total energy
     endfor
   endif
   
; calculate differential number density 
   
   if dtype(1) eq 0 and dtype(3) eq 2 then begin
     for i=0,sp-1 do begin                           ; all time steps
       ymat = 0.0       
       for chh=en_ch1,en_ch2 do begin
         deltaen = (en_array(chh)*1.1 - en_array(chh)*0.9) ; delta energy
         energy  = en_array(chh)                        ; energy
         numden = lsum(i,chh,*)*charge(dtype(0))*deltaen / $
           sqrt(charge(dtype(0))*energy) ; number density
         ymat = ymat + numden                       ; sum energies
       endfor
       zarr(i,*) = ymat                             ; total energy
     endfor
   endif
   
; calculate differential energy density

   if (dtype(1) eq 0 and dtype(3) eq 3) or $
      (dtype(1) eq 0 and dtype(3) eq 4) or $
      (dtype(1) eq 0 and dtype(3) eq 5) or $
      (dtype(1) eq 0 and dtype(3) eq 6) then begin
     for i=0,sp-1 do begin                           ; all time steps
       ymat = 0.0       
       for chh=en_ch1,en_ch2 do begin
         deltaen = (en_array(chh)*1.1 - en_array(chh)*0.9) ; delta energy
         energy  = en_array(chh)                        ; energy
         numden = lsum(i,chh,*)*charge(dtype(0))*deltaen / $
           sqrt(charge(dtype(0))*energy) ; number density
         enden = numden * charge(dtype(0)) * energy $
           *coeff(dtype(0))     ; energy density
         ymat = ymat + enden                       ; sum energies
       endfor
       zarr(i,*) = ymat                             ; total energy
     endfor
   endif
   
; restrict data to actual time range requested - this makes for faster zooming
   
   index=where((time gt xut1) and (time lt xut2),count)
   if count gt 1 then begin
     time=time(index)
     xarr=xarr(index,*)
     zarr=zarr(index,*)
   endif
   
   index=where(xarr(1:*,0)-xarr(0:*,0) gt 0.,count) + 1
   if count gt 1 then begin
     time=time(index)
     xarr=xarr(index,*)
     zarr=zarr(index,*)
   endif

   yst=yarr(0,0)
   yen=yarr(nobins-1,1)

; restrict data to actual time range requested - this makes for faster zooming

   index=where((xarr(*,0) ge xut1) and (xarr(*,1) le xut2),count)
   if count ne 0 then begin
      zarr=zarr(index,*)
      time=time(index)
      xarr=xarr(index,*)
   endif
   
; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data

; the procedure papco_autorange (in papco_XX/papco_lib/autoscaling is
; optional and sets the zlimits automatically for the data, attempting
; to use the full dynamic range of the color table. you may need to
; supply your own procedure for your own data.

; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling

   if (zscl(panel(0),0) eq 1) then begin
     zmin=zscl(panel(0),1)
     zmax=zscl(panel(0),2)
   endif else begin
     papco_autorange,zarr,zmin,zmax, $
       log=zscl(panel(0),3), exclude=0, nodata=1e-20
     zscl(panel(0),1)=zmin
     zscl(panel(0),2)=zmax
   endelse

   if (yscl(panel(0),0) eq 1) then begin
     yst=yscl(panel(0),1)
     yen=yscl(panel(0),2)
   endif else begin
     yscl(panel(0),1)=yst
     yscl(panel(0),2)=yen
   endelse
   
; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO:
;
;    !P.CHARSIZE  = widgetData.default_charsize    
;    !P.CHARTHICK = widgetData.default_charthick
;    !P.FONT      = -1
;    !P.NOERASE   = 1
;    !P.TICKLEN   = widgetData.default_TICKLEN
;    !X.TICKS     = widgetData.default_XTICKS
;    !X.STYLE     = widgetData.default_XSTYLE
;    !Y.STYLE     = 1
;    !X.MINOR     = widgetData.default_XMINOR
;
; Overriding these values by setting keywords is of course possible
; and allowed but discouraged, since 
; then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure.

   extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zmin,zmax], $
                  ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
                  ytitle:uytitle, xtickformat:'noticks'}

; check if the passed-in structure extra_par was set. If not, create
; it and set color to default black; which for the color table used by
; papco is 1. If extra_par was set, see if the tag color exists. If it
; does, leave it unchanged, if not, add it and set color to default
; black. 

  if n_elements(extra_par) EQ 0 then extra_par={color:1} else begin
    names=tag_names(extra_par)
    res=where(names eq 'COLOR',count)
    if count eq 0 then extra_par=create_struct({color:1}, extra_par)
  endelse  

; add keyword structure set locally and the one passed in

  extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; change the xtickformat if the panel is the bottom one and needs to
; plot a time axis   
   
  if panel(0) eq 0 then extra_plotPar.xtickformat='mjd_dec_2'
   
; You might have to add your own xtickformat for time axis
; labeling. If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; You can at this point also define a custom ytickformat, or leave it out for
; IDL default plot behaviour.
   
; "down" sets the axis labeling level. 
  
  down=0

;do color plot and color bar plot. nodata specifies the "nodata" flag
;in your data. Here we do not plot axis and data seperate, as
;papco_plot_colorspec cannot be used as an overplot!

  papco_plot_colorspec,zarr,xarr,yarr, $
    nodata=1e-20, _extra=extra_plotPar
  papco_colorbar,zmin,zmax,0,0,uztit
  
; store the coordinate information into common block

  plotted_x = !x
  plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label - 0r use any other function you define yourself.
; The routine x_side_label puts x-axis label to the right of the axis.
  
  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
    extra_plotPar.xtickformat='mjd_dec_1'
    extra_plotPar.color=1
    down=1
    plot,time,time, _extra=extra_plotPar, /NODATA
    extra_plotPar.xtickformat='crres_mics_orb_ticks'
    extra_plotPar.color=1
    down=1
    plot,time,time, _extra=extra_plotPar, /NODATA
    x_side_label,panel,'Time(UT)!CDate!COrbit'
  endif
   
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.

  right_side_label,panel,utitle,/rot90
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  
  if KEYWORD_SET(OUTPUT) then begin
    print,'% p_crres_mics_overview: Writing plot data out to file'
    description='crres_mics_overview template sample data - Energy spectra, 10 channels'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, xarr, zarr, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
  endif  
   
end 

;------------------------------------------------------------------------------
; This function labels orbits given that a mag-lav file was read

function crres_mics_orb_ticks,axis,index,t
  
  common crres_mics_overview, info, lsum, valdat
  common crres_mics_overview_extra, intime, sp, type,lbin, lrange
  
  time=(intime(*,0)+intime(*,1))/2.
  res=min(abs(time-t),mindex)
  orbit=info(mindex,3)
  if res lt 1.0 then $
    return,string(orbit,format="('!C!C',i4.4)") $
  else $
    return,''
  
end  
