;******************************************************************************
;* file p_crres_mics_plaspar.pro
;*
;* this file is a template for a plot routine which can be used by papco.
;* it includes all the periphery needed, the user simply has to add in
;* the actual plot statement for his/her kind of data.
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_crres_mics_plaspar, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_PLOTINFO.switch (three element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_crres_mics_plaspar, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;******************************************************************************
pro p_crres_mics_plaspar, panel, plotinfo, $
                          OUTPUT=OUTPUT, $
                          OVERPLOT=OVERPLOT, $
                          PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                          SUBTABLE=SUBTABLE, $
                          _EXTRA=extra_par

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_crres_mics_plaspar, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags which are meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type, switch. Replace type in the call with whatever variables
; you need, or add more

; The following common blocks are needed:
  common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
  common yscale, yscl			;man/auto yscaling
  common shift_label, down             ;common for x-axis label shifting
  common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_crres_mics_plaspar. 
  common crres_mics_data, mics_header, mics_data
  common ephem, xutepf,xmlt,xilat,xlshel,xrad,ndat
  common get_error, get_err_no, get_err_msg
   
  panelset,panel			;sets the panel position viewport
  
  dtype=plotinfo.typevector
  options=plotinfo.options
  
; set the begin end end times.
  convert_t90_to_Date, mjdt_start, year,doy,hr,min,sec,mon,cmon,doym, /mjdt 
  start_tai=UTC2TAI({YEAR:year, MONTH:mon, DAY:doym, HOUR:hr, MINUTE:min, $
                     SECOND:sec, MILLISECOND:0})
  convert_t90_to_Date, mjdt_end, year,doy,hr,min,sec,mon,cmon,doym, /mjdt 
  end_tai=UTC2TAI({YEAR:year, MONTH:mon, DAY:doym, HOUR:hr, MINUTE:min, $
                   SECOND:sec, MILLISECOND:0})
  
; restrict data to actual time range requested - this makes for faster zooming
  index=where((mics_data.time ge start_tai) AND $
              (mics_data.time le end_tai),count)
  if count ne 0 then data=mics_data(index) else data=mics_data
  
  xut1=0  &  xut2=end_tai-start_tai  

  utit = ['H+','He+','He++','O+','O++', 'Proton','DCR', 'O+ / Proton', $
          'O+ / H+', 'He++ / Proton', 'He++ / H+', 'He+ / Proton', 'He+ / H+']
   
  en_array = [1.2, 1.5, 2.0, 2.6, 3.4, 4.4, 5.8, 7.7, 10.1, 13.4, 17.7, $
              23.4, 31.2, 41.2, 54.6, 72.4, 81.0, 90.5, 101.0, 113.0, $
              125.3, 140.3, 157., 175., 197., 220., 240., 275., 305., $
              341., 381., 426.]
   
  coeff    = [1.43059e-7, 2.86135e-7, 2.8612e-7, 5.7213e-7, $
              5.7213e-7, 1.43059e-7, 1.43059e-7]
   
  charge   = [1, 1, 2, 1, 2, 1, 1]
   
  type_ind = [1,2,3,4,5,8,11]
   
  dtype_orig = dtype(0)
  ratio_flag = 0
  RATIO :
   if dtype_orig ge 7 and dtype_orig le 12 then begin ; calculate ratios
     ratio_flag = ratio_flag + 1
     if ratio_flag eq 1 then begin
       if dtype_orig eq 7 or dtype_orig eq 8 then dtype(0) = 3       ; oxygen
       if dtype_orig eq 9 or dtype_orig eq 10 then dtype(0) = 2      ; Alphas
       if dtype_orig eq 11 or dtype_orig eq 12 then dtype(0) = 1     ; He+
     endif else begin
       if dtype_orig eq 7 or $
         dtype_orig eq 9 or $
         dtype_orig eq 11 then begin
         dtype(0) = 5           ; Protons
       endif else begin
         dtype(0) = 0           ; H+
       endelse
     endelse
   endif
   
   av_int = dtype(2)
   
; set time, data array
   time=data.time-start_tai
   zmat = data.data(*,type_ind(dtype(0)),*)
   zmat = transpose(zmat)
   
   time_av = fltarr(n_elements(time))
   zmat_av = fltarr(n_elements(time),32)
   iind_av = 0
   for iav=0, n_elements(time)-1, av_int do begin
     if (iav+av_int-1) le n_elements(time)-1 then begin
       time_av(iind_av) = total(time(iav:iav+av_int-1))/float(av_int)
       zmat_av(iind_av,*) = $
         total(zmat(iav:iav+av_int-1,*),1)/float(av_int)
       iind_av = iind_av + 1
     endif
   endfor
   time=time_av(0:iind_av-1)
   zmat=zmat_av(0:iind_av-1,*)
   xlshel=interpol(data.xlshel,data.time-start_tai,time)
   
; Select energy ranges   
; for the calculation of total energy take into account the charge
   if dtype(0) eq 2 or dtype(0) eq 4 then begin ; for He++ & O++
     en_ch1=12                  ;  31.2 x 2 =  62.4 keV
     en_ch2=22                  ; 157.0 x 2 = 314.0 keV
   endif else BEGIN
;change by RF to get equal energy ranges for ratios!
     if dtype(0) eq 0 or dtype(0) eq 5 then begin ; for H+ and protons
       ;en_ch1=15                ;  23.4 keV 
       en_ch1=14                ;  NEW 54.6 keV 
       en_ch2=28                ; 305.0 keV
       if dtype(0) eq 5 then en_ch2=22 ; 157.0 keV (due to bg)
     endif else begin
       if dtype(0) eq 3 then begin ; for O+
         en_ch1=14              ;  54.6 keV 
         en_ch2=28              ; 305.0 keV
       endif else begin
         if dtype(0) eq 1 then begin ; for He+
           en_ch1=12            ;  31.2 keV 
           en_ch2=28            ; 305.0 keV
         endif else begin
           if dtype(0) eq 6 then begin ; for DCR
             en_ch1=11          ;  23.4 keV 
             en_ch2=22          ; 157.0 keV (due to bg)
           endif
         endelse
       endelse
     endelse
   endelse
                     
   sp=n_elements(zmat(*,0))
   yray = fltarr(sp)
   
   for i=0,sp-1 do begin
     ymat=0.0
     for chh=en_ch1,en_ch2 do begin
       deltaen = (en_array(chh)*1.1 - en_array(chh)*0.9) ; delta energy
       energy  = en_array(chh)  ; energy
       numden = zmat(i,chh)*charge(dtype(0))*deltaen / $
         sqrt(charge(dtype(0))*energy) ; number density
       enden = numden * charge(dtype(0)) * energy $
         *coeff(dtype(0))       ; energy density
       ymat = ymat + enden      ; sum energies
     endfor
     yray(i)=ymat
   endfor
   
   if dtype_orig ge 7 and dtype_orig le 12 then begin ; calculate ratios
     if ratio_flag eq 1 then begin
       yray_ratio = yray
       goto, RATIO
     endif else begin
       yray = yray_ratio / yray
     endelse
   endif
   
   uytitle='ED(keV/cm!U3!N)'
   utitle=utit(dtype(0))
   
   if dtype_orig ge 7 and dtype_orig le 12 then begin ; calculate ratios
     yst = 0 
     yen = 1     
     uytitle='ED ratio'
     utitle=utit(dtype_orig)
   endif

; limit data in L - set unwanted data to nodata flag.
  nodata=1e-20 
  index=where( (xlshel le options(2)) OR (xlshel ge options(3)), c )
  if c ne 0 then begin
    yray(index)=nodata
    str_lrange=string(options(2),options(3),$
                       format="('L-range:!C',f4.1,'-',f4.1)")
  endif else begin
    message, 'L-range not found, plotting all', /cont 
    str_lrange=''
  endelse
  
  utitle=utitle+'!C'+str_lrange
  
; check for any other bad data points  
  idx=where(yray eq 'inf',c)  & if c ne 0 then yray(idx)=nodata
  
; the variables used above have the following meaning:
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;
; the procedure YLIM3 is optional and sets the ylimit automatically for the
; data leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data. 
; (in $papco/papco_Lib/ylim.pro)

; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for "linear" and 1 for "log"  scale
; when scaling is set to automatic, the automatic y-limits are loaded in.
   
  if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
  endif else begin
    papco_autorange,yray,yst,yen, LOG=yscl(panel(0),3), NODATA=nodata
    yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
  endelse
  
; use auto y-labeling routine to get "good" y-lables
  papco_y_label, yst, yen, LOG=yscl(panel(0),3)   
  
; plot y-axis label at left of plot. Use scalable routine!  
  left_side_label,panel,uytitle,/rot90  
     
; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO:
;
;    !P.CHARSIZE  = widgetData.default_charsize    
;    !P.CHARTHICK = widgetData.default_charthick
;    !P.FONT      = -1
;    !P.NOERASE   = 1
;    !P.TICKLEN   = widgetData.default_TICKLEN
;    !X.TICKS     = widgetData.default_XTICKS
;    !X.STYLE     = widgetData.default_XSTYLE
;    !Y.STYLE     = 1
;    !X.MINOR     = widgetData.default_XMINOR
;
; Overriding these values by setting keywords is of course possible
; and allowed but discouraged, since 
; then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure.
   extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen],  $
                  ylog:yscl(panel(0),3), xtickformat:'noticks'}
   
; add keyword structure set here with the one passed in
   extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; use papco rotuine to draw time axis. This checks for bottom plot
; and uses the user's  xtickformat if it is something other than 'noticks'.
  down=0
  papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar  

; now do the actual data plot, data only, no axis
  ;plot, time, yray, _extra=extra_plotPar, xstyle=5, ystyle=5
  papco_gap_plot, time, yray, nodata, _extra=extra_plotPar, xstyle=5, ystyle=5

; store the coordinate information into common block for slices
  plotted_x = !x  &  plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; The routine x_side_label puts x-axis label to the right of the axis.
; If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; Or use any other function you define yourself.
; down sets the axis labeling level
  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin

  endif

; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION
  if not keyword_set(overplot) then begin
    right_side_label,panel,utitle,_extra=extra_Par
  endif else begin
    for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C'+utitle
    right_side_label,panel,utitle,_extra=extra_Par
  endelse
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
  
  if KEYWORD_SET(OUTPUT) then begin
    print,'% p_crres_mics_plaspar: Writing plot data out to file'
    description='crres_mics_plaspar data - single species/ratio'
    time_label='TAI'
    ;channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, $
                      RS_LABEL=RS_LABEL
  endif  

end 
