;******************************************************************************
;* file p_crres_mics_spectra.pro
;*
;* This file is a template for a plot routine which can be used by PAPCO.
;* It includes all the periphery needed. This routine makes use of a
;* color spectrogram routine that is supplied with PAPCO:
;* papco_XX/papco_lib/papco_plot_colorspec.pro. This routine should
;* meet all the requirements for the display of 3-D data (tyhe third
;* dimension being color). This routine also adheres to the PAPCO color scheme.
;* You of course can use your own color plot rotuine if desired!
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_crres_mics_spectra, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (and/or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure PAPCD_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.switch (three element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to make use of
;*              an extra common block.
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_crres_mics, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;******************************************************************************
pro p_crres_mics_spectra, panel, type, $
                      OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Line styles are also not useful. So the keywords 
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,_EXTRA=extra_par
; are not used at all here, but are in the call for consistency...

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_crres_mics_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags which are meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type and/or switch. Replace type in the call with whatever variables
; you need, or add more. 

; The following common blocks are needed for PAPCO:
  common mjdt, mjdt_start, mjdt_end ;common time limit in mjdt
  common yscale, yscl           ;man/auto yscaling
  common zscale, zscl           ;man/auto zscaling
  common shift_label, down      ;common for x-axis label shifting
  common coordinateSystems, plotted_x, plotted_y  
                                ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_crres_mics. 
  common crres_mics_data, mics_header, mics_data
  common ephem, xutepf,xmlt,xilat,xlshel,xrad,ndat
  common get_error, get_err_no, get_err_msg
  
  papco_Set_SubTable, SUBTABLE  ;sets the papco color table index
  panelset,panel                ;sets the panel position viewport
  
; set the begin end end times.
  convert_t90_to_Date, mjdt_start, year,doy,hr,min,sec,mon,cmon,doym, /mjdt 
  start_tai=UTC2TAI({YEAR:year, MONTH:mon, DAY:doym, HOUR:hr, MINUTE:min, $
                     SECOND:sec, MILLISECOND:0})
  convert_t90_to_Date, mjdt_end, year,doy,hr,min,sec,mon,cmon,doym, /mjdt 
  end_tai=UTC2TAI({YEAR:year, MONTH:mon, DAY:doym, HOUR:hr, MINUTE:min, $
                   SECOND:sec, MILLISECOND:0})
  
; restrict data to actual time range requested - this makes for faster zooming
  index=where((mics_data.time ge start_tai) AND $
              (mics_data.time le end_tai),count)
  if count ne 0 then data=mics_data(index) else data=mics_data
  
  xut1=0  &  xut2=end_tai-start_tai  
  
; set time, data array
  time=data.time-start_tai
  zmat = data.data(*,type(0))
  zmat = transpose(zmat)  
  
  utit = ['R0','R1 H+','R2 He+','R3 He++','R4 O+','R5 O++',  $
          'R6','R7','Proton','HVY','FSR','DCR','TCR','MSS']

  yarr=mics_header.energy  
  ndiff = n_elements(mics_data.data(0,0))

  if type(1) eq 0 then begin
    uztit='Num Flux'
  endif else begin
    zmat(0:ndiff-1,ind2) = $
      zmat(0:ndiff-1,ind2)*mics_header.energy(ind2)
    uztit='En Flux'
  endelse
  
  uytitle='E/Q (kV/q)'
  utitle=utit(type(0))
  
  yst=yarr(0)
  yen=yarr(n_elements(yarr)-1)
   
; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data

; the procedure papco_autorange (in papco_XX/papco_lib/autoscaling is
; optional and sets the zlimits automatically for the data, attempting
; to use the full dynamic range of the color table. you may need to
; supply your own procedure for your own data.

; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling
   
   if (zscl(panel(0),0) eq 1) then begin
     zmin=zscl(panel(0),1)
     zmax=zscl(panel(0),2)
   endif else begin
     papco_autorange,zmat,zmin,zmax, $
       log=zscl(panel(0),3), exclude=0, nodata=1e-20
     zscl(panel(0),1)=zmin
     zscl(panel(0),2)=zmax
   endelse

   if (yscl(panel(0),0) eq 1) then begin
     yst=yscl(panel(0),1)
     yen=yscl(panel(0),2)
   endif else begin
     yscl(panel(0),1)=yst
     yscl(panel(0),2)=yen
   endelse
   
; use auto y-labeling routine to get "good" y-lables
   papco_y_label, yst, yen, LOG=yscl(panel(0),3)   
   
; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO:
;
;    !P.CHARSIZE  = widgetData.default_charsize    
;    !P.CHARTHICK = widgetData.default_charthick
;    !P.FONT      = -1
;    !P.NOERASE   = 1
;    !P.TICKLEN   = widgetData.default_TICKLEN
;    !X.TICKS     = widgetData.default_XTICKS
;    !X.STYLE     = widgetData.default_XSTYLE
;    !Y.STYLE     = 1
;    !X.MINOR     = widgetData.default_XMINOR
;
; Overriding these values by setting keywords is of course possible
; and allowed but discouraged, since 
; then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure.
   extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zmin,zmax], $
                  ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
                  ztitle:uztit, $
                  xtickformat:'noticks', $
                  ztickformat:'papco_color_bar_log_ticks'}

; add keyword structure set locally and the one passed in
  extra_plotPar=create_struct(extra_plotPar, extra_par)

;do color plot and color bar plot. nodata specifies the "nodata" flag
;in your data. 
  
  ;check the routine papco_plot_colorspec in papco_XX\papco_lib\color_plot
  ;for all the possible options of papco_plot_colorspec.
  ;Hint: you might want to use the resx, resy keywords, which set the
  ;resolution to plot. Without these, the routine will try to use
  ;device resolution, which for postscript may result in HUGE files!
  ;resx, resy are in data coordinates. If the axis is logarithmic, 
  ;resx, resy need to be too.  
  
  papco_plot_colorspec,zmat,time,yarr, $
    /xcenter, /ycenter, nodata=nodata, xstyle=5, ystyle=5,_extra=extra_plotPar
  
; use papco rotuine to draw time axis. This checks for bottom plot
; and uses the user's  xtickformat if it is something other than 'noticks'.
  down=0
  papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar  
  
  papco_color_bar, _extra=extra_plotPar
  
; store the coordinate information into common block
  plotted_x = !x  &  plotted_y = !y

; plot y-axis label at left of plot. Use scalable routine!  
  left_side_label,panel,uytitle,/rot90  
  
; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label - 0r use any other function you define yourself.
; The routine x_side_label puts x-axis label to the right of the axis.
  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
     ;extra_plotPar.xtickformat='n_mltticks'
     ;extra_plotPar.color=1     
     ;down=1
     ;plot,time,time, _extra=extra_plotPar, /NODATA
     ;extra_plotPar.xtickformat='n_ilatticks'
     ;extra_plotPar.color=1     
     ;down=2
     ;plot,time,time, _extra=extra_plotPar, /NODATA
     ;extra_plotPar.xtickformat='n_lshelticks'
     ;extra_plotPar.color=1     
     ;down=3
     ;plot,time,time, _extra=extra_plotPar, /NODATA     
     ;extra_plotPar.xtickformat='n_radticks'
     ;extra_plotPar.color=1     
     ;down=4
     ;plot,time,time, _extra=extra_plotPar, /NODATA     
     ;x_side_label,panel,'time (UT)!CMLT!CILAT!CL!CRadius'
  endif
   
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot. 
  right_side_label,panel,utitle,/rot90
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  
  if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file', /cont
    description='crres_mics template sample data - Energy spectra, 10 channels'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, time, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
  endif  
   
end 



;-----------------------------------------------------------------------------
function n_mltticks,axis,index,t

  common crres_eph_data, input_header, input_data
  
  xutepf = input_data.time
  xmlt   = input_data.xmlt
  
  difft=abs(xutepf-t)
  result=min(difft,minindex)
  
  mlt=xmlt(minindex)
  
  return,string(mlt,format="('!C',f4.1)")
  
end

;-----------------------------------------------------------------------------
function n_ilatticks,axis,index,t

  common crres_eph_data, input_header, input_data
  
  xutepf = input_data.time
  xilat  = input_data.xilat
  
  difft=abs(xutepf-t)
  result=min(difft,minindex)
  
  ilat=xilat(minindex)
  
  return,string(ilat,format="('!C!C',f5.1)")
  
end 

;-----------------------------------------------------------------------------
function n_lshelticks,axis,index,t
  
  common crres_eph_data, input_header, input_data
  
  xutepf = input_data.time
  xlshel = input_data.xlshel 
  
  difft=abs(xutepf-t)
  result=min(difft,minindex)
  
  lshel=xlshel(minindex)
  
  return,string(lshel,format="('!C!C!C',f3.1)")
  
end 

;-----------------------------------------------------------------------------
function n_radticks,axis,index,t
  
  common crres_eph_data, input_header, input_data
  
  xutepf = input_data.time
  xrad   = input_data.xrad
  
  difft=abs(xutepf-t)
  result=min(difft,minindex)
  
  rad=xrad(minindex)
  
  return,string(rad,format="('!C!C!C!C',i5)")
  
end 






