;******************************************************************************
;*  NAME:        r_crres_mics
;*
;*  DESCRIPTION: Routine to read crres_mics files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       orbit number
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        August 1997
;*
;*  AUTOR:       C. Mouikis
;*               modified Jan 2001 to read multiple orbits / start_end
;*               time, R. Friedel
;******************************************************************************
pro r_crres_mics, in_orbit, VERBOSE=VERBOSE

  common crres_mics_data, mics_header, mics_data
  common get_error, get_err_no, get_err_msg

; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.
  get_err_no=0  &  get_err_msg='crres_mics read succesfully'
   
; check env. variable pointing to data directory
  if not (papco_check_data_env('crres_mics_data', PATH=path)) then return
   
; check VERBOSE keyword and set
  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
  mics_header=0  &  mics_data=0   

; loop through from start to end time concatonating data files into one
; array. Here data are stored by orbit, so we need to find the orbit
; range corresponding to the start and end times.
  time=0  &  no_files_read=0
   
; check if orbit is non-zero, then read that orbit 
; - otherwise get list of orbits to read
  if in_orbit eq 0 then orbit_list=papco_getOrbitList('CRRES') $
    else orbit_list=[in_orbit]
      
  for i=0,n_elements(orbit_list)-1 do begin
    orbit=orbit_list(i)
    verbose=1
    ; read one orbit of data
    r_crres_mics_oneorbit, orbit, path, VERBOSE=VERBOSE
        
    ; check for an error in get_error - and then get the file remotely!
    ; comment this section out if you do not want this feature.
    if get_err_no ne 0 then begin      
      c_url=papco_getenv('CRRES_MICS_FTP_ARCHIVE')
    ; assemble the data file from the orbit requested
      orbitstr=string(format='(i4.4)',orbit)
      fln='bin'+orbitstr
      c_file=fln+'.idl'
      c_download_dir=path
      papco_get_remote_data, c_url, c_file, c_download_dir
      ;check if file "arrived" and then read it
      if get_err_no eq 0 then $
        r_crres_mics_oneorbit_idlsunbin, path+fln, /VERBOSE $
      else begin
        get_err_msg='% r_crres_mics: No data locally, remote get failed!C  ' $
          + get_err_msg
        continue
      endelse  
    endif   
     
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
      if no_files_read eq 0 then begin   ;start the array buffer
        mics_data_buf=temporary(mics_data)
        ndat=mics_header.ndat
      endif else begin                   ;otherwise concatonate arrays
        mics_data_buf=[temporary(mics_data_buf),temporary(mics_data)]
        ndat=ndat+mics_header.ndat
      endelse        
      mics_data=0                        ;clear array for reuse
      no_files_read=no_files_read+1
    endif
  endfor         

; check if any files were read   
  if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    mics_data=0  &  mics_data=mics_data_buf  &  mics_data_buf=0
    mics_header.ndat=ndat
  endif else begin
    get_err_no=1
    get_err_msg='no files found for timerange'
  endelse      
  message, get_err_msg, /cont
  close, /all

end

;******************************************************************************
;*  NAME:        r_crres_mics_oneorbit, orbit
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures mics_header and
;*               mics_data
;*
;*  INPUT:       orbit  - orbit number
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        August 1997
;*
;*  AUTOR:       C. Mouikis
;******************************************************************************
pro r_crres_mics_oneorbit, orbit, path, VERBOSE=VERBOSE

  common crres_mics_data, mics_header, mics_data
  common get_error, get_err_no, get_err_msg

  get_err_no=0  &  get_err_msg='CRRES_mics read succesfully'
  
  ;---------------------------------------------------------------------------
  ;--Initialise variables
  ;---------------------------------------------------------------------------
  
  iorbit = long(0)
  iday   = long(0)
  iyear  = long(0)
  iutorb = long(0)
  ndiff  = long(0)
  nepf   = long(0)
  xutdif = fltarr(2)

  file=path+'r'+string(format='(i4.4)',orbit)+'.bin'

  rm_flag=0
  filthere=findfile(file, count=nofiles)
  if nofiles gt 0 then goto, READ_FILE

  file=file+'.gz'
  filethere=findfile(file, count=nofiles)
  if nofiles eq 0 then begin
    get_err_no=2
    get_err_msg='File '+file+' does not exist'
    print, get_err_msg
    return
  endif
  
  ;---------------------------------------------------------------------------
  ;--Uzip the file
  ;---------------------------------------------------------------------------

  ff = findfile(file)
  print, 'unzip '+file
  spawn, 'gunzip -c '+file+'>'+strmid(file,0,strlen(file)-3)
  rm_flag=1
  file = strmid(file,0,strlen(file)-3)

  READ_FILE:
  ;---------------------------------------------------------------------------
  ;--Open the file
  ;---------------------------------------------------------------------------
  
  on_ioerror, error_return
  
  if keyword_set(VERBOSE) then message, file, /cont
  ;original data files written on SOLARIS - hence SWAP_IF_LITTLE_ENDIAN
  openr, unit, file, /get_lun, /f77_unformatted, /SWAP_IF_LITTLE_ENDIAN

  readu, unit, iorbit, iday, iyear, iutorb
  readu, unit, xutdif, ndiff, nepf
  ;---------------------------------------------------------------------------
  ;--Define Arrays/Structures
  ;---------------------------------------------------------------------------
 
  xut         = fltarr(ndiff)
  xutepf      = fltarr(nepf)
  xmlt        = fltarr(nepf)
  xilat       = fltarr(nepf)
  xlshel      = fltarr(nepf)
  xrad        = fltarr(nepf)
  dummy       = fltarr(ndiff)
  
  data_dummy = {CRRES_MICS_DATA_TAI, $
                time:0.0d, $
                xlshel:0.0, $
                data:fltarr(32,14)}
  mics_data = replicate(data_dummy,ndiff)
  
  mics_header={MICS_HEADER_TAI, $
               year:0, $
               doy:0, $
               orbit:0, $
               dp:0, $
               ndat:0l, $
               energy:fltarr(32)}
  ;---------------------------------------------------------------------------
  ;--Define the time axis
  ;---------------------------------------------------------------------------
  mics_header.dp=ndiff
  mics_header.year=iyear
  mics_header.doy=iday
  mics_header.orbit=iorbit
  xt1 = xutdif(0)
  xt2 = xutdif(1)
  xdiff = 0
  if (ndiff gt 1) then xdiff = (xt2 - xt1)/(ndiff-1)

  for k=0, ndiff-1 do begin
    mics_data(k).time=xt1 + (k-1)*xdiff
  endfor

  ;---------------------------------------------------------------------------
  ;--Define the energy levels
  ;---------------------------------------------------------------------------
  mics_header.energy = $
    [   1.2,   1.5,   2.0,   2.6,   3.4,   4.4,   5.8,   7.7, $
       10.1,  13.4,  17.7,  23.4,  31.2,  41.2,  54.6,  72.4, $
       81.0,  90.5, 101.0, 113.0, 125.3, 140.3, 157.0, 175.0, $
      197.0, 220.0, 240.0, 275.0, 305.0, 341.0, 381.0, 426.0]
  
  ;---------------------------------------------------------------------------
  ;--Read the rates
  ;---------------------------------------------------------------------------
   
  species = ['R0','H+','He+','He++','O+','O++',  $
          'O+++','O2+','P','ALPHA','FSR','DCR','TCR','MSS']
  
  for ind1=0, 13 do begin
    conv_fact, iorbit, species(ind1), 1, factor
    for ind2=0, 31 do begin
      on_ioerror, error_return
      readu, unit, dummy
      mics_data(0:ndiff-1).data(ind2,ind1) = dummy*factor(ind2)
    endfor
  endfor

  readu, unit, xutepf
  readu, unit, xmlt
  readu, unit, xilat
  readu, unit, xlshel
  readu, unit, xrad
  
  mics_data.xlshel=interpol(xlshel,xutepf,mics_data.time) 
  
  ;---new rf Jan 2001. To concatonate orbit files, need
  ; to have a better continuous time variable   
  ; use TAI time. It's a double, so time variable in   mics_data
  ; needs to be double too.  
  message,'converting to TAI...', /cont
  utc=doy2utc(IDAY,1900+IYEAR)
  ndat=n_elements(mics_data)
  for i=0,ndat-1 do begin
    utc.time=mics_data(i).time*1000
    mics_data(i).time=utc2tai(utc)
  endfor    
  
  return

  error_return:

  close, unit & free_lun, unit
  if rm_flag eq 1 then spawn, '/bin/rm '+file
  get_err_no = 1
  get_err_msg = !ERROR_STATE.MSG
  message,  !ERROR_STATE.MSG, /cont

end
