;******************************************************************************
;*  NAME:        r_plot_type
;*
;*  DESCRIPTION: Routine to read plot_type files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
pro r_plot_type, VERBOSE=VERBOSE

   common plot_type_data, input_header, input_data
   common mjdt, mjdt_start, mjdt_end
   common get_error, get_err_no, get_err_msg

; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

   get_err_no=0
   get_err_msg='plot_type read succesfully'
   
; check for env. variable pointing to data directory
   data_dir='plot_type_DATA'
   path=papco_getenv(data_dir)
   if strlen(path) eq 0 then begin
     get_err_no=1
     get_err_msg='Environmental variable '+data_dir+' not set'
     print,get_err_msg
     return
   endif
   
; check for existence of data directory
   cc=papco_finddir(path)
   if cc eq 0 then begin
     get_err_no=2
     get_err_msg='Data directory '+path+' not found'
     print,get_err_msg
     return
   endif
   
; check VERBOSE keyword and set
   if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. papco standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
   
   time=mjdt_start  &  no_files_read=0  &  day_count=0.0
   
   while mjdt_end.mjd ge time.mjd do begin
     
     ; assemble the data file from the time range requested
     convert_t90_to_date, mjdt_start, year, day_oy, hour, minute, second,$
                          month, cmonth, day_om, /mjdt   
     datestr=string(year-1900,month,day_om,format='(i2.2,i2.2,i2.2)')
     fln='plot_type_'+datestr
     
     ; read one day of data
     r_plot_type_onefile, path+fln, /VERBOSE
   
     ; check for "OPENR: Error opening file" in error string - and then get 
     ; the file remotely!
     ; comment this section out if you do not want this feature.
     res=strpos(!err_string,'OPENR: Error opening file')
     if res ne -1 then begin
       c_url=papco_getenv('plot_type_FTP_ARCHIVE')
       c_file=fln+'.idl'
       c_download_dir=path
       papco_get_remote_data, c_url, c_file, c_download_dir
       ;check if file "arrived" and then read it
       if get_err_no eq 0 then r_plot_type_onefile, path+fln, /VERBOSE
     endif   
     
     ;if a file was succesfully read, concatonate!
     if get_err_no eq 0 then begin 
       if no_files_read eq 0 then begin   ;start the array buffer
         input_data.time=input_data.time+86400.0*day_count
         input_data_buf=temporary(input_data)
       endif else begin                   ;otherwise concatonate arrays
         input_data.time=input_data.time+86400*day_count
         input_data_buf=[temporary(input_data_buf),temporary(input_data)]
       endelse        
       input_data=0                     ;clear array for reuse
       no_files_read=no_files_read+1
     endif
     day_count=day_count+1
     time.mjd=time.mjd+1                  ; next day
   endwhile         
   
; check if any files were read   
   if (no_files_read ne 0) and (get_err_no ne 1) then begin
     get_err_no=0
     get_err_msg='% r_plot_type: '+string(no_files_read,format="(i2)")+ $
       ' files read succesfully'
     ;clear arrays and load buffer into output array
     input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
   endif else begin
     get_err_no=1
     get_err_msg='% r_plot_type: no files found for timerange'
     return
   endelse      
   
; restrict data to actual time range requested
   ut1=mjdt_start.t
   ut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
   index=where((input_data.Time ge ut1) and (input_data.Time le ut2),count)
   if count ne 0 then begin
      input_data=input_data(index)
   endif   

end

;******************************************************************************
;*  NAME:        r_plot_type_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_plot_type_onefile, fln, VERBOSE=VERBOSE
  
   common plot_type_data, input_header, input_data
   common get_error, get_err_no, get_err_msg
   
   get_err_no=0
   get_err_msg='plot_type onefile read succesfully'
   
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
   on_ioerror, read_original
   
   restore,fln+'.idl',/verbose
   dealloc_lun,/quiet    ;frees logical units used by save/restore
   print,'% r_plot_type_oneday: idl saveset read succesfully'
   return

   read_original:
   ; if the .idl save file could not be found, then try to read the original
   r_plot_type_onefile_original, fln, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_plot_type_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_plot_type_onefile_original, fln, VERBOSE=VERBOSE
  
   common plot_type_data, input_header, input_data
   common get_error, get_err_no, get_err_msg  
   
   get_err_no=0
   get_err_msg='plot_type_onefile_orignal read succesfully'
   
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BU USER!!!!
  
; define the header and data structures. Edit to suit your data!
   max_no_rec=1000              ;the maximum number time records in your data
   ext='.txt'                   ;file extention of your dat files
   input_header={plot_type_header,tag1:0,tag2:0}
   dummy={plot_type_data, time:0.0d, val1:0.0, val2:0.0}
   input_data=replicate(dummy, max_no_rec)
   
   on_ioerror, out  
   openr,unit,fln+ext,/get_lun
   while not eof(unit) do begin
     ;read in data and store in input_data
     readu,unit,dummy
     input_data(i)=dummy
   endwhile

; end of user written section
;------------------------------------------------------------------------------
   
   
; save read data with correct time. Rename data to *.idl
  save,filename=fln+'.idl',input_header,input_data,/verbose  
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return
  
  out:
  ; if there was an error, report it. Use the IDL error variables. Error
  ; 171 is for "file not fond" which is used to invoke the remote get.  
  get_err_no=!ERROR
  get_err_msg='r_plot_type_onefile:!C '+!ERR_STRING
   
end
