;******************************************************************************
;* file papcoadd_crres_protel
;*      papcoadd_crres_protel
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for a crres_protel. 
;*
;* THIS IS A WORKING EXAMPLE MODULE WHICH CAN BE INCLUDED INTO papco AS IS
;* (it doesn't plot very exiting data though!)
;*
;* THIS TEMPLATE IS INTENDED AS THE STARTING POINT FOR DEVELOPING OWN MODULES.
;* COPY THE WHOLE crres_protel DIRECTORY TREE AND RENAME EVERY OCCURRENCE OF 
;* "crres_protel" WITH YOUR MODULE NAME, IN ALL THE FILES AND FILENAMES.
;*
;* If the file starting with "papcoadd" exist in the module directory, then the
;* routines contained thereinare utilized to add that plot type into PAPCO.
;* This makes papco "configurable" by adding or leaving out modules.
;*
;* ALL that is needed to include the module is to include the path to the
;* module directory tree in the IDL search path. This is done automatically 
;* by the PAPCO configure widget when you start PAPCO with
;* 
;* idl papco_setup
;*
;* If the plot type is in itself a sub-menu containing more crres_protels, then
;* these must be listed at the top of this file, as it is read by papco.
;* the entries for each sub-plot-type must come after the line 
;* "file papcoadd_crres_protel" and before the "end" line, one entry per line.
;* Some of the routines in this file must then be duplicated for each 
;* sub-plot-type used: see some of the exisiting modules for an example.
;*
;* Apart from editing this template the user has to write a panel 
;* editor widget for his crres_protel. Use the file 
;*    $papco_XX/papco_modules/crres_protel/papco_interface/crres_protel_panel.pro
;* as the template.
;*
;* If slice functions are to be supported a slice program needs to be written
;* too. Use the template file
;*    $papco_XX/papco_modules/crres_protel/papco_interface/crres_protel_slice.pro.
;*
;* Finally, a data read and data plot routine are required as well. The data
;* read routine will be specific for each crres_protel. An example for a standard
;* line plot is given in the file 
;*    $papco_XX/papco_modules/crres_protel/plot_read/p_data_type_line.pro
;* and for a standard color spectrogram in 
;*    $papco_XX/papco_modules/crres_protel/plot_read/p_data_type_spec.pro. 
;* These templates can also be used as a basis for more complex plots.
;*
;*
;* The Way these add-in routines work:
;* ===================================
;*
;* On initializing IDL for PAPCO, the routine find_modules.pro is used to
;* parse the IDL search path, and all the init_module.pro batch files found in 
;* the module sub-directories \papco_interface are written to the 
;* papco_add_init.pro batch file and then executed: In this way the user can 
;* specify all the routines that need to be compiled for his module at startup.
;* 
;* On starting papco, all module sub-directories \papco_interface are searched
;* for a file beginning with "papcoadd_". From the names of this file the 
;* crres_protel name is stripped off and used to construct all the procedure 
;* names contained in the file. The first routine, "addplottype_crres_protel", is
;* used to add the button for this crres_protel to the papco main window. Since
;* "addplottype_crres_protel" simply returns widget ID of the button, the button
;* itself can contain further sub-menus for plot sub-types.
;*
;* To inform papco of all these sub-plot types (and to avoid having to write a 
;* separate file for each sub-plot type) all the types have to be listed in the
;* header of the file papcoadd_crres_protel.pro. These entries have to finish off 
;* with the line ";* end". All crres_protel names are stored in the string array
;* crres_protels and are placed in a common block:
;*
;*    COMMON ADDINS, crres_protels
;*
;* If there are no sub-plot types, then the crres_protel as used in the file name
;* for papcoadd_crres_protel.pro is used to construct the calls to the procedures
;* in that file. The convention for naming these procedures is
;* "procedure_name_crres_protel" , so that papco can always construct the right
;* procedure call when needed. 
;*
;* When there are sub-plot types, their names get used to construct the
;* required calls. There are 6 procedures that need to be called for each
;* crres_protel, so that a set of these 6 procedures has to be written for each
;* sub-plot type. In the same way, you need to write a panel editor for each
;* plot type (sometimes several plot sub-types may share the same editor,
;* e.g. multiple line plots or color spectra, see papcoadd_crres_plots).
;* 
;* In PAPCO, whenever an action for a crres_protel is taken, the information
;* stored in the structure widgetData is queried. In this structure there
;* is an array of PlotInfo structures, which contain all the relevant 
;* information for a chosen panel. The field plotInfo.panelkind is loaded
;* with the name of the crres_protel. This is set by the user in this file in
;* the routine addpanel_crres_protel ( "panelKind='crres_protel'" ). THIS MUST
;* BE SET CORRECTLY AS ALL THE ROUTINE NAMES USED IN THIS FILE ARE
;* CONSTRUCTED USING THIS INFORMATION!
;*
;* Whenever papco has to undertake some action for a crres_protel, the field 
;* plotInfo.panelkind is compared to the entries in the (COMMON ADDINS) 
;* crres_protels array. If a match is found, the required procedure name is
;* constructed and called If not, an error occurs.
;*
;*    These procedures are contained:
;*
;*    PRO addplottype_crres_protel,plottypes_base,pb_plottype
;*        ->  adds the button for the new plot type to the main window.
;*
;*    PRO plottable_crres_protel, plottable
;*        -> Sets a variable to indicate plottable without orbit.
;*
;*    PRO canget_crres_protel, canget
;*        -> Sets a variable to indicate output to file is supported.
;*
;*    PRO descriptor_crres_protel, plotInfo, descriptStr
;*        -> Constructs the string used to label plots in the panels window
;*
;*    PRO addpanel_crres_protel
;*        -> Calls the panel editor for this plottype and adds the plot to the
;*            panel window.
;*
;*    PRO editor_crres_protel, plotNumber, newPlotData
;*        -> This procedure calls the panel editor when called from 
;*           papco_EditPlot (papco_paneledit.pro).
;*
;*    PRO  draw_crres_protel, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;*     updated for OUTPUT functionality, June 1998, Reiner Friedel
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_crres_protel,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*      This procedure adds a button for this crres_protel to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*      package, so that they can be automatically called when needed.
;*      The button itself can contain access to a sub-menu for further
;*      crres_protels.
;*
;* INPUTS:       
;*      plottypes_base  the widget ID of the parent widget to contain the
;*                      plottype buttons.
;*
;* OUTPUTS: 
;*      pb_plottype     the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for sub-types.
;*
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      addplottype_crres_protel,plottypes_base,buttonGeometry,pb_plottype
;******************************************************************************
pro addplottype_crres_protel,plottypes_base,buttonGeometry,pdm_crres_protel
  
; This example is for a simple button with a submenu. 
; **NEW** from PAPCO 09 is the inclusion of the configure button as
; the first entry of the module submenu. Further entries into the
; submenu are easily added using
;
;        { CW_PDMENU_S, 0, ' crres_protel submenu ' }, $
;  
; The second entry in this structure defines the following:
;                       3   -   label for the button
;                       0   -   submenu entry
;                       2   -   last submenu entry  
;
; The third entry in this structure is the name of the submenu entry.

  MenuDesc = [ $
               { CW_PDMENU_S, 3, strcenter('CRRES-PROT',10) }, $ 
               { CW_PDMENU_S, 0, 'configure plot type' }, $
               { CW_PDMENU_S, 2, 'crres_protel ' } $
             ]  
  
  pdm_crres_protel = CW_PDMENU(plottypes_base, MenuDesc, $
                            UVALUE='pdm_crres_protel')
  
; The name used for the submenu entries is user by PAPCO to construct
; the procedure/function calls needed for that part of the module.
  
; Given here is one submenu entry. IF YOU ADD FURTHER SUBMENU
; ENTRIES, THEN YOU WILL NEED TO WRITE ONE EACH THE
; plottable_crres_protel, canget_crres_protel, descriptor_crres_protel,  
; editor_crres_protel AND draw_crres_protel 
; ROUTINES THAT FOLLOW FOR EACH SUBMENU ENTRY, MAKING SURE YOU ADHERE
; TO THE CORRECT NAMING CONVENTION.
;
; You will also have to add another case to the case statement in
; addpanel_crres_protel.

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_crres_protel
;*
;* DESCRIPTION:  
;*      Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      Using an instance of the papco_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled. The
;*      desired default values for the plotinfo tags need to be edited by
;*      the user. This plotinfo structure will be passed to the panel editor
;*      When it is first called for the crres_protel.
;*      "panelEditor" must be set to the name of the panel editor routine.
;*
;* INPUTS:       
;*      event           event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      addpanel_crres_protel
;***************************************************************************
pro addpanel_crres_protel, event

  common get_error, get_err_no, get_err_msg
  
; The case statement here needs an entry for every submenu type you
; added in  the first procedure in this file (addplottype_crres_protel).
; For each submenu entry, you will need to specify the panelEditor to use.  
  
; The FIRST time you use this module you will have to configure it,
; i.e. set the data paths etc. Once you have done this PAPCO will
; write a crres_protel.config file into your papco_user_XX/papco_modules_defaults
; directory. 
; So here we check if this crres_protel.config file exists, and if not,
; we disallow the execution of any other option apart from "configure
; crres_protel". In this way the user is forced to configure the module
; before using it the first time!
  
  panelKind='crres_protel'
  panelKind=strlowcase(panelKind)
  
  r_module_config, panelKind, config_data, msg
  if (get_err_no ne 0) and (event.value ne 1) then begin
    message,'You need to configure the module first', /cont
    papco_module_configure,panelKind
    return
  endif  
   
  CASE event.value of
    1: BEGIN
        papco_module_configure, panelKind
    end
    2: BEGIN
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
        defaults.typeVector=[0,0,0,0]      ;set some tags to desired defaults
        defaults.channel=0            
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
    end
    else: begin
      print,'%addpanel_crres_protel: Nothing defined for this submenu entry'
      return
    end  
  endcase
      
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO plottable_crres_protel, plottable
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate plottable with / without orbit.
;*      PAPCO provides "set time by orbit" feature for CRRES and POLAR
;*      so far. Included for compatibility with older CRRES modules
;*      which ONLY use orbit as input to their read/plot routines
;*      (plotting over orbit boundaries is not enabled for CRRES)
;* 
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "plottable_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_crres_protel.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      plottable        a boolean 0 if plottable only by orbit
;*                                 1 if plottable by time also.
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;******************************************************************************
pro plottable_crres_protel, plottable

  plottable=1     ; 1 -> plotable without orbit info, by time 
                  ; 0 -> plotable only by orbit number not by time 
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO canget_crres_protel, canget
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate that crres_protel support the OUTPUT
;*      functionality. 
;*      
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "canget_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_crres_protel.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      canget         a boolean 0 if this feature is not available
;*                               1 if this feature is available
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      canget_crres_protel, canget
;******************************************************************************
pro canget_crres_protel, canget

  canget=  1      ; 1 -> plot routine can return just data to file
                  ; 0 -> feature disabled
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_crres_protel, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*      Constructs the string used to label plots in the panel window.
;*      The plotinfo structure passed in here is the panel's plotinfo
;*      as returned by the panel editor for this crres_protel. You can use
;*      the information in that structure to construct some informative
;*      label for the panel which then appears in the Panel Compose list.
;*
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "descriptor_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_crres_protel.
;*
;* INPUTS:       
;*      plotInfo        structure containing all data for this plot.
;*                      (structure type: papco_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*      descriptStr     a string containing the label
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      descriptor_crres_protel, plotInfo, descriptStr
;******************************************************************************
pro descriptor_crres_protel, plotInfo, descriptStr
  
  if PlotInfo.typeVector(0) ne 0 then $
     descriptStr=descriptStr+'Lineplot Channel : ' + $
       strtrim(string(plotInfo.typevector(0)), 2) $
  else $
    descriptStr=descriptStr+'Spectrum plot'
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO editor_crres_protel, plotNumber, newPlotData
;* 
;* DESCRIPTION:  
;*      This procedure calls the panel editor when called from 
;*      papco_EditPlot (papco_paneledit.pro). Provided here to allow
;*      several crres_protels to share a common panel editor. Normally this
;*      procedure does not need to be edited beyond the required name
;*      change for crres_protel.
;*
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "editor_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_crres_protel.
;*
;* INPUTS:       
;*      plotNumber      anInteger. 0<=plotNumber<=widgetData.numberOfPlots
;*                      the index of the plot in widgetData.plotInfos.
;*            
;* OUTPUTS: 
;*      newPlotData     the new structure returned by the panel editor
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      editor_crres_protel, plotNumber, newPlotData
;******************************************************************************
PRO editor_crres_protel, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

  plotInfo=widgetData.plotInfos(plotNumber)
  newPlotData=crres_protel_panel(plotInfo, $
                              ACTION='Update '+plotInfo.panelKind, $
                              GROUP=widgetData.main_base)
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO draw_crres_protel, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:  
;*      This procedure performs the actual data read call and the plot call
;*      for this plottype. The user need to pass to his plot routine all the
;*      parameters needed, typically these parameters are stored in the 
;*      currplotinfo structure or constructed from them. The currplotinfo
;*      structure tags normally used for passing information from the panel
;*      editor to the plot routine are:
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          swittch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      
;*      These tags have grown out of traditional usage and can be used for
;*      anything the user wants to use them for. There are plenty variables
;*      for most applications: Adding a new tag to the plotinfo structure
;*      is not user-definable but need to be done at the papco core level.
;* 
;*      The panelVector MUST be passed to your plotroutine as it is used
;*      to set the panel position.
;*      The _EXTRA=extra_PlotInfo should be passed if you intend to make use
;*      of papco functionality such as overplotting, linestyle choice and
;*      color bar choice.
;*
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "draw_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_crres_protel.
;*
;* INPUTS:       
;*      panelVector    position of the plot (see papco_conventions.text, 6)
;*      currPlotInfo   structure containing all data for this plot.
;*                     (structure papco_PLOTINFO, see papco_variables.text) 
;*
;* OUTPUTS: 
;*      get_data_call  a string containing the data call. Used by other
;*                     routines that need to know how to get the data
;*                     in a panel, e.g. to draw spectra, distribution functions
;*      IsRead         a boolean which is set when the the data is read the
;*                     first time. Useful if a given data-read puts data
;*                     into commons which can be plotted in many ways.
;*                     When set, data does not have to be read again.
;   slice_type:   Selects which slices are supported and for which slice
;                 routines have been written.
;                 This is an integer logical with 16 swittches. Currently
;                 there are 5 papco slices. You may extend this by
;                 further USER slices up to a total of 16 slices.
;                     0 = disable all slices
;                     1 = papco Zoom                 (2^0)
;                     2 = papco View Slice           (2^1)
;                     4 = papco Access level Zero    (2^2)
;                     8 = papco Write Paneldata      (2^3)
;                    16 = papco Hair Cursor          (2^4)
;                    32 = USER slice A or not used   (2^5)
;                    64 = USER slice B or not used   (2^6)
;                    "     "     "   "  "  "    "      "    
;                 32768 = USER slice Z or not used  (2^15)
;                        
;   You can support more than one type of slice by arithmetically adding the
;   options, eg slice_type=1+4  support Zoom and Level Zero
;*
;* KEYWORDS:
;*      OUTPUT        if set then plot routine should output data to file,
;*                    as well as to screen.
;*
;* CALLING SEQUENCE:
;*      draw_crres_protel, panelVector, currPlotInfo, orbitNo, $
;*                      get_data_call, slice_type, IsRead, NODATA=NODATA
;******************************************************************************
pro draw_crres_protel, panelVector, currPlotInfo, orbitNo, $
                    get_data_call, slice_type, IsRead, $
                    OUTPUT=OUTPUT

  COMMON extra, extra_PlotInfo  ; passes in papco-set keywords structure
    
  get_data_call = ''            ;use new method
  
; as papco dynamically allocates slice values for the user defined
; slices, one needs to be able to find out this assignment! Use the
; following function to find out the slice values for any slices you
; might have added for your module, so that you know which values to enable!
  
  ;slice_assigned_value=papco_slice_value('USER Defined Slice A')
  
; set up which mouse functions to allow
  slice_type=1+16

  IF currPlotInfo.ioptions(10) eq 1 then IsRead=1
  
  IF NOT IsRead THEN BEGIN                ; check if data has already been read
      CASE currplotinfo.ioptions(0) OF ; read data
          0:r_crres_protel,  orbitNo,  currPlotInfo 
          1:r_crres_protel_lav, currPlotInfo
      ENDCASE 
      ; check the read for success.
      IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)       THEN IsRead=1 $
      ELSE IsRead=0
  ENDIF

  IF IsRead THEN BEGIN
      CASE currplotinfo.ioptions(0) OF
          0: p_crres_protel_line, panelVector, currPlotInfo, OUTPUT=OUTPUT, $
            _EXTRA=extra_PlotInfo 
          1: p_crres_protel_lav, panelVector, currPlotInfo, OUTPUT=OUTPUT, $
             _EXTRA=extra_PlotInfo
      ENDCASE
  ENDIF

  ;IF currPlotInfo.ioptions(5) eq 1 then IsRead=IsRead else $
  ;  IsRead=0 ; this data ALWAYS has to be re-read - unless overwritten.

; Note: The structure extra_PlotInfo passed to the plot routine contains the
;       following tags: 
;                  overPlot               - overplot swittch
;                  plots_AtThisPosition   - no. of overplots here
;                  subtable               - index of papco colortable 
;                  PSYM                   - linestyle keyword for plot
;                  COLOR                  - linecolor keyword for plot
;               
;       They are set by papco and your plot routine can use them if you
;       want to support that functionality.

END
