;******************************************************************************
;* PROCEDURE:     
;*      p_crres_protel_lav, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (and/or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure PAPCD_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to make use of
;*              an extra common block.
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_crres_protel, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written January 2001, Reiner Friedel
;******************************************************************************
pro p_crres_protel_lav, panel, plotinfo, OUTPUT=OUTPUT, $
                        OVERPLOT=OVERPLOT, $
                        PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                        SUBTABLE=SUBTABLE, _EXTRA=extra_par
  
; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Line styles are also not useful. So the keywords 
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,_EXTRA=extra_par
; are not used at all here, but are in the call for consistency...

; The following common blocks are needed for PAPCO:
  common mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
  common yscale, yscl			;man/auto yscaling
  common zscale, zscl			;man/auto zscaling
  common shift_label, down              ;common for x-axis label shifting
  common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_crres_protel. 
  common crres_protel_lav, input_header, input_data
  common papco_color_names
  
  papco_Set_SubTable, SUBTABLE          ;sets the papco color table index
  panelset,panel		        ;sets the panel position viewport

  n_channels=160                        ;no of L-bins, 1/20th of an L 
  yarr=fltarr(n_channels,2)   
  yarr(*,0)=(findgen(160)+1)*0.05+0.95
  yarr(*,1)=(findgen(160)+2)*0.05+0.95
  yst=yarr(0,0)  &  yen=yarr(n_channels-1,1)

  ;prepare time array 
  convert_t90_to_date, mjdt_start, yr,doy,hr,min,sec,mo,cmo,dom, /mjdt
  tai_1=UTC2TAI({YEAR:yr, MONTH:mo, DAY:dom, HOUR:hr, MINUTE:min, $
                   SECOND:sec, MILLISECOND:0})
  convert_t90_to_date, mjdt_end, yr,doy,hr,min,sec,mo,cmo,dom, /mjdt
  tai_2=UTC2TAI({YEAR:yr, MONTH:mo, DAY:dom, HOUR:hr, MINUTE:min, $
                   SECOND:sec, MILLISECOND:0})
  ;restrict data in time - PROTEL LAV are in in TAI already 
  index=where((input_data.time ge tai_1) AND (input_data.time le tai_2),c)
  if c gt 2 then data=input_data(index) ELSE BEGIN
      message, 'No data to plot', /cont
      papco_Draw_ErrorPanel, panel, 'No data to plot'
      return
  ENDELSE 
  xut1 = 0  &  xut2 = tai_2-tai_1  
  ndat = n_elements(data)
  time = data.TIME-tai_1

  ;select data to plot depending on pitch, energy choices
  plot_type = plotinfo.ioptions(1)
  pitch     = plotinfo.ioptions(2)-1
  energy    = plotinfo.ioptions(3)-1
  CASE plot_type OF  
      0:BEGIN                   ;Energy(s) @Pitch Angle
          zmat = transpose(data.data(energy,pitch,*))
          uytitle='L-value'
      end
      1:BEGIN                   ;Pitch Angle(s) @Energy
          zmat = transpose(data.data(energy,pitch,*))
          uytitle='L-value'
          
      END
  ENDCASE

  nodata = -99
  uztit='flux'
  utitle='CRRES PROTEL!CPA: '+varprt(input_header.pitch(pitch))+$
    '!CEk: '+string(input_header.energy(energy), format = "(f4.1)")+' MeV'
                                              
; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data

; the procedure papco_autorange (in papco_XX/papco_lib/autoscaling is
; optional and sets the zlimits automatically for the data, attempting
; to use the full dynamic range of the color table. you may need to
; supply your own procedure for your own data. Nodata sets the nodata
; flag used for the dataset.

; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling

  if (zscl(panel(0),0) eq 1) then begin
    zmin=zscl(panel(0),1)  &  zmax=zscl(panel(0),2)
  endif else begin
    papco_autorange,zmat,zmin,zmax, $
      log=zscl(panel(0),3), exclude=0, nodata=[nodata, -55],  /verb
    zscl(panel(0),1)=zmin  &  zscl(panel(0),2)=zmax
  endelse

  if (yscl(panel(0),0) eq 1) then begin 
    yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
  endif else begin
    yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
  endelse
  
; use auto y-labeling routine to get "good" y-lables. This sets the
; global IDL graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS -
; so if you want to use this feature, avoid setting these or their
; corresponding keywords in your plot.
  papco_y_label, yst, yen, log=yscl(panel(0),3)
     
  extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zmin,zmax], $
                 ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
                 ztitle:uztit, $
                 xtickformat:'noticks',ztickformat:'papco_color_bar_log_ticks'}
  
; add keyword structure set locally and the one passed in
  extra_plotPar=create_struct(extra_plotPar, extra_par)
  
; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this rotuine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
  down=0
  papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar    
   
; do color plot and color bar plot. nodata lavifies the "nodata" flag
; in your data. Here we do not plot axis and data seperate, as
; papco_plot_colorlav cannot be used as an overplot!
  
  ;check the routine papco_plot_colorlav in papco_XX\papco_lib\color_plot
  ;for all the possible options of papco_plot_colorlav.
  ;Hint: you might want to use the resx, resy keywords, which set the
  ;resolution to plot add. Without these, the routine will try to use
  ;device resolution, which for postscript may result in HUGE files!
  ;resx, resy are in data coordinates. If the axis is logarithmic, 
  ;resx, resy need to be too.
  resx=10000  &  resy=0.05
  diff = time(1:ndat-1) - time(0:ndat-2)
  diff = diff(where(diff GT 0))
  resx=min(diff)/5.0
  diff = alog10(yarr(*, 1))-alog10(yarr(*, 0))
  resy=min(diff)/5.0

  ;set up special colors for plot. nodata=-99, grey
  ;"0" really mean no flux was measured - 
  ;-55 means that pitch angle bin was not covered, white
  sc = [[-55, white]]

  papco_plot_colorspec,zmat,time,yarr, nodata=nodata, resx=resx, resy=resy, $
    xstyle=5, ystyle=5, _extra=extra_plotPar, special_colors = sc
  papco_color_bar, _extra=extra_plotPar
  
; plot y-axis label at left of plot. Use scalable routine!  
  left_side_label,panel,uytitle,/rot90
   
; store the coordinate information into common block
  plotted_x = !x  &  plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. 
  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
    ;extra_plotPar.xtickformat='MJDT_DATE_3'
    ;extra_plotPar.color=1
    ;down=down+1
    ;plot,time,time, _extra=extra_plotPar, /NODATA
    ;x_side_label,panel,'time (UT)!CDate'
  endif
   
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.
  right_side_label,panel,utitle,/rot90
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  if KEYWORD_SET(OUTPUT) then begin
    print,'% p_crres_protel_lav: Writing plot data out to file'
    description='crres_protel template sample data - Energy lavtra, 10 channels'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, time, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
  endif
   
end 
