;******************************************************************************
;*  NAME:        r_crres_protel
;*
;*  DESCRIPTION: Routine to read crres_protel files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
pro r_crres_protel, in_orbit, plotinfo, VERBOSE=VERBOSE

  common crres_protel_data, input_header, input_data
  common mjdt, mjdt_start, mjdt_end
  common get_error, get_err_no, get_err_msg
   
; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

  get_err_no=0  &  get_err_msg='crres_protel read succesfully'
   
; check VERBOSE keyword and set
  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
  input_header={crres_meb, orbit:0, dim1:0l, dim2:0l, ndat:0l}
  input_header.orbit=in_orbit
   
; loop through from start to end time concatonating data files into one
; array. Here data are stored by orbit, so we need to find the orbit
; range corresponding to the start and end times.
  time=0  &  no_files_read=0
   
; check if orbit is non-zero, then read that orbit 
; - otherwise get list of orbits to read
  if in_orbit eq 0 then orbit_list=papco_getOrbitList('CRRES') $
    else orbit_list=[in_orbit]

  for i=0,n_elements(orbit_list)-1 do begin
    orbit=orbit_list(i)

    ; read one orbit of data
    r_crres_protel_onefile, orbit, plotinfo, VERBOSE=VERBOSE
  
    ;check for an error in get_error - and then get the file remotely!
    ;comment this section out if you do not want this feature.
    ;if get_err_no ne 0 then begin      
    ;  c_url=papco_getenv('CRRES_PROTEL_FTP_ARCHIVE')
    ;  c_file=fln+'.idl'  &  c_download_dir=path
    ;  papco_get_remote_data, c_url, c_file, c_download_dir
    ;  ;check if file "arrived" and then read it
    ;  if get_err_no eq 0 then begin
    ;    r_crres_protel_onefile, path+fln, VERBOSE=VERBOSE
    ;  endif else begin
    ;    get_err_msg='No data locally, remote get failed!C  ' + get_err_msg
    ;    return
    ;  endelse  
    ;endif   
     
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
      if no_files_read eq 0 then begin   ;start the array buffer
        input_data.time=input_data.time+86400.0*day_count
        input_data_buf=temporary(input_data)
      endif else begin                   ;otherwise concatonate arrays
        input_data.time=input_data.time+86400*day_count
        input_data_buf=[temporary(input_data_buf),temporary(input_data)]
      endelse        
      input_data=0                      ;clear array for reuse
      no_files_read=no_files_read+1
    endif
    day_count=day_count+1
    time.mjd=time.mjd+1                 ; next day
  endfor      
   
; check if any files were read   
  if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=temporary(input_data_buf)
  endif else begin
    get_err_no=1
    get_err_msg='No files found for timerange'
    return
  endelse      
   
; restrict data to actual time range requested
  ut1=mjdt_start.t
  ut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
  index=where((input_data.Time ge ut1) and (input_data.Time le ut2),count)
  if count ne 0 then input_data=input_data(index)

end

;******************************************************************************
;*  NAME:        r_crres_protel_onefile, in_orbit, plotinfo
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_crres_protel_onefile, orbit, plotinfo, VERBOSE=VERBOSE
  
  common crres_protel_data, input_header, input_data
  common get_error, get_err_no, get_err_msg
   
  get_err_no=0  &  get_err_msg='crres_protel onefile read succesfully'

; check env. variable pointing to data directory
  if not (papco_check_data_env('CRRES_PROTEL_DATA', PATH=path)) then return
  
  path = path+'savesets/'
  CASE plotinfo.ioptions(0) OF
      0:ext = '.min'
      1:ext = '.pad'
  ENDCASE
  fln = 'd08'+string(orbit, format = "(i4.4)")+ext
 
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
  on_ioerror, read_original
  restore,path+fln+'.idl', VERBOSE=VERBOSE
  dealloc_lun,/quiet   ;frees logical units used by save/restore
  message,'idl saveset '+fln+' read', /cont
  return

  read_original:       ;if the .idl file is not found, try to read the original
  r_crres_protel_onefile_original, orbit, plotinfo, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_crres_protel_onefile_original, in_orbit, plotinfo
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_crres_protel_onefile_original, orbit, plotinfo, VERBOSE=VERBOSE
  
  common crres_protel_data, input_header, input_data
  common get_error, get_err_no, get_err_msg  
   
  get_err_no=0  &  get_err_msg='crres_protel_onefile_orignal read succesfully'
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BY USER!!!!
  
; check env. variable pointing to data directory
  if not (papco_check_data_env('CRRES_PROTEL_DATA', PATH=path)) then return
  
  CASE plotinfo.ioptions(0) OF
      0:BEGIN ;minute data
          message, 'not implemented yet'
          fln = 'd08'+string(orbit, format = "(i4.4)")+'.min'
      end
      
      1:BEGIN ;L-averaged data
          fln = 'd08'+string(orbit, format = "(i4.4)")+'.pad'
          input_header={crres_protel_head, $
                orb:0l, year:0l, day:0l, nrec:0l, maxbin:0l, energy:fltarr(24)}
          dummy={crres_protel1_pad, leg:0, sh:0, sec:0l, $
                 flux:fltarr(24, 19), nobf:intarr(24, 19)}
  
          ;initialize variables to be read from fortran file
          ;header stuff
          orb = 0l & year = 0l & day = 0l & nrec = 0l & maxbin = 0l
          energy = fltarr(24) & zeroes = lonarr(657)
          ;data record stuff
          leg = 0 & sh = 0 & sec = 0l & flux = fltarr(24, 19)
          nobf = intarr(24, 19)

          on_ioerror, out  
          openr,unit,path+fln, /get_lun, /swap_if_little_endian

          readu, unit, orb, year, day, nrec, maxbin, energy, zeroes
          input_header.orb = orb
          input_header.year = year
          input_header.day = day
          input_header.nrec = nrec
          input_header.energy = energy
          
          print, orb, year, day, nrec, maxbin
          IF nrec EQ 0 THEN GOTO, out
          input_data=replicate(dummy, nrec)
  
          FOR i = 0l, nrec-1 DO begin
            readu, unit, leg, sh, sec, flux, nobf
            input_data(i).leg = leg
            input_data(i).sh = sh
            input_data(i).sec = sec 
            input_data(i).flux = flux 
            input_data(i).nobf = nobf
            ;print, i, leg, sh, sh*0.05+0.95, sec
          endfor
          close, unit & free_lun, unit
      END
  ENDCASE


; end of user written section
;------------------------------------------------------------------------------
   
; save read data with correct time. Rename data to *.idl
  save,filename=path+'savesets/'+fln+'.idl', input_header, input_data, $
    /compress, VERBOSE=VERBOSE
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return
  
  out:
  ; if there was an error, report it. 
  get_err_no=!ERROR
  get_err_msg='!C '+!ERR_STRING
   
end


;******************************************************************************
;*  NAME:        r_crres_protel_onefile, in_orbit, plotinfo
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_crres_protel_lav, plotinfo, VERBOSE=VERBOSE
  
  common crres_protel_lav, input_header, input_data
  common get_error, get_err_no, get_err_msg
   
  get_err_no=0  &  get_err_msg='crres_protel onefile read succesfully'

  common crres_protel_data, protel_header, protel_data
  common mjdt, mjdt_start, mjdt_end
   
; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

  get_err_no=0  &  get_err_msg='crres_protel_lav read succesfully'
   
; check VERBOSE keyword and set
  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; check env. variable pointing to data directory
  if not (papco_check_data_env('CRRES_PROTEL_DATA', PATH=path)) then return

  ;read LAV saveset if it exists
  ON_IOERROR, make_from_original
  fln = 'd08_lav.pad'
  message, 'Restoring PROTAl LAV saveset', /cont
  restore,path+'savesets/'+fln+'.idl', VERBOSE=VERBOSE
  return

  make_from_original:
  message, 'Making PROTEL LAV data ... slow', /cont

; make data structure for L-averaged PROTEL data.
  data = fltarr(24, 19, 180) & data(*, *, *) = -99
  dummy = {crres_protel_lav2,  time:0.0d, orbit:-99, leg:0, $
           data:data}
 
; loop through from start to end time assembling L-averaged data into
; an array. Here data are stored by orbit, so we need to find the orbit
; ranges corresponding to the start and end times.
  no_files_read = 0
   
; get list of orbits to read
  orbit_list=papco_getOrbitList('CRRES')
  input_data = replicate(dummy, n_elements(orbit_list)*2)
  count = 0

  for i=0,n_elements(orbit_list)-1 do begin
    ; read one orbit of data
    r_crres_protel_onefile, orbit_list(i), plotinfo, VERBOSE=VERBOSE
    IF get_err_no NE 0 THEN continue
    
    ;find which data bin zeroes are due to no measurment
    FOR n = 0, n_elements(protel_data)-1 DO BEGIN
        nobf = protel_data(n).nobf
        idx = where(nobf EQ 0, c)
        IF c NE 0 THEN  protel_data(n).flux(idx) = -55 ;set to "no meas. flag"
    ENDFOR

    ;assign time to the L-cut: average of cut   
    FOR j = 1, 2 DO BEGIN
        index = where(protel_data.leg EQ j, c) ;select one leg
        IF c gt 2 THEN BEGIN
            result = moment(protel_data(index).sec)
            sec = result(0)
            ;convert time of slice to TAI
            result = convert_TZERO_TO_T90(sec, protel_header.day, $
                                    protel_header.year, /mjdt)
            tai = UTC2TAI({mjd:result.mjd, time:result.t*1000})
            print, TAI2UTC(tai, /VMS)
            ;fold data into LAV array
            input_data(count).time = tai
            input_data(count).leg = j
            input_data(count).orbit = orbit_list(i)
            
            FOR k = 0,n_elements(index)-1 do BEGIN 
                input_data(count).data(*, *, protel_data(index(k)).sh) = $
                  protel_data(index(k)).flux
            ENDFOR
            
            count = count+1

        ENDIF 
    ENDFOR 
  endfor

  input_data = input_data(0:count-1)
  input_header = {crres_protel_lav_headr, $
                  energy:protel_header.energy,  $
                  pitch:indgen(19)*5}

  ;save LAV data in saveset.
  fln = 'd08_lav.pad'
  save,filename=path+'savesets/'+fln+'.idl', input_header, input_data, $
    /compress, VERBOSE=VERBOSE
  dealloc_lun,/quiet                
  

end   
