;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  GOES_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the PAPCO_PLOTINFO-type
;*                      this structure is edited in the panel  editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*      All the functions in $PAPCO_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at last some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I 
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      switch:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, switch and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          switch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commincate between your
;*     panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which PAPCO supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      PAPCO internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $                       
;*                      xut2:0.0}                           
;*
;*                  
;*      For an enhanced description read 
;*                          $PAPCO_XX/PAPCO_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=PAPCO_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION goes_panel, aPlotInfo, GROUP=group, ACTION=ACTION

COMMON goes_panel_data, widgetData

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
title = aPlotInfo.panelKind + ' - ' + papco_module_version('goes')
panel_base=papco_widget_base(TITLE=title, COLUMN=1, GROUP_LEADER=group)

; All PAPCO_panel_Add* routines are PAPCO panel widgets that can be included
; in the panel editor or not. They provide functionality that PAPCO supports.
; They dynamically extend the panel editors widget structure, and all the 
; events they generate are supported by PAPCO_panel_Event, transparent to the
; user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
widgets=PAPCO_panel_AddHeightWidgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)

;
; <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
;  widget to chose the plot (magnetic field, ion, electron)
;
   type_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=2) 
   typeNames=['1-8 Ang X-ray Flux', $
              '.5-4 Ang X-ray Flux', $
              'Northward Magnetic Field', $
              'Earthward Magnetic Field', $
              'Eastward Magnetic Field', $
              'Total Magnetic Feld', $
              '>= 2 MeV electrons', $
              '.6 -   4.2 MeV protons', $
              '4.2 -   8.7 MeV protons', $
              '8.7 -  14.5 MeV protons', $
              '15.0 -  44.0 MeV protons', $
              '39.0 -  82.0 MeV protons', $
              '84.0 - 200.0 MeV protons', $
              '110.0 - 500.0 MeV protons']
   X_base = WIDGET_BASE(type_base, /FRAME, /COLUMN, /NONEXCLUSIVE) 
   B_base = WIDGET_BASE(type_base, /FRAME, /COLUMN, /NONEXCLUSIVE) 
   E_base = WIDGET_BASE(type_base, /FRAME, /COLUMN, /NONEXCLUSIVE) 
   P_base = WIDGET_BASE(type_base, /FRAME, /COLUMN, /NONEXCLUSIVE) 


   spb_type=lonarr(N_ELEMENTS(typeNames))

   FOR i=0, 1 DO $
      spb_type(i)=WIDGET_BUTTON(X_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))
   FOR i=2, 5 DO $
      spb_type(i)=WIDGET_BUTTON(B_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))
   spb_type(6)=WIDGET_BUTTON(E_base, $
			VALUE=typeNames(6), $
			UVALUE='spb_type'+strtrim(string(6), 2))
   FOR i=7, 13 DO $
      spb_type(i)=WIDGET_BUTTON(P_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))

   index = where(aPlotInfo.ioptions eq 1)
   for i=0,n_elements(index)-1 do begin
     WIDGET_CONTROL, spb_type(index(i)), SET_BUTTON=1
   endfor
;
;
;--- now, add the widgets for choosing an satellite----------------------------
;
;
   sat_base=WIDGET_BASE(panel_base, column=3, /frame)
   goes_Name=CW_FIELD(sat_base, $
                         title='Satellite ID:', $
                         value=aPlotInfo.filename)
   goes_LIST=WIDGET_LIST(sat_base, $
                        ysize=5, xsize=8, $
                        UVALUE='goes_LIST')
   label_list_base=WIDGET_LABEL(sat_base, $
                          value='satellites for the time chosen')

;
; <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
; for examples of standard buttons or sliders look at other panel editors.
; some standard buttons which have been recycled before can be found in
; $PAPCO_MODULES/panels&slice/more_panel_widgets

; As the same panel is called when you edit a plot, interrogate the tags in the
; plotInfoStruct (aplotInfo) passed in and set your widgets accordingly. 
; In this way the widget is always reinitialized to its last state.

; The next four PAPCO panel widgets will be added only if you want to
; support that functionality, and if they make sense for your goes.

;--- add the  PAPCO panel widget for choosing the plot-style for line plots ---
   widgets=PAPCO_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
   additionalWData=create_struct(widgets, additionalWData)
;--- add the PAPCO panel widget for choosing the y-axis-ranges ----------------
   widgets=PAPCO_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors.
;    if you want to make use of the configure feature, then add the
;    keyword CONFIG
   widgets=PAPCO_panel_AddActionButtons(panel_base, Action, aPlotInfo, /CONFIG)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'. You should also actually create 
;    that helpfile in the papco_interface directory of your module (a template
;    for this help file exist in the goes example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
satnames = strarr(4)
thisWidgetsData={helpFile:'goes_panel.help', $
                    typevector:aPlotInfo.typevector, $
                    ioptions:aPlotInfo.ioptions, $
                    satName:aPlotInfo.filename, $                    
                    goes_Name: goes_Name, $
                    goes_LIST: goes_LIST, $
                    spb_type:spb_type}
widgetData=PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
PAPCO_Panel_Realize, panel_base, widgetData
WIDGET_CONTROL, panel_base, HOURGLASS=1
goes_refresh, /ALL      
WIDGET_CONTROL, panel_base, HOURGLASS=0

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reported but the 
; data contained in aPlotInfo would be returned.
papco_xmanager, 'goes_panel', panel_base, /MODAL

; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - the typeVector-tag and switch-tag must
; always be included, but you can add more. However, if possible stick to the 
; tags provided in aPlotInfo, it makes life easier. 
returnData={typevector:widgetData.typevector, $
                filename:WidgetData.satname, $
                 ioptions:WidgetData.ioptions}
additionalRData=PAPCO_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)
END


;******************************************************************************
;* PROCEDURE:     
;*      PRO goes_refresh, ALL=ALL, SATID=SATID, SATLIST=SATLIST
;* 
;* DESCRIPTION:  
;*      update the panelplot-editor display
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      if set (e.g. /SATID), only the widgets influenced by the information
;*      that the keyword's name describes, are updated.
;*      If the keyword /ALL is set, all of the window is updated.
;*      
;*      SATELLITE ID
;*      ALL
;*
;* CALLING SEQUENCE:
;*      have a look the the event-routine los_alamos_Panel_event
;*
;* MODIFICATION HISTORY:       
;*      october 1995, written by R. Friedel
;*      june    1997, modified by S. Bourdarie
;******************************************************************************
PRO goes_refresh, ALL=ALL, SATID=SATID, SATLIST=SATLIST
                   
COMMON goes_panel_data, widgetdata
COMMON goes_list, satnames

IF KEYWORD_SET(ALL) THEN BEGIN
    SATID = 1
    SATLIST = 1
    CHANNEL = 1
ENDIF

IF KEYWORD_SET(SATLIST) THEN BEGIN
    satnames=goes_satlist()
    WIDGET_CONTROL,WidgetData.goes_LIST, SET_VALUE=[satnames]
ENDIF

IF KEYWORD_SET(SATID) THEN BEGIN
    WIDGET_CONTROL,WidgetData.goes_Name, $
         SET_VALUE=WidgetData.satname
ENDIF

END 

;******************************************************************************
;* PROCEDURE:     
;*      PRO goes_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO goes_panel_event, event

COMMON goes_panel_DATA, widgetData
COMMON goes_list,satnames

WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to 
; 'PAPCO_panel_Event', which handles the events generated
; by the PAPCO-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'PAPCO_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.

IF strleft(uval, 8) EQ 'spb_type' THEN BEGIN
    press = fix(strfrom(uval, 8))
    idx = where(WidgetData.ioptions eq 1)
    IF (idx(0) le 1 and press le 1) THEN begin
        if WidgetData.ioptions(press) eq 1 then $
             WidgetData.ioptions(press) = 0 else $
             WidgetData.ioptions(press) = 1
        return
    ENDIF  
    if (idx(0) gt 1 and idx(0) le 5 and press gt 1 and press le 5) THEN BEGIN
        ;Mag field choices
        if WidgetData.ioptions(press) eq 1 then $
             WidgetData.ioptions(press) = 0 else $
             WidgetData.ioptions(press) = 1
        return
    ENDIF
      if (idx(0) eq 6 and press eq 6) THEN begin
         if WidgetData.ioptions(press) eq 1 then $
             WidgetData.ioptions(press) = 0 else $
             WidgetData.ioptions(press) = 1
         return
      endif
      if (idx(0) gt 6 and idx(0) le 13 and press gt 6 and press le 13) THEN begin
         if WidgetData.ioptions(press) eq 1 then $
             WidgetData.ioptions(press) = 0 else $
             WidgetData.ioptions(press) = 1
         return
      endif
      WidgetData.ioptions(idx) = 0
      WidgetData.ioptions(press) = 1
      for i=0,n_elements(idx)-1 do begin
         WIDGET_CONTROL, WidgetData.spb_type(idx(i)), SET_BUTTON=0
      endfor

    return
ENDIF

   CASE uval OF
      'goes_LIST' : IF N_ELEMENTS(size(satnames)) EQ 4 THEN BEGIN
         WidgetData.satname=satnames(event.index)
         goes_refresh, /SATID
      END 

     'pb_done' : BEGIN
        uval=''
        idx = where(widgetData.ioptions eq 1,count)
        if count eq 0 then begin
          dummy=messagebox(['If you want to exit without choosing '+ $
                            'a plot', $
                            'use the Cancel-button.'],$
                           ['O.K,'], $
                           title='no plot selected')
           return
        endif
        if WidgetData.satName eq '' then begin
          dummy=messagebox(['If you want to exit without choosing '+ $
                            'a satellite name', $
                            'use the Cancel-button.'],$
                           ['O.K,'], $
                           title='no satellite ID')
        endif else uval='pb_done'
      END

     'pb_done_and_cont' : BEGIN
        uval=''
        idx = where(widgetData.ioptions eq 1,count)
        if count eq 0 then begin
          dummy=messagebox(['If you want to exit without choosing '+ $
                            'a plot', $
                            'use the Cancel-button.'],$
                           ['O.K,'], $
                           title='no plot selected')
           return
        endif
        if WidgetData.satName eq '' then begin
          dummy=messagebox(['If you want to exit without choosing '+ $
                            'a satellite name', $
                            'use the Cancel-button.'],$
                           ['O.K,'], $
                           title='no satellite ID')
        endif else uval='pb_done_and_cont'
      END

      'pb_cancel' : BEGIN
        WidgetData=PAPCO_Panel_event(event,WidgetData)        
      END

      ELSE : begin
        WidgetData=PAPCO_Panel_event(event,WidgetData)        
      END

  ENDCASE

IF (uval eq 'pb_done') or (uval eq 'pb_done_and_cont') then $
                       WidgetData=PAPCO_Panel_event(event,WidgetData)

END

