;******************************************************************************
;* file papcoadd_goes
;*      papcoadd_goes
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for a GOES. 
;*
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_GOES,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*      This procedure adds a button for this GOES to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*      package, so that they can be automatically called when needed.
;*      The button itself can contain access to a sub-menu for further
;*      GOESs.
;*
;* INPUTS:       
;*      plottypes_base  the widget ID of the parent widget to contain the
;*                      plottype buttons.
;*
;* OUTPUTS: 
;*      pb_plottype     the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for sub-types.
;*
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      addplottype_GOES,plottypes_base,buttonGeometry,pb_plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro addplottype_goes, plottypes_base, buttonGeometry, pdm_goes

MenuDesc = [ { CW_PDMENU_S, 3, strcenter('NOAA GOES',12) }, $ 
             { CW_PDMENU_S, 0, 'configure goes' }, $
             { CW_PDMENU_S, 2, 'goes' } ]  
  
pdm_goes = CW_PDMENU(plottypes_base, MenuDesc, $
                     UVALUE='pdm_goes')

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_goes
;*
;* DESCRIPTION:  
;*      Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      Using an instance of the PAPCO_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled. The
;*      desired default values for the plotinfo tags need to be edited by
;*      the user. This plotinfo structure will be passed to the panel editor
;*      When it is first called for the goes.
;*      "panelEditor" must be set to the name of the panel editor routine.
;*
;* INPUTS:       
;*      event           event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      addpanel_goes
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;***************************************************************************
pro addpanel_goes,event

COMMON PLOT_COMPOSER, widgetData
COMMON get_error, get_err_no, get_err_msg

panelKind='goes'
panelKind=strlowcase(panelKind)

r_module_config, panelKind, config_data, msg
if (get_err_no ne 0) and (event.value ne 1) then begin
    message,'You need to configure the module first', /cont
    papco_module_configure,panelKind
    return
endif  

CASE event.value of
    1: BEGIN
        papco_module_configure, panelKind
    end
    2: BEGIN
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
        defaults.typeVector=[0,0,0,0]      ;set some tags to desired defaults
        defaults.ioptions(*) = 0
        defaults.ioptions(0) = 1
        defaults.channel=0            
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
    end
    else: begin
      message,'Nothing defined for this submenu entry', /cont
      return
    end  
endcase

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO plottable_goes, plottable
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate plottable without orbit.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      plottable        a boolean 0 if plottable only by CRRES orbit
;*                                1 if plottable by time also.
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      plottable_goes, plottable
;*
;* MODIFICATION HISTORY:       
;*      written November 1995, Reiner Friedel
;******************************************************************************
pro plottable_goes, plottable

plottable=1     ; 1 -> plotable without orbit info, by time 
                ; 0 -> plotable only by orbit number not by time 
END

;******************************************************************************
pro canget_goes, canget

canget=  1      ; 1 -> plot routine can return just data 
                ; 0 -> feature disabled
END
;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_goes, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*      Constructs the string used to label plots in the panel window.
;*      The plotinfo structure passed in here is the panel's plotinfo
;*      as returned by the panel editor for this goes. You can use
;*      the information in that structure to construct some informative
;*      label for the panel which then appears in the Panel Compose list.
;*
;* INPUTS:       
;*      plotInfo        structure containing all data for this plot.
;*                      (structure type: PAPCO_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*      descriptStr     a string containing the label
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      descriptor_goes, plotInfo, descriptStr
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro descriptor_goes, plotInfo, descriptStr

;   typeNames=['1-8 Ang X-ray Flux', $
;              '.5-4 Ang X-ray Flux', $
;              'Northward Magnetic Field', $
;              'Earthward Magnetic Field', $
;              'Eastward Magnetic Field', $
;              'Total Magnetic Feld', $
;              '>= 2 MeV electrons', $
;              '.6 -   4.2 MeV protons', $
;              '4.2 -   8.7 MeV protons', $
;              '8.7 -  14.5 MeV protons', $
;              '15.0 -  44.0 MeV protons', $
;              '39.0 -  82.0 MeV protons', $
;              '84.0 - 200.0 MeV protons', $
;              '110.0 - 500.0 MeV protons']
;   descriptStr=descriptStr+typeNames(plotInfo.typeVector(0))$
;               +' - ' + plotInfo.filename
   index = where(plotInfo.ioptions eq 1)
   descriptStr=descriptStr+'['
   for i=0,n_elements(index)-2 do begin
       descriptStr=descriptStr+string(index(i),format='(i2.2)')+','
   endfor
   descriptStr=descriptStr+string(index(n_elements(index)-1),format='(i2.2)')
   descriptStr=descriptStr+'] - ' + plotInfo.filename
   
END


;******************************************************************************
;* PROCEDURE:     
;*      PRO draw_goes, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:  
;*      This procedure performs the actual data read call and the plot call
;*      for this plottype. The user need to pass to his plot routine all the
;*      parameters needed, typically these parameters are stored in the 
;*      currplotinfo structure or constructed from them. The currplotinfo
;*      structure tags normally used for passing information from the panel
;*      editor to the plot routine are:
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          switch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      
;*      These tags have grown out of traditional usage and can be used for
;*      anything the user wants to use them for. There are plenty variables
;*      for most applications: Adding a new tag to the plotinfo structure
;*      is not user-definable but need to be done at the PAPCo core level.
;* 
;*      The panelVector MUST be passed to your plotroutine as it is used
;*      to set the panel position.
;*      The _EXTRA=extra_PlotInfo should be passed if you intend to make use
;*      of PAPCO functionality such as overplotting, linestyle choice and
;*      color bar choice.
;*
;* INPUTS:       
;*      panelVector    position of the plot (see papco_conventions.text, 6)
;*      currPlotInfo   structure containing all data for this plot.
;*                     (structure PAPCO_PLOTINFO, see papco_variables.text) 
;*
;* OUTPUTS: 
;*      get_data_call  a string containing the data call. Used by other
;*                     routines that need to know how to get the data
;*                     in a panel, e.g. to draw spectra, distribution functions
;*      IsRead         a boolean which is set when the the data is read the
;*                     first time. Useful if a given data-read puts data
;*                     into commons which can be plotted in many ways.
;*                     When set, data does not have to be read again.
;*      slice_type      Selects which slices are supported and for which
;*                      slice routines have been written.
;*                      This is a byte logical with 8 switches. Currently
;*                      Three are supported (this may be extended later)
;*                        1 = mouse zooms (utility slice, always possible)
;*                        2 = mouse slices
;*                        4 = mouse calls level zero
;*                        8 = mouse views slice (polar)  
;*                       16 = mouse views ephemeris
;*                       32 = mouse writes paneldata (utility slice)
;*                       64, 128 = not supported yet
;*                      you can support more than one type of slice
;*                      by arithmetically adding the options, eg
;*                      slice_type=5  support zoom and level zero
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      draw_goes, panelVector, currPlotInfo, orbitNo, $
;*                                 get_data_call, slice_type, IsRead
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro draw_goes, panelVector, currPlotInfo, orbitNo, $
               get_data_call, slice_type, IsRead, OUTPUT=OUTPUT

COMMON extra, extra_PlotInfo  ; passes in PAPCO-set keywords structure

get_data_call="r_goes"   ; data read call used by slices

; set parameter for plot routine. In this example, channel 1 -10 chooses the 
; channel number for a line plot, channel=0 chooses a spectrum color plot.
channel = where(currPlotInfo.ioptions eq 1)
; set up which mouse functions to allow
slice_type=1+8+16

IsRead = 0
 
IF NOT IsRead THEN BEGIN      ; check if data has already been read
    r_goes,currPlotInfo.filename             ; read data
    ; check the read for success.
    IF PAPCO_Draw_CheckResult(currPlotInfo.panelKind, $
                                panelVector, orbitNr, $
                                currPlotInfo.typeVector, $
                                currPlotInfo.channel)   THEN IsRead=1 
ENDIF


IF IsRead THEN BEGIN
    p_goes_line, panelVector, channel,currPlotInfo.fileName, $
      OUTPUT=OUTPUT, _EXTRA=extra_PlotInfo
ENDIF

; Note: The structure extra_PlotInfo passed to the plot routine contains the
;       following tags: 
;                  overPlot               - overplot switch
;                  plots_AtThisPosition   - no. of overplots here
;                  subtable               - index of PAPCO colortable 
;                  PSYM                   - linestyle keyword for plot
;                  COLOR                  - linecolor keyword for plot
;               
;       They are set by PAPCO and your plot routine can use them if you
;       want to support that functionality.

END

