;******************************************************************************
;* file p_GOES_line.pro
;*
;* this file is a template for a plot routine which can be used by PAPCO.
;* it includes all the periphery needed, the user simply has to add in
;* the actual plot statement for his/her kind of data.
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_GOES_line, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in PAPCO.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.switch (three element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_GOES_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;******************************************************************************
PRO p_goes_line, panel, channel,satID,OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                      SUBTABLE=SUBTABLE, _EXTRA=extra_par

; The following common blocks are needed:

COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
COMMON yscale, yscl			;man/auto yscaling
COMMON shift_label, down             ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse
COMMON papco_color_names, black, red, green, yellow, blue, magenta, cyan, $
  white, burgundy, olive, dark_green, teal, $
  royal_blue, violet, dark_grey, grey

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_GOES. 
COMMON goes_data, input_header, input_data
COMMON time, xut1,xut2

panelset,panel			;sets the panel position viewport

; make color array
couleur_arr = [black, red, green, yellow, blue, magenta, cyan, $
white, burgundy, olive, dark_green, teal, $
royal_blue, violet, dark_grey, grey]

;for plotting, always start at zero to get maximum
;resolution (plot works in float only, large values get rounded!) 
tai_1 = mjdt_start.t                       
tai_2 = long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t

xut1 = 0 &  xut2=tai_2-tai_1 & time = input_data.time-tai_1

;time between two points (needed not to plot missing data)
tmin = 500.

;restrict data to actual time range requested - this makes for faster zooming
index = where((time GE xut1) AND (time LE xut2),count)
IF count NE 0 THEN BEGIN 
    data = input_data(index)
    time = time(index)
ENDIF

yray = fltarr(n_elements(time),n_elements(channel))
utitle = strarr(n_elements(channel))
ind = -1
utitle1=satID+'!C5 min. Av.'

FOR i=0,n_elements(channel)-1 DO BEGIN
    ind = ind+1
    CASE channel(i) OF
         0:begin
            yray(*,ind)=data.XL
            utitle(ind)='1-8 Ang'
            uytitle='[Watts/m!e2!n]'
            nodat=3.27E+04
         end
         1:begin
            yray(*,ind)=data.XS
            utitle(ind)='.5-4 Ang'
            uytitle='[Watts/m!e2!n]'
            nodat=3.27E+04
         end
         2:begin
           yray(*,ind)=data.Hp
           utitle(ind)='H!Dp!N'
           uytitle='[nT]'
           nodat=32711.
         end
         3:begin
           yray(*,ind)=data.He
           utitle(ind)='H!De!N'
           uytitle='[nT]'
           nodat=32711.
         end
         4:begin
           yray(*,ind)=data.Hn
           utitle(ind)='H!Dn!N'
           uytitle='[nT]'
           nodat=32711.
         end
         5:begin
           yray(*,ind)=data.Ht
           utitle(ind)='B!DTOT!N'
           uytitle='[nT]'
           nodat=32711.
         end
         6:begin
           yray(*,ind)=data.E1/1000.
           utitle(ind)='>= 2 MeV!Celectrons'
           uytitle='[cm!e-2!ns!e-1!nkeV!e-1!nsr!e-1!n]'
           nodat=3.27E+01
         end
         7:begin
           yray(*,ind)=data.P1/1000.
           utitle(ind)='.6 - 4.2 MeV'
           uytitle='[cm!e-2!ns!e-1!nkeV!e-1!nsr!e-1!n]'
           nodat=3.27E+01
         end
         8:begin
           yray(*,ind)=data.P2/1000.
           utitle(ind)='4.2 - 8.7 MeV'
           uytitle='[cm!e-2!ns!e-1!nkeV!e-1!nsr!e-1!n]'
           nodat=3.27E+01
         end
         9:begin
           yray(*,ind)=data.P3/1000.
           utitle(ind)='8.7 - 14.5 MeV'
           uytitle='[cm!e-2!ns!e-1!nkeV!e-1!nsr!e-1!n]'
           nodat=3.27E+01
         end
         10:begin
           yray(*,ind)=data.P4/1000.
           utitle(ind)='15.0 - 44.0 MeV'
           uytitle='[cm!e-2!ns!e-1!nkeV!e-1!nsr!e-1!n]'
           nodat=3.27E+01
         end
         11:begin
           yray(*,ind)=data.P5/1000.
           utitle(ind)='39.0 - 82.0 MeV'
           uytitle='[cm!e-2!ns!e-1!nkeV!e-1!nsr!e-1!n]'
           nodat=3.27E+01
         end
         12:begin
           yray(*,ind)=data.P6/1000.
           utitle(ind)='84.0 - 200.0 MeV'
           uytitle='[cm!e-2!ns!e-1!nkeV!e-1!nsr!e-1!n]'
           nodat=3.27E+01
         end
         13:begin
           yray(*,ind)=data.P7/1000.
           utitle(ind)='110.0 - 500.0 MeV'
           uytitle='[cm!e-2!ns!e-1!nkeV!e-1!nsr!e-1!n]'
           nodat=3.27E+01
         end
    endcase
ENDFOR  

if channel(0) le 1 then utitle1 = utitle1 + '!CX-ray Flux'
if channel(0) ge 7 then utitle1 = utitle1 + '!Cprotons'

ind = -1
FOR i=0,n_elements(channel)-1 DO BEGIN
    ind = ind+1
    CASE channel(i) OF
         0:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim_io,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 1
         end
         1:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim_io,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 1
         end
         2:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim4,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 0
         end
         3:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim4,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 0
         end
         4:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim4,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 0
         end
         5:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim4,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 0
         end
         6:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim_io,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 1
         end
         7:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim_io,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 1
         end
         8:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim_io,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 1
         end
         9:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim_io,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 1
         end
         10:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim_io,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 1
         end
         11:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim_io,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 1
         end
         12:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim_io,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 1
         end
         13:begin
           yvect = yray(*,ind)
           index=where(yvect ne nodat, count)
           if count ne 0 then begin
             yvect=yvect(index)
             t=time(index)
           endif else t=time
           ylim_io,t,yvect,ymin,ymax
           if (ind eq 0) then begin
              yst = ymin
              yen = ymax
           endif else begin
              if ymin lt yst then yst = ymin
              if ymax gt yen then yen = ymax
           endelse
           yscl(*,3) = 1
         end
    ENDCASE
ENDFOR 

; the variables used above have the following meaning:
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;
; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for "linear" and 1 for "log"  scale
; when scaling is set to automatic, the automatic y-limits are loaded in.

; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for "linear" and 1 for "log"  scale
; when scaling is set to automatic, the automatic y-limits are loaded in.
   
IF (yscl(panel(0),0) EQ 1) THEN BEGIN 
    yst=yscl(panel(0),1)  &   yen=yscl(panel(0),2)
ENDIF ELSE BEGIN
    papco_ylim, time, yray, yst, yen, BAD_DATA = nodata
    yscl(panel(0),1)=yst  &   yscl(panel(0),2)=yen
ENDELSE 

; use auto y-labeling routine to get "good" y-lables. This sets the
; global IDL graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS -
; so if you want to use this feature, avoid setting these or their
; corresponding keywords in your plot.
papco_y_label, yst, yen, log=yscl(panel(0),3)

; set up extra plot keywords, common ones first. 
; yrange, xrange  are always needed. 
; the rest you can set if you want to override PAPCO defaults.
; See $PAPCO_BASE/PAPCO/papco_draw for the definition of the PAPCO default
; graphic system variable settings.
extra_plotPar={yrange:[yst, yen], xrange:[xut1, xut2], $
               ytitle:uytitle, ylog:yscl(panel(0),3), xtickformat:'noticks'}

; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)

; use papco routine to draw time axis. This checks for bottom plot
; and uses the user's  xtickformat if it is something other than
; 'noticks'. It's always drawn in black
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar  

; now do the actual data plot, data only, no axis   

;create new structure to use PLOTS 
extra_gapplotPar = create_struct({PSYM:0},{COLOR:0L},{THICK:1},{LINESTYLE:0})
extra_gapplotPar.PSYM = extra_plotPar.PSYM
extra_gapplotPar.COLOR = extra_plotPar.COLOR
extra_gapplotPar.THICK = extra_plotPar.THICK
extra_gapplotPar.LINESTYLE = extra_plotPar.LINESTYLE

plot, [0, 1], [0, 1], _extra=extra_plotPar, ystyle = 5, xstyle = 5, /nodata

; now do the actual data plot, data only, no axis
couleur_start = extra_plotPar.color
couleur_index = where(couleur_start EQ couleur_arr)
IF couleur_index(0) EQ -1 THEN couleur_index(0) = 0
couleur = extra_plotPar.color
ind = -1
FOR i=0,n_elements(channel)-1 DO BEGIN 
    ind = ind + 1
    yvect = yray(*,ind)
    index=where(yvect ne nodat, count)
    if count ne 0 then begin
        yvect=yvect(index)
        t=time(index)
    endif
    extra_gapplotPar.color = couleur
    goes_gap_oplot,tmin,t,yvect, _extra=extra_gapplotPar
    couleur = couleur_arr(couleur_index(0)+ind+1)
;       couleur = couleur+1
;       if (couleur eq 2) then couleur = 171
;       if (couleur eq 185) then couleur = 1
    endfor

; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y

; plot info right of the panel. If you want overplots to have their own labels
; you need to add position-control for this label using the keyword
; PLOTS_ATTHISPOSITION
extra_Par.color = couleur_start
couleur = extra_Par.color
if not keyword_set(overplot) then begin
    extra_Par.color = 1
    right_side_label,panel,utitle1,_extra=extra_Par
    utitle2 = '!C!C'
    if (channel(0) le 1 or channel(0) ge 7 ) then utitle2 = '!C!C!C'
    ind = -1
    for i=0,n_elements(channel)-1 do begin
        ind = ind+1
        extra_Par.color = couleur
        utitleS = utitle2 + '!C' + utitle(ind)
        right_side_label,panel,utitleS,_extra=extra_Par
        utitle2 = utitle2 + '!C'
;        couleur = couleur + 1
        couleur = couleur_arr(couleur_index(0)+ind+1)
;        if (couleur eq 2) then couleur = 171
;        if (couleur eq 185) then couleur = 1
    endfor
endif else begin
    for i=0, PLOTS_ATTHISPOSITION-1 do utitle='!C'+utitle1
    right_side_label,panel,utitle,_extra=extra_Par
endelse

; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  
IF KEYWORD_SET(OUTPUT) THEN BEGIN
    message,'Writing plot data out to file', /info
;
; transform seconds since start day into decimal year
;
    convert_t90_to_date, mjdt_start, year, day_oy, hour, minute, second,$
                          month, cmonth, day_om, /mjdt   
    Njours = 365.0d0
    if (year eq 4*(year/4)) then Njours=366.0d0
    time = double(year)+double(day_oy-1)/double(Njours)+double(time)/double(86400*Njours)
    time_label='Decimal year'
    NODATA = nodat
    if channel(0) lt 7 then begin
       description = ['1-8 Ang X-ray Flux', $
              '.5-4 Ang X-ray Flux', $
              'Northward Magnetic Field', $
              'Earthward Magnetic Field', $
              'Eastward Magnetic Field', $
              'Total Magnetic Feld', $
              '>= 2 MeV electrons']
       description = description(channel)
       y_label=uytitle
       rs_label=utitle1
       papco_write_data, time, yray, $
                      TIME_LABEL=time_label, $
                      DESCRIPTION=description, Y_LABEL=y_label, $
                      RS_LABEL=rs_label,NODATA=nodata
    endif else begin
      channels = fltarr(n_elements(channel),2)
      yminarr = [0.6,4.2,8.7,15.,39.,84.,110.]
      ymaxarr = [0.6,4.2,8.7,15.,39.,84.,500.]
      channels(*,0) = yminarr(channel-7)
      channels(*,1) = ymaxarr(channel-7)
      y_label=uytitle
      rs_label=utitle1
      papco_write_data, time, yray, $
                      TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, $
                      RS_LABEL=rs_label,NODATA=nodata
    endelse
ENDIF 

END 


pro ylim_io,time,yray,yst,yen

common time, xut1,xut2

index=where((time ge xut1) and (time le xut2) and (yray gt 0.),count)

if count gt 2 then begin
  yen=max(yray(index),min=yst)
  yst = 10.^floor(ALOG10(yst))
  yen = 10.^ceil(ALOG10(yen))
endif else begin
  yen=10.
  yst=1.
endelse

end


;******************************************************************************
;* PROCEDURE:     
;*      goes_gap_oplot, x, y, _EXTRA=e
;* 
;* DESCRIPTION:  
;*      This procedure plots data without connecting data point if they are
;*      further than 5 min. apart.
;*
;* INPUTS:       
;*      x,y     arrays to be plotted
;*              
;* OUTPUTS: 
;*      none
;* 
;* CALLING SEQUENCE:
;*      goes_gap_oplot, x, y, _EXTRA=e
;*
;* MODIFICATION HISTORY:       
;*     written July 1997, Haje Korth
;*     modified February 1998, Sebastien Bourdarie
;******************************************************************************
pro goes_gap_oplot,step,x,y,_EXTRA=e

; work out jumps in y. If a jump is larger than step, don't plot

   ndat=n_elements(x)
   if (ndat eq 0) then return
   for i=0l,ndat-2l do begin
      int=abs(x(i+1)-x(i))
      if int lt step then begin
        plots,x(i),y(i) 
        plots,x(i+1),y(i+1),/continue,noclip=0 ,_EXTRA=e
      endif
    endfor

end


