;****************************************************************************
;*  NAME:        r_GOES
;*
;*  DESCRIPTION: Routine to read GOES files. Adheres to
;*               PAPCO conventions: Error common block
;*                                  Uses MJDT time
;*                                  PAPCO read philosophy
;*
;*  INPUT:       none (uses common block time from PAPCO)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*                              running example module on it's own
;****************************************************************************
pro r_goes,sat, VERBOSE=VERBOSE

   common goes_data, input_header, input_data
   common mjdt, mjdt_start, mjdt_end
   common get_error, get_err_no, get_err_msg

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. Papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; empty panel with the error message get_err_msg is plotted.

   get_err_no=0
   get_err_msg='GOES read succesfully'

; check for env. variable pointing to data directory
   data_dir='GOES_DATA'
   c_path=getenv(data_dir)
   if strlen(c_path) eq 0 then begin
      get_err_no=1
      get_err_msg='Environmental variable '+data_dir+' not set'
      print,get_err_msg
      return
   endif
   
; check for existence of data directory
   res=findfile(c_path,count=cc)
   if cc eq 0 then begin
      get_err_no=2
      get_err_msg='Data directory '+c_path+' not found'
      print,get_err_msg
     return
   endif
   


; check VERBOSE keyword and set
   if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. papco standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
   
   time=mjdt_start  &  no_files_read=0  &  day_count=0.0
   
   sat_ID = strmid(sat,5,2)
   while mjdt_end.mjd ge time.mjd do begin
     
     ; assemble the data file from the time range requested
     convert_t90_to_date, time, year, day_oy, hour, minute, second,$
                          month, cmonth, day_om, /mjdt   
     datestr=string(year-1900,month,day_om,format='(i2.2,i2.2,i2.2)')
     fln = sat+'/EPS_MAG/'+string(year,format="(i4.4)")+'/data/z' $
           +sat_ID+'5'+string(year-1900,month,format="(i2.2,i2.2)")
;print,fln     
     ; read one day of data
     r_goes_onefile, c_path+fln, /VERBOSE
   
     ; check for "OPENR: Error opening file" in error string - and then get 
     ; the file remotely!
     ; comment this section out if you do not want this feature.
;     res=strpos(!err_string,'OPENR: Error opening file')
;     if res ne -1 then begin
;       c_url=papco_getenv('goes_FTP_ARCHIVE')
;       c_file=fln+'.idl'
;       c_download_dir=path
;       papco_get_remote_data, c_url, c_file, c_download_dir
;       ;check if file "arrived" and then read it
;       if get_err_no eq 0 then r_goes_onefile, path+fln, /VERBOSE
;     endif   
     
     ;if a file was succesfully read, concatonate!
     if get_err_no eq 0 then begin 
       if no_files_read eq 0 then begin   ;start the array buffer
         input_data_buf=temporary(input_data)
         input_header_buf = input_header
       endif else begin                   ;otherwise concatonate arrays
         input_data_buf=[temporary(input_data_buf),temporary(input_data)]
       endelse        
       input_data=0                     ;clear array for reuse
       no_files_read=no_files_read+1
     endif
     month = month+1
     day_om=1
     time=convert_date_to_t90(year=year, month=month, day=day_om ,$
                        hour=hour , minute = minute , second = second,  /MJDT)
   endwhile         
   
; check if any files were read   
   if (no_files_read ne 0) and (get_err_no ne 1) then begin
     get_err_no=0
     get_err_msg='% r_goes: '+string(no_files_read,format="(i2)")+ $
       ' files read succesfully'
     ;clear arrays and load buffer into output array
     input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
     input_header = input_header_buf

;converting time to PAPCO plot standard
     input_data(*).time=(input_data(*).mjdt_time.mjd-mjdt_start.mjd) $
                     *86400+input_data(*).mjdt_time.t
   endif else begin
     get_err_no=1
     get_err_msg='% r_goes: no files found for timerange'
     return
   endelse      
   
; restrict data to actual time range requested
   ut1=mjdt_start.t
   ut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
   index=where((input_data.Time ge ut1) and (input_data.Time le ut2),count)
   if count ne 0 then begin
      input_data=input_data(index)
   endif   
end

;******************************************************************************
;*  NAME:        r_goes_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_goes_onefile, fln, VERBOSE=VERBOSE
  
   common goes_data, input_header, input_data
   common get_error, get_err_no, get_err_msg
   
   get_err_no=0
   get_err_msg='goes onefile read succesfully'
   
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
   on_ioerror, read_original
   
   restore,fln+'.idl',/verbose
   dealloc_lun,/quiet    ;frees logical units used by save/restore
   print,'% r_goes_onemonth: idl saveset read succesfully'
   return

   read_original:
   ; if the .idl save file could not be found, then try to read the original
   r_goes_onemonth_original, fln
      
end   

;****************************************************************************
;*  NAME:        getmonth_onefile
;*
;*  DESCRIPTION: Routine to read GOES files. Adheres to
;*               PAPCO conventions: Error common block
;*                                  Uses MJDT time
;*                                  PAPCO read philosophy
;*
;*  INPUT:       none (uses common block time from PAPCO)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        February 1998
;*
;*  AUTOR:       S. BOURDARIE
;****************************************************************************
pro   r_goes_onemonth_original, fln

   common goes_data, input_header, input_data
   common get_error, get_err_no, get_err_msg

;  clear exisiting data array
   input_data=0

   print,'% r_goes_ascii: Reading ',fln+'.asc',' (slow)'

   time={MJDT, MJD:0l, T:0.0}
   dummystr=''
   dummytime=' 0 0 0'
   dummy_struct={GOE,  XL:0.0, XS:0.0, Hp:0.0, $
                       He:0.0, Hn:0.0, Ht:0.0, $
                       E1:0.0, P1:0.0, P2:0.0, $
                       P3:0.0, P4:0.0, P5:0.0, $
                       P6:0.0, P7:0.0, mjdt_time:time, $
                       time:0l}

   input_data=replicate(dummy_struct,9000)
   i=-1

   zipped = 0
   openr,f_Unit,fln+'.asc', /GET_LUN, ERROR=err

   IF err NE 0 THEN BEGIN
      openr,f_Unit,fln+'.asc.gz',/GET_LUN, ERROR=err
      IF err EQ 0 THEN BEGIN
         close,f_Unit
         zipped=1
         print,'% read_goes: File ',fln,' comressed - uncompressing'
         spawn,'cp '+fln+'.asc.gz'+' ~/temp.gz'
         spawn,'gunzip '+fln+'.asc.gz'
         spawn,'mv ~/temp.gz '+fln+'.asc.gz'  
         openr,f_Unit,fln+'.asc'
       ENDIF ELSE begin
         print, '% readgoes: '+fln+'.asc'+'  :  not available.'
         get_err_no=2
         get_err_msg=!ERR_STRING
         print, !ERR_STRING
         RETURN
       ENDELSE
   ENDIF 

   for i=1,23 do readf,f_Unit,dummystr  ; read the header of the file

   while not eof(f_Unit) do begin 
      i=i+1
      readf,f_Unit,dummystr
      timestr=strmid(dummystr,0,15)
      dummystr=strmid(dummystr,15,strlen(dummystr)-15)
      dummystr=dummystr+dummytime
      reads,dummystr,dummy_struct
      year     =fix(strmid(timestr,0,2))
      hour     =fix(strmid(timestr,7,2))
      minutes  =fix(strmid(timestr,9,2))
      doy      =fix(strmid(timestr,12,3))
      dummy_struct.mjdt_time=convert_date_to_t90(doy=doy, year=year,$
                                                 hour=hour,minutes=minutes, /MJDT)
      input_data(i)=dummy_struct
   endwhile

   close,f_Unit
   free_lun,f_Unit
   if (zipped eq 1) then spawn,'rm '+fln+'.asc'
   input_data=temporary(input_data(0:i))

; save read data with correct time. Rename data to *.idl
  save,filename=fln+'.idl',input_data,/verbose

END
