;******************************************************************************
;* PROCEDURE:     
;*      p_goes_eph_ticks__line, panel, PlotInfo, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      PlotInfo    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_lanlgeo_eph_ticks__line, panel, PlotInfo, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_goes_eph_ticks, panel, PlotInfo, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par

COMMON plot_composer, widgetData
COMMON mjdt, mjdt_start, mjdt_end               ;time limit in mjdt
COMMON yscale, yscl                             ;man/auto yscaling
COMMON shift_label, down                        ;for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  ;info on coords

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_lanlgeo_eph_ticks_. 
COMMON goes_eph_data, input_header, input_data
COMMON goes_eph_tick, time, yray, mag_model, tick_plotinfo
COMMON goes_eph
  
panelset, panel			;sets the panel position viewport

; restrict data to actual time range requested - this makes for faster zooming
; input data is in TAI. For plotting, always start at zero to get maximum
; resolution (plot works in float only, large values get rounded!) 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

; restrict data in time
index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)

if c GT 0 then data=input_data(index) ELSE GOTO, noplot
xut1=0  &  xut2=tai_2-tai_1  & time=data.tai-tai_1

control = *plotinfo.USR_PTR1
sat_id = sat_names(control.sc_id)

utitle='goes_eph_ticks_ '+sat_names(PlotInfo.typevector(0))

extra_plotPar={xrange:[xut1,xut2], yrange:[0,1], ystyle:5, xstyle:9, $
               xtickformat:'noticks'}
      
; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; use papco rotuine to draw time axis. This checks for bottom plot
; and uses the user's  xtickformat if it is something other than 'noticks'.
; "down" sets the axis labeling level. 
down=0 & p=[0,panel(1),panel(2)] & p = panel
papco_draw_time_axis, p, OVERPLOT=0, NTICKS=n, XTICKVALS=vals, MINOR=mn, $
                         _extra=extra_plotPar

; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y

; plot extra x-axis labels
extra_plotPar.xtickformat='goes_eph_ticks' & lbl=''
tick_plotinfo = plotinfo
tick_control = *tick_plotinfo.USR_PTR1
FOR i=0, down-1 DO lbl=lbl+'!C'
FOR i=0ll, n_elements(eph_names)-1 DO BEGIN
    IF (control.product AND 2ll^i) EQ 2ll^i THEN BEGIN
        tick_control.product = i
        tick_plotinfo.USR_PTR1 = ptr_new(tick_control)
        papco_draw_time_axis, p, OVERPLOT=0, /FORCE, _extra=extra_plotPar
        lbl = lbl + eph_names(i) + ' (' + yunits(i) + ')!C'
        down=down+1
    ENDIF
ENDFOR 

mag_names = papco_make_choice_names(control.MAG_COORDS_INFO, TITLE = title)
FOR i=0ll, n_elements(mag_names)-1 DO BEGIN
    tick_control.orig = 1
    IF (control.MAG_COORDS AND 2ll^i) EQ 2ll^i THEN BEGIN
        tick_control.MAG_COORDS = i
        tick_plotinfo.USR_PTR1 = ptr_new(tick_control)
        papco_draw_time_axis, p, OVERPLOT=0, /FORCE, _extra=extra_plotPar
        lbl = lbl + mag_names(i) + ' (' + mag_model + ')!C'
        down=down+1
    ENDIF
ENDFOR 

; x_side_label puts x-axis label to the right of the axis.
x_side_label, panel, lbl

return

;jump to here if no data to plot 
noplot: 
message,'No data to plot', /cont
plotted_x = !x  &  plotted_y = !y

end 

;-----------------------------------------------------------------------------
; ONE function for ALL lanl_geo eph ticks.
FUNCTION goes_eph_ticks, axis, index, t

COMMON mjdt, mjdt_start, mjdt_end
COMMON goes_eph_tick, time, yray, mag_model, tick_plotinfo
COMMON goes_eph
COMMON shift_label, down             ;common for x-axis label shifting
  
down_str=''  &  for i=0,down-1 do down_str=down_str+'!C'
  
;get data by calling p_lanl_geo_line to get data only.
IF index EQ 0 THEN p_goes_eph, panel, tick_plotinfo, OUTPUT = 2

;t starts at startsec of day from papco_draw_time_axis.
;time is from p_geo_eph_line, so is normalized to zero.
; Need to subtract. 
tt = t-mjdt_start.t

difft=abs(time-tt) & result=min(difft,minindex) & tick=yray(minindex)

IF tick EQ -999 THEN return, ''
  
case tick_plotinfo.typevector(1) of
    0: out_str=string(tick,format="(f5.1)")
    1: out_str=string(tick,format="(f6.1)")
    2: out_str=string(tick,format="(f4.1)")
    3: out_str=string(tick,format="(f4.1)")
    4: out_str=string(tick,format="(f4.1)")
    5: out_str=string(tick,format="(f5.1)")
    else: out_str=string(tick)
endcase
  
return, down_str+out_str
  
END 
