;******************************************************************************
;*  NAME:        r_goes_eph
;*
;*  DESCRIPTION: Routine to read goes_eph files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
PRO r_goes_eph, plotinfo, AVERAGE = AVERAGE, VERBOSE=VERBOSE

COMMON goes_eph_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg

get_err_no=0  &  get_err_msg='goes_eph read succesfully'

; check env. variable pointing to data directory
if not (papco_check_data_env('GOES_EPH_DATA', path=PATH)) then return
   
; check VERBOSE keyword and set
IF keyword_set(VERBOSE) THEN verbose=VERBOSE ELSE verbose=0     

; clear data arrays for reuse
input_header=0  &  input_data=0

; loop from start to end time concatonating data files into one array.   
time=mjdt_start  &  no_files_read=0
    
WHILE mjdt_end.mjd GE time.mjd DO BEGIN

    r_goes_eph_one, plotinfo, time, AVERAGE = AVERAGE, VERBOSE=VERBOSE

    ;if a file was succesfully read, concatonate!
    IF get_err_no EQ 0 THEN BEGIN 
        IF no_files_read eq 0 THEN BEGIN   ;start the array buffer
            input_data_buf = temporary(input_data)
        ENDIF ELSE BEGIN                   ;otherwise concatonate arrays
            input_data_buf = [temporary(input_data_buf),temporary(input_data)]
        ENDELSE         
        input_data = 0                     ;clear array for reuse
        no_files_read = no_files_read+1
    ENDIF 
    next_day:
    time.mjd = time.mjd+1                   ; next day
    ; check for any time left on last day - skip if none
    IF (mjdt_end.mjd EQ time.mjd) AND (mjdt_end.t EQ 0) THEN BREAK

ENDWHILE
  
; check if any files were read   
IF (no_files_read NE 0) THEN BEGIN 
    msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    get_err_no=0  &  get_err_msg=msg
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
ENDIF ELSE BEGIN 
    msg='no files found for timerange'
    get_err_no=1  &  get_err_msg=msg
ENDELSE     
message, msg, /cont

END 

;******************************************************************************
;*  NAME:        r_goes_eph_one,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       sat_id, date  - string of sate name, date yyyymmdd
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_goes_eph_one, plotinfo, time, AVERAGE = AVERAGE, VERBOSE = VERBOSE
  
COMMON goes_eph_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
COMMON goes_eph
   
get_err_no=0 & get_err_msg='goes_eph one read succesfully'
   
; check keywords and set
if keyword_set(VERBOSE) then VERBOSE=1 else VERBOSE=0

; check env. variable pointing to data directory
if not (papco_check_data_env('GOES_EPH_DATA', path=PATH)) then return

;assemble the data file from the date requested
utc = {MJD:long(time.mjd), TIME:long(time.t*1000)}
datestr = ANYTIM2CAL(utc, FORM = 8, /DATE)
  
control = *plotinfo.USR_PTR1
sat_id = sat_names(control.sc_id)

file_path = path+strmid(datestr, 0, 4)+papco_get_pathsep()
fln = file_path+datestr+'_'+sat_id+'_eph.idl'
eph_struct_flname = file_path+'eph_struct.idl'

;test if subdir exists, create if necessary
papco_create_dir, file_path

; clear data arrays for reuse
input_header=0  &  input_data=0

;read existing file, or make it
filethere=findfile(fln, count = c)

IF c EQ 0 THEN BEGIN
    get_err_no=2 & get_err_msg='File '+fln+' not found'
    message, get_err_msg, /cont
    ;then read original data, extract ephemeris
    r_goes_eph_one_original, plotinfo, time, AVERAGE = AVERAGE

    IF get_err_no EQ 0 THEN BEGIN 
        ;if no gps_eph_struct exists, write the original one
        result = findfile(eph_struct_flname, count = c)
        IF c EQ 0 THEN BEGIN
            eph_struct = input_data(0)
            save, file = eph_struct_flname, eph_struct
        ENDIF
    ENDIF ELSE BEGIN
        return
    ENDELSE     
ENDIF ELSE BEGIN
    restore, fln,  VERBOSE = VERBOSE
    message, 'idl saveset read succesfully', /cont
    print, '  ', fln
ENDELSE

;now make/add/check mag coords in input_data.
;need to "hand over" time in TAI,coord system info, actual eph coords. 

;GEO eph in in geographic lat/long
cor_type='SPH' & cor_sys='GED'
tai = input_data.tai
ndat =  n_elements(input_data)
pos = dblarr(3, ndat)
pos(0, *) = input_data.SC_POS_LL(2)           ;rad, km
pos(1, *) = 90.0d - input_data.SC_POS_LL(0)   ;lat -> colat
pos(2, *) = input_data.SC_POS_LL(1)           ;long

papco_add_mag_unilib, control, cor_type, cor_sys, $
                      eph_struct_flname, new_struct_name, $
                      tai, pos, input_data, error

IF error GT 0 THEN return ;unilib not loaded

input_header = {model_name:new_struct_name}

IF error EQ 2 THEN return ;no new structures added to input_data

;save the new eph data to file
message, 'Writing new', /cont
print, '  ', fln
save, file = fln, input_header, input_data, /COMPRESS

;save the new gps eph structure to file
message, 'Writing new structure contents to file', /cont
eph_struct = input_data(0)
save, file = eph_struct_flname, eph_struct
      
END 
  
;******************************************************************************
;*  NAME:        r_goes_eph_one_original, plotinfo, date, flname
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_goes_eph_one_original, plotinfo, time, VERBOSE=VERBOSE, $
                             AVERAGE = AVERAGE
    
COMMON mjdt, mjdt_start, mjdt_end
COMMON goes_eph_data, input_header, input_data
COMMON goes_eph
COMMON get_error, get_err_no, get_err_msg    
 
; check keywords and set
if keyword_set(VERBOSE) then VERBOSE=1 else VERBOSE=0

; check env. variable pointing to data directory
if not (papco_check_data_env('GOES_EPH_DATA', path=PATH)) then return

;read goes mag cdf file, extract ephemeris info. Do for one day only
orig_mjdt_start = mjdt_start & orig_mjdt_end = mjdt_end
mjdt_start = time & mjdt_end = time & mjdt_end.mjd = mjdt_start.mjd+1

;assemble the data file from the date requested
utc = {MJD:long(time.mjd), TIME:long(time.t*1000)}
datestr = ANYTIM2CAL(utc, FORM = 8, /DATE)
  
control = *plotinfo.USR_PTR1
sat_id = sat_names(control.sc_id)

file_path = path+strmid(datestr, 0, 4)+papco_get_pathsep()
fln = file_path+datestr+'_'+sat_id+'_eph.idl'

new_plotinfo = papco_getplotinfostruct()
new_control = control
new_plotinfo.USR_PTR1 = ptr_new(new_control)

get_goes_eph, control

IF get_err_no NE 0 THEN GOTO, done

;save the new eph data to file
message, 'Writing new', /cont
print, '  ', fln
save, file = fln, input_header, input_data, /COMPRESS

done:

;restore original time range
mjdt_start = orig_mjdt_start & mjdt_end = orig_mjdt_end

return

;now eph data is read. Average as required
if keyword_set(AVERAGE) then av=AVERAGE else av=300 ;deault 5 min
message, 'averaging to '+varprt(av)+' sec resolution', /cont

END 
